/****************************************************************************
 File: GoMath.h

 (c) Copyright 1992, GO Corporation,  All Rights Reserved

 $Revision:   1.5  $
   $Author:   eyee  $
	 $Date:   21 Feb 1992 18:12:26  $

 This file contains the API definition for fixed point arithmetic.
 The functions described in this file are contained in PENPOINT.LIB.

 The API in this file is all function oriented.
****************************************************************************/

#ifndef GOMATH_INCLUDED
#define GOMATH_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *			   Math Operation Error Codes								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
#define stsUnderflow		MakeStatus(clsGOMath, 1)
#define stsOverflow			MakeStatus(clsGOMath, 2)
#define stsMathInvOp		MakeStatus(clsGOMath, 3)
#define stsMathInvStrOp		MakeStatus(clsGOMath, 4)
#define stsMathEqual		MakeStatus(clsGOMath, 5)
#define stsMathFirstHigher	MakeStatus(clsGOMath, 6)
#define stsMathFirstLower	MakeStatus(clsGOMath, 7)
#define stsZeroDivide		MakeStatus(clsGOMath, 8)
//	The following two values are used by the runtime.lib as ERRNO values
#define stsMathDomain		MakeStatus(clsGOMath, 9) // Argument too large
#define stsMathRange		MakeStatus(clsGOMath, 10) // Result too large

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *			   Math Constants											   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
#define GoFx0 ((FIXED) 0x00000000)		// 0.0
#define GoFx1 ((FIXED) 0x00010000)		// 1.0
#define GoFxMinus1 ((FIXED) 0xffff0000) // -1.0

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *			   Fixed-point Function Prototypes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 FxCmp returns S16
	Compares two FIXED.

 Return Value
	-1:	  if a < b.
	0:	  if a = b.
	1:	  if a > b.
*/
S16 PASCAL FxCmp(FIXED a, FIXED b);

/****************************************************************************
 FxAdd returns STATUS
	Adds two FIXED numbers, producing a FIXED.

 Return Value
	stsOverflow:	The integer part of the result overflows a 16-bit signed.
*/
STATUS PASCAL FxAdd(FIXED a, FIXED b, P_FIXED pC);

/****************************************************************************
 FxAddSC returns FIXED
	Macro form of FxAdd with no overflow detection.
*/
#define FxAddSC(_f1,_f2) ((FIXED)((_f1) + (_f2)))

/****************************************************************************
 FxSub returns STATUS
	Subtracts two FIXED numbers, producing a FIXED.

 Return Value
	stsOverflow:	The integer part of the result overflows a 16-bit signed.
*/
STATUS PASCAL FxSub(FIXED a, FIXED b, P_FIXED pC);

/****************************************************************************
 FxSubSC returns FIXED
	Macro form of FxSub with no overflow detection.
*/
#define FxSubSC(_f1,_f2) ((FIXED)((_f1) - (_f2)))

/****************************************************************************
 FxNegate returns FIXED
	Negates a FIXED.
*/
#define FxNegate(_f) ((FIXED)(-(_f)))

/****************************************************************************
 FxMul returns STATUS
	Multiplies two FIXED numbers, producing a FIXED.

 Return Value
	stsOverflow:	The integer part of the result overflows a 16-bit signed.
*/
STATUS PASCAL FxMul(FIXED a, FIXED b, P_FIXED pC);

/****************************************************************************
 FxMulSC returns FIXED
	Multiplies two FIXED numbers returning the product.

 No overflow detection is performed.
*/
FIXED PASCAL FxMulSC(FIXED a, FIXED b);

/****************************************************************************
 FxMulInt returns STATUS
	Multiplies a FIXED number by an S32, producing a FIXED.

 Return Value
	stsOverflow:	The integer part of the result overflows a 16-bit signed.
*/
STATUS PASCAL FxMulInt(FIXED a, S32 b, P_FIXED pC);

/****************************************************************************
 FxMulIntSC returns FIXED
	Multiplies a FIXED number by an S32, returning the FIXED product.

 No	overflow detection is performed. 
*/
#define FxMulIntSC(_a,_b) ((FIXED)(_a*_b))

/****************************************************************************
 FxMulIntToInt returns STATUS
	Multiplies a FIXED number by an S32, producing a rounded S32 product.

 Return Value
	stsOverflow:	The integer part of the result overflows a 32-bit signed.
*/
STATUS PASCAL FxMulIntToInt(FIXED a, S32 b, P_S32 pC);

/****************************************************************************
 FxMulIntToIntSC returns S32
	Multiplies a FIXED number by an S32, returning a rounded S32 product.

 No	overflow detection is performed. 
*/
S32 PASCAL FxMulIntToIntSC(FIXED a, S32 b);

/****************************************************************************
 FxDiv returns STATUS
	Divides two FIXED numbers, producing a FIXED quotient.

 Return Value
	stsOverflow:	The integer part of the result overflows a 16-bit signed.
	stsZeroDivide:	The input divisor is zero.
*/
STATUS PASCAL FxDiv(FIXED top, FIXED bottom, P_FIXED pC);

/****************************************************************************
 FxDivSC returns FIXED
	Divides two FIXED numbers, returning a FIXED quotient.

 No overflow or zero-divide detection is performed. 
*/
FIXED PASCAL FxDivSC(FIXED top, FIXED bottom);

/****************************************************************************
 FxDivInts returns STATUS
	Divides two 32-bit signed integers, producing a FIXED quotient.

 Return Value
	stsOverflow:	The integer part of the result overflows a 16-bit signed.
	stsZeroDivide:	The input divisor is zero.
*/
STATUS PASCAL FxDivInts(S32 top, S32 bottom, P_FIXED pC);

/****************************************************************************
 FxDivIntsSC returns FIXED
	Divides two FIXED numbers, returning a FIXED quotient.

 No overflow or zero-divide detection is performed. 
*/
FIXED PASCAL FxDivIntsSC(S32 top, S32 bottom);

/****************************************************************************
 FxDivIntToInt returns STATUS
	Divides an S32 by a FIXED, producing a rounded S32 quotient.

 Return Value
	stsOverflow:	The integer part of the result overflows a 16-bit signed.
	stsZeroDivide:	The input divisor is zero.
*/
STATUS PASCAL FxDivIntToInt(S32 top, FIXED bottom, P_S32 pC);

/****************************************************************************
 FxDivIntToIntSC returns S32
	Divides an S32 by a FIXED, producing a rounded S32 quotient.

 No overflow or zero-divide detection is performed. 
*/
S32 PASCAL FxDivIntToIntSC(S32 top, FIXED bottom);

/****************************************************************************
 FxSin returns FIXED
	Returns the sine of an integer angle in degrees.
*/
FIXED PASCAL FxSin(S16 angle);

/****************************************************************************
 FxCos returns FIXED
	Returns the cosine of an integer angle in degrees.
*/
FIXED PASCAL FxCos(S16 angle);

/****************************************************************************
 FxTan returns FIXED
	Returns the tangent of an integer angle in degrees.
*/
FIXED PASCAL FxTan(S16 angle);

/****************************************************************************
 FxSinFx returns FIXED
	Returns the sine of a FIXED angle in degrees.
*/
FIXED PASCAL FxSinFx(FIXED angle);

/****************************************************************************
 FxCosFx returns FIXED
	Returns the cosine of a FIXED angle in degrees.
*/
FIXED PASCAL FxCosFx(FIXED angle);

/****************************************************************************
 FxTanFx returns FIXED
	Returns the tangent of a FIXED angle in degrees.
*/
FIXED PASCAL FxTanFx(FIXED angle);

/****************************************************************************
 FxArcTanInt returns FIXED
	Returns an arctangent value as a FIXED angle.

 Computes a FIXED angle whose tangent is the value given by the quotient of
 the two signed 32-bit integers, top / bottom. The value returned ranges from
 0 to 359 degrees.
*/
FIXED PASCAL FxArcTanInt(S32 top, S32 bottom);

/****************************************************************************
 FxArcTanFx returns FIXED
	Returns an arctangent value as a FIXED angle.

 Computes a FIXED angle whose tangent is the value given by the quotient of
 the two signed 32-bit numbers, top / bottom. The value returned ranges from
 0 to 359 degrees.
*/
FIXED PASCAL FxArcTanFx(S32 top, S32 bottom);

/****************************************************************************
 FxAbs returns FIXED
	Takes the absolute value of a FIXED.
*/
#define FxAbs(_f) (((_f)<0)?FxNegate(_f):(_f))

/****************************************************************************
 FxRoundToInt returns S32
	Rounds a FIXED number to a 32-bit signed integer.
*/
S32 PASCAL FxRoundToInt(FIXED fx);

/****************************************************************************
 FxRoundToIntSC returns S16
	Rounds a FIXED number to a 16-bit signed integer.

 No overflow detection is performed.
*/
#define FxRoundToIntSC(_f) (S16)(((_f)+0x8000)>>16)

/****************************************************************************
 FxChop returns S16
	Returns the 16-bit signed integer part of a FIXED.
*/
#define FxChop(_f) (S16)((_f)>>16)
#define FxChopSC(_f) (S16)((_f)>>16)

/****************************************************************************
 FxFraction returns U16
	Returns the 16-bit fractional part of the absolute value a FIXED.
*/
#define FxFraction(_f) (U16)(FxAbs(_f))

/****************************************************************************
 FxIntToFx returns FIXED
	Converts a 16-bit signed integer into a FIXED.
*/
#define FxIntToFx(_i) ((FIXED)(((S32)(_i))<<16))

/****************************************************************************
 FxMakeFixed returns FIXED
	Makes a FIXED with an S16 (integer) and a U116(fraction).

	FIXED PASCAL FxMakeFixed(S16 whole, U16 frac); (now in go.h)
*/

/****************************************************************************
 FxBinToStr returns nothing
	Converts a FIXED format value into an ascii string in decimal.

 The string will have the format:
	{-}xxxxx.xxxxx	 or	  {-}xx,xxx.xxxxx.
 The number of digits to the left of the decimal point is the minimum
 number required, and the number of digits to the right of the decimal point
 is specified in fracDigits.  The last digit is rounded accurately. If the
 string will not fit within maxLen bytes, then the string "*******"
 (maxLen-1 *'s) will be returned; maxLen = 9+fracDigits is sufficient,
 although any higher number is also acceptable. If showCommas is true, then
 commas will separate the thousands.
*/
void PASCAL FxBinToStr(
	FIXED	a,
	P_CHAR	pStr,
	U8		fracDigits,
	U8		maxLen,
	BOOLEAN	showCommas
);

/****************************************************************************
 FxStrToBin returns STATUS
	Converts a null-terminated ascii string to a FIXED.

 The fractional portion will be rounded to fit within 16 bits.

 Return Value
	stsOverflow:	 The integer part of the result overflows a 16-bit signed.
	stsMathInvStrOp: A character in the string does not represent a valid
					 number. *pC is set to zero.
*/
STATUS PASCAL FxStrToBin(
	P_CHAR	pStr,
	P_FIXED	pC
);

#endif // GOMATH_INCLUDED
