/****************************************************************************
 File: frame.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.71  $
   $Author:   cmeyer  $
     $Date:   16 Mar 1992 17:34:30  $

 This file contains the API definition for clsFrame.

 clsFrame inherits from clsShadow.
 Frames support a single client window, surrounded by a host of optional
 "decorations" -- title bar, menu bar, close box, tab bar, command bar, etc.
****************************************************************************/
#ifndef FRAME_INCLUDED
#define FRAME_INCLUDED

													#ifndef SHADOW_INCLUDED
#include <shadow.h>
													#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT	FRAME;

typedef struct FRAME_STYLE {
	U16	titleBar		: 1,	// show/don't show decoration
		menuBar			: 1,	//		"
		closeBox		: 1,	//		"
		cmdBar			: 1,	// 		"
		tabBar			: 1,	// 		"
		pageNum			: 1,	// 		"
		zoomable		: 1,	// true => zoom is allowed
		clipBoard		: 1,	// true => look like a clip board
		maskTitleLine	: 1,	// mask out the closeBox, titleBar, pageNum
		maskMenuLine	: 1,	// mask out the menuBar
		maskAll			: 1,	// mask out title, menu and cmd lines
		maskCmdLine		: 1,	// mask out the cmdBar
		useAltVisuals	: 1,	// use alternate border visuals
		spare1			: 3;	// unused (reserved)

	U16	spare2			: 16;	// unused (reserved)
} FRAME_STYLE, *P_FRAME_STYLE;

/*
 Default FRAME_STYLE:
//{
 	titleBar			= true
 	menuBar		 		= false
 	closeBox			= true
 	cmdBar				= false
 	tabBar				= false
 	pageNum				= false
 	zoomable			= true
 	clipBoard			= false
 	maskTitleLine		= false
 	maskMenuLine		= false
 	maskAll				= false
 	useAltVisuals		= false
//}
*/

/* for msgFrameZoomOK, msgFrameZoomed */
typedef struct FRAME_ZOOM {
	FRAME		frame;		// in: Frame to zoom.
	BOOLEAN		up;			// in: True=zoom up, False=zoom down
	WIN			toWin;		// out: Window to zoom to
	U32			spare;		// unused (reserved)
} FRAME_ZOOM, *P_FRAME_ZOOM;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew		takes P_FRAME_NEW, returns STATUS
	category: class message
	Creates a frame window.  Passes back the resulting FRAME_METRICS in
	pArgs->frame.

 clsFrame creates an instance of clsFrameBorder as the frame's border window
 to be the parent of all of the frame decorations (except the tabBar, which
 is a direct child of the frame).  The border window is inserted as a
 child of the frame.

 If pArgs->frame.style.clipBoard is true, the frame is made opaque and
 many of the border.style values are changed to produce a clipboard style
 look.

 For each of the decoration visibility style bits (e.g. style.titleBar),
 the following is done:

	If the style value is true, and the corresponding decoration window (e.g.
	metrics.titleBar) is not objNull, the window provided is inserted as
	a child of the frame border window.
 
	If the style value is true and no window is provided (e.g. titleBar
	is objNull), a default instance of the decoration is created (e.g. msgNew
	to clsTitleBar) and inserted as a child of the frame border window.
 
	If the style value is false, the provided decoration window is remembered
	for use when the style value is set to true.

 If style.menuBar is true, the border style of the menuBar is altered to
 have a bottom edge with thickness bsThicknessDouble and borderInk
 bsInkGray66.

 If style.titleBar is true, the border style of the titleBar is altered to
 have a bottom edge with thickness bsThicknessDouble (if style.menuBar is
 false) or bsThicknessSingle (if style.menuBar is true) and borderInk
 bsInkGray66.

 If style.closeBox is true, the border style of the closeBox is altered to
 match that of the titleBar.

 If style.cmdBar is true and style.clipBoard is false, the border style of
 the cmdBar is altered to have a top edge with thickness bsThicknessDouble
 and borderInk bsInkGray33.

 If style.maskTitleLine is true, style.closeBox, style.titleBar and
 style.pageNum are all treated as though they are false.

 If style.maskMenuLine is true, style.menuBar is treated as though it is false.

 If style.maskCmdLine is true, style.cmdBar is treated as though it is false.

 If style.maskAll is true, style.maskTitleLine, style.maskMenuLine, and
 style.maskCmdLine are all is treated as though they are true.
*/

typedef struct FRAME_NEW_ONLY {
	FRAME_STYLE		style;
	WIN				clientWin;
	WIN				titleBar;
	WIN				menuBar;
	WIN				closeBox;
	WIN				cmdBar;
	P_CHAR			pTitle;			// in only for msgNew
	OBJECT			client;
	WIN				tabBar;
	WIN				pageNum;		// page number
	U32				spare1;			// unused (reserved)
	U32				spare2;			// unused (reserved)
} FRAME_NEW_ONLY, *P_FRAME_NEW_ONLY,
  FRAME_METRICS, *P_FRAME_METRICS;

#define	frameNewFields	\
	shadowNewFields		\
	FRAME_NEW_ONLY		frame;

typedef struct FRAME_NEW {
	frameNewFields
} FRAME_NEW, *P_FRAME_NEW;

/****************************************************************************
 msgNewDefaults				takes P_FRAME_NEW, returns STATUS
	category: class message
	Initializes the FRAME_NEW structure to default values.

 Zeroes out pArgs->frame and sets
//{
	pArgs->win.flags.style &= ~wsParentClip;
	pArgs->win.flags.style |= wsClipChildren | wsClipSiblings;

	pArgs->embeddedWin.style.selection = ewSelect;

	pArgs->frame.style.titleBar	= true;
	pArgs->frame.style.closeBox	= true;
	pArgs->frame.style.zoomable	= true;
//}
*/

/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Causes an object to file itself in an object file.

 If the client of the frame is OSThisAPP(), this is remembered
 and reinstated in msgRestore.  In any case, the client is not saved.

 Each of the frame decorations, including the clientWin, with
 WIN_METRICS.flags.style.wsSendFile on is filed, even if the
 corresponding visibility style bit (e.g. style.titleBar) is false.
*/

/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	Creates and restores an object from an object file.
	
 clsFrame restores the instance from the file.  If the client
 of the frame was OSThisApp() when filed, the client is set to
 OSThisApp(), otherwise objNull.

 Each of the filed decoration windows and the clientWin are restored.
 If the frame was zoomed when filed, the frame is unzoomed as in
 msgFrameZoom(false).

 For each of the following, if the corresponding child windows were not
 filed (i.e. wsSendFile was not on), and the visibility style is on, default
 instances will not be created and the visibility style will be set to false:
 menuBar, cmdBar, and tabBar.  For example, if the frame was filed with 
 style.menuBar true and the menuBar did not have wsSendFile on, the restored
 frame will have style.menuBar false, and the menuBar in FRAME_METRICS set to
 objNull.
*/

/****************************************************************************
 msgFree	takes OBJ_KEY, returns STATUS
	Sent as the last of three msgs to destroy an object.

 All children of the frame border window are destroyed.  Decoration
 windows with visibility style bits off are also destroyed.
*/

/****************************************************************************
 msgFrameGetMetrics		takes P_FRAME_METRICS, returns STATUS
	Passes back the metrics.
*/
#define msgFrameGetMetrics		MakeMsg(clsFrame, 1)

/****************************************************************************
 msgFrameSetMetrics		takes P_FRAME_METRICS, returns STATUS
	Sets the metrics.

 clsFrame replaces existing decoration windows with those provided.
 For example, if pArgs->titleBar specifies a new titleBar, the existing
 titleBar is extracted from the window tree and the new titleBar
 inserted as a child of the frame border window.

 Note that the old decoration windows are not destroyed and are no longer
 referenced by the frame (the client is free to destroy them at this point).

 Frame style values are changed as in msgFrameSetStyle.
*/
#define msgFrameSetMetrics		MakeMsg(clsFrame, 2)

/****************************************************************************
 msgFrameGetStyle		takes P_FRAME_STYLE, returns STATUS
	Passes back the current style values.

*/
#define msgFrameGetStyle		MakeMsg(clsFrame, 22)

/****************************************************************************
 msgFrameSetStyle		takes P_FRAME_STYLE, returns STATUS
	Sets the style.

 The new decoration visibility style bits (e.g. style.titleBar) are
 treated as in msgNew.  Setting a visibility bit to false results in
 extracting the corresponding decoration window (e.g. metrics.titleBar)
 from the frame border window.  Note that the extracted decoration window
 is not destroyed; but remembered for later use when the visibility bit is
 set to true.

 If style.useAltVisuals is changed from false to true, the alternate frame
 border visuals are applied to the frame's border style.

 If style.useAltVisuals is changed from true to false, the normal frame
 border visuals are applied to the frame's border style.

 Note that changing style.clipBoard is not implemented.
*/
#define msgFrameSetStyle		MakeMsg(clsFrame, 23)

/****************************************************************************
 msgFrameGetClientWin		takes P_WIN, returns STATUS
	Passes back metrics.clientWin.
*/
#define msgFrameGetClientWin	MakeMsg(clsFrame, 24)

/****************************************************************************
 msgFrameSetClientWin		takes WIN, returns STATUS
	Sets metrics.clientWin.

 The old clientWin, if any, is not destroyed and is no longer referenced
 by the frame.
*/
#define msgFrameSetClientWin	MakeMsg(clsFrame, 25)

/****************************************************************************
 msgFrameGetMenuBar		takes P_WIN, returns STATUS
	Passes back metrics.menuBar.
*/
#define msgFrameGetMenuBar	MakeMsg(clsFrame, 26)

/****************************************************************************
 msgFrameSetMenuBar		takes WIN, returns STATUS
	Sets metrics.menuBar; also sets style.menuBar to true if pArgs is not
	objNull, else false.

 The menuBar is changed as in msgFrameSetMetrics.
*/
#define msgFrameSetMenuBar	MakeMsg(clsFrame, 27)

/****************************************************************************
 msgFrameDestroyMenuBar		takes VOID, returns STATUS
	Sets style.menuBar to false and destroys the existing menu bar, if any.
*/
#define msgFrameDestroyMenuBar	MakeMsg(clsFrame, 28)

/****************************************************************************
 msgFrameSetTitle		takes P_CHAR, returns STATUS
	Sets the string in the metrics.titleBar.

 This results in msgLabelSetString to metrics.titleBar.
*/
#define msgFrameSetTitle		MakeMsg(clsFrame, 3)

/****************************************************************************
 msgFrameGetClient		takes P_OBJECT, returns STATUS
	Passes back metrics.client.

*/
#define msgFrameGetClient  		MakeMsg(clsFrame, 4)

/****************************************************************************
 msgFrameSetClient		takes OBJECT, returns STATUS
	Sets metrics.client.

*/
#define msgFrameSetClient  		MakeMsg(clsFrame, 5)

/****************************************************************************
 msgFrameGetAltVisuals		takes P_BORDER_STYLE, returns STATUS
	Passes back the alternate border visuals.

*/
#define msgFrameGetAltVisuals  		MakeMsg(clsFrame, 29)

/****************************************************************************
 msgFrameSetAltVisuals		takes P_BORDER_STYLE, returns STATUS
	Sets the alternate border visuals.

 If style.useAltVisuals is true, the new alternate visuals are applied to
 the frame's border style.
*/
#define msgFrameSetAltVisuals  		MakeMsg(clsFrame, 30)

/****************************************************************************
 msgFrameGetNormalVisuals		takes P_BORDER_STYLE, returns STATUS
	Passes back the normal border visuals.

 This is equivalent to msgBorderGetStyle if style.useAltVisuals is false.
*/
#define msgFrameGetNormalVisuals  		MakeMsg(clsFrame, 31)

/****************************************************************************
 msgFrameSetNormalVisuals		takes P_BORDER_STYLE, returns STATUS
	Sets the normal border visuals.

 If style.useAltVisuals is false, the new normal visuals are applied to
 the frame's border style.
*/
#define msgFrameSetNormalVisuals  		MakeMsg(clsFrame, 32)

/****************************************************************************
 msgFrameShowSelected		takes BOOLEAN, returns STATUS
	Makes the frame look selected or not.
*/
#define msgFrameShowSelected	MakeMsg(clsFrame, 17)

/****************************************************************************
 msgFrameMoveEnable		takes BOOLEAN, returns STATUS
	Enables or disables UI for moving.

 clsFrame alters the border.style.drag of the metrics.titleBar to be
 bsDragHoldDown if pArgs is true, bsDragNone otherwise.
*/
#define msgFrameMoveEnable		MakeMsg(clsFrame, 19)

/****************************************************************************
 msgFrameResizeEnable		takes BOOLEAN, returns STATUS
	Enables or disables UI for resizing.

 clsFrame alters the border.style.resize of self to be bsResizeCorner if
 pArgs is true, bsResizeNone otherwise.
*/
#define msgFrameResizeEnable	MakeMsg(clsFrame, 20)

/****************************************************************************
 msgFrameIsZoomed		takes P_BOOLEAN, returns STATUS
	Passes back true if the frame is currently zoomed.
*/
#define msgFrameIsZoomed		MakeMsg(clsFrame, 21)

/****************************************************************************
 msgFrameDelete		takes nothing, returns STATUS
	Asks the frame's client to delete the frame.

	clsFrame forwards this message to the client with self as the pArgs.
*/
#define msgFrameDelete			MakeMsg(clsFrame, 7)

/****************************************************************************
 msgFrameClose		takes nothing, returns STATUS
	Asks the frame's client to close the frame.

	clsFrame forwards this message to the client with self as the pArgs.
*/
#define msgFrameClose 			MakeMsg(clsFrame, 8)

/****************************************************************************
 msgFrameFloat		takes VOID, returns STATUS
	Asks the frame's client to float the frame.

	clsFrame forwards this message to the client with self as the pArgs.
*/
#define msgFrameFloat 			MakeMsg(clsFrame, 9)

/****************************************************************************
 msgFrameZoom		takes BOOLEAN, returns STATUS
	Zooms the frame up or down.

 If style.zoomable is false, nothing is done and stsOK is returned.
 
 Otherwise, msgFrameZoomOK is sent to the client with the following
 FRAME_ZOOM parameters:
//{
		frame	= self;
		up		= pArgs;
		toWin	= objNull;
//}

 If the client returns stsRequestDenied or does not set the FRAME_ZOOM.toWin,
 the client's status is returned.

 If the frame is already zoomed as pArgs requests, nothing is done and
 stsOK is returned.

 If pArgs is true and style.clipBoard is false, the frame is zoomed up as
 follows:
	-: The frame is made opaque by turning off wsTransparent in
	   WIN_METRICS.flags.style and turning off inputTransparent in
	   WIN_METRICS.flags.input.

	-: The border edges, shadow, margin and resize handles on the frame
	   are all turned off.

	-: The current frame window bounds and parent are remembered for
	   restoration in unzoom.

	-: The frame is extracted from its current parent and inserted as a child
	   of the FRAME_ZOOM.toWin with  a window bounds computed to zoom the
	   inner rect of the frame into the FRAME_ZOOM.toWin.  The inner rect is
	   computed using msgBorderGetOuterOffsets on the frame.

 If pArgs is false and style.clipBoard is false, the frame is zoomed down as
 follows:
	-: The frame is made transparent by turning on wsTransparent in
	   WIN_METRICS.flags.style and turning on inputTransparent in
	   WIN_METRICS.flags.input.

	-: The border edges, shadow, margin and resize handles on the frame
	   are all restored to their values before the zoom.

	-: The frame is extracted from its current parent and inserted in its
	   original parent with its original window bounds.
	  
 After the frame is zoomed/unzoomed it is layed out via msgWinLayout to self.

 clsFrame then sends the following notifications of the zoom/unzoom:
	-: self-send msgFrameZoomed with the FRAME_ZOOM as pArgs.
	-: msgFrameZoomed to its client with the FRAME_ZOOM as pArgs.
	-: self-sends msgNotifyObservers with the following OBJ_NOTIFY_OBSERVERS
	   parameters:
//{
			msg		= msgFrameZoomed;
			pArgs	= address of FRAME_ZOOM used to zoom/unzoom;
			lenSend	= sizeof(FRAME_ZOOM);
//}
*/
#define msgFrameZoom  			MakeMsg(clsFrame, 6)

/****************************************************************************
 msgFrameSelect		takes VOID, returns STATUS
	Selects the frame.

	msgFrameSelectOK(self) is sent to the client.
*/
#define msgFrameSelect 			MakeMsg(clsFrame, 18)

/****************************************************************************
 msgFrameZoomOK		takes P_FRAME_ZOOM, returns STATUS
	category: client notification
	Sent to the client when msgFrameZoom is received.
*/
#define msgFrameZoomOK	   		MakeMsg(clsFrame, 11)

/****************************************************************************
 msgFrameSelectOK		takes FRAME, returns STATUS
	category: client notification
	Sent to the client when msgFrameSelect is received.

	The client should alter the frame to look selected.
*/
#define msgFrameSelectOK		MakeMsg(clsFrame, 16)

/****************************************************************************
 msgFrameZoomed		takes P_FRAME_ZOOM, returns STATUS
	category: client & observer notification
	Sent to client and observers after frame is zoomed.
*/
#define msgFrameZoomed	   		MakeMsg(clsFrame, 12)

/****************************************************************************
 msgFrameClosed		takes WIN, returns STATUS
	category: client & observer notification
	Sent to client and observers after frame is closed.  pArgs is the frame.

 Note: not implemented.
*/
#define msgFrameClosed	   		MakeMsg(clsFrame, 13)

/****************************************************************************
 msgFrameFloated		takes VOID, returns STATUS
	category: client & observer notification
	Sent to client and observers after frame is floated.

 Note: not implemented.
*/
#define msgFrameFloated	   		MakeMsg(clsFrame, 14)

/****************************************************************************
 msgFrameTopped		takes VOID, returns STATUS
	category: client & observer notification
	Sent to client and observers after frame is brought to top.

 Note: not implemented.
*/
#define msgFrameTopped	   		MakeMsg(clsFrame, 15)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from Other Classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
 msgGWinForwardedGesture:	takes P_GWIN_GESTURE, returns STATUS
	Called to process the gesture.

 clsFrame maps certain gestures forwarded from the frame's titleBar
 into self-sent messages.  Other gestures are forwarded to the frame's client.

 If the pArgs->uid is not metrics.titleBar or a direct child of
 metrics.titleBar, msgGWinForwardedGesture(pArgs) will be sent to the frame's
 client.  clsFrame will return the client's return status from this message.

 The value of pArgs->msg is processed as follows:

	-: If xgsFlickUp/Down and the system preference with tag tagPrDocZooming
	   is prDocZoomingOn, msgFrameZoom(true/false) is self-sent.

	-: If xgsCross, msgFrameDelete(pNull) is self-sent.

	-: If xgsPlus, msgFrameSelect(pNull) is self-sent.

	-: If xgs2Tap, msgFrameFloat(pNull) is self-sent.

	-: If xgs3Tap, the frame's WIN_METRICS.flags.style.wsMaskWrapWidth/Height
	   flags are cleared and msgWinLayout(WIN_METRICS.options=wsLayoutDefault)
	   is self-sent.  This results in a re-layout to the frame's desired
	   size.

	-: If xgsTrplFlickUp and the DEBUG version of tk.dll is installed,
	   msgWinDumpTree is self-sent with pArgs of self or theRootWindow if
	   the '!' debug flag has value 1.  Note that msgWinDumpTree requires
	   the debug version of win.dll to be installed.  This is usefull for
	   debugging window layout problems.

	-: All other gestures result in msgGWinForwardedGesture(pArgs) to
	   the frame's client.
*/

/****************************************************************************
 msgTrackProvideMetrics			takes P_TRACK_METRICS, returns STATUS
	category: third-party notification
	Sent to a tracker client before tracker is created.

 If pArgs->minWH and pArgs->maxWH allow the width to change,
 pArgs->minWH.w is set to a small value to prevent the frame from being
 resized to zero.

 If pArgs->minWH and pArgs->maxWH allow the height to change,
 pArgs->minWH.h is set to prevent the frame from being resized smaller
 than the sum of the metrics.titleBar and metrics.menuBar heights.

 The value of pArgs->style.draw is altered to present the proper visual
 given the frame's style.tabBar and style.cmdBar.

 msgTrackProvideMetrics(pArgs) is sent to the frame's client.
*/

/****************************************************************************
 msgWinSetFlags takes P_WIN_METRICS, returns STATUS
    Sets the window flags.

 clsFrame alters the metric.clientWin's window flags to match the
 wsShrinkWrapWidth/Height flags of the frame.
*/

/****************************************************************************
 msgCstmLayoutGetChildSpec		takes P_CSTM_LAYOUT_CHILD_SPEC, returns STATUS
	category: self-sent
	Passes back the current spec for the specified child.

 clsFrame responds by providing the custom layout constraints for
 metrics.tabBar, metrics.cmdBar, and the frame's border window.

 Note that the decoration windows and the metrics.clientWin are actually
 children of the frame's border window, which is an instance of clsFrameBorder.
 clsFrameBorder responds to msgCstLayoutGetChildSpec by providing the
 custom layout constraints for its children (e.g. titleBar at the top,
 menuBar below titleBar, etc.).
*/

/****************************************************************************
 msgWinSend 	takes WIN_SEND, returns STATUS
    Sends a message up a window ancestry chain.

 If pArgs->msg is msgBorderProvideDeltaWin and the frame is zoomed,
 clsFrame returns stsOK.  This prevents a zoomed frame from being resized.
*/

#endif	// FRAME_INCLUDED
