/****************************************************************************

 File: export.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.27  $
 $Author:   DYEE  $
 $Date:   03 Mar 1992 15:05:00  $

 This file contains the API definition for clsExport.

 clsExport inherits from clsObject.
 clsExport is the abstract class defining the API for exporting data 
 to external disks.

 The clsExport API provides a common mechanism for documents to translate
 themselves into foreign file formats and place the file on external disks.

****************************************************************************/

/**** Overview ****/
/*
 The export protocol is initiated from the move/copy protocol
 (see embedwin.h).  All moves/copies from the TOC to non-bookshelf
 views of the DiskViewer are implicitly exports.

 More specifically, export happens after msgSelCopySelection 
 reaches the DiskViewer, which is the destination of the copy, and 
 the source of the copy includes clsExport as an item in the list
 returned by msgXferList.  Anything moveable/copyable can potentially
 invoke export.(See xfer.h and sel.h for information on PenPoint's
 move/copy protocol and selection management.)

 The DiskViewer will send the source of the copy (the selection) 
 msgExportGetFormats.  The source should pass back an array of possible 
 export formats.  From the information in msgExportGetFormats clsApp 
 generates the export dialog box.  If the user selects the external export
 format and taps the Move/Copy button, the export class sends msgExport
 to the appropriate translator specified in msgExportGetFormats.
 If user selects the PenPoint format and taps the Move/Copy button,
 the move/copy is equivalent to msgAppMgrMove/msgAppMgrCopy (see appmgr.h).

 If the source of the export is in the TOC, the DiskViewer activates the 
 source document and sends it msgExportGetFormats.
*/

/**** How to Be an Exporting Application ****/
/*
 Any application that wants to export must have its subclass of clsApp
 respond to msgExportGetFormats and msgExport.
*/

#ifndef EXPORT_INCLUDED
#define EXPORT_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef UID_INCLUDED
#include <uid.h>
#endif

#ifndef FS_INCLUDED
#include <fs.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                    Common #defines and typedefs                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/**** Status codes ****/
#define stsExportActivateSource         MakeWarning(clsExport, 1)
#define stsExportFailed                 MakeWarning(clsExport, 2)
#define stsExportFailedUserNotified     MakeWarning(clsExport, 3)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgExportGetFormats    takes P_EXPORT_LIST, returns STATUS
    category: client responsibility
    Passes back the export format array from from the source of the export.

 The DiskViewer sends this message to the selection.

 The recipient should allocate global memory to hold the EXPORT_FORMAT
 array which is passed back to the DiskViewer in the format field.
 The sender of msgExportGetFormats must free the memory.   

 If the source returns stsExportActivateSource, the DiskViewer will
 treat the source as an inactive document (This is how the TOC behaves
 when it is the source of export).  The source will be activated using
 msgAppMgrActivate and the activated doc will be sent msgExportGetFormats.
*/
#define msgExportGetFormats             MakeMsg(clsExport, 1)

typedef struct {
    TAG         documentType;               // Source document type.
    TAG         exportType;                 // Export destination type.
    OBJECT      translator;                 // Object which to send msgExport.
    CHAR        exportName[nameBufLength];  // Name of export type for
                                            // display in dialog box.
} EXPORT_FORMAT, *P_EXPORT_FORMAT;

typedef struct {
    P_EXPORT_FORMAT     format;         // Array of formats, must be SHARED
                                        // memory, freed by caller.
    U16                 numEntries;     // Number of elements in format array.
} EXPORT_LIST, *P_EXPORT_LIST;

/****************************************************************************
 msgExport              takes P_EXPORT_DOC, returns STATUS
	category: client responsibility
    Initiates export by the translator.  
    
 This message is sent to the translator specified in EXPORT_FORMAT.
 The translator is passed an open file handle to which the translator can
 write exported data or the translator can get the path of the file, 
 destroy the file and replace it with its own file structure.

 If the export fails, it is the exporter's reponsibility for removing
 invalid and/or partial files created during the failed export.
 The minimum the client should do is send msgFSDelete to pArgs->destination
 to remove the file created for the exportation.

 If the exporter wishes to put their custom dialog box to query the user
 for more information, the exporter should do this in response to msgExport.
 If the custom dialog allows the user to cancel the export operation,
 then the exporter should return stsExportFailedUserNotified which will
 cause PenPoint to suppress any error of the aborted export.
*/
#define msgExport		                MakeMsg(clsExport, 2)

typedef struct {
    TAG             exportType;             // Corresponds to exportType from 
                                            // msgExportGetFormats EXPORT_FORMAT.
    FS_LOCATOR      source;                 // Source document or null if
                                            // source is not a document.
    FILE_HANDLE     destination;            // Destination file handle.
                                            // If you don't want to export to 
                                            // this file, use msgFSGetPath to 
                                            // retrieve the destination and
                                            // destroy this file handle.
    CHAR            path[fsPathBufLength];  // Source path.
    TAG             documentType;           // Corresponds to documentType from
                                            // msgExportGetFormats EXPORT_FORMAT.
    U32             spare1;                 // Spare: reserved
    U32             spare2;                 // Spare: reserved
} EXPORT_DOC, *P_EXPORT_DOC;

/****************************************************************************
 msgExportName          takes P_EXPORT_FORMAT, returns STATUS
    Passes back a possibly modified destination name from the translator.

 This message is sent to the translator specified in EXPORT_FORMATS whenever
 the user chooses a new export type in the dialog box.  When the translator
 receives the message, export name is set to the source document name.
 The translator should set export name exportName should be set to the 
 "correct" destination file name.  For instance the extension '.RTF' or 
 '.WKS' may be appended to the name.

 If the translator ignores this message the destination name will
 remain unchanged (so this message can safely be ignored).
*/
#define msgExportName                   MakeMsg(clsExport, 3)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Miscellaneous							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/**** Help tags ****/
/*
 * These are help tags on various pieces of the standard export dialog box.
*/
#define hlpExportSheet                  MakeTag(clsExport, 50)
#define hlpExportName                   MakeTag(clsExport, 51)
#define hlpExportNewName                MakeTag(clsExport, 52)
#define hlpExportChoice                 MakeTag(clsExport, 53)

#endif // EXPORT_INCLUDED
