/****************************************************************************
 dvtimer.h

 Copyright 1992, GO Corporation,   All Rights Reserved

 The public definition of the timer device interface.

 $Revision:   1.8  $
   $Author:   thall  $
     $Date:   24 Mar 1992 15:10:20  $
****************************************************************************/
#ifndef DVTIMER_INCLUDED
#define DVTIMER_INCLUDED

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 This section contains definitions specific to the timer device. 
 The definitions contained here are data structures that are necessary
 for the MIL timer specific functions along with standard
 MIL device data structures with additional fields specific to the
 timer device.
*/

/**** Timer Device Specific Function Numbers ****/

/*
 The following definitions are the device specific functions supported
 by the timer device. The function numbers start immediately after
 the last function number used by the functions that are common to all
 MIL devices. These function numbers correspond to the Function Transfer
 Table (FTT) entries.

 All but the milTimerTickCont function must be implemented as
 single-staged functions. In addition, the timer device should
 be initialized and functional after the call to its device block
 initialization routine. The base initialization function should do
 nothing. Of course, none of the timer device functions can be staged
 on time.
*/

#define milTimerGetDevParameters	milLastCommonFunction+1
#define milTimerTickCont			milLastCommonFunction+2
#define milTimerSetMultiplier		milLastCommonFunction+3
#define milTimerGetTicks			milLastCommonFunction+4

#define fnCountTimer				fnCountBase+4

/**** Timer Device Function Transfer Table ****/
/*
 This structure is the declaration of the Timer Device Function
 Transfer Table. It includes one entry for each of the MIL driver
 functions common to all MIL devices along with one entry for
 each of the device specific functions for the MIL Timer device.
 Each entry contains fields that define the size of the request
 block for that function, the staging requirements of the function,
 and the address of the actual function. 
*/


typedef struct MIL_FTT_TIMER {
	funcTransTableCommonFields
	MIL_FUNCTION_DESC functionDesc[fnCountTimer];
} MIL_FTT_TIMER, *P_MIL_FTT_TIMER;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 					Timer Device Specific Functions	   	 				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 The following function definitions are the timer device specific
 functions. Access to these functions is provided by the FTT defined
 above. All of the functions defined below will be passed 3 pointers.

	pCommonData:
		This is a pointer to the common data area.

	pRB:
		This is a pointer to the function request block.

	pDeviceBlock:
		This is a pointer to the device block for the device

 All of the functions return STATUS. Any STATUS other than staging 
 statuses will cause the request to terminate. The staging statuses
 will result in the specified staging operation to be performed.
*/


/****************************************************************************
 milTimerGetDevParameters		returns STATUS
 	Provides device specific information about the timer device.

 This function is a single stage function that provides device
 specific information about the timer device.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milTimerGetDevParameters funtion request block contains a data 
 structure that defines the timer device specific parameters. All
 timer intervals are in microseconds.
	
	timerFlags:
		This field is intended to contain bits that describe
		timer capabilities and timer status. Currently there
		are not bits defined for this field.

	timerTicksPerInterval:
		This field indicates the number of timer ticks that occur
		during each timer interval. This is calculated as 
		timerInputFreq * timerBaseInterval / timerRateMultiplier.

	timerBaseInterval:
		This field is the period of one timer base interval.

	timerInterval:
		This field is the period of one timer interval. It is 
		calculated as timerBaseInterval / timerRateMultiplier.

	timerRateMultiplier:
		The variable timerRateMultiplier is only used by PenPoint 
        in special cases where it is desireable to increase the 
		tick rate. For example, this feature is used when the
		debugger is profiling a program.

	maxTimerInterval:
		This field specifies the maximum interval allowed by the
		hardware or the driver.
		
	minTimerInterval:
		This field specifies the minimum interval allowed by the
		hardware or the driver.

	timerInputFreq:
		This field specifies the timer input frequency in Hertz. This
		indicates how many times the timer ticks in a second.


	timerIntervalCount:
		This is the number of intervals that have occurred since 
		the system was booted. This is the low 32 bits of a 64 bit
		value.

	timerIntervalCountOverflow:
		This is the number of intervals that have occurred since 
		the system was booted. This is the high 32 bits of a 64 bit
		value.

 For standard PC-compatible hardware, the values of these parameters are
 (timerRateMultiplier = 1):

	timerInputFreq:	1.193180 megahertz
	timerBaseInterval:	54925 microseconds
	timerInterval:	54925 microseconds
	timerTicksPerInterval:	65535
	maxTimerInterval:	54925 microseconds
	minTimerInterval:	5000 microseconds

 A simplier hardware configuration could support:

	timerInputFreq:	20 hertz
	timerBaseInterval:	50000 microseconds
	timerInterval:	50000 microseconds
	timerTicksPerInterval:	1
	maxTimerInterval:	50000 microseconds
	minTimerInterval:	50000 microseconds

*/

typedef struct MIL_TIMER_DEVICE_PARAMETERS {
	U16 timerFlags;
	U32	timerTicksPerInterval;
	U32 timerBaseInterval;
	U32 timerInterval;
	U32 timerRateMultiplier;
	U32 maxTimerInterval;
	U32 minTimerInterval;
	U32 timerInputFreq;
	U32 timerIntervalCount;
	U32 timerIntervalCountOverflow;
	U16 reservedU16[2];
	U32 reservedU32[2];
} MIL_TIMER_DEVICE_PARAMETERS, *P_MIL_TIMER_DEVICE_PARAMETERS;

/*
 rbPublicFieldsTimerGetDevParameters

 This define merges the device specific data structure with the standard
 MIL device parameters reserved fields.

*/

#define rbPublicFieldsTimerGetDevParameters \
	MIL_TIMER_DEVICE_PARAMETERS timerDevParms; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];
/*
 
 milTimerGetDevParameters Request Block

*/

typedef struct MIL_RB_PUB_TIMER_GET_DEV_PARAMETERS {
	milRequestBlockCommonFields
	rbPublicFieldsTimerGetDevParameters
} MIL_RB_PUB_TIMER_GET_DEV_PARAMETERS, *P_MIL_RB_PUB_TIMER_GET_DEV_PARAMETERS;


/****************************************************************************
 milTimerTickCont		returns STATUS
 	Basic periodic tic of the system.

 This function is a continuous multi-stage function that provides the
 basic periodic tick of the system.

*/

/**** Device Specific Request Block Fields ****/
/*

 The milTimerTickCont funtion request block contains the
 following fields.

	pAsyncEventFunc:
		This field is a pointer to a function that should be
		called ever interval.
	
	timerIntervalCount:
		This is the number of intervals that have occurred since 
		the system was booted. This is the low 32 bits of a 64 bit
		value.

	timerIntervalCountOverflow:
		This is the number of intervals that have occurred since 
		the system was booted. This is the high 32 bits of a 64 bit
		value.

*/

#define rbPublicFieldsTimerTickCont \
	P_MIL_ASYNC_EVENT_FUNC pAsyncEventFunc; \
	U32 timerIntervalCount; \
	U32 timerIntervalCountOverflow; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

typedef struct MIL_RB_PUB_TIMER_TICK_CONT {
	milRequestBlockCommonFields
	rbPublicFieldsTimerTickCont
} MIL_RB_PUB_TIMER_TICKCONT, *P_MIL_RB_PUB_TIMER_TICK_CONT;


/****************************************************************************
 milTimerSetMultiplier		returns STATUS
 	Set the low level interval.

 The multiplier determines how many actual intervals make up the actual
 base interval. Normally this value is set to 1. The variable
 timerRateMultiplier is only used by PenPoint in special cases where
 it is desireable to increase the tick rate. For example, this feature
 is used when the debugger is profiling a program.

*/

/**** Device Specific Request Block Fields ****/
/*

 The milTimerSetMultiplier funtion request block contains the
 following fields.

	timerRateMultiplier:
		This field is the value to be used as the timerRateMultiplier.

*/

#define rbPublicFieldsTimerSetMultiplier \
	U32 timerRateMultiplier; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milTimerSetMultiplier Request Block

*/

typedef struct MIL_RB_PUB_TIMER_SET_MULTIPLIER {
	milRequestBlockCommonFields
	rbPublicFieldsTimerSetMultiplier
} MIL_RB_PUB_TIMER_SET_MULTIPLIER, *P_MIL_RB_PUB_TIMER_SET_MULTIPLIER;



/****************************************************************************
 milTimerGetTicks		returns STATUS
 	Gets the number of ticks in the current interval.

 This single stage function returns the number of ticks that have passed
 in the current interval.		

*/

/**** Device Specific Request Block Fields ****/
/*

 The milTimerGetTicks funtion request block contains the
 following fields.

	timerTicks:
		This field is the current number of timer ticks from the timer
		hardware.

	timerIntervalCount:
		This is the number of intervals that have occurred since 
		the system was booted. This is the low 32 bits of a 64 bit
		value.

	timerIntervalCountOverflow:
		This is the number of intervals that have occurred since 
		the system was booted. This is the high 32 bits of a 64 bit
		value.


	timerTicksPerInterval:
		This is the number of timer ticks that make up an interval.

*/


#define rbPublicFieldsTimerGetTicks \
	U32	timerTicks; \
	U32 timerIntervalCount; \
	U32 timerIntervalCountOverflow; \
	U32	timerTicksPerInterval; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milTimerGetTicks Request Block

*/


typedef struct MIL_RB_PUB_TIMER_GET_TICKS {
	milRequestBlockCommonFields
	rbPublicFieldsTimerGetTicks
} MIL_RB_PUB_TIMER_GET_TICKS, *P_MIL_RB_PUB_TIMER_GET_TICKS;

#endif
