/****************************************************************************
 dvsound.h

 Copyright 1992, GO Corporation,   All Rights Reserved

 The public definition of the MIL sound device interface.

 $Revision:   1.7  $
   $Author:   thall  $
     $Date:   24 Mar 1992 15:10:04  $
****************************************************************************/
#ifndef DVSOUND_INCLUDED
#define DVSOUND_INCLUDED


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 This section contains definitions specific to the sound device. 
 The definitions contained here are data structures that are necessary
 for the MIL sound specific functions along with standard
 MIL device data structures with additional fields specific to the
 sound device.
*/

/**** Sound Device Specific Function Numbers ****/

/*
 The following definitions are the device specific functions supported
 by the sound device. The function numbers start immediately after
 the last function number used by the functions that are common to all
 MIL devices. These function numbers correspond to the Function Transfer
 Table (FTT) entries.
*/

#define milSoundGetDevParameters	milLastCommonFunction+1
#define milSoundStartTone			milLastCommonFunction+2

#define fnCountSound				fnCountBase+2

/**** Sound Device Function Transfer Table ****/
/*
 This structure is the declaration of the Sound Device Function
 Transfer Table. It includes one entry for each of the MIL driver
 functions common to all MIL devices along with one entry for
 each of the device specific functions for the MIL Sound device.
 Each entry contains fields that define the size of the request
 block for that function, the staging requirements of the function,
 and the address of the actual function. 
*/

typedef struct MIL_FTT_SOUND {
	funcTransTableCommonFields
	MIL_FUNCTION_DESC functionDesc[fnCountSound];
} MIL_FTT_SOUND, *P_MIL_FTT_SOUND;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 					Sound Device Specific Functions	   	 				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 The following function definitions are sound device specific
 functions. Access to these functions is provided by the FTT defined
 above. All of the functions defined below will be passed 3 pointers.

	pCommonData:
		This is a pointer to the common data area.

	pRB:
		This is a pointer to the function request block.

	pDeviceBlock:
		This is a pointer to the device block for the device

 All of the functions return STATUS. Any STATUS other than staging 
 statuses will cause the request to terminate. The staging statuses
 will result in the specified staging operation to be performed.
*/


/****************************************************************************
 milSoundGetDevParameters		returns STATUS
 	Provides device specific information about the sound device.

 This function is a single stage function that provides device specific
 information about the sound device.

*/ 

/**** Device Specific Request Block Fields ****/
/*

 This data structure contains information specific to sound device. It
 is made up of the following fields.

	soundFlags:
		These flags are intended to contain state and capability
		information about the sound device. Currently there are
		no flags defined for the sound device.

	highFreq:
		This field defines the highest frequency the sound device 
		an emit. The value is given in hertz.

	highLevel:
		This field defines the highest volume level supported by
		the sound device.


*/

typedef struct MIL_SOUND_DEVICE_PARAMETERS {
	U16 soundFlags;				// out: state and capability flags
	U16 highFreq;				// out: highest supported frequency
	U16 highLevel;				// out: highest volume supported
	U16 reservedU16[2];
	U32 reservedU32[2];
} MIL_SOUND_DEVICE_PARAMETERS, *P_MIL_SOUND_DEVICE_PARAMETERS;

/*
 rbPublicFieldsSoundGetDevParameters

 This define merges the device specific data structure with the standard
 MIL device parameters reserved fields.

*/

#define rbPublicFieldsSoundGetDevParameters \
	MIL_SOUND_DEVICE_PARAMETERS soundDevParms; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];
/*
 
 milSoundGetDevParameters Request Block

*/

typedef struct MIL_RB_PUB_SOUND_GET_DEV_PARAMETERS {
	milRequestBlockCommonFields
	rbPublicFieldsSoundGetDevParameters
} MIL_RB_PUB_SOUND_GET_DEV_PARAMETERS, *P_MIL_RB_PUB_SOUND_GET_DEV_PARAMETERS;

/****************************************************************************
 milSoundStartTone		returns STATUS
	Produces a tone of the specified frequency, level, and duration. 	

 This function is a multi-stage function that provides the capability
 of generating a tone on the sound device. The function	allows the caller
 to specify the frequency, volume level, and duration of the tone.

*/ 

/**** Device Specific Request Block Fields ****/
/*
 This definition creates the device specific fields for the 
 milSoundStartTone used in the request block.

	frequency:
		This field specifies the tone frequency in Hertz.

	level:
		This field specifies the volume level. The value should be
		in the range of 0 to the highLevel field indicated in the
		milSoundGetDevParameters function.

	duration:
		This field specifies the duration of the tone. The value
		is provided in millisconds.
	
	soundTypeHint:
		This field is advice to the sound device as to how the
		sound should be generated. The values defined for this
		field are given below.

*/

#define rbPublicFieldsSoundStartTone \
	U16 frequency; \
	U16 level; \
	U16 soundTypeHint; \
	U16 duration; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 soundTypeHint Values

 The following values are advice to sound producing hardware to 
 specify how the tone should be generated (if possible).

	milSoundWarn:
		The tone is a warning. This sound is generated when
		the user is being informed of something that may or
		may not be a problem.

	milSoundError:
		This tone is associated with an error. This sound 
		is generated when the user is bing informed of an
		error situation.

	milSoundChime:
		The tone should be emitted as a chime.

	milSoundBong:
		The tone should be emitted as a bong.

*/

#define milSoundWarn	0
#define milSoundError	1
#define milSoundChime	2
#define milSoundBong	3

/*
 
 milSoundStartTone Request Block

*/

typedef struct MIL_RB_PUB_SOUND_START_TONE {
	milRequestBlockCommonFields
	rbPublicFieldsSoundStartTone
} MIL_RB_PUB_SOUND_START_TONE, *P_MIL_RB_PUB_SOUND_START_TONE;

#endif

