/****************************************************************************
 File: dialenv.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.7  $
   $Author:   cleu  $
     $Date:   24 Feb 1992 12:01:32  $

 This file contains the API for clsDialEnv, clsDialEnvOptCard, and clsDialEnvField.

 clsDialEnv inherits from clsService.

 clsDialEnv maintains telephone dialing related information
 pertinent to a specific geographic location/environment.

 The intent of clsDialEnv is to relieve client data communication
 programs of having to replicate the code for maintaining their
 own seperate telephone dialing-related data and logic.  clsDialEnv
 is designed to provide the "intelligence" and data needed for
 dialing from/to a variety of environments (to/from local in-house
 to/from international).

	 ===================================================================

 clsDialEnvOptCard inherits from clsCustomLayout.

 clsDialEnvOptCard provides a default behavior of observing the
 dialing environment and refreshing dialing environment option
 cards when the dialing environment changes.

	 ===================================================================

 clsDialEnvField inherits from clsField.

 clsDialEnvField alters the a default behavior of ancestor clsField by
 specifying a character list template for coercing its field input.


 Dialing environments are a location type service and therefore
 managed by a service manager called theLocations.  Each instance
 of a dialing environment is identified by the name of a location
 to which the dialing environment pertains (NOTE: for PenPoint 1.0
 there is only a single location/dialing environment).  Objects
 wishing to communicate with a dialing environment do so by sending
 messages to the current location service.  The UID of the current
 location is obtained by querying theLocations via standard install
 manager and service manager messages.  The following block of code
 provides one example of how a client might obtain dialing environment
 data.

//{
	{
		OBJECT			handleCurrentLoc,
						theCurrentLocation;
		SM_QUERY_LOCK	lock;
		SM_QUERY_UNLOCK	unlock;
		DIALENV_COUNTRY	country;
		IM_GET_SET_NAME	getName;
		CHAR			locationName[nameBufLength];

		//
		//	Get the handle and UID of the current location.
		//	Lock the current location to guarantee exclusive access to
		//	  location data.
		//	Get the country code for the current location (from the dialing
		//	  environment for the current location).
		//	Unlock the current location so that other clients may access it.
		//	Get the name of the current location.
		//
		ObjCallJmp(msgIMGetCurrent, theLocations, &handleCurrentLoc, s, Problem);
		lock.handle = unlock.handle = handleCurrentLoc;
		ObjCallJmp(msgSMQueryLock, theLocations, &lock, s, Problem);
		theCurrentLocation = lock.service;
		ObjCallJmp(msgDialEnvGetCountry, theCurrentLocation, &country, s, Problem);
		ObjectCall(msgSMQueryUnlock, theLocations, &unlock);
		getName.handle = handleCurrentLoc;
		getName.pName = locationName;
		ObjCallJmp(msgIMGetName, theLocations, &getName, s, Problem);
	}
//}

 For PenPoint 1.0 an application or service requiring dialing environment
 services should install the dialing environment dll via a SERVICE.INI file.

 **** Future Direction Ideas ****

 In a future release of PenPoint, dialing environments will be subsumed
 by a location service.  The location service will manage all of the
 objects which provide location-dependent behavior to the PenPoint
 environment/applications.  Current plans are for the user to access
 location services via the configuration notebook.  Because dialing
 environments will be a constituent of a location service it won't be
 necessary for a dialing environment to be included by an application's
 or service's SERVICE.INI file.

 The location service will maintain the list of locations the user has
 created (GO may ship pre-configured locations; however a user will be
 able to create and modify locations).  A user will select a location
 by name, and all of the unique properties regarding that location will
 take effect.

 For each location there may be a dialing environment.  Thus, whenever
 the user selects a new location, a different dialing environment may
 take effect (it is possible that two different locations will share
 the same dialing environment, or that a location doesn't have a dialing
 environment).  When a user creates a new location, the user will be given
 the opportunity to specify a dialing environment for the new location,
 or to select one of the currently available dialing environments and bind
 it to the new location.

 The dialing environment will be enhanced to provide clients with information
 regarding valid city/area codes and dialing rules for specific countries.
 This information can be presented to the user for UI pick-lists, used to
 coerce input to only valid combinations of codes, and to enforce the rules
 which national telephone systems impose on computer software which interacts
 with the public telephone system.

 **** End of Future Direction Ideas ****

 clsDialEnvOptCard provides a default behavior of observing the
 dialing environment and refreshing dialing environment option
 cards when the dialing environment changes.  A client needn't
 provide any special code support to have such option cards track
 dialing environment changes.  Note: A client shouldn't insert
 a dialing environment option card into an option sheet or any
 window tree with a modal filter (e.g. option sheet with a style
 modality set to either osModalApp or osModalSystem).

 The following block of code provides one example of creating a
 dialing environment option card.

//{
	{
		//
		//	Create an option card for dialing environment settings.
		//
		STATUS					s;
		DIALENV_OPTCARD_NEW		don;
		OBJECT					handleCurrentLoc;
		IM_GET_SET_NAME			getName;
		CHAR					locationName[nameBufLength];

		//
		//	Get the handle and name of the current location.  Create
		//	  a dialing environment option card for the current location.
		//
		ObjCallRet(msgIMGetCurrent, theLocations, &handleCurrentLoc, s);
		getName.handle = handleCurrentLoc;
		getName.pName = locationName;
		ObjCallRet(msgIMGetName, theLocations, &getName, s);
		ObjCallRet(msgNewDefaults, clsDialEnvOptCard, &don, s);
		don.win.tag = tagDialEnvOptionCard;
		strcpy(don.dialenvOptCard.dialEnv.name, locationName);
		ObjCallRet(msgNew, clsDialEnvOptCard, &don, s);
	}
//}

 clsDialEnvField alters the a default behavior of ancestor clsField by
 specifying a character list template for coercing its field input.

		
	Defined within this header file:

	o	defines and typedefs for dial environment data.
	o	function prototypes.
	o	messages & status values.

****************************************************************************/
#ifndef DIALENV_INCLUDED
#define DIALENV_INCLUDED

#ifndef	GO_INCLUDED
#include <go.h>
#endif
#ifndef	CLSMGR_INCLUDED
#include <clsmgr.h>
#endif
#ifndef	SERVICE_INCLUDED
#include <service.h>
#endif
#ifndef CLAYOUT_INCLUDED
#include <clayout.h>
#endif
#ifndef	FIELD_INCLUDED
#include <field.h>
#endif


/****	Defines and typedefs   ****/

/*
 Class UIDs for:

		clsDialEnv			-	The dialing environment service.
		clsDialEnvOptCard	-	Dialing environment option cards.
		clsDialEnvField		-	Field for entering and coercing dialing codes/numbers.
		theLocations		-	Service manager for dialing environments.

*/
#define clsDialEnv					MakeWKN(2576,1,wknGlobal)
#define	clsDialEnvOptCard			MakeWKN(2577,1,wknGlobal)
#define	clsDialEnvField				MakeWKN(2578,1,wknGlobal)
#define	theLocations				MakeWKN(2579,1,wknGlobal)

typedef	OBJECT						DIALENV,
									*P_DIALENV,
									DIALENV_OPTCARD,
									*P_DIALENV_OPTCARD;


/*
	Dialing environment and telephone number data component dimensions.
*/
#define	lenDialEnvCountry			3			// 1 to 3 digits.
#define	lenDialEnvAreaCity			4			// 0 to 4 digits.
#define	lenDialEnvTeleNumber		12			// 1 to 12 digits.
#define	lenDialEnvIntlAccess		4			// 1 to 4 digits.
#define	lenDialEnvLongDist			6			// 0 to 6 digits.
#define	lenDialEnvOutsideLine		6			// 0 to 6 digits.
#define	lenDialEnvSuffix			12			// 0 to 12 symbols.
#define	lenDialEnvMacroCode			20			// 0 to 20 symbols.
#define	numDialEnvMacroCodes		4			// # of dialing codes.
#define	lenDialEnvPostConnect		20			// 0 to 20 symbols.
#define	lenDialEnvDialString		75			// 0 to 75 symbols.


/*
	Tags for resources/windows ...
*/

/*
 Dialing Environment Quick Help ...

	  Quick help is stored in clsDialEnv resource list 0.
	  Each quick help entry is located by its index/position
		within resource list 0.

*/
#define	resListDialEnvQHelp			0
#define	MakeDialEnvQHelpResId(x)	MakeIndexedResId(clsDialEnv,resListDialEnvQHelp,x)

/*
	Tags for dialing environment option card windows & quick help.
*/
#define tagDialEnvOptCard			MakeTag(clsDialEnv, 1)
#define	hlpDialEnvOptCard			MakeDialEnvQHelpResId(0)

#define tagDialEnvDialEnvTable		MakeTag(clsDialEnv, 17)
#define	hlpDialEnvDialEnv			MakeDialEnvQHelpResId(0)

#define tagDialEnvCurrentLocLabel	MakeTag(clsDialEnv, 18)
#define tagDialEnvCurrentLocTable	MakeTag(clsDialEnv, 19)
#define	hlpDialEnvCurrentLoc		MakeDialEnvQHelpResId(6)

#define tagDialEnvDialLabel			MakeTag(clsDialEnv, 24)
#define tagDialEnvDial				MakeTag(clsDialEnv, 25)
#define tagDialEnvDialTone			MakeTag(clsDialEnv, 26)
#define tagDialEnvDialPulse			MakeTag(clsDialEnv, 27)
#define	hlpDialEnvDial				MakeDialEnvQHelpResId(7)

#define tagDialEnvAreaCityLabel		MakeTag(clsDialEnv, 32)
#define tagDialEnvAreaCity 			MakeTag(clsDialEnv, 33)
#define	hlpDialEnvAreaCity			MakeDialEnvQHelpResId(9)

#define tagDialEnvCountryLabel		MakeTag(clsDialEnv, 40)
#define tagDialEnvCountry  			MakeTag(clsDialEnv, 41)
#define	hlpDialEnvCountry			MakeDialEnvQHelpResId(8)

#define tagDialEnvOutsideLineLabel	MakeTag(clsDialEnv, 48)
#define tagDialEnvOutsideLine		MakeTag(clsDialEnv, 49)
#define	hlpDialEnvOutsideLine		MakeDialEnvQHelpResId(1)

#define tagDialEnvLongDistLabel		MakeTag(clsDialEnv, 56)
#define tagDialEnvLongDist			MakeTag(clsDialEnv, 57)
#define	hlpDialEnvLongDist			MakeDialEnvQHelpResId(2)

#define tagDialEnvIntlAccessLabel	MakeTag(clsDialEnv, 64)
#define tagDialEnvIntlAccess		MakeTag(clsDialEnv, 65)
#define	hlpDialEnvIntlAccess		MakeDialEnvQHelpResId(3)

#define tagDialEnvSuffixLabel		MakeTag(clsDialEnv, 72)
#define tagDialEnvSuffix			MakeTag(clsDialEnv, 73)
#define	hlpDialEnvSuffix			MakeDialEnvQHelpResId(4)

#define tagDialEnvMacroCodesLabel	MakeTag(clsDialEnv, 80)
#define tagDialEnvMacroCodes		MakeTag(clsDialEnv, 81)
#define	hlpDialEnvMacroCodes		MakeDialEnvQHelpResId(5)
#define	tagDialEnvSetCodes			MakeTag(clsDialEnv, 82)
#define tagDialEnvMacroCodeALabel	MakeTag(clsDialEnv, 83)
#define tagDialEnvMacroCodeA		MakeTag(clsDialEnv, 84)
#define tagDialEnvMacroCodeBLabel	MakeTag(clsDialEnv, 85)
#define tagDialEnvMacroCodeB		MakeTag(clsDialEnv, 86)
#define tagDialEnvMacroCodeCLabel	MakeTag(clsDialEnv, 87)
#define tagDialEnvMacroCodeC		MakeTag(clsDialEnv, 88)
#define tagDialEnvMacroCodeDLabel	MakeTag(clsDialEnv, 89)
#define tagDialEnvMacroCodeD		MakeTag(clsDialEnv, 90)
#define tagDialEnvMacroCodesFrame	MakeTag(clsDialEnv, 91)
#define	deMaxMacroCodes				4


/****	Exported function prototypes from dialenv.dll   ****/
/*
	None currently defined.
 */

/*
	>>>>>>>>>>>>>>>>>>>>>>>>>>> clsDialEnv <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
*/

/****	Message definitions   ****/
/*
	NOTE: msg #1 is reserved for private use.
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	 					Observer Notification Messages					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgDialEnvChanged			takes OBJECT,			returns STATUS
	category: observer notification
	Notification sent to observers to indicate a dialing environment change.

 The pArgs indicates the object which initiated the change to the dialing
 environment.  pArgs of objNull indicates that the dialing environment is
 being destroyed.

 Observers which receive this message should refresh any local
 dialing environment information or view of such information.

 Error Return Values:	N/A.
*/
#define msgDialEnvChanged					MakeMsg(clsDialEnv,	2)



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	 						Action Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgDialEnvGetCountry		takes P_DIALENV_COUNTRY, returns STATUS
	category: service action request
	Passes back the country code from the current dialing environment.

 Error Return Values:	none, always returns stsOK.
*/
#define msgDialEnvGetCountry				MakeMsg(clsDialEnv,	3)

/*
	Country code.
*/
typedef struct DIALENV_COUNTRY
{
	CHAR	symbols[lenDialEnvCountry+1];

} DIALENV_COUNTRY, *P_DIALENV_COUNTRY;


/****************************************************************************
 msgDialEnvIsCountryNorthAmerican takes P_DIALENV_COUNTRY, returns STATUS
	category: service action request
	Indicates whether or not the specified country code is North American.

 NOTES:	This message is provided so a client may alter its UI and/or
		enforce editing rules unique to North American phone numbers.

 Returns stsOK if the specified country is North American,
	 otherwise stsDialEnvNoMatch.
*/
#define msgDialEnvIsCountryNorthAmerican	MakeMsg(clsDialEnv,	6)


/****************************************************************************
 msgDialEnvGetEnvironment	takes P_DIALENV_ENVIRONMENT, returns STATUS
	category: service action request
	Passes back the current dialing environment settings.

 Error Return Values:	none, always returns stsOK.
*/
#define msgDialEnvGetEnvironment			MakeMsg(clsDialEnv,	4)

/*
	Dial mode (type of phone/dialing).
*/
typedef TAG				DIALENV_DIAL_MODE;
#define	deTone			tagDialEnvDialTone		// Touch tone dialing.
#define	dePulse			tagDialEnvDialPulse		// Pulse code dialing.

/*
	Outside line.

	  Symbols prefixed to a dialing string to gain
		access to the general switched telephone network.
*/
typedef struct DIALENV_OUTSIDE_LINE
{
	CHAR	symbols[lenDialEnvOutsideLine+1];

} DIALENV_OUTSIDE_LINE, *P_DIALENV_OUTSIDE_LINE;

/*
	Area/City code.
*/
typedef struct DIALENV_AREA_CITY
{
	CHAR	symbols[lenDialEnvAreaCity+1];

} DIALENV_AREA_CITY, *P_DIALENV_AREA_CITY, **PP_DIALENV_AREA_CITY;

/*
	International access code.
*/
typedef struct DIALENV_INTL_ACCESS
{
	CHAR	symbols[lenDialEnvIntlAccess+1];

} DIALENV_INTL_ACCESS, *P_DIALENV_INTL_ACCESS;

/*
	Long distance access code.
*/
typedef struct DIALENV_LONG_DIST
{
	CHAR	symbols[lenDialEnvLongDist+1];

} DIALENV_LONG_DIST, *P_DIALENV_LONG_DIST;

/*
	Suffix.

	  Symbols appended to a dialing string.  Typically
		for credit card billing/call accounting purposes.
*/
typedef struct DIALENV_SUFFIX
{
	CHAR	symbols[lenDialEnvSuffix+1];

} DIALENV_SUFFIX, *P_DIALENV_SUFFIX;

/*
	Macro code.

	  Multi-purpose codes for specifying credit card #s, account
		billing codes, or altering environment dependent behavior.
		  When a client requests to build a dial string, the symbols
			from a macro code get expanded into the resultant dial string.
*/
typedef struct DIALENV_MACRO_CODE
{
	CHAR	symbols[lenDialEnvMacroCode+1];

} DIALENV_MACRO_CODE, *P_DIALENV_MACRO_CODE;

/*
	Dialing environment.
*/
typedef	struct	DIALENV_ENVIRONMENT
{
	DIALENV_DIAL_MODE	dialMode;		// Dial mode (tone/pulse).
	DIALENV_OUTSIDE_LINE outsideLine;	// Outside line/net access.
	DIALENV_AREA_CITY	areaCity;		// Area/City call originates from.
	DIALENV_COUNTRY		country;		// Country call originates from.
	DIALENV_INTL_ACCESS	intlAccess;		// International access code.
	DIALENV_LONG_DIST	longDist;		// Long distance access code.
	DIALENV_SUFFIX		suffix;			// Suffix applied to dial strings.
	DIALENV_MACRO_CODE	macroCode[numDialEnvMacroCodes];// Macro/expand codes.

} DIALENV_ENVIRONMENT, *P_DIALENV_ENVIRONMENT;


/****************************************************************************
 msgDialEnvBuildDialString	takes P_DIALENV_BUILD_DIALSTR, returns STATUS
	category: service action request
	Construct a dial string based upon the current dialing environment.
		
 NOTE: The order in which macro codes are processed is significant.
 All like macro codes are expanded before the next macro code is
 expanded.  Thus if expansion of macro code N results in symbols
 for a subsequent macro code (e.g. N+1) to be inserted into the
 dial string, such symbols will be interpretted as and expanded
 as macro codes.

 Error Return Values:	stsDialEnvDialStrTooLarge
*/
#define msgDialEnvBuildDialString			MakeMsg(clsDialEnv,	5)

/*
	Telephone number definition/components.
*/
typedef	struct	DIALENV_TELEPHONE_NUMBER
{
	CHAR	country[lenDialEnvCountry+1];		// Cntry call originates from.
	CHAR	areaCity[lenDialEnvAreaCity+1];		// Area/City call origs from.
	CHAR	teleNumber[lenDialEnvTeleNumber+1];	// Destination telephone #.
	CHAR	postConnect[lenDialEnvPostConnect+1];//Post connect destination
												//	 network navigation code.
} DIALENV_TELEPHONE_NUMBER, *P_DIALENV_TELEPHONE_NUMBER;

/*
	Dial string.

	  The resultant string of symbols a dialer sends to either clsModem,
		the phone network, or another server which performs the dialing.
*/
typedef struct DIALENV_DIAL_STRING
{
	CHAR	symbols[lenDialEnvDialString+1];

} DIALENV_DIAL_STRING, *P_DIALENV_DIAL_STRING;

/*
	Convert a raw telephone number into a dial string.
*/
typedef	struct	DIALENV_BUILD_DIALSTR
{
	P_DIALENV_TELEPHONE_NUMBER	pTeleNumber;	// In:	Raw tele # to dial.
	P_DIALENV_DIAL_STRING		pDialString;	// Out:	Resultant dial str.

} DIALENV_BUILD_DIALSTR, *P_DIALENV_BUILD_DIALSTR;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *		 					Class Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew					takes P_DIALENV_NEW,			returns STATUS
	category: class message
	Creates an instance of a dialing environment.

 Error Return Values:	percolated up from other classes,
						none from clsDialEnv.
*/

typedef	DIALENV_ENVIRONMENT DIALENV_NEW_ONLY, *P_DIALENV_NEW_ONLY;

#define dialenvNewFields	\
		serviceNewFields	\
		DIALENV_NEW_ONLY	dialEnv;

typedef struct DIALENV_NEW
{
    dialenvNewFields
		
} DIALENV_NEW, *P_DIALENV_NEW;


/****************************************************************************
 msgNewDefaults				takes P_DIALENV_NEW,		returns STATUS
	category: class message
	Initializes the DIALENV_NEW structure to default values.

 Sets:
//{
	pArgs->svc.style.waitForTarget	=
	pArgs->svc.style.exclusiveOpen	=
	pArgs->svc.style.autoOwnTarget	=
	pArgs->svc.style.autoOpen		=
	pArgs->svc.style.autoMsgPass	=
	pArgs->svc.style.checkOwner		= false;
	pArgs->svc.pManagerList			= pManagerList;	// theLocations
	pArgs->svc.numManagers			= 1;
	memset(&(pArgs->dialEnv), 0, sizeof(pArgs->dialEnv));
	pArgs->dialEnv.dialMode	= deTone;		// Tone dialing.
											// All remaining struct dialEnv
											//  fields are set to zero/null.
//}

 Error Return Values:	percolated up from other classes,
						none from clsDialEnv.
*/


/****************************************************************************
 msgDialEnvGetMacroIds	takes P_DIALENV_MACRO_IDS,	returns STATUS
	category: class message
	Passes back a string of symbols which identify dialing macro codes.

 Error return values:	percolated up from other classes,
						none from clsDialEnv.
*/
#define msgDialEnvGetMacroIds				MakeMsg(clsDialEnv,	6)

typedef	struct DIALENV_MACRO_IDS
{
	CHAR	symbols[numDialEnvMacroCodes+1];
	
} DIALENV_MACRO_IDS, *P_DIALENV_MACRO_IDS;


/****	clsDialEnv non-error status values   ****/

/* 
 None currently defined 
 */


/****	clsDialEnv error status values   ****/

/*
 The request sent to the dialing environment has been
 denied because the request isn't supported by this
 dialing environment.
 */
#define	stsDialEnvRequestDenied				MakeStatus(clsDialEnv, 1)

/*
 The request sent to the dialing environment specified
 an invalid country code.
 */
#define	stsDialEnvInvalidCountry			MakeStatus(clsDialEnv, 2)

/*
 The request sent to the dialing environment contained
 data which didn't match the specified constraints.
 */
#define	stsDialEnvNoMatch					MakeStatus(clsDialEnv, 3)

/*
 The dial string resulting from msgDialEnvBuildDialString is
 too large to be contained within struct DIALENV_DIAL_STRING.
 */
#define	stsDialEnvDialStrTooLarge			MakeStatus(clsDialEnv, 4)



/*
	>>>>>>>>>>>>>>>>>>>>>>> clsDialEnvOptCard <<<<<<<<<<<<<<<<<<<<<<<<<<<
*/

/****	Message definitions ....  ****/
/*
	NOTE: msg #1 reserved for private use.
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	 						Action Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgDialEnvOptCardRefresh	takes nothing,			returns STATUS
	category: action request
	Refreshes a dialing environment option card (self) with the current
	dialing environment settings.

 A client should send msgDialEnvOptCardRefresh to a dialing
 environment option card when it receives msgOptionRefreshCard
 and the card tag matches that assigned to the dialing environment
 option card.

 Error Return Values:	percolated up from other classes,
						none from clsDialEnv.
*/
#define msgDialEnvOptCardRefresh			MakeMsg(clsDialEnvOptCard,	2)


/****************************************************************************
 msgDialEnvOptCardApply		takes nothing,			returns STATUS
	category: action request
	Updates the dialing environment with current settings from a
	dialing environment option card (self).

 A client should send msgDialEnvOptCardApply to a dialing
 environment option card when it receives msgOptionApplyCard
 and the card tag matches that assigned to the dialing environment
 option card.

 Error Return Values:	percolated up from other classes,
						none from clsDialEnv.
*/
#define msgDialEnvOptCardApply				MakeMsg(clsDialEnvOptCard,	3)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *		 					Class Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew				takes P_DIALENV_OPTCARD_NEW,		returns STATUS
	category: class message
	Creates an instance of a dialing environment option card.
		
 A client may add the dialing environment option card to its stack of
 of option cards, and create it in reponse to msgOptionProvideCard via
 this message.  Clients may create multiple cards and insert them into
 any window.  The cards needn't be part of an option card stack.

 NOTES:	It is possible for one or more clients to create multiple
		dial environment option cards.  Because of this, dialing
		environment option cards observe the dialing environment.
		When the dialing environment changes, all dialing environment
		cards get refreshed with current dialing environment settings.
		
		The requestor must fill in the pArgs->dialEnv with the name
		of the location which will supply the option card with dialing
		environment settings.

 Error Return Values:	percolated up from other classes,
						stsDialEnvOptCardBadEnvironment.
*/

typedef	struct LOCATION_NAME
{
	CHAR			name[nameBufLength];	// Name of a location.

} LOCATION_NAME, *P_LOCATION_NAME;

											// Name of a dialing environment.
typedef	LOCATION_NAME	DIALENV_NAME, *P_DIALENV_NAME;


typedef struct DIALENV_OPTCARD_NEW_ONLY
{
	DIALENV_NAME	dialEnv;				// Name of DialEnv supplying info.
	U32				spare1;					// unused (reserved).
	U32				spare2;					// unused (reserved).

} DIALENV_OPTCARD_NEW_ONLY, *P_DIALENV_OPTCARD_NEW_ONLY;

#define	dialenvOptCardNewFields	\
	customLayoutNewFields		\
	DIALENV_OPTCARD_NEW_ONLY	dialenvOptCard;

typedef struct DIALENV_OPTCARD_NEW
{
    dialenvOptCardNewFields
		
} DIALENV_OPTCARD_NEW, *P_DIALENV_OPTCARD_NEW;


/****************************************************************************
 msgNewDefaults		takes P_DIALENV_OPTCARD_NEW,		returns STATUS
	category: class message
	Initializes the DIALENV_OPTCARD_NEW structure to default values.

 Sets:

//{
	 memset(pArgs->dialenvOptCard.dialEnv.name, Nil(CHAR),
		 sizeof(pArgs->dialenvOptCard.dialEnv.name));
//}
*/



/****	clsDialEnvOptCard non-error status values   ****/

/* 
 None currently defined 
 */


/****	clsDialEnvOptCard error status values   ****/

/*
 An internal system error was encountered creating an
 instance of clsDialEnvOptCard.
 */
#define	stsDialEnvOptCardProblem			MakeStatus(clsDialEnvOptCard, 1)

/*
 The arguments specified via msgNew to clsDialEnvOptCard
 didn't specify a dialing environment (from which data
 for the option card is obtained).
 */
#define	stsDialEnvOptCardBadEnvironment		MakeStatus(clsDialEnvOptCard, 2)

/*
 An internal system error was encountered unfiling
 clsDialEnvOptCard from a resource file.
 */
#define	stsDialEnvOptCardBadResFile			MakeStatus(clsDialEnvOptCard, 3)

/*
 An internal system error was encountered when attempting
 to locate a window (containing option data) withing a
 dialing environment option card.
 */
#define	stsDialEnvOptCardNoSuchOption		MakeStatus(clsDialEnvOptCard, 4)



/*
	>>>>>>>>>>>>>>>>>>>>>>>> clsDialEnvField <<<<<<<<<<<<<<<<<<<<<<<<<<<<
*/

/****	Message definitions   ****/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *		 					Class Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew				takes P_DIALENV_FIELD_NEW,		returns STATUS
	category: class message
	Creates an instance of a dialing environment field.

 clsDialEnvField logic within its msgInit method:

//{
	DIALENV_MACRO_IDS	macroIds;
	CHAR				fieldCharList[20+numDialEnvMacroCodes+1];
	XTM_ARGS			template;
	P_STRING			fieldChars = "0123456789()-,*#;!";
	
	//
	//	If the client hasn't modified the default field template value,
	//	establish a template to coerce dialing environment field input.
	//
	//	Query clsDialEnv to obtain the symbols identifying macro
	//	  codes.  Append them to base dialing type characters.
	//

	if (pArgs->field.xlate.pTemplate == pNull  &&
		pArgs->field.style.xlateType == fstXlateTemplate)
	{
		macroIds.symbols[0] = Nil(CHAR);
		ObjCallWarn(msgDialEnvGetMacroIds, clsDialEnv, &macroIds);
		strcpy(fieldCharList, fieldChars);
		strcat(fieldCharList, macroIds.symbols);

		template.xtmType = xtmTypeCharList;	//	Char list type template.
		template.xtmMode = xtmModeDefault;	//	No special template mode.
		template.pXtmData = fieldCharList;	//	The character list.

		pArgs->field.xlate.pTemplate	= &template;
	}

	// Call our ancestor to create the object.
	return ObjectCallAncestor(msg, self, pArgs, ctx);

//}

 Error Return Values:	percolated up from other classes,
*/

#define	dialenvFieldNewFields	\
	fieldNewFields

typedef struct DIALENV_FIELD_NEW
{
    dialenvFieldNewFields
		
} DIALENV_FIELD_NEW, *P_DIALENV_FIELD_NEW;


/****************************************************************************
 msgNewDefaults		takes P_DIALENV_FIELD_NEW,		returns STATUS
	category: class message
	Initializes the DIALENV_FIELD_NEW structure to default values.

 Sets:
//	{
		//
		//	Establish defaults for an instance of clsDialEnvField.
		//
		pArgs->field.style.veto			=
		pArgs->field.style.noSpace		=
		pArgs->field.style.upperCase	= true;
		pArgs->field.style.xlateType	= fstXlateTemplate;
		pArgs->field.xlate.pTemplate	= &template;
		pArgs->label.style.numCols		=
		pArgs->label.style.numRows		= lsNumAbsolute;
		pArgs->label.cols				= 12;
		pArgs->label.rows				= 1;
		pArgs->border.style.edge		= bsEdgeBottom;
		pArgs->border.style.topMargin	=
		pArgs->border.style.bottomMargin= bsMarginMedium;
		pArgs->border.style.borderInk	= bsInkGray66;
//	}
*/

#endif  // DIALENV_INCLUDED

