/****************************************************************************
 File: counter.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.14  $
   $Author:   kcatlin  $
     $Date:   17 Mar 1992 14:40:22  $

 This file contains the API definition for clsCounter.

 clsCounter inherits from clsTableLayout.

 Counters are general components which display a current count and provide
 up/down arrows for the user to alter the count.

 Counters are used as notebook frame decorations to provide
 up/down arrows to move between pages.
****************************************************************************/
#ifndef COUNTER_INCLUDED
#define COUNTER_INCLUDED

													#ifndef TLAYOUT_INCLUDED
#include <tlayout.h>
													#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define tagCounterDecArrow 		MakeTag(clsCounter, 1)
#define tagCounterLabel	  		MakeTag(clsCounter, 2)
#define tagCounterIncArrow	  	MakeTag(clsCounter, 3)

#define hlpCounterDecArrow 		tagCounterDecArrow
#define hlpCounterLabel	  		tagCounterLabel	  
#define hlpCounterIncArrow	  	tagCounterIncArrow

typedef OBJECT COUNTER;

/****  Show Style  ****/
#define	csShowCount				0	// show "count" only     (e.g. "24")
#define	csShowCountSlashTotal	1	// show "count/total"    (e.g. "1/24")
#define	csShowCountOfTotal		2	// show "count of total" (e.g. "1 of 24")

typedef struct COUNTER_STYLE {
	U16	numCols		: 4,	// number of columns for shrink-wrap
		show		: 3,	// what to show
		spare		: 9;	// unused (reserved)
} COUNTER_STYLE, *P_COUNTER_STYLE;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew		takes P_COUNTER_NEW, returns STATUS
	category: class message
	Creates a counter window.
*/

typedef struct COUNTER_NEW_ONLY {
	COUNTER_STYLE	style;
	OBJECT			client;		// client to notify
	S32				value;		// initial count
	S32				total;		// total to display
	U32				spare1;		// unused (reserved)
	U32				spare2;		// unused (reserved)
} COUNTER_NEW_ONLY, *P_COUNTER_NEW_ONLY;

#define	counterNewFields	\
	tableLayoutNewFields	\
	COUNTER_NEW_ONLY		counter;

typedef struct COUNTER_NEW {
	counterNewFields
} COUNTER_NEW, *P_COUNTER_NEW;

/****************************************************************************
 msgNewDefaults				takes P_COUNTER_NEW, returns STATUS
	category: class message
	Initializes the COUNTER_NEW structure to default values.

 Zeroes out pArgs->counter and sets
//{
	pArgs->border.style.leftMargin 		= bsMarginNone;
	pArgs->border.style.rightMargin 	= bsMarginNone;
	pArgs->border.style.bottomMargin	= bsMarginSmall;
	pArgs->border.style.topMargin		= bsMarginMedium;

	pArgs->tableLayout.style.growChildWidth		= false;
	pArgs->tableLayout.style.growChildHeight	= false;

	pArgs->counter.style.numCols = 1;
//}
*/
/*
 Default COUNTER_STYLE:
//{
 	numCols		= 1
 	show		= csShowCount
//}
*/


/****************************************************************************
 msgCounterGetStyle		takes P_COUNTER_STYLE, returns STATUS
	Passes back the current style values.
*/
#define msgCounterGetStyle	  	MakeMsg(clsCounter, 1)

/****************************************************************************
 msgCounterSetStyle		takes P_COUNTER_STYLE, returns STATUS
	Sets the style values.

 If style.numCols requires the counter to be wider, clsCounter will self-send
 msgWinLayout to relayout.
*/
#define msgCounterSetStyle	  	MakeMsg(clsCounter, 2)

/****************************************************************************
 msgCounterGetClient		takes P_OBJECT, returns STATUS
	Passes back the current counter client.
*/
#define msgCounterGetClient	  	MakeMsg(clsCounter, 7)

/****************************************************************************
 msgCounterSetClient		takes OBJECT, returns STATUS
	Sets the client.
*/
#define msgCounterSetClient	  	MakeMsg(clsCounter, 8)

/****************************************************************************
 msgCounterGetValue		takes P_S32, returns STATUS
	Passes back the current count value.
*/
#define msgCounterGetValue		  	MakeMsg(clsCounter, 3)

/****************************************************************************
 msgCounterSetValue		takes S32, returns STATUS
	Sets the current counter value.

 If the new value requires the counter to be wider, clsCounter will self-send
 msgWinLayout to relayout.
*/
#define msgCounterSetValue		  	MakeMsg(clsCounter, 4)

/****************************************************************************
 msgCounterGetTotal		takes P_S32, returns STATUS
	Passes back the current total value.
*/
#define msgCounterGetTotal		  	MakeMsg(clsCounter, 11)

/****************************************************************************
 msgCounterSetTotal		takes S32, returns STATUS
	Sets the current total value.

 If the new total value requires the counter to be wider, clsCounter will
 self-send msgWinLayout to relayout.
*/
#define msgCounterSetTotal		  	MakeMsg(clsCounter, 12)

/****************************************************************************
 msgCounterIncr		takes S32, returns STATUS
	Increments the current counter value by adding in pArgs.

 If the new value requires the counter to be wider, clsCounter will self-send
 msgWinLayout to relayout.
*/
#define msgCounterIncr		  	MakeMsg(clsCounter, 5)

/****************************************************************************
 msgCounterGoto		takes S32, returns STATUS
	Sends msgCounterNotify to the counter's client to alter the counter's value.

 clsCounter will send msgCounterNotify to the counter's client with the
 following COUNTER_NOTIFY parameters:
//{
	counter		= self;
	initValue	= current counter value;
	action		= csActionAccept;
	value		= pArgs;
//}
 The client can alter the value parameter to goto a different value, if
 desired.

 A common use for this message is to create a menu with individual menu
 buttons representing particular counter values, and set the (msg, data)
 pair for each menu button to be (msgCounterGoto, desired value) and
 set the menu button's client to be the counter.
*/
#define msgCounterGoto		  	MakeMsg(clsCounter, 9)

/****************************************************************************
 msgCounterGetLabel		takes P_WIN, returns STATUS
	Passes back the counter label window uid.

 The label is an instance of clsMenuButton, and can be given a menu
 by setting the CONTROL_STYLE.previewEnable to true and using
 msgMenuButtonSetMenu.
*/
#define msgCounterGetLabel		  MakeMsg(clsCounter, 10)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                  Messages Counters Send to Clients					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgCounterNotify		takes P_COUNTER_NOTIFY, returns STATUS
	category: client notification
	Sent to the client when an arrow repeats, finishes or cancels.

 If the user presses or continues to hold down on the decrement arrow,
 pArgs->action will be set to csActionDecrement.

 If the user presses or continues to hold down on the increment arrow,
 pArgs->action will be set to csActionIncrement.

 If the user pen's-up over either arrow, pArgs->action will be set to
 csActionAccept.

 If the user drags out of either arrow, pArgs->action will be set to
 csActionCancel.

 For any action, pArgs->value will be the current value of the counter and
 pArgs->initValue will be the initial value of the counter when the first
 csActionIncrement/Decrement was sent out.

 Clients should change pArgs->value to the new desired value.
 Note that clsCounter does not change the value of the counter, other
 than copying back pArgs->value.

 If pArgs->value is not changed by the client, the value of the counter
 will not be changed.  This allows clients to use msgCounterIncr or
 msgCounterSetValue to alter the value during msgCounterNotify.
*/
#define msgCounterNotify		  	MakeMsg(clsCounter, 6)

Enum16(COUNTER_ACTION) {
	csActionIncrement	= 0,	// increment the counter
	csActionDecrement	= 1,	// decrement the counter
	csActionCancel		= 2,	// cancel the increment/decrement
	csActionAccept		= 3,	// accept the increment/decrement 
};

typedef struct COUNTER_NOTIFY {
	OBJECT			counter;	// in: counter calling out
	S32				initValue;	// in: initial value before repeat
	COUNTER_ACTION	action;		// in: what happened
	S32				value;		// in/out: current value
	S32				total;		// in: current total value
	U32				spare1;		// unused (reserved)
	U32				spare2;		// unused (reserved)
} COUNTER_NOTIFY, *P_COUNTER_NOTIFY;

#endif	// COUNTER_INCLUDED
