/****************************************************************************
 File: button.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.54  $
   $Author:   kcatlin  $
     $Date:   09 Mar 1992 19:16:28  $

 This file contains the API definition for clsButton.

 clsButton inherits from clsLabel.

 Buttons are labels, but with input behavior.  Buttons also have a state
 value: on or off.  Buttons notify their client when certain input events
 occur.  clsButton make extensive use of its ancestors display
 capabilities, particularly clsBorder and clsLabel.
****************************************************************************/
#ifndef BUTTON_INCLUDED
#define BUTTON_INCLUDED

													#ifndef LABEL_INCLUDED
#include <label.h>
													#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT			BUTTON;

/****  Contact Styles  ****/
/* Use one of these values in button's style.contact. */
#define	bsContactMomentary		0	// push-on, release-off
#define	bsContactToggle			1	// push-on, push-off
#define	bsContactLockOn			2	// push-on, stays on
//								3	// unused (reserved)

/****  Feedback Styles  ****/
/* Use one of these values in button's style.feedback. */
#define	bsFeedbackInvert		0	// invert on/off
#define	bsFeedbackDecorate		1	// use onDecoration/offDecoration
#define	bsFeedbackNone			2	// no feedback
#define	bsFeedback3D			3	// 3D shadow effect
#define	bsFeedbackBox			4	// boxed outline 
//								5	// unused (reserved)
//								..	// unused (reserved)
//								7	// unused (reserved)

/****  pArgs Styles  ****/
/* Use one of these values in button's style.pArgs. */
#define	bsPargsData				0	// pArgs is data
#define	bsPargsValue			1	// pArgs is current value
#define	bsPargsUID				2	// pArgs is button's UID
//								3	// unused (reserved)

/****  Manager Styles  ****/
/* Use one of these values in button's style.manager. */
#define	bsManagerNone			0	// no manager
#define	bsManagerParent			1	// parent is the manager
#define	bsManagerClient			2	// client is the manager
//								3	// unused (reserved)

typedef struct BUTTON_STYLE {
	U16	contact		 	: 2,	// push behavior
	   	feedback		: 4,	// invert, decorate, etc.
	   	notifyDetail 	: 1,	// notify manager of BeginPreview etc. 
	   	notifyWithMsg	: 1,	// send specified msg & data
	   	on				: 1,	// button state: true == on
	   	manager			: 2,	// button manager style
		pArgs			: 2,	// which pArgs to send with msg
	    halfHeight		: 1,    // half-height borders
	    spare1 			: 2;	// unused (reserved)

	U16	onDecoration 	: 5,	// label decoration when on (see label.h)
		offDecoration   : 5,	// label decoration when off (see label.h)
	    spare  			: 6;	// unused (reserved)
} BUTTON_STYLE, *P_BUTTON_STYLE;

/*
 Default BUTTON_STYLE:
//{
	contact			= bsContactMomentary
	feedback		= bsFeedbackInvert
	onDecoration	= lsDecorationNone
	offDecoration	= lsDecorationNone
	notifyDetail 	= false
	notifyWithMsg	= true
	pArgs			= bsPargsData
	on				= false
	halfHeight 		= false
//}
*/

typedef struct BUTTON_NOTIFY {
	OBJECT	button;	// uid of sender
	MESSAGE	msg;	// defined message or some other data
	U32		data;	// pArgs for message or some other data
	MESSAGE	detail;	// msgButtonBeginPreview, etc.
	U32		spare;	// unused (reserved)
} BUTTON_NOTIFY, *P_BUTTON_NOTIFY;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew		takes P_BUTTON_NEW, returns STATUS
	category: class message
	Creates a button window.

 The rest of this description uses the following abbreviations:
//{
	on 		= pArgs->button.style.on;
	pButton	= &pArgs->button.style;
	pBorder	= &pArgs->border.style,
	pLabel	= &pArgs->label.style,
//}

 If pButton->feedback is bsFeedbackInvert, sets
//{
	pBorder->preview = on;
//}

 If pButton->feedback is bsFeedback3D, sets
//{
	pBorder->join			= bsJoinSquare;
	pBorder->previewAlter	= bsAlterNone;
	pBorder->edge			= bsEdgeTop | bsEdgeLeft;
	pBorder->shadowGap		= bsGapNone;
	pBorder->preview		= on;
	if (on) {
		pBorder->borderInk		= bsInkBlack;
		pBorder->backgroundInk	= bsInkGray66;
		pBorder->shadow			= bsShadowThinWhite;
	} else {
		pBorder->borderInk		= bsInkWhite;
		pBorder->backgroundInk	= bsInkGray33;
		pBorder->shadow			= bsShadowThinGray;
	}
//}

 If pButton->feedback is bsFeedbackDecorate, sets
//{
		pLabel->decoration = on ?
		  pArgs->button.style.onDecoration : 
		  pArgs->button.style.offDecoration;
//}
*/
typedef struct BUTTON_NEW_ONLY {
	BUTTON_STYLE	style;			// overall style
	MESSAGE			msg;			// message to send or other data
	U32				data;			// pArgs for msg or other data
	U16				onCustomGlyph;	// glyph to use for 
									// lsDecorationCustomLeft/Right
	U16				offCustomGlyph;	// glyph to use for 
									// lsDecorationCustomLeft/Right
	U32				spare;			// unused (reserved)
} BUTTON_NEW_ONLY, BUTTON_METRICS,
  *P_BUTTON_NEW_ONLY, *P_BUTTON_METRICS;

#define	buttonNewFields		\
	labelNewFields			\
	BUTTON_NEW_ONLY		button;

typedef struct BUTTON_NEW {
	buttonNewFields
} BUTTON_NEW, *P_BUTTON_NEW;

/****************************************************************************
 msgNewDefaults				takes P_BUTTON_NEW, returns STATUS
	category: class message
	Initializes the BUTTON_NEW structure to default values.

 Zeroes out pArgs->button and sets:
//{
	pArgs->win.flags.input |= inputTip | inputEnter | inputExit;
	pArgs->win.flags.style |= wsFileInline;

	pArgs->border.style.edge = bsEdgeAll;
	pArgs->border.style.join = bsJoinSquare;
	pArgs->border.style.shadow = bsShadowThinBlack;
	pArgs->border.style.borderInk = bsInkGray66;

	pArgs->control.style.previewEnable = true;

	pArgs->label.style.xAlignment = lsAlignCenter;
	pArgs->label.style.yAlignment = lsAlignCenter;

	pArgs->button.style.notifyWithMsg = true;
//}
*/

/****************************************************************************
 msgButtonGetMetrics		takes P_BUTTON_METRICS, returns STATUS
	Passes back the current metrics.
*/
#define msgButtonGetMetrics		MakeMsg(clsButton, 1)

/****************************************************************************
 msgButtonSetMetrics		takes P_BUTTON_METRICS, returns STATUS
	Sets the metrics.

 If style.on changes, the button does the following:

    -:  If style.contact != bsContactMomentary, self-sends
        msgControlSetDirty, true.

    -:  Self-sends msgButtonNotifyManager with msg = msgButtonDone.

    -:  Self-sends msgButtonNotify with detail of msgButtonAcceptPreview.
        This results in either msgButtonNotify or a client-specified
        message to the client.  Alters border and label styles to reflect
        the new "on" value (see msgNew description).

 Changes to style.feedback and style.on/offDecoration result in appropriate
 changes to the Border and Label styles.
*/
#define msgButtonSetMetrics		MakeMsg(clsButton, 2)

/****************************************************************************
 msgButtonGetStyle		takes P_BUTTON_STYLE, returns STATUS
	Passes back the current style values.
*/
#define msgButtonGetStyle		MakeMsg(clsButton, 3)

/****************************************************************************
 msgButtonSetStyle		takes P_BUTTON_STYLE, returns STATUS
	Sets the style values.

 Reacts to changes in style.on and other style values as in
 msgButtonSetMetrics.
*/
#define msgButtonSetStyle		MakeMsg(clsButton, 4)

/****************************************************************************
 msgButtonGetMsg		takes P_MESSAGE, returns STATUS
	Passes back metrics.msg.
*/
#define msgButtonGetMsg			MakeMsg(clsButton, 5)

/****************************************************************************
 msgButtonSetMsg		takes MESSAGE, returns STATUS
	Sets metrics.msg.
*/
#define msgButtonSetMsg			MakeMsg(clsButton, 6)

/****************************************************************************
 msgButtonGetData		takes P_U32, returns STATUS
	Passes back metrics.data.
*/
#define msgButtonGetData		MakeMsg(clsButton, 7)

/****************************************************************************
 msgButtonSetData		takes U32, returns STATUS
	Sets metrics.data.
*/
#define msgButtonSetData		MakeMsg(clsButton, 8)

/****************************************************************************
 msgButtonSetNoNotify		takes BOOLEAN, returns STATUS
	Sets the value of the button (i.e. style.on) without notifying.

 pArgs must be true or false.  The button's manager and client are not
 notified.  Alters border and label styles to reflect new on value (see
 msgNew description).
*/
#define msgButtonSetNoNotify	MakeMsg(clsButton, 17)

/****************************************************************************
 msgButtonButtonShowFeedback		takes BOOLEAN, returns STATUS
	category: self-sent
	Shows the feedback for an on/off button if pArgs is true/false.

 This message is self-sent by clsButton to change the on/off feedback shown
 to the user.  For example, when a button with a contact style of
 bsContactToggle is pressed and msgControlBeginPreview is received,
 clsButton self-sends msgButtonShowFeedback(!style.on) to show the user
 what will happen when the pen is lifted.

 Subclasses can handle the message and show the appropriate feedback for
 the new state.
*/
#define msgButtonShowFeedback	MakeMsg(clsButton, 19)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages Sent to Button's Manager					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgButtonDone		takes UID, returns STATUS
	category: manager notification
	Sent via msgWinSend to the manager when button receives
	msgControlAcceptPreview.
*/
#define msgButtonDone				MakeMsg(clsButton, 16)

/****************************************************************************
 msgButtonBeginPreview		takes UID, returns STATUS
	category: manager notification
	Sent via msgWinSend to the manager when button receives
	msgControlBeginPreview.

 Only sent if style.notifyDetail is true.
*/
#define msgButtonBeginPreview		MakeMsg(clsButton, 9)

/****************************************************************************
 msgButtonUpdatePreview		takes UID, returns STATUS
	category: manager notification
	Sent via msgWinSend to the manager when button receives
	msgControlUpdatePreview.

 Only sent if style.notifyDetail is true.
*/
#define msgButtonUpdatePreview		MakeMsg(clsButton, 10)

/****************************************************************************
 msgButtonRepeatPreview		takes UID, returns STATUS
	category: manager notification
	Sent via msgWinSend to the manager when button receives
	msgControlRepeatPreview.

 Only sent if style.notifyDetail is true.
*/
#define msgButtonRepeatPreview		MakeMsg(clsButton, 11)

/****************************************************************************
 msgButtonCancelPreview		takes UID, returns STATUS
	category: manager notification
	Sent via msgWinSend to the manager when button receives
	msgControlCancelPreview.

 Only sent if style.notifyDetail is true.
*/
#define msgButtonCancelPreview		MakeMsg(clsButton, 12)

/****************************************************************************
 msgButtonAcceptPreview		takes UID, returns STATUS
	category: manager notification
	Sent via msgWinSend to the manager when button receives
	msgControlAcceptPreview.

 Only sent if style.notifyDetail is true.
*/
#define msgButtonAcceptPreview		MakeMsg(clsButton, 13)

/****************************************************************************
 msgButtonNotifyManager		takes P_BUTTON_NOTIFY, returns STATUS
	category: self-sent
	Sent to self when button wants to notify its manager.

 A button responds to this by sending msgWinSend with the following
 WIN_SEND parameters to its manager:
//{
	flags	= wsSendDefault;
	lenSend	= SizeOf(WIN_SEND);
	msg		= pArgs->msg;
	data[0] = pArgs->data;
//}
*/
#define msgButtonNotifyManager		MakeMsg(clsButton, 18)

/****************************************************************************
 msgButtonNotify		takes P_BUTTON_NOTIFY, returns STATUS
	category: client notification
	Sent to self when button wants to notify its client.

 If style.notifyWithMessage is true, pArgs->msg is sent to the button's
 client with the pArgs of pArgs->data or as specified by style.pArgs.

 Otherwise, msgButtonNotify is sent to the button's client with the 
 following BUTTON_NOTIFY parameters:
//{
	button	= self;
	msg		= pArgs->msg;
	data	= pArgs->data;
	detail	= pArgs->detail;
//}
*/
#define msgButtonNotify				MakeMsg(clsButton, 14)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages Defined by Other Classes					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgBorderGetForegroundRGB		takes P_SYSDC_RGB, returns STATUS
	Passes back foreground RGB to use given current visuals.
 
 If style.feedback is bsFeedback3D and border.style.look is bsLookInactive,
 passes back sysDcRGBGray66.  Otherwise, calls ancestor.
*/

/****************************************************************************
 msgControlBeginPreview		takes P_EVENT, returns STATUS
	category: self-sent
	Self-sent when msgPenDown is received.
 
 Button computes new on value according to style.feedback (e.g.
 bsContactToggle results in on = !style.on).

 Alters border and label styles to reflect new on value and self-sends
 msgWinUpdate to repaint immediately.  style.on is not changed.

 If style.contact is not bsContactMomentary, self sends msgControlSetDirty,
 true.

 If style.notifyDetail is true, self-sends msgButtonNotifyManager, which
 results in msgWinSend to the manager.  Also, if
 control.style.previewRepeat is true, self-sends msgButtonNotify which
 results in client notification.
*/

/****************************************************************************
 msgControlUpdatePreview		takes P_EVENT, returns STATUS
	category: self-sent
	Self-sent when msgPenMoveDown is received.

 If style.notifyDetail is true, notifies manager and client as in
 msgControlBeginPreview.
*/

/****************************************************************************
 msgControlRepeatPreview		takes P_EVENT, returns STATUS
	category: self-sent
	Self-sent if style.repeatPreview is true.
	Initial delay is 600ms, then immediate repeat until msgPenUp.

 If style.notifyDetail is true, notifies manager and client as in
 msgControlBeginPreview.
*/

/****************************************************************************
 msgControlCancelPreview		takes P_EVENT, returns STATUS
	category: self-sent
	Self-sent when control.style.previewGrab is false and msgPenExitDown is
	received.

 Clients or subclasses can send this to a control to cancel existing
 preview.

 Alters border and label styles to reflect current style.on value and
 self-sends msgWinUpdate to repaint immediately.  This undoes the visual
 effects of msgControlBeginPreview.

 If style.notifyDetail is true, notifies manager and client as in
 msgControlBeginPreview.
*/

/****************************************************************************
 msgControlAcceptPreview		takes P_EVENT, returns STATUS
	category: self-sent
	Self-sent when msgPenUp is received.

 If gestures are enabled this message is not sent until msgGWinGesture is
 received with xgs1Tap.

 Self-sends msgControlSetValue with on value computed as in
 msgControlBeginPreview.
*/

/****************************************************************************
 msgControlSetValue		takes S32, returns STATUS
	Sets style.on.

 Updates visuals to reflect new on value as in msgButtonSetMetrics.

 Self-sends msgButtonNotifyManager with the following BUTTON_NOTIFY
 parameters (this results in msgWinSend to the manager):
//{
	button	= self;
	msg		= msgButtonDone;
	data	= self;
//}

 Self-sends msgButtonNotify with the following BUTTON_NOTIFY parameters
 (this results in client notification):
//{
	button	= self;
	msg		= metrics.msg;
	data	= metrics.data;
	detail	= msgButtonAcceptPreview;
//}
*/

/****************************************************************************
 msgControlGetValue		takes P_S32, returns STATUS
	Passes back the style.on.
*/
#endif	// BUTTON_INCLUDED
