/****************************************************************************
 File: border.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.68  $
   $Author:   cmeyer  $
     $Date:   13 Mar 1992 19:04:00  $

 This file contains the API for clsBorder.

 clsBorder inherits from clsEmbeddedWin.
 clsBorder supports drawing borders, backgrounds and shadows.
 Support is also provided for resize, drag and top window management.

****************************************************************************/
#ifndef BORDER_INCLUDED
#define BORDER_INCLUDED

													#ifndef WIN_INCLUDED
#include <win.h>
													#endif
													#ifndef EWNEW_INCLUDED
#include <ewnew.h>
													#endif
													#ifndef INPUT_INCLUDED
#include <input.h>
													#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define hlpBorderResizeBottom  	MakeTag(clsBorder, 1)
#define hlpBorderResizeCorner  	MakeTag(clsBorder, 2)
#define hlpBorderResizeRight  	MakeTag(clsBorder, 3)

typedef OBJECT			BORDER;

/****  Edge Styles  ****/
#define	bsEdgeNone					0		// no borders
#define	bsEdgeLeft					flag0	// border on the left
#define	bsEdgeRight					flag1	// border on the right
#define	bsEdgeTop					flag2	// border on the top
#define	bsEdgeBottom				flag3	// border on the bottom

// Borders on all edges
#define	bsEdgeAll					(bsEdgeLeft | bsEdgeRight | \
								 	bsEdgeTop  | bsEdgeBottom)
/****  Join Styles  ****/
#define	bsJoinSquare				0	// right-angle rectangle
#define	bsJoinRound					1	// round corner rectangle
#define	bsJoinEllipse				2	// ellipse instead of rectangle
//									3	// unused (reserved)

/****  Line Styles  ****/
#define	bsLineSingle			0	// solid ink
#define	bsLineDouble			1	// ink-white-ink lines
#define	bsLineMarquee			2	// flowing dashed lines
#define	bsLineDashed			3	// dashed lines
#define	bsLineDoubleMarquee		4	// double flowing dashed lines
#define	bsLineDoubleDashed		5	// double dashed lines
//								6	// unused (reserved)
//								..	// unused (reserved)
//								15	// unused (reserved)

/****  Edge and Background Colors  ****/
#define bsInkTransparent			0	// no ink
#define bsInkBlack					1	// black
#define bsInkGray75					2	// 75% gray
#define bsInkGray66					3	// 66% gray
#define bsInkGray50					4	// 50% gray
#define bsInkGray33					5	// 33% gray
#define bsInkGray25					6	// 25% gray
#define	bsInkWhite					7	// white
#define	bsInkAsIs					8	// use appropriate dc value
#define	bsInkRGB					9	// use custom RGB value
#define	bsInkBackground				10	// use the background ink
//									11	// unused (reserved)
//									..	// unused (reserved)
//									31	// unused (reserved)

/* 
 bsInkExclusive can be or'ed into any ink to indicate that
 the specified ink should only be used if the window exclusively paints
 its pixels.  If the window is transparent or shares clipping with its
 parent, bsInkTransparent will be used (i.e. nothing will be painted).
*/
#define	bsInkExclusive				flag4

/* BorderInk extracts the base ink from a border ink */
#define	BorderInk(ink)				((ink) & 0xF)

/****  Shadow Styles:  drawn on the bottom and right  ****/
#define	bsShadowNone				0	// no shadow
#define	bsShadowThinGray			1	// one line gray
#define	bsShadowThickGray			2	// two line gray
#define	bsShadowThinBlack			3	// one line black
#define	bsShadowThickBlack			4	// two line black
#define	bsShadowThinWhite			5	// one line white
#define	bsShadowThickWhite			6	// two line white
#define	bsShadowCustom				7	// use shadowThickness and shadowInk
//									8	// unused (reserved)
//									..	// unused (reserved)
//									15	// unused (reserved)

/****  Units  ****/
#define	bsUnitsLayout				0	// values are in layout units
#define	bsUnitsDevice				1	// values are in device units
#define	bsUnitsTwips				2	// values are in twips
#define	bsUnitsPoints				BorderUnitsCustom(bsUnits20x, bsUnitsTwips)
										// values are in points = 20 x twips
#define	bsUnitsRules				3	// values are in rules
#define	bsUnitsLines				BorderUnitsCustom(bsUnits20x, bsUnitsRules)
										// values are in lines = 20 x rules
#define	bsUnitsMetric				4	// values are in .01 mm
#define	bsUnitsMil					5	// values are in .001 inch
#define	bsUnitsFitWindow			6	// values not specified --
										// compute to fit window
#define	bsUnitsFitWindowProper		7	// values not specified --
										// compute to fit window w/proper
										// aspect ratio
//									8	// unused (reserved)
//									..	// unused (reserved)
//									15	// unused (reserved)

/****  Units Multiplier  ****/
/*
 These values can be used with BorderUnitsCustom()  to produce new units
 e.g. BorderUnitsCustom(bsUnits20x, bsUnitsTwips) indicates units are
 20 x twips
*/
#define	bsUnits1x					0	//    1x
#define	bsUnits20x					1	//   20x
#define	bsUnits100x					2	//  100x
#define	bsUnits1000x				3	// 1000x

#define	BorderUnitsCustom(mult, units)	((mult << 4) | (units))

/* macros to extract base units and multiplier values */
#define	BorderUnits(units)			((units) & 0x0F)
#define	BorderUnitsMult(units)		((units) >> 4)

/****  Common Margin Values  ****/
#define	bsMarginNone				0	// no inner margin
#define	bsMarginSmall				1	// 1 unit
#define	bsMarginMedium				2	// 2 units
#define	bsMarginLarge				8	// 8 units

/****  Resize Handles  ****/
#define	bsResizeNone				0		// no resize handles
#define	bsResizeCorner				flag0	// lower-right corner
#define	bsResizeBottom				flag1	// center-bottom
#define	bsResizeRight				flag2	// center-right
#define	bsResizeAll					(bsResizeCorner | bsResizeBottom |\
									bsResizeRight)

/****  Drag Styles  ****/
#define	bsDragNone					0		// no drag
#define	bsDragHoldDown				1		// drag on penHoldDown
#define	bsDragDown					2		// drag on penDown
#define	bsDragMoveDown				3		// drag on penMoveDown beyond range

/****  Top Styles  ****/
#define	bsTopNone					0		// never top the window
#define	bsTopUp						1		// top on penUp
#define	bsTopDrag					2		// top after drag
//									3		// unused (reserved)

/****  Shadow Gap Styles  ****/
#define	bsGapNone					0		// no shadow gap
#define	bsGapWhite					1		// cleared to white
#define	bsGapTransparent			2		// unpainted
//									3		// unused (reserved)

/****  Look Styles  ****/
#define	bsLookActive				0		// usually black foreground color
#define	bsLookInactive				1		// usually gray66 foreground color
//									2		// unused (reserved)
//									3		// unused (reserved)

/****  Alter Styles for preview and selected  ****/
#define	bsAlterNone					0		// don't alter anything
#define	bsAlterBackground			1		// alter the background ink
#define	bsAlterBorders				2		// alter the border ink
//									3		// unused (reserved)


typedef struct BORDER_STYLE {
	U16 edge			: 4,	// edges to border
		top				: 2,	// top style (e.g. bsTopUp)
		drag			: 2,	// drag style (e.g bsDragDown)
		resize			: 5,	// resize handles (e.g. bsResizeCorner|bsResizeBottom)
   		maskBorders		: 1,	// mask out edge, shadow, resize
   		getDeltaWin		: 1,	// use msgBorderProvideDeltaWin
		spare1			: 1;	// unused (reserved)

	U16	leftMargin		: 8,	// margin in marginUnits
		rightMargin		: 8;	// 		"	

	U16	bottomMargin	: 8,	// 		"
		topMargin		: 8;	// 		"

	U16	borderInk 		: 6,	// edge line color
		backgroundInk	: 6,	// background fill color
		previewAlter 	: 2,	// what to alter when previewing
		selectedAlter 	: 2;	// what to alter when selected

	U16	marginInk		: 6,	// ink for margin area (not implemented)
		marginUnits		: 6,	// units for left, right, bottom, top margins
		preview			: 1,	// true/false
		selected		: 1,	// true/false
		look			: 2;	// active/inactive

	U16	shadow			: 4,	// type of shadow
		shadowGap		: 2,	// type of shadow gap
		shadowThickness	: 8,	// custom shadow thickness, in lineUnits
		spare4			: 2;	// unused (reserved)

	U16	shadowInk		: 6,	// custom shadow ink
		lineStyle		: 4,	// edge line style (e.g. bsLineSingle)
		spare5			: 6;	// unused (reserved)

	U16	lineUnits		: 6,	// units for lineThickness and shadowThickness
		lineThickness	: 8,	// line thickness, in lineUnits
		join			: 2;	// how edges join together

	U16	propagateVisuals: 1,	// propagate changes in visuals to children
		notifyVisuals	: 1,	// send msgBorderSetVisuals to observers
		spare3			: 14;	// unused (reserved)

} BORDER_STYLE, *P_BORDER_STYLE;

/*
 *	Default BORDER_STYLE:
 *	edge			= bsEdgeNone
 *	join			= bsJoinSquare
 *	lineStyle		= bsLineSingle
 *	marginUnits		= bsUnitsLayout
 *	resize			= bsResizeNone
 *	move			= bsDragNone
 *  top				= bsTopNone
 *	leftMargin		= bsMarginNone
 *	rightMargin		= bsMarginNone
 *	bottomMargin 	= bsMarginNone
 *	topMargin		= bsMarginNone
 *	look			= bsLookActive
 *	preview			= false
 *	selected		= false
 *	propagateVisuals= false
 *	notifyVisuals	= false
 *	getDeltaWin		= false
 *	maskBorders		= false
 *	borderInk		= bsInkBlack
 *  backgroundInk	= bsInkWhite
 *	marginInk		= bsInkBackground
 *	shadow			= bsShadowNone
 *	shadowGap		= bsGapWhite
 *	shadowInk		= bsInkBlack
 *	shadowThickness	= 1
 *	lineUnits		= bsUnitsLines
 *	lineThickness	= 1
 *	previewAlter	= bsAlterNone
 *	selectedAlter	= bsAlterNone
 */

/*
 Input event flags returned in INPUT_EVENT.flags
 indicates event was used to move/resize
*/
#define evBorderTaken		evFlag0

/*
 Tags used by resize or drag tracks.
 These will be the tags in TRACK_METRICS of msgTrackProvideMetrics and
 msgTrackDone.
*/
#define	tagBorderResize		MakeTag(clsBorder, 4)
#define	tagBorderDrag		MakeTag(clsBorder, 5)

/****************************************************************************
 msgNew		takes P_BORDER_NEW, returns STATUS
	category: class message
	Creates a border window.

 If pArgs->border.style.maskBorders is true, style.resize is treated as though
 it is bsResizeNone, style.edge is treated as though it is bsEdgeNone, and
 style.shadow is treated as though it is bsShadowNone.

 If pArgs->style.resize is not bsResizeNone, pArgs->win.flags.input is altered
 to enable events needed for resizing.

 If pArgs->style.drag is not bsDragNone, pArgs->win.flags.input is altered
 to enable events needed for draging.

 If pArgs->style.top is not bsTopNone, pArgs->win.flags.input is altered
 to enable events needed for topping.
*/

typedef struct BORDER_NEW_ONLY {
	BORDER_STYLE			style;	// overall style
	U32						spare1;	// unused (reserved)
	U32						spare2;	// unused (reserved)
} BORDER_NEW_ONLY, BORDER_METRICS,
  *P_BORDER_NEW_ONLY, *P_BORDER_METRICS;

#define	borderNewFields		\
	embeddedWinNewFields 	\
	BORDER_NEW_ONLY			border;

typedef struct {
	borderNewFields
} BORDER_NEW, *P_BORDER_NEW;

/****************************************************************************
 msgNewDefaults				takes P_BORDER_NEW, returns STATUS
	category: class message
	Initializes the BORDER_NEW structure to default values.

 Zeroes out pNew->border and sets...

//{
	pArgs->win.flags.style			   |= wsSendFile;

	pArgs->border.style.shadowInk		= bsInkBlack;
	pArgs->border.style.borderInk		= bsInkBlack;
	pArgs->border.style.marginInk		= bsInkBackground;
	pArgs->border.style.backgroundInk	= bsInkWhite;
	pArgs->border.style.lineUnits		= bsUnitsLines;
	pArgs->border.style.lineThickness	= 1;
	pArgs->border.style.shadowThickness = 1;
	pArgs->border.style.shadowGap		= bsGapWhite;

	pArgs->border.style.previewAlter	= bsAlterNone;
	pArgs->border.style.selectedAlter	= bsAlterNone;
//}
*/

/****************************************************************************
 msgBorderGetStyle	takes P_BORDER_STYLE, returns STATUS
	Passes back the current style values.

*/
#define msgBorderGetStyle		MakeMsg(clsBorder, 1)

/****************************************************************************
 msgBorderSetStyle	takes P_BORDER_STYLE, returns STATUS
	Sets all of the style values.

 Self-sends msgWinDirtyRect(pNull) if painting styles change.
 If only the edge painting style changes, self-sends msgWinDirtyRect 
 with pArgs specifying the rectangle around each border.

 Self-sends msgWinSetLayoutDirty(true), if new style results in new layout.

 If style.propagateVisuals is true, and propagateVisuals or any of the
 visual styles (look, backgroundInk, previewAlter, selectedAlter, preview,
 or selected) change, msgBorderSetVisuals(pArgs) is sent to each child of
 self.

 If style.notifyVisuals is true and notifyVisuals or any of the visual styles
 change, msgNotifyObservers is self-sent with the following
 OBJ_NOTIFY_OBSERVERS parameters:
	msg		= msgBorderSetVisuals;
	pArgs	= pointer to new style struct;
	lenSend	= SizeOf(BORDER_STYLE);
*/
#define msgBorderSetStyle		MakeMsg(clsBorder, 2)

/****************************************************************************
 msgBorderSetStyleNoDirty	takes P_BORDER_STYLE, returns STATUS
	Sets all of the style values.

 This message is the same as msgBorderSetStyle, except msgWinDirtyRect or
 msgWinSetLayoutDirty will not be self-sent, even if they new style
 parameters require repaint or relayout.
*/
#define msgBorderSetStyleNoDirty	MakeMsg(clsBorder, 26)

/****************************************************************************
 msgBorderGetLook		takes P_U16, returns STATUS
	Passes back value of style.look.

*/
#define msgBorderGetLook 		MakeMsg(clsBorder, 13)

/****************************************************************************
 msgBorderSetLook		takes U16 (bsLook...), returns STATUS
	Sets style.look as in msgBorderSetStyle.

*/
#define msgBorderSetLook 		MakeMsg(clsBorder, 12)

/****************************************************************************
 msgBorderSetPreview		takes BOOLEAN, returns STATUS
	Sets style.preview as in msgBorderSetStyle.

*/
#define msgBorderSetPreview		MakeMsg(clsBorder, 8)

/****************************************************************************
 msgBorderGetPreview		takes P_BOOLEAN, returns STATUS
	Passes back value of style.preview.

*/
#define msgBorderGetPreview		MakeMsg(clsBorder, 9)

/****************************************************************************
 msgBorderSetSelected		takes BOOLEAN, returns STATUS
	Sets style.selected as in msgBorderSetStyle.

*/
#define msgBorderSetSelected	MakeMsg(clsBorder, 16)

/****************************************************************************
 msgBorderGetSelected		takes P_BOOLEAN, returns STATUS
	Passes back value of style.selected.

*/
#define msgBorderGetSelected	MakeMsg(clsBorder, 17)

/****************************************************************************
 msgBorderPropagateVisuals		takes nothing, returns STATUS
	Propagates visuals to children.

 Sends msgBorderSetVisuals(&style), where style is self's current
 style, to each child.
*/
#define msgBorderPropagateVisuals	MakeMsg(clsBorder, 15)

/****************************************************************************
 msgBorderSetDirty	takes BOOLEAN, returns STATUS
	Sends msgBorderSetDirty(pArgs) to each child.

 clsBorder will pass this message along to each of its children.
 Child windows can alter their visuals to display a clean/dirty look.
 For example, clsControl will self-send msgControlSetDirty(pArgs) when
 receiving this message.
*/
#define msgBorderSetDirty		MsgNoError(MakeMsg(clsBorder, 37))

/****************************************************************************
 msgBorderGetDirty	takes P_BOOLEAN, returns STATUS
 	Passes back true if any child responds to msgBorderGetDirty with true;
	otherwise passes back false.

 clsBorder will pass this message along to each of its children.
 The first child that responds with true will result in an answer of true.
 If no children are dirty, or there are no children, false will be returned.

 This message can be used to check the overall dirty/clean visual state
 of a tree of border windows.  clsControl will respond by passing back the
 value of visual dirty bit, style.dirty.
*/
#define msgBorderGetDirty		MsgNoError(MakeMsg(clsBorder, 38))

/****************************************************************************
 msgBorderGetForegroundRGB		takes P_SYSDC_RGB, returns STATUS
	Passes back foreground RGB to use given current visuals.
 
 Subclasses should use this message to determine the correct foreground
 color to use.  For example, clsLabel will self-send
 msgBorderGetForegroundRGB in its response to msgWinRepaint to make sure and
 get the correct foreground color.

*/
#define msgBorderGetForegroundRGB		MakeMsg(clsBorder, 27)

/****************************************************************************
 msgBorderGetBackgroundRGB		takes P_SYSDC_RGB, returns STATUS
	Passes back background RGB to use given current visuals.

*/
#define msgBorderGetBackgroundRGB		MakeMsg(clsBorder, 28)

/****************************************************************************
 msgBorderInkToRGB		takes P_SYSDC_RGB, returns STATUS
	Maps ink value (bsInkGray66, etc.) to RGB.

 For example, bsInkGray66 maps to sysDcRGBGray66.
*/
#define	msgBorderInkToRGB	   			MakeMsg(clsBorder, 29)

/****************************************************************************
 msgBorderRGBToInk		takes P_SYSDC_RGB, returns STATUS
	Maps RGB value to ink (bsInkGray66, etc).

 For example, sysDCRGBGray66 maps to bsInkGray66.

 If pArgs has no matching ink value, bsInkTransparent is passed back.
*/
#define	msgBorderRGBToInk	   			MakeMsg(clsBorder, 30)

/****************************************************************************
 msgBorderConvertUnits		takes P_BORDER_UNITS, returns STATUS
	catagory:	class or instance message
	Converts values from one unit to another.

 This message can be sent to clsBorder or an instance of clsBorder.
 clsBorder will convert pArgs->size from pArgs->fromUnits to pArgs->toUnits.
 If bsUnitsDevice is specified, pArgs->win should be set to a window on the
 corresponding device.
*/
#define msgBorderConvertUnits		MakeMsg(clsBorder, 39)

typedef struct BORDER_UNITS {
	WIN		win;		// in: window on target device
	U16		fromUnits;	// in: units for initial size.w/h
	U16		toUnits;	// in: units for final size.w/h
	SIZE32	size;		// in/out: initial/converted value
	U32		spare;		// unused (reserved)
} BORDER_UNITS, *P_BORDER_UNITS;

/****************************************************************************
 msgBorderSetVisuals		takes P_BORDER_STYLE, returns STATUS
	Sets only the visual fields from pArgs.

 Sets style.look, style.preview, and style.selected from pArgs as in
 msgBorderSetStyle.

 If style.backgroundInk is not currently bsInkTransparent, sets
 style.backgroundInk from pArgs as in msgBorderSetStyle.
*/
#define msgBorderSetVisuals	   		MakeMsg(clsBorder, 22)

/****************************************************************************
 msgBorderGetBorderRect		takes P_RECT32, returns STATUS
	Passes back the rect on the border.

 The first pixel of this rect is on the border.  This is the rectangle on
 which the border edges will be drawn, which is outside the inner margin.
 pArgs is in device units.
*/
#define msgBorderGetBorderRect		MakeMsg(clsBorder, 3)

/****************************************************************************
 msgBorderInsetToBorderRect		takes P_RECT32, returns STATUS
	Assumes given rect is window bounds, insets to border rect as in
	msgBorderGetBorderRect.

 You can send this message to determine where the border rect would be
 with the given bounds.

 clsBorder will self-send this message during msgWinRepaint to determine
 the rect on which the border edges should be drawn.

 pArgs should be in device units.
*/
#define msgBorderInsetToBorderRect	MakeMsg(clsBorder, 7)

/****************************************************************************
 msgBorderGetInnerRect		takes P_RECT32, returns STATUS
	Passes back the rect after the inner margin.

 The first pixel of this rect is inside the shadow, border edges and margin
 area.  This is the outer-most usable area. pArgs is in device units.
 Subclasses should use this message to determine the area available to draw
 in after clsBorder has drawn all the shadows and borders.
*/
#define msgBorderGetInnerRect		MakeMsg(clsBorder, 4)

/****************************************************************************
 msgBorderInsetToInnerRect		takes P_RECT32, returns STATUS
	Assumes given rect is window bounds, insets to inner rect as in
	msgBorderGetInnerRect.

*/
#define msgBorderInsetToInnerRect	MakeMsg(clsBorder, 18)

/****************************************************************************
 msgBorderGetMarginRect		takes P_RECT32, returns STATUS
	Passes back the rect after the border.

 The first pixel of this rect is the start of the margin area.
 pArgs is in device units.
*/
#define msgBorderGetMarginRect		MakeMsg(clsBorder, 31)

/****************************************************************************
 msgBorderInsetToMarginRect		takes P_RECT32, returns STATUS
	Assumes given rect is window bounds, insets to margin rect as in
	msgBorderGetMarginRect.

*/
#define msgBorderInsetToMarginRect	MakeMsg(clsBorder, 35)

/****************************************************************************
 msgBorderGetOuterSize		takes P_SIZE32, returns STATUS
	Passes back the sum of the border, margin and shadow sizes for width 
	and height.
 
 Values are in device units.  Subclasses can use this message to determine
 the space needed for the border area.  For example, clsLabel will use this
 number to compute its total shrink-wrap size.
*/
#define msgBorderGetOuterSize		MakeMsg(clsBorder, 5)

/****************************************************************************
 msgBorderGetOuterSizes		takes P_RECT32, returns STATUS
	Passes back the breakdown of the outer size requirements.

 OuterSizes are insets from outer edge to inner rect.
 Note that this is not a true rectangle, each field (x, y, w, h) is
 a distance from the outer edge.
 The sum x+w is equivalent to the OuterSize w, the sum y+h is
 equivalent to the OuterSize h. Values are in device units.
*/
#define msgBorderGetOuterSizes		MakeMsg(clsBorder, 36)

/****************************************************************************
 msgBorderGetOuterOffsets		takes P_RECT32, returns STATUS
	Passes back the distance from the outer edge to the border rect in
	each dimension.

 OuterOffsets are insets from outer edge to inner rect.
 Note that this is not a true rectangle, each field (x, y, w, h) is
 a distance from the outer edge.

 Values are in device units.

 This message may be subclassed to return the visual outer offsets.
 For example, clsFrame will return the outer offsets to the frame border
 window.

*/
#define msgBorderGetOuterOffsets	MakeMsg(clsBorder, 25)

/****************************************************************************
 msgBorderXOR		takes U16, returns STATUS
	Sets the raster-op to XOR and paints the background.

 The U16 passed in is used as backgroundInk.  Using pArgs of
 bsInkWhite yields a true XOR, bsInkGray66 gives a graying effect.
*/
#define msgBorderXOR				MakeMsg(clsBorder, 33)

/****************************************************************************
 msgBorderPaint		takes VOID, returns STATUS
	Paints the border, background, shadow, etc. using msgWinBeginPaint.

 See Also
	msgBorderXOR
	msgBorderPaintForeground
*/
#define msgBorderPaint				MakeMsg(clsBorder, 34)

/****************************************************************************
 msgBorderProvideDeltaWin		takes P_WIN, returns STATUS
	catagory: ancestor window responsibility
	Receiver must provide window to be dragged, resized or topped.

 clsBorder will respond by self-sending msgWinSend with the following
 WIN_SEND parameters:

//{
	ws.flags = wsSendDefault;
	ws.lenSend = SizeOf(WIN_SEND);
	ws.msg = msgBorderProvideDeltaWin;
	ws.data[0] = objNull;
//}

 *pArgs will be set to ws.data[0].
 
 This message is used by clsBorder if style.getDeltaWin is true to determine
 which window to drag/resize/top.

 See Also
 	msgWinSend
	msgInputEvent
*/
#define msgBorderProvideDeltaWin	MakeMsg(clsBorder, 23)

/****************************************************************************
 msgBorderProvideBackground		takes P_BORDER_BACKGROUND, returns STATUS
	catagory: subclass responsibility
	Receiver must provide rect and ink for drawing background.

 Self-sent during msgWinRepaint if style.preview or style.selected is
 true.  pArgs defaults to current border rect, background and border inks.

 A subclass can catch this message and change any of the parameters.
 For example, clsMenuButton will alter the background rect if the menu button
 has a top or bottom border on, to back away previewing feedback from the
 border edge.
*/

typedef struct BORDER_BACKGROUND {
	RECT32		rect;		// in/out: background rect to fill
	U16			ink;		// in/out: ink to fill with (e.g. bsInkWhite)
	U16			borderInk;	// in/out: ink to draw border with (e.g. bsInkBlack)
	U32			spare;		// unused (reserved)
} BORDER_BACKGROUND, *P_BORDER_BACKGROUND;

#define msgBorderProvideBackground	MakeMsg(clsBorder, 24)

/****************************************************************************
 msgBorderPaintForeground		takes VOID, returns STATUS
	catagory: subclass window responsibility
	Receiver must paint the foreground, if any.

 clsBorder never sends this message.  A subclass may send this message
 to force an ancestor class (e.g. clsLabel) to paint the foreground.

 clsBorder responds by doing nothing and returning stsOK.

 See Also
 	msgBorderPaint
	msgBorderXOR
*/
#define msgBorderPaintForeground	MakeMsg(clsBorder, 32)


/****************************************************************************
 msgBorderFlash	takes VOID, returns STATUS
	Flashes self's window by drawing a thick border and erasing it.

 clsBorder will flash a border around self's window.  This is used by
 msgBorderTop to hilight a window that is already on top.

 See Also
 	msgBorderTop
*/
#define msgBorderFlash	MakeMsg(clsBorder, 40)

/****************************************************************************
 msgBorderTop takes U32, returns STATUS
	Tops the border window with optional UI feedback and/or bottoming.

 If self is not already on top of its siblings, clsBorder will bring self
 to top.

 If pArgs has bsTopFlash on and self is already on top, clsBorder
 will self-send msgBorderFlash to flash a border around self.

 If pArgs has bsTopBottom on and self is already on top, clsBorder
 will re-insert self at the "bottom".  The bottom is defined as
 the first child of the mainWin of theDesktop.  If theDesktop does
 not exist, or it has no mainWin, self is placed at the bottom of its
 sibling stack.  If self is not a sibling of the mainWin of
 theDesktop, nothing is done.

 See Also
 	msgBorderFlash
	
*/

#define	bsTopFlash		((U32)flag0)	// msgBorderFlash if already on top
#define	bsTopBottom		((U32)flag1)	// send to bottom if already on top

#define msgBorderTop	MakeMsg(clsBorder, 41)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from other classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgWinSend 	takes WIN_SEND, returns STATUS
    Sends a message up a window ancestry chain.

 If pArgs->msg is msgBorderProvideDeltaWin and style.getDeltaWin is true,
 clsBorder will set pArgs->data[0] to self and return stsOK.  This will
 result in self being the window that is dragged/resized/topped.
*/

/****************************************************************************
 msgInputEvent 					takes P_INPUT_EVENT, returns STATUS
	Notification of an input event. 

 clsBorder will respond to input events that trigger dragging, resizing,
 or topping.

 If pArgs->devCode is msgPenHoldTimeout and style.drag is bsDragHoldDown,
 or pArgs->devCode is msgPenDown and style.drag is bsDragDown,
 or pArgs->devCode is msgPenMoveDown and style.drag is bsDragMoveDown and the
 pen has moved beyond a small threshold since the last msgPenDown, the
 following is done:

	msgGWinAbort(pNull) is self-sent to terminate any gesture in progress.

	If style.getDeltaWin is true, msgBorderProvideDeltaWin is self-sent
	via msgWinSend to determine the window to be dragged.  Otherwise,
	self is used as the window to be dragged.

	msgTrackProvideMetrics is sent to the window to be dragged with
	TRACK_METRICS parameters as follows:

		msgNewDefaults is sent to clsTrack to initialize a TRACK_METRICS
		struct and then:

//{
		style.startThickness	= tsThicknessDouble;
		win						= parent of window to be dragged;
		client					= self;
		clientData				= window to be dragged;
		initRect				= bounds of window to be dragged;
		constrainRect.size		= size of window to be dragged;
		keepRect				= rect around grabbed point;
		tag						= tagBorderDrag;
//}

	An instance of clsTrack is created and started via msgTrackStart.

 If pArgs->devCode is msgPenUp and style.top is bsTopUp and
 gWin.style.gestureEnable is false, a window to be topped is determined
 as in the window to be dragged above, and msgBorderTop(bsTopBottom) is
 sent to the window to bring it to top (or take it to bottom if already
 on top).

 If pArgs->devCode is one of msgPenInProxUp, msgPenEnterUp, or msgPenMoveUp,
 and style.resize is not bsResizeNone, and pArgs->xy is in one of the resize
 handle areas, the following is done:

	msgGWinAbort(pNull) is self-sent to terminate any gesture in progress.

	A window to be resized is determined as in the window to be dragged above,
	and an instance of clsGrabBox is created with this window as its client.
	The grabBox is sent msgGrabBoxShow(true) to start the resize feedback.

 If a drag or resize is done, pArgs->flags will have evBorderTaken turned on
 to indicate that clsBorder "took" the event.
*/

/****************************************************************************
 msgTrackDone					takes P_TRACK_METRICS, returns STATUS
	category: client notification
	Sent by a tracker when it's done.

 If pArgs->tag is not tagBorderDrag, nothing is done and the message
 is passed to ancestor.

 Otherwise, clsBorder assumes pArgs->clientData is a window to be dragged
 and sends msgWinDelta to this window to change its origin to one based
 on pArgs->rect.origin.

 If style.top is bsTopDrag, the window to be dragged is also topped
 (brought to front) by sending it msgBorderTop(0).
*/

/****************************************************************************
 msgTimerNotify takes P_TIMER_NOTIFY, returns nothing
	category: advisory message
	Notifies the client that the timer request has elapsed.	 

 If self's lineStyle is bsLineMarquee or bsLineDoubleMarquee, clsBorder
 will animate the marquee and set the timer again.
*/

/****************************************************************************
 msgSelSelect	takes nothing, returns STATUS
	Sets self to be the selection.

 clsBorder responds by self-sending msgBorderSetSelected(true).
*/

/****************************************************************************
 msgSelYield	takes BOOLEAN, returns STATUS
	The Selection Manager requires the release of the selection.

 clsBorder responds by self-sending msgBorderSetSelected(false).
*/

/****************************************************************************
 msgGWinGesture:	takes P_GWIN_GESTURE, returns STATUS
	Called to process the gesture.

 If pArgs->msg is xgs1Tap and style.top is bsTopUp, a window to be topped
 is determined and topped as in response to the input event msgPenUp.

 If pArgs->msg is xgsQuestion and style.resize is not bsResizeNone and
 pArgs->hotPoint falls over one of the resize handle areas, quick help
 for the resize handle is shown.

 See Also
 	msgInputEvent
*/

/****************************************************************************
 msgWinRepaint takes nothing, returns STATUS
    category: descendant responsibility
    Tells a window to repaint itself.

 clsBorder responds by painting the background, shadow, resize handles,
 and border edges.

 msgBorderInsetToBorderRect will be self-sent with a default of the current
 window bounds to allow the subclass to alter the rect on which the border
 will be drawn.

 If style.preview or style.selected are true, msgBorderProvideBackground
 is self-sent with the following BORDER_BACKGROUND parameters:
	rect	= rectangle on which the border will be drawn, in device units;
	ink		= backgroundInk to be used;

 The resulting rect and ink are used during painting.

 If any of the specified inks have bsInkExclusive or-ed in, and the border
 window does not exclusively paint the pixels in its window, bsInkTransparent
 will be used.  The test for a window exclusively painting the pixels in its
 window is as follows:

//{
 	define selfStyle to be self's window style flags
	define parentStyle to be parent's window style flags

	if (selfStyle & wsTransparent)
		return false;

	if (selfStyle & (wsClipSiblings | wsClipChildren))
		return true;

	if (!(selfStyle & wsParentClip))
		return true;

	if (parentStyle & wsTransparent)
		return true;

	if (parentStyle & wsClipChildren)
		return true;

	return false;
//}

 If any of the specified inks are bsInkTransparent, nothing will be painted
 for that feature (e.g. backgroundInk of bsInkTransparent results in no
 paint on the background).
*/

/****************************************************************************
 msgScrollWinProvideDelta		takes P_SCROLL_WIN_DELTA, returns STATUS
	category: descendant/client responsibility
	Self-sent when scrollWin.style.getDelta is set to true so that descendant
	or client can normalize the scroll if desired.

 clsBorder responds by computing a new origin based on pArgs->action
 and normalizing to prevent scrolling into part of a row or column.

 clsBorder will enumerate the leaf-level children and try to compute
 the row/column structure from the placement of the children.

 Normalization will occur in the direction of the scroll.  For example,
 if the scroll action is moving upward (e.g sbLineUp), normalization will
 occur at the top of the view.
*/

#endif	// BORDER_INCLUDED
