(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Tue Jun 16 12:56:33 PDT 1992 by muller                   *)
(*      modified on Fri Mar 27 02:25:03 1992 by steveg   *)
(*      modified on Sun Nov 17 14:53:10 PST 1991 by mhb      *)

MODULE PaintOpCache;

IMPORT PaintOp, RefanyToRefanyTable, RGB, Word;

TYPE
  Cache = MUTEX OBJECT
    t: RefanyToRefanyTable.T;
  END;
  
VAR
  tTable  := NEW(Cache, t := RefanyToRefanyTable.New(THash, TEqual, 16));
  cqTable := NEW(Cache, t := RefanyToRefanyTable.New(PairHash, PairEqual, 16));
  csTable := NEW(Cache, t := RefanyToRefanyTable.New(PairHash, PairEqual, 16));


TYPE
  TEntry = REF RECORD
    r, g, b: REAL;
    mode: PaintOp.Mode;
    gray: REAL;
    bw: PaintOp.BW;
  END;

  PairEntry = REF RECORD 
    bg, fg: PaintOp.T;  
  END;


PROCEDURE FromRGB(
   rgb: RGB.T; 
   mode := PaintOp.Mode.Normal;
   gray := -1.0;
   bw := PaintOp.BW.UseIntensity): PaintOp.T =
  BEGIN
    RETURN FromRGBCoords(rgb[0], rgb[1], rgb[2], mode, gray, bw)
  END FromRGB;

PROCEDURE FromRGBCoords(
   r, g, b: REAL; 
   mode := PaintOp.Mode.Normal;
   gray := -1.0;
   bw := PaintOp.BW.UseIntensity): PaintOp.T =
  VAR
    e := NEW(TEntry, r:=r, g:=g, b:=b, mode:=mode, gray :=gray, bw := bw);
    value: REFANY;
    paintOp: REF PaintOp.T;
  BEGIN
    LOCK tTable DO
      IF tTable.t.in(e, value) THEN
        RETURN NARROW(value, REF PaintOp.T)^
      ELSE
        paintOp := NEW(REF PaintOp.T);
        paintOp^ := PaintOp.FromRGB(e.r, e.g, e.b, e.mode, e.gray, e.bw);
        EVAL tTable.t.put(e, paintOp);
        RETURN paintOp^;
      END
    END;
  END FromRGBCoords;

PROCEDURE MakeColorQuad (bg, fg: PaintOp.T): PaintOp.ColorQuad =
  VAR
    e :=   NEW(PairEntry, bg := bg, fg := fg);
    value: REFANY;
    cq:    PaintOp.ColorQuad;
  BEGIN
    LOCK cqTable DO
      IF cqTable.t.in(e, value) THEN
        RETURN value;
      ELSE
        cq := PaintOp.MakeColorQuad(bg, fg);
        EVAL cqTable.t.put(e, cq);
        RETURN cq;
      END
    END
  END MakeColorQuad;

PROCEDURE MakeColorScheme (bg, fg: PaintOp.T): PaintOp.ColorScheme =
  VAR
    e :=   NEW(PairEntry, bg := bg, fg := fg);
    value: REFANY;
    cs:    PaintOp.ColorScheme;
  BEGIN
    LOCK csTable DO
      IF csTable.t.in(e, value) THEN
        RETURN value;
      ELSE
        cs := PaintOp.MakeColorScheme(bg, fg);
        EVAL csTable.t.put(e, cs);
        RETURN cs;
      END
    END
  END MakeColorScheme;


(****************************** Table Methods ******************************)

CONST
  Multiplier =  -1664117991;
      (* = LOOPHOLE( Round( .6125423371 * 2^32 ), INTEGER ) *)

(* FIX THIS WHEN STOLFI FIXES REAL!!!!!!! *)
PROCEDURE RealHash(r: REAL): INTEGER =
  BEGIN RETURN ROUND(r) MOD 311117 END RealHash;

PROCEDURE THash (key: REFANY): Word.T =
  VAR
    e      := NARROW(key, TEntry);
    result := 0;
  BEGIN
    result := result * Multiplier + RealHash(e.r);
    result := result * Multiplier + RealHash(e.g);
    result := result * Multiplier + RealHash(e.b);
    result := result * Multiplier + ORD(e.mode);
    result := result * Multiplier + RealHash(e.gray);
    result := result * Multiplier + ORD(e.bw);
    RETURN result;
  END THash;

PROCEDURE TEqual (key1, key2: REFANY): BOOLEAN =
  VAR
    e1 := NARROW(key1, TEntry);
    e2 := NARROW(key2, TEntry);
  BEGIN
    RETURN e1^ = e2^
  END TEqual;

PROCEDURE PairHash (key: REFANY): Word.T =
  VAR e := NARROW(key, PairEntry);
  BEGIN
    RETURN e.bg.op * Multiplier + e.fg.op;
  END PairHash;

PROCEDURE PairEqual (key1, key2: REFANY): BOOLEAN =
  VAR
    e1 := NARROW(key1, PairEntry);
    e2 := NARROW(key2, PairEntry);
  BEGIN
    RETURN e1^ = e2^
  END PairEqual;

BEGIN
END PaintOpCache.


