(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Tue Jun 16 12:55:24 PDT 1992 by muller                   *)
(*      modified on Wed Jun  3 10:39:45 1992 by mhb    *)
(*      modified on Fri Nov 29 15:48:45 PST 1991 by meehan *)
(*      modified on Thu Sep 12 10:04:53 PDT 1991 by steveg *)

(* The "FlexShape" interface defines a {\em flexible size} that
   is is used to compute a VBT's size based on a {\em natural}
   size with some {\em stretch} and {\em shrink}.  All units are
   specified in points (see the "Pts" interface).  This interface
   is used by "FlexVBT" to compute a "VBT.SizeRange" for a VBT as
   follows:

|       preferred := natural,
|       lo        := natural - shrink, and
|       hi        := natural + stretch + 1.

   However, if any of these values are left unspecified, a size
   range for VBT v is computed consistent with its child's shape
   according to the following rules:\begin{itemize}

   \item if natural is missing, use child's preferred;

   \item if shrink is missing, use 0 if natural is set, or
   child's lo (but no more than v's preferred) otherwise;

   \item if strech is missing, use 0 if natural is set, or
   child's hi (but no less than v's preferred) otherwise.

   \end{itemize} This interface defines some common types of
   flexible size ranges, such as adding inifinite stretch to a
   child's size, removing all stretch from a child, setting a
   size with neither stretch nor shrink, and so on.

   This interface is similar to "RigidVBT", but more powerful in
   that one can spcify a size based on a child's size.  Also, it
   presents a slightly different model to the client: In
   "RigidVBT", one thinks in terms of the low and high bounds of
   some range.  Here, one thinks in terms of the amount the a
   value can stretch and shrink.  Finally, don't forget that
   values in this interface are specified in point, whereas
   "RigidVBT" uses millimeters. *)

INTERFACE FlexShape;

IMPORT Axis;

CONST
  Large    = 99999.0;
  Missing  = -Large;
  Infinity = Large;

TYPE
  SizeRange = RECORD natural, shrink, stretch: REAL END;
  Shape = ARRAY Axis.T OF SizeRange;

CONST

(* Use child's size info: *)
  Default = 
    SizeRange {natural := Missing,
               shrink  := Missing, 
               stretch := Missing};

(* Use child's preferred; remove all shrink and stretch: *)
  Fixed = 
    SizeRange {natural := Missing, 
               shrink  := 0.0, 
               stretch := 0.0};

(* Use child's preferred and shrink; give infinite stretch: *)
  Stretchy = 
    SizeRange {natural := Missing,
               shrink  := Missing, 
               stretch := Infinity};


PROCEDURE Rigid (natural: REAL): SizeRange;
(* Return a "SizeRange" with the specified natural amount and
   with no stretch or shrink.  Really, just a convenience
   procedure for
|  SizeRange{natural := natural,
|            shrink  := 0.0,
|            stretch := 0.0} 
*)

CONST (* for both dimensions: *)
  DefaultShape  = Shape{Default, Default};
  FixedShape    = Shape{Fixed, Fixed};
  StretchyShape = Shape{Stretchy, Stretchy};

END FlexShape.



