(* Copyright (C) 1990, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Last modified on Mon Oct 12 16:38:11 PDT 1992 by muller     *)
(*      modified on Thu Mar  5 10:15:58 PST 1992 by kalsow     *)


UNSAFE MODULE RTException;

IMPORT RTMisc, Csetjmp, Thread, ThreadF, Text, SmallIO;

TYPE HC = HandlerClass;

PROCEDURE Raise (en: ExceptionName;  arg: ExceptionArg) RAISES ANY =
  VAR
    i: CARDINAL;
    h := LOOPHOLE (ThreadF.currentHandlers, Handler);
  BEGIN
    LOOP
      IF (h = NIL) THEN NoHandler (en, raises := FALSE) END;
      (* check this handler *)
      CASE h.class OF
      | HC.Except => 
          i := 0;
          WHILE (h.handles [i] # NIL) DO
            IF (h.handles [i] = en) THEN RaiseForSure (en, arg) END;
            INC (i);
          END;
      | HC.ExceptElse =>
          RaiseForSure (en, arg);
      | HC.Raises =>
          (* check that this procedure does indeed raise 'en' *)
          i := 0;
          LOOP
            IF (h.handles[i] = NIL) THEN  NoHandler (en) END;
            IF (h.handles[i] = en)  THEN  (* ok, it passes *) EXIT  END;
            INC (i);
          END;
      | HC.RaisesNone =>
          NoHandler (en);
      | HC.Finally, HC.FinallyProc, HC.Lock => 
          (* ignore for this pass *)
      ELSE
          BadStack ();
      END;

      (* move to the next handler (in a paranoid way) *)
      h := h.next;
    END;
  END Raise;

PROCEDURE RaiseForSure (en: ExceptionName;  arg: ExceptionArg) RAISES ANY =
  VAR
    h := LOOPHOLE (ThreadF.currentHandlers, Handler);
    i : CARDINAL;
  BEGIN
    (* scan the handler stack *)
    LOOP
      IF (h = NIL) THEN BadStack () END;
      CASE h.class OF
      | HC.ExceptElse, HC.Finally =>
          ThreadF.currentHandlers := h.next;  (* cut to the new handler *)
          h.current := en;                    (* record the exception *)
          h.arg := arg;                       (* and its argument *)
          Csetjmp.ulongjmp (h.jmp_buf, 1);    (* and jump... *)
      | HC.Except =>
          i := 0;
          WHILE (h.handles[i] # NIL) DO
            IF (h.handles[i] = en) THEN
              (* we found the handler *)
              ThreadF.currentHandlers := h.next;  (* cut to the new handler *)
              h.current := en;                    (* record the exception *)
              h.arg := arg;                       (* and its argument *)
              Csetjmp.ulongjmp (h.jmp_buf, 1);    (* and jump... *)
            END;
            INC (i);
          END;
      | HC.Raises =>
          (* ignore *)
      | HC.RaisesNone =>
          BadStack ();
      | HC.FinallyProc =>
          ThreadF.currentHandlers := h.next;  (* cut to this handler *)
          VAR x := LOOPHOLE (h, FinallyProcHandler); BEGIN
            x.proc (x.frame);
          END;
      | HC.Lock => 
          ThreadF.currentHandlers := h.next;  (* cut to this handler *)
          VAR x := LOOPHOLE (h, LockHandler); BEGIN
            Thread.Release (x.mutex);
          END;
      ELSE
          BadStack ();
      END;

      h := h.next;
    END;
  END RaiseForSure;

PROCEDURE NoHandler (en: ExceptionName;  raises := TRUE) =
  BEGIN
    DumpStack ();
    IF (raises)
      THEN RTMisc.RaisesFault (EName (en));
      ELSE RTMisc.HandlerFault (EName (en));
    END;
  END NoHandler;

PROCEDURE SanityCheck () =
  <*FATAL ANY*>
  CONST Min_HC = ORD (FIRST (HC));
  CONST Max_HC = ORD (LAST (HC));
  VAR h := LOOPHOLE (ThreadF.currentHandlers, Handler);
  VAR i: INTEGER;
  BEGIN
    WHILE (h # NIL) DO
      i := ORD (h.class);
      IF (i < Min_HC) OR (Max_HC < i) THEN BadStack () END;
      h := h.next;
    END;
  END SanityCheck;

PROCEDURE BadStack () =
  BEGIN
    DumpStack ();
    RTMisc.FatalError (NIL, 0, "corrupt exception stack");
  END BadStack;

PROCEDURE DumpStack () =
  VAR h := LOOPHOLE (ThreadF.currentHandlers, Handler);
  BEGIN
    Txt ("****************** EXCEPTION HANDLER STACK *********************\n");
    WHILE (h # NIL) DO
      Addr (h);

      CASE h.class OF
      | HC.Except =>
          Txt  (" TRY-EXCEPT ");
          DumpHandles (h.handles);
      | HC.ExceptElse =>
          Txt  (" TRY-EXCEPT-ELSE ");
          DumpHandles (h.handles);
      | HC.Finally =>
          Txt  (" TRY-FINALLY ");
      | HC.Raises =>
          Txt  (" RAISES ");
          DumpHandles (h.handles);
      | HC.RaisesNone =>
          Txt  (" RAISES {}");
      | HC.FinallyProc =>
          VAR x := LOOPHOLE (h, FinallyProcHandler); BEGIN
            Txt  (" TRY-FINALLY  proc = ");
            Addr (LOOPHOLE (x.proc, ADDRESS));
            Txt  ("   frame = ");
            Addr (x.frame);
          END;
      | HC.Lock =>
          VAR x := LOOPHOLE (h, LockHandler); BEGIN
            Txt  (" LOCK  mutex = ");
            Addr (LOOPHOLE (x.mutex, ADDRESS));
          END;
      ELSE
         Txt (" *** BAD EXCEPTION RECORD, class = ");
         Int (ORD (h.class));
         Txt (" ***\n");
         EXIT;
      END;
      Txt ("\n");
      h := h.next;
    END;
    Txt ("****************************************************************\n");
  END DumpStack;

PROCEDURE DumpHandles (x: UNTRACED REF ARRAY LOTS OF ExceptionName) =
  BEGIN
    Txt ("{");
    IF (x # NIL) THEN
      FOR i := FIRST (LOTS) TO LAST (LOTS) DO
        IF (x[i] = NIL) THEN EXIT END;
        IF (i # FIRST (LOTS)) THEN Txt (", ") END;
        Txt (EName (x[i]));
      END;
    END;
    Txt ("}");
  END DumpHandles;

PROCEDURE Txt (t: TEXT) =
  BEGIN
    SmallIO.PutText (SmallIO.stderr, t)
  END Txt;

PROCEDURE Int (i: INTEGER) =
  BEGIN
    SmallIO.PutInt (SmallIO.stderr, i)
  END Int;

PROCEDURE Addr (a: ADDRESS) =
  BEGIN
    SmallIO.PutHexa (SmallIO.stderr, LOOPHOLE (a, INTEGER))
  END Addr;

PROCEDURE EName (en: ExceptionName): TEXT =
  VAR i: CARDINAL := 0;
  BEGIN
    WHILE (en^^[i] # '\000') DO INC (i) END;
    RETURN Text.FromChars (SUBARRAY (en^^, 0, i));
  END EName;

BEGIN
END RTException.

