(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)
(*                                                             *)
(* Last modified on Sun Jul 19 23:06:18 PDT 1992 by meehan     *)
(*      modified on Tue Jun 16 21:55:41 PDT 1992 by muller     *)
(*      modified on Fri Oct 18 14:49:19 PDT 1991 by mhb        *)
<* PRAGMA LL *>

(* This interface maintains a cache of "FormsVBT"s. The client
   can register which ".fv" files will be in the cache, and a
   background thread will parse the files into runtime objects.
   When the client needs a form, it is removed from the cache
   and given to the client.  If the form isn't in the cache,
   it is generated synchronously.  When the client no longer
   needs the form, it can return the form to the cache.
*)

INTERFACE FormsCache;

IMPORT FormsVBT;

EXCEPTION BadForm;

CONST DefaultNumberOfWorkers = 2;
(* No threads are actually forked until the first call to
   "ActiveThreads" or "Prepare". *)

PROCEDURE ActiveThreads (n: CARDINAL); <* LL = 0 *>
(* Change the number of worker threads for generating forms.
   Initially, there are no worker threads.  The first call to
   "Prepare" will cause "DefaultNumberOfWorkers" to be forked.  If
   "ActiveThreads" is called before "Prepare", then "n" threads are
   started initially rather than "DefaultNumberOfWorkers".
   Naturally, setting "n=0" causes there to be no background
   threads. *)

PROCEDURE Prepare (name: TEXT; copies: CARDINAL := 1); <* LL = 0 *>
(* Schedule the specified number of copies of the specified file
   to be parsed and cached.  The file is opened using
   "FormsVBT.NewFromFile", thereby making use of the "Rsrc"
   search path machinery.  If any exceptions are raised when
   "name" is parsed, nothing is added to the cache; a subsequent
   "Get" of this form will cause "FormsVBT" to reparse the file,
   thus raising the exception for the client to handle.  The
   first time that this is called, "DefaultNumberOfWorkers" will
   be started, unless "ActiveThreads" has been previously
   called. *)

PROCEDURE Get (name: TEXT; restock := FALSE):
  FormsVBT.T RAISES {FormsVBT.Error}; <* LL = 0 *>
(* Look for "name" in the cache.  If it's there, return the runtime
   version of name.  Otherwise, parse name into a runtime object
   and return it.  (Parsing "name" may cause the exception to be
   raised.) However, in either case, if "restock" is "TRUE"
   and if there are no more cached or scheduled copies of name,
   call "Prepare(name)" just before returning. *)

PROCEDURE Assoc (name: TEXT; body: TEXT); <* LL = 0 *>
(* Use this procedure if "name" should refer to some text rather
   than to a file.  Thus, "FormsVBT.T.init(name)", rather than
   "FormsVBT.InitFromFile(name)", will be used to parse "body" in
   "Prepare" and "Get".  This must be called before passing
   "name" to "Prepare" or "Get". *)

PROCEDURE Return (fv: FormsVBT.T) RAISES {BadForm}; <* LL = 0 *>
(* A client can return a form that it had gotten by calling "Get".
   The form must be uninstalled in the "VBT" tree.  The exception
   is raised if "fv" was not created by a call to "Get". *)

PROCEDURE Flush (name := ""); <* LL = 0 *>
(* Delete all cached forms for "name", as well as any in the
   waiting list.  If "name" is the empty string, then all entries
   in the cache and waiting list are deleted. *)

END FormsCache.


