/*
	Copyright 1987-1990 XVT Software Inc. All rights reserved.
	May be used freely by licensed and registered users of XVT.
	May be distributed in source form only when embedded in an
	XVT user's application.

	Example XVT application, showing how to track with the mouse.  During
	dragging, the mouse should be trapped.  To see what happens when you
	don't do this:
		1. Choose Trap Mouse from the Options menu (so that it is unchecked).
		2. Start dragging, move the mouse into the menu bar area, and release
		   the button.
		3. With the button up, move the mouse back into the XVT window.  It will
		   continue to drag the rectangle, because the application doesn't know
		   that the button was released.
		4. Click the button.  A black rectangle will be left in the window.
	Now, try the same procedure with trapping enabled.

	You will also notice that, when the mouse is trapped, the cursor stays as a
	plus sign even when moved out of the application's window, but, with
	trapping off, it changes back and forth between a plus sign and an arrow.

	Do to the automatic grab of the mouse by the X-Windows System and the
	Character based system, the strange behavior of that occurs when the mouse
	isn't trapped is not seen.
*/
#include "xvt.h"					/* standard XVT header */
#if (XVTCC == MWCCC) && (XVTOS == CTOOS)
pragma Calling_convention(CTOS_CALLING_CONVENTIONS);
#endif
#include "xvtmenu.h"				/* standard XVT menu tags */
#include "track.h"					/* this app's menu tags */

#ifdef PROTO
STATICFCN void do_menu(MENU_TAG, BOOLEAN, BOOLEAN);
STATICFCN void mark_point(PNT *);
#else
STATICFCN void do_menu();
STATICFCN void mark_point();
#endif
/* Required application setup structure. */
APPL_SETUP appl_setup = {
	0,								/* menu bar resource ID (use default) */
	0,								/* about box resource ID (use default) */
	"track",						/* application's name */
	W_DOC,							/* type of initial window */
	TRUE,							/* size box on initial window? */
	FALSE,							/* vert. scroll bar on initial window? */
	FALSE,							/* horz. scroll bar on initial window? */
	FALSE,							/* close box on initial window? */
	TRUE,							/* want std. font menu? (includes sizes) */
	TRUE							/* want std. style menu? */
};

static FONT cur_font;				/* current font */
static BOOLEAN trap_it = TRUE;		/* trap the mouse? */

/*
	Function to handle all menu commands.
*/
#ifdef FPROTO
static void do_menu(MENU_TAG cmd, BOOLEAN shift, BOOLEAN control)
#else
static void do_menu(cmd, shift, control)
MENU_TAG cmd;						/* menu tag */
BOOLEAN shift;						/* was shift key down? */
BOOLEAN control;					/* was control key down? */
#endif
{
	NOREF(shift);
	NOREF(control);
	switch (cmd) {
	case M_FILE_CLOSE:
		if (std_win != NULL_WIN)
			close_window(std_win);
		break;
	case M_FILE_QUIT:
		terminate();
		break;
	case M_OPT_TRAP:
		trap_it = !trap_it;
		menu_check(M_OPT_TRAP, trap_it);
	}
}

/* Application initialization.  Set current font and font menu. */
BOOLEAN XVTENTRY appl_init BTCENTRY(void)
{
	cur_font = normal_font;
	set_font_menu(&cur_font);
	menu_check(M_OPT_TRAP, trap_it);
	return(TRUE);
}

/*
	Function to mark a point with a shape.  It is XORed so that a second call
	for the same point will put the pixels back the way they were.
*/
#ifdef FPROTO
static void mark_point(PNT *p)
#else
static void mark_point(p)
PNT *p;
#endif
{
	RCT rct;
	CBRUSH black_cbrush;

	black_cbrush.pat = PAT_SOLID;
	black_cbrush.color = COLOR_GRAY;
	set_cpen(&hollow_cpen);
	set_cbrush(&black_cbrush);
	set_draw_mode(M_XOR);
	set_rect(&rct, p->h - 10, p->v - 10, p->h + 10, p->v + 10);
	draw_rect(&rct);
}

/* Main application entry point. */
#ifdef PROTO
void XVTENTRY main_event BTCENTRY(WINDOW win, EVENT_PTR ep)
#else
void XVTENTRY main_event BTCENTRY(win, ep)
WINDOW win;
EVENT_PTR ep;
#endif
{
	RCT rct;
	DRAW_CTOOLS ctools;
	static BOOLEAN dragging = FALSE;
	static BOOLEAN prev_pnt_valid;
	static PNT prev_pnt;
	CBRUSH gray_cbrush;

	switch (ep->type) {
	case E_MOUSE_DOWN:
	case E_MOUSE_DBL:
		dragging = TRUE;
		prev_pnt_valid = FALSE;
		set_cursor(std_win, CURSOR_PLUS);
		if (trap_it)
			trap_mouse(std_win);
		break;
	case E_MOUSE_UP:
		if (dragging) {
			if (prev_pnt_valid)
				mark_point(&prev_pnt);		/* clear the mark */
			dragging = FALSE;
			release_mouse();
			set_cursor(std_win, CURSOR_ARROW);
		}
		break;
	case E_MOUSE_MOVE:
		if (dragging) {
			/* do nothing if we didn't move */
			if (prev_pnt_valid && ep->v.mouse.where.v == prev_pnt.v &&
			  ep->v.mouse.where.h == prev_pnt.h)
				break;
			if (prev_pnt_valid)
				mark_point(&prev_pnt);		/* clear the old mark... */
			mark_point(&ep->v.mouse.where);	/* ...and show the new one */
			prev_pnt = ep->v.mouse.where;
			prev_pnt_valid = TRUE;
		}
		break;
	case E_UPDATE:
		get_normal_ctools(&ctools);
		ctools.pen = hollow_cpen;
		ctools.font = cur_font;
		ctools.brush = white_cbrush; /* ?????? */
		set_draw_ctools(&ctools);
		get_client_rect(win, &rct);
		draw_rect(&rct);
		draw_text(4, 75, "Hello World!", -1);
		set_rect(&rct, 200, 20, 400, 120);
		gray_cbrush.pat = PAT_SOLID;
		gray_cbrush.color = COLOR_GRAY;
		set_cbrush(&gray_cbrush);
		draw_rect(&rct);
		break;
	case E_ACTIVATE:
		if (ep->v.active)
			set_font_menu(&cur_font);
		break;
	case E_COMMAND:
		do_menu(ep->v.cmd.tag, ep->v.cmd.shift, ep->v.cmd.control);
		break;
	case E_CLOSE:
		close_window(win);
		break;
	case E_FONT:
		cur_font = ep->v.font.font;
		set_font_menu(&cur_font);
		invalidate_rect(std_win, NULL);
		break;
	case E_QUIT:
		if (ep->v.query)
			quit_OK();
		else
			terminate();
		break;
	}
}

/* Application cleanup.  Nothing to do. */
void XVTENTRY appl_cleanup BTCENTRY(void)
{
}
