/*
	Copyright 1987-1990 XVT Software Inc. All rights reserved.
	May be used freely by licensed and registered users of XVT.
	May be distributed in source form only when embedded in an
	XVT user's application.

	Example XVT application, showing how to handle E_HSCROLL and E_VSCROLL
	events.
*/
#include "xvt.h"					/* standard XVT header */
#if (XVTCC == MWCCC) && (XVTOS == CTOOS)
pragma Calling_convention(CTOS_CALLING_CONVENTIONS);
#endif
#include "xvtmenu.h"				/* standard menu tags */

#ifdef PROTO
STATICFCN void set_scroll_bars(void);
STATICFCN void do_close(WINDOW);
STATICFCN void do_menu(MENU_TAG, BOOLEAN, BOOLEAN);
STATICFCN void do_update(void);
STATICFCN void do_scroll(SCROLL_TYPE, SCROLL_CONTROL, int);
#else
STATICFCN void set_scroll_bars();
STATICFCN void do_close();
STATICFCN void do_menu();
STATICFCN void do_update();
STATICFCN void do_scroll();
#endif

/* Required application setup structure. */
APPL_SETUP appl_setup = {
	0,								/* menu bar resource ID (use default) */
	0,								/* about box resource ID (use default) */
	"scroll",						/* application's name */
	W_DOC,							/* type of initial window */
	TRUE,							/* size box on initial window? */
	TRUE,							/* vert. scroll bar on initial window? */
	TRUE,							/* horz. scroll bar on initial window? */
	TRUE,							/* close box on initial window? */
	FALSE,							/* want std. font menu? (includes sizes) */
	FALSE							/* want std. style menu? */
};

#define HRANGE		10000			/* horizontal range (app coordinates) */
#define VRANGE		5000			/* vertical range (app coordinates) */
#define HINTERVAL	104				/* horizontal scrolling interval */
#define VINTERVAL	48				/* vertical scrolling interval */
#define GAP			8				/* gap between rectangles */

static PNT org = {0, 0};			/* app coord. at upper left of client area */
static PNT range;					/* actual range (app coordinates) */

/*
	Function to set scroll bar positions.
*/
#ifdef FPROTO
static void set_scroll_bars(void)
#else
static void set_scroll_bars()
#endif
{
	RCT rct;

	get_client_rect(std_win, &rct);
	set_scroll_range(std_win, HSCROLL, 0, range.h = HRANGE - rct.right);
	set_scroll_range(std_win, VSCROLL, 0, range.v = VRANGE - rct.bottom);
	set_scroll_pos(std_win, HSCROLL, org.h);
	set_scroll_pos(std_win, VSCROLL, org.v);
}

/* Application initialization. */
BOOLEAN XVTENTRY appl_init BTCENTRY(void)
{
	set_scroll_bars();
	return(TRUE);
}

/* Function to close a window.  We quit when the window is closed. */
#ifdef FPROTO
static void do_close(WINDOW win)
#else
static void do_close(win)
WINDOW win;
#endif
{
	NOREF(win);
	terminate();
}

/* Function to handle all menu commands. */
#ifdef FPROTO
static void do_menu(MENU_TAG cmd, BOOLEAN shift, BOOLEAN control)
#else
static void do_menu(cmd, shift, control)
MENU_TAG cmd;						/* menu tag */
BOOLEAN shift;						/* was shift key down? */
BOOLEAN control;					/* was control key down? */
#endif
{
	WINDOW win;

	NOREF(shift);
	NOREF(control);
	switch (cmd) {
	case M_FILE_CLOSE:
		if ((win = get_front_window()) != NULL_WIN)
			do_close(win);
		break;
	case M_FILE_QUIT:
		terminate();
		break;
	}
}

/* Function to update the window. */
#ifdef FPROTO
static void do_update(void)
#else
static void do_update()
#endif
{
	RCT rct;
	int i, j, i_start, j_start, i_limit, j_limit;
	char buf[20];

	get_client_rect(std_win, &rct);
	set_cbrush(&white_cbrush);
	set_cpen(&hollow_cpen);
	draw_rect(&rct);
	set_cpen(&black_cpen);
	set_font(&small_font, FALSE);
	i_start = (org.h / HINTERVAL) * HINTERVAL;
	i_limit = min(org.h + rct.right, HRANGE - HINTERVAL);
	j_start = (org.v / VINTERVAL) * VINTERVAL;
	j_limit = min(org.v + rct.bottom, VRANGE - VINTERVAL);
	for (i = i_start; i < i_limit; i += HINTERVAL)
		for (j = j_start; j < j_limit; j += VINTERVAL) {
			set_rect(&rct, i - org.h, j - org.v, i + HINTERVAL - GAP - org.h,
			  j + VINTERVAL - GAP - org.v); /* client coordinates */
			if (needs_update(std_win, &rct)) {
				draw_rect(&rct);
				sprintf(buf, "%d, %d", i, j);
				draw_text(rct.left + GAP, rct.bottom - GAP, buf, -1);
			}
		}
}

/*
	Function to handle scroll events.  Note that we set the new origin
	before calling scroll_rect, because main_event will be called recursively
	from that function with E_UPDATE events, and we must have an accurate
	origin in order to update the window.

	Also, note the if statements for the SC_LINE_DOWN cases to ensure that we
	don't decrease the value of org.h or org.v, which would happen otherwise if
	the range were made smaller due to the window client area being increased.
	Without the if statement, the following can happen:
		1. The window is made fairly small.
		2. The thumb is moved all the way to the end of the range.
		3. The window is made larger (leaving an unpainted margin).
		4. The user clicks the right or down arrows, and the contents jump
		   to the right or down (opposite to what the user expects).
*/
#ifdef FPROTO
static void do_scroll(SCROLL_TYPE stype, SCROLL_CONTROL ctl, int pos)
#else
static void do_scroll(stype, ctl, pos)
SCROLL_TYPE stype;
SCROLL_CONTROL ctl;
int pos;
#endif
{
	RCT rct;
	PNT old_org;

	old_org = org;
	get_client_rect(std_win, &rct);
	switch (stype) {
	case HSCROLL:
		switch (ctl) {
		case SC_LINE_UP:
			org.h = max(0, old_org.h - HINTERVAL);
			break;
		case SC_LINE_DOWN:
			if (old_org.h < range.h) /* avoid rightward jump */
				org.h = min(range.h, old_org.h + HINTERVAL);
			break;
		case SC_PAGE_UP:
			org.h = max(0, old_org.h - rct.right);
			break;
		case SC_PAGE_DOWN:
			org.h = min(range.h, old_org.h + rct.right);
			break;
		case SC_THUMB:
			org.h = pos;
		}
		break;
	case VSCROLL:
		switch (ctl) {
		case SC_LINE_UP:
			org.v = max(0, old_org.v - VINTERVAL);
			break;
		case SC_LINE_DOWN:
			if (old_org.v < range.v) /* avoid downward jump */
				org.v = min(range.v, old_org.v + VINTERVAL);
			break;
		case SC_PAGE_UP:
			org.v = max(0, old_org.v - rct.bottom);
			break;
		case SC_PAGE_DOWN:
			org.v = min(range.v, old_org.v + rct.bottom);
			break;
		case SC_THUMB:
			org.v = pos;
		}
	}
	if (old_org.h != org.h || old_org.v != org.v) {
		set_scroll_bars();
		scroll_rect(&rct, old_org.h - org.h, old_org.v - org.v);
	}
}

/* Main application entry point. */
#ifdef PROTO
void XVTENTRY main_event BTCENTRY(WINDOW win, EVENT_PTR ep)
#else
void XVTENTRY main_event BTCENTRY(win, ep)
WINDOW win;
EVENT_PTR ep;
#endif
{
	switch (ep->type) {
	case E_HSCROLL:
		do_scroll(HSCROLL, ep->v.scroll.what, ep->v.scroll.pos);
		break;
	case E_VSCROLL:
		do_scroll(VSCROLL, ep->v.scroll.what, ep->v.scroll.pos);
		break;
	case E_SIZE:
		set_scroll_bars();
		break;
	case E_UPDATE:
		do_update();
		break;
	case E_COMMAND:
		do_menu(ep->v.cmd.tag, ep->v.cmd.shift, ep->v.cmd.control);
		break;
	case E_CLOSE:
		do_close(win);
		break;
	case E_QUIT:
		if (ep->v.query)
			quit_OK();
		else
			terminate();
		break;
	}
}

/* Application cleanup.  Nothing to do. */
void XVTENTRY appl_cleanup BTCENTRY(void)
{
}
