/*
	Copyright 1987-1990 XVT Software Inc. All rights reserved.
	May be used freely by licensed and registered users of XVT.
	May be distributed in source form only when embedded in an
	XVT user's application.
*/

#include "xvt.h"					/* standard XVT header */
#include "xvtmenu.h"				/* standard XVT menu tags */
#include "dlg.h"					/* this app's menu tags and dialog IDs */

#define E_R_MOUSE_DOWN 1001 		/* app event generated by event_hook */

#if defined (MSDOS) || defined (UNIX) /* conditional because Mac C compilers */
#include <sys/types.h>				/* aren't ANSI C compliant */
#include <sys/stat.h>
#endif
#include <time.h>

#ifdef PROTO
STATICFCN void multitask(void);
STATICFCN void change_num_wins(BOOLEAN);
STATICFCN void build_window(WIN_TYPE);
STATICFCN BOOLEAN XVTENTRY cb_search BTCENTRY (int, CONTROL_INFO *);
STATICFCN void shorten(char *);
STATICFCN void do_search(void);
STATICFCN void limit(int, int);
STATICFCN BOOLEAN XVTENTRY cb_lbox BTCENTRY(int, CONTROL_INFO *);
STATICFCN void lbox_dialog(void);
STATICFCN int cvt_index(void);
STATICFCN BOOLEAN XVTENTRY cb_lbexer BTCENTRY(int, CONTROL_INFO *);
STATICFCN void lbexer_dialog(void);
STATICFCN void opt_dialog(WIN_TYPE); /* needed for mutual recursion */
STATICFCN BOOLEAN XVTENTRY cb_opt BTCENTRY(int, CONTROL_INFO *);
STATICFCN BOOLEAN XVTENTRY cb_scroll BTCENTRY(int, CONTROL_INFO *);
STATICFCN void scroll_dialog(void);
STATICFCN void do_close(WINDOW);
STATICFCN void do_menu(MENU_TAG, BOOLEAN, BOOLEAN);
STATICFCN void do_update(WINDOW);
#else
STATICFCN void multitask();
STATICFCN void change_num_wins();
STATICFCN void build_window();
STATICFCN BOOLEAN XVTENTRY cb_search BTCENTRY ();
STATICFCN void shorten();
STATICFCN void do_search();
STATICFCN void limit();
STATICFCN BOOLEAN XVTENTRY cb_lbox BTCENTRY();
STATICFCN void lbox_dialog();
STATICFCN int cvt_index();
STATICFCN BOOLEAN XVTENTRY cb_lbexer BTCENTRY();
STATICFCN void lbexer_dialog();
STATICFCN void opt_dialog(); /* needed for mutual recursion */
STATICFCN BOOLEAN XVTENTRY cb_opt BTCENTRY();
STATICFCN BOOLEAN XVTENTRY cb_scroll BTCENTRY();
STATICFCN void scroll_dialog();
STATICFCN void do_close();
STATICFCN void do_menu();
STATICFCN void do_update();
#endif
/*
	This function tests the primitive multitasking possible by frequent calling
	of process_events.
*/
#ifdef FPROTO
static void multitask(void)
#else
static void multitask()
#endif
{
	static BOOLEAN running = FALSE;
	RCT rct;
	char buf[25];
	static int count;
	DRAW_CTOOLS tools;
	WINDOW win;
	CBRUSH cbrsh;
	
	cbrsh.pat = PAT_SOLID;
	cbrsh.color = COLOR_WHITE;

	if (running) {
		set_menu_text(M_DLG_TASK, "Start Multitasking");
		running = FALSE;
	}
	else {
		if ((win = std_win) == NULL_WIN && (win = get_front_window()) == NULL_WIN) {
			note("No window open to display count.");
			return;
		}
		set_menu_text(M_DLG_TASK, "Stop Multitasking");
		running = TRUE;
		count = 0;
		while (running) {
			set_cur_window(win);
			get_draw_ctools(&tools);
			set_rect(&rct, 50, 99, 150, 125);
			set_cpen(&black_cpen);
			set_cbrush(&cbrsh);
			set_draw_mode(M_COPY);
			set_font(&normal_font, FALSE);
			draw_rect(&rct);
			sprintf(buf, "%d", count++);
			draw_text(90, 115, buf, -1);
			set_draw_ctools(&tools);
			process_events();
		}
		do_update(win);
	}
}

/* Required application setup structure. */
APPL_SETUP appl_setup = {
	0,								/* menu bar resource ID (use default) */
	0,								/* about box resource ID (use default) */
	"dlg",  						/* application's name */
	W_PLAIN,						/* type of initial window */
	FALSE,							/* size box on initial window? */
	FALSE,							/* vert. scroll bar on initial window? */
	FALSE,							/* horz. scroll bar on initial window? */
	FALSE,							/* close box on initial window? */
	TRUE,							/* want std. font menu? (includes sizes) */
	TRUE							/* want std. style menu? */
};

/*
	Following structure contains the user's current window options,
	changed only when the OK button in the options dialog is clicked.
*/
static struct s_opt {				/* effective window options */
	BOOLEAN close;					/* want close box? */
	BOOLEAN size;					/* want size box? */
	BOOLEAN hscroll;				/* want horizontal scroll bar? */
	BOOLEAN vscroll;				/* want vertical scroll bar? */
	WIN_TYPE type;				/* type of dialog wanted */
	BOOLEAN build_win;				/* set to TRUE on dlg return if OK */
} win_opt = {
	FALSE,
	FALSE,
	FALSE,
	FALSE,
	DLG_MODAL,
	FALSE
};

/*
	Function to keep track of the number of open windows, so that the Close
	item on the File menu is enabled only when there is a window to close.
	Note that we set the menu item all the time -- this is easier than
	remembering its current state.
*/
#ifdef FPROTO
static void change_num_wins(BOOLEAN add_one)
#else
static void change_num_wins(add_one)
BOOLEAN add_one;					/* add a window, or subtract one? */
#endif
{
	static int num_wins = 0;

	num_wins += (add_one ? 1 : -1);
	menu_enable(M_FILE_CLOSE, num_wins > 0);
}

/* Function to build a new window according to the user's specifications. */
#ifdef FPROTO
static void build_window(WIN_TYPE wtype)
#else
static void build_window(wtype)
WIN_TYPE wtype;						/* type of window to build */
#endif
{
	WINDOW win;
	RCT rct;

	switch (wtype) {
	case W_DOC:
		set_rect(&rct, 100, 100, 400, 175);
		break;
	case W_PLAIN:
		set_rect(&rct, 10, 100, 100, 200);
		break;
	case W_DBL:
		set_rect(&rct, 180, 100, 260, 200);
		break;
	default:
		fatal("Invalid window type [build_window].");
	}
	if ((win = new_window(&rct, "A Window", wtype, win_opt.size,
	  win_opt.vscroll, win_opt.hscroll, win_opt.close)) == NULL_WIN)
		error("Can't create window.");
	else
		change_num_wins(TRUE);
	caret_on(win, 10, 20);
}

/*
	Callback function for search dialog box. Sample only -- doesn't actually
	search.
*/
#ifdef FPROTO
static BOOLEAN XVTENTRY cb_search BTCENTRY (int control_id, CONTROL_INFO *cip)
#else
static BOOLEAN XVTENTRY cb_search BTCENTRY (control_id, cip)
int control_id;
CONTROL_INFO *cip;
#endif
{
	static char buf[100] = "Mickey Mouse";

	NOREF(cip);
	switch(control_id) {
		case DLG_INIT:
			set_item_text(SEARCH_TEXT, buf);
			select_item_text(SEARCH_TEXT, 0, 32767);
			break;
		case DLG_OK:
			get_item_text(SEARCH_TEXT, buf);
			note("You typed '%s'.", buf);
			/* fall through */
		case DLG_CANCEL:
			set_item_text(SEARCH_TEXT, buf);
			select_item_text(SEARCH_TEXT, 0, 32767);
			break;
		case DLG_CHAR:
			note("Got char %d", cip->v.chr.ch);
			break;
		case DLG_CLOSE:
			return (FALSE);
	}
	return(TRUE);
}

/* Function to display search dialog box. */
#ifdef FPROTO
static void do_search(void)
#else
static void do_search()
#endif
{
	if (!new_dialog(DLG_MODELESS, SEARCH_DLG, cb_search, 0L))
		error("Can't display dialog box for searching.");
}

/* Function to lop end off of ctime date/time string. */
#ifdef FPROTO
static void shorten(char *s)
#else
static void shorten(s)
char *s;
#endif
{
	s[16] = ' ';
	s[17] = s[22];
	s[18] = s[23];
	s[19] = '\0';
}

/*
	Function to limit typed data to specified size. Called on every change
	to edit field.
*/
#ifdef FPROTO
static void limit(int id, int n)
#else
static void limit(id, n)
int id;
int n;
#endif
{
	char buf[100];

	get_item_text(id, buf);
	if (strlen(buf) > n) {
		beep();
		buf[n] = '\0';
		set_item_text(id, buf);
	}
}

/* Callback function for list box dialog. */
#ifdef FPROTO
static BOOLEAN XVTENTRY cb_lbox BTCENTRY(int control_id, CONTROL_INFO *cip)
#else
static BOOLEAN XVTENTRY cb_lbox BTCENTRY(control_id, cip)
int control_id;						/* ID of control that got activated */
CONTROL_INFO *cip;					/* pointer to control info */
#endif
{
	SLIST x;
	SLIST_ELT e;
	char buf[100], ctm[30], mtm[30], *name;
	WINDOW win;
	struct stat st;
	char type[5];
	long ltype;
	DIRECTORY dir;
	struct s_state {
		int contents;
		int box;
		int box_id;
		BOOLEAN dirs;
		BOOLEAN docs;
		char type[5];
		char pat[21];
	} *state = (struct s_state *)get_app_data(cip->win);

	type[4] = '\0';
	switch (control_id) {
	case DLG_INIT:
		if ((state = (struct s_state *)malloc(sizeof(struct s_state))) == NULL) {
			error("Not enough memory.");
			return(FALSE);
		}
		set_app_data(cip->win, PTR_LONG(state));
		state->contents = LST_FILES;
		state->box = LST_BIG;
		state->box_id = LST_BOX1;
		state->dirs = TRUE;
		state->docs = TRUE;
		state->type[0] = '\0';
		state->pat[0] = '\0';
		set_title (cip->win, "LISTS OF THINGS");
		break;
	case DLG_OK:
		wait_cursor();
		switch (state->contents) {
		case LST_FACES:
			lbox_clear(state->box_id);
			if ((x = list_faces()) == NULL)
				error("Can't list faces.");
			else {
				if (!lbox_add(state->box_id, -1, (char *)x))
					error("Can't add to list box.");
			}
			slist_dispose(x);
			break;
		case LST_FILES:
			get_item_text(LST_TYPE, buf);
			strncpy(state->type, buf, sizeof(state->type) - 1);
			state->type[sizeof(state->type) - 1] = '\0';
			get_item_text(LST_PAT, buf);
			strncpy(state->pat, buf, sizeof(state->pat) - 1);
			state->pat[sizeof(state->pat) - 1] = '\0';
			if ((x = list_files(state->type, state->pat, state->dirs)) == NULL)
				error("Can't list files.");
			lbox_clear(state->box_id);
			lbox_suspend(state->box_id);
			for (e = slist_first(x); e != NULL; e = slist_next(x, e)) {
				if (stat(name = slist_get(x, e, &ltype), &st) == -1) {
					error("Can't get stat info [%d].", xvterrno);
					break;
				}
				memcpy(type, (char *)&ltype, 4);
				strcpy(ctm, ctime((unsigned long *)&st.st_ctime));
				shorten(ctm);
				strcpy(mtm, ctime((unsigned long *)&st.st_mtime));
				shorten(mtm);
				sprintf(buf, "%cr%c%c%7ld  %s  %s  %s",
				  ((st.st_mode & S_IFDIR) == S_IFDIR) ? 'd' : '-',
				  ((st.st_mode & S_IWRITE) == S_IWRITE) ? 'w' : '-',
				  (strcmp(type, "APPL") == 0) ? 'x' : '-',
				  st.st_size, ctm, mtm, name);
				if (!lbox_add(state->box_id, -1, buf)) {
					error("Can't add to list box.");
					break;
				}
			}
			lbox_resume(state->box_id);
			slist_dispose(x);
			break;
		case LST_STATES:
			lbox_clear(state->box_id);
			if ((x = list_res_str(STR_STATES1, STR_STATES2)) == NULL)
				error("Can't list states.");
			else {
				if (!lbox_add(state->box_id, -1, (char *)x))
					error("Can't add states to list box.");
				slist_dispose(x);
			}
			break;
		case LST_WINDOWS:
			if ((x = list_windows()) == NULL)
				break;
			lbox_clear(state->box_id);
			for (e = slist_first(x); e != NULL; e = slist_next(x, e)) {
				if ((name = slist_get(x, e, (long *)&win)) == NULL) {
					error("Can't get window name from list.");
					break;
				}
				if (state->docs)
					switch (get_window_type(win)) {
					case W_DOC:
					case W_PLAIN:
					case W_DBL:
					case W_NO_BORDER:
						break;
					default:
						continue; /* non-document window */
					}
				if (!lbox_add(state->box_id, -1, name))
					error("Can't add to list box.");
			}
			slist_dispose(x);
			break;
		default:
			error("Invalid state->contents.");
		}
		break;
	case LST_DIRS:
		state->dirs = !state->dirs;
		/* fall through */
	case LST_FILES:
		state->contents = LST_FILES;
		break;
	case LST_TYPE:
		limit(LST_TYPE, sizeof(state->type) - 1);
		state->contents = LST_FILES;
		break;
	case LST_PAT:
		limit(LST_PAT, sizeof(state->pat) - 1);
		state->contents = LST_FILES;
		break;
	case LST_FACES:
		state->contents = LST_FACES;
		break;
	case LST_STATES:
		state->contents = LST_STATES;
		break;
	case LST_DOCSONLY:
		state->docs = !state->docs;
		/* fall through */
	case LST_WINDOWS:
		state->contents = LST_WINDOWS;
		break;
	case LST_BIG:
		state->box = LST_BIG;
		state->box_id = LST_BOX1;
		break;
	case LST_SMALL:
		state->box = LST_SMALL;
		state->box_id = LST_BOX2;
		break;
	case LST_CHGDIR:
		if (!get_dir(&dir) || !dir_to_str(&dir, buf, sizeof(buf)))
			error("Can't get current directory.");
		else if (get_str_response("Directory?", buf, sizeof(buf)) != NULL) {
			if (!str_to_dir(buf, &dir))
				error("Can't convert dir to str.");
			if (!chg_dir(&dir))
				error("Can't change directory.");
		}
		break;
	case DLG_CANCEL:
	case DLG_CLOSE:
		free((char *)state);
		return(FALSE);
	default:
		return(TRUE);
	}
	check_radio_button(state->contents, LST_FILES, LST_WINDOWS);
	check_radio_button(state->box, LST_BIG, LST_SMALL);
	check_box(LST_DIRS, state->dirs);
	check_box(LST_DOCSONLY, state->docs);
	return(TRUE);
}

/* Function to put up list box dialog. */
#ifdef FPROTO
static void lbox_dialog(void)
#else
static void lbox_dialog()
#endif
{
	if (!new_dialog(DLG_MODELESS, LST_DLG, cb_lbox, 0L))
		error("Can't put up list box dialog.");
}

/* Convert to convert numeric edit field (list box index) to integer. */
#ifdef FPROTO
static int cvt_index(void)
#else
static int cvt_index()
#endif
{
	char buf[100];
	register int i;

	if (get_item_text(LBEXER_INDEX_TEXT, buf)[0] == '\0')
		i = -1;
	else {
		for (i = 0; buf[i] != '\0'; i++)
			if (!isdigit(buf[i])) {
				i = -1;
				break;
			}
		if (i != -1) {
			i = atoi(buf);
			if (i < 0 || i >= lbox_count_all(LBEXER_BOX1))
				i = -1;
		}
	}
	if (i == -1)
		select_item_text(LBEXER_INDEX_TEXT, 0, SHRT_MAX);
	return(i);
}

/* Callback function for list box exerciser. */
#ifdef FPROTO
static BOOLEAN XVTENTRY cb_lbexer BTCENTRY(int control_id, CONTROL_INFO *cip)
#else
static BOOLEAN XVTENTRY cb_lbexer BTCENTRY(control_id, cip)
int control_id;						/* ID of control that got activated */
CONTROL_INFO *cip;					/* pointer to control info */
#endif
{
	SLIST x;
	SLIST_ELT e;
	char buf[100];
	int i, adjust, count;
	long index;
	struct s_state {
		int contents;
		int source;
		int index;
		char str[100];
		int count;
	} *state = (struct s_state *)get_app_data(cip->win);

	switch (control_id) {
	case DLG_INIT:
		if ((state = (struct s_state *)malloc(sizeof(struct s_state))) == NULL) {
			error("Not enough memory.");
			return(FALSE);
		}
		set_app_data(cip->win, PTR_LONG(state));
		state->contents = LBEXER_FACES;
		state->source = LBEXER_ALL;
		state->count = -1;
		state->index = 0;
		state->str[0] = '\0';
		break;
	case LBEXER_ADD:
		get_item_text(LBEXER_ADD_TEXT, buf);
		if ((count = lbox_count_all(LBEXER_BOX1)) == 0)
			i = -1;
		else
			i = 0;
		if (i != -1 && lbox_count_sel(LBEXER_BOX1) != 1)
			note("Exactly one item must be selected.");
		else if (i != -1 && (i = lbox_get_sel_index(LBEXER_BOX1)) == -1)
			error("Error getting selection.");
		else {
			if (++i >= count) /* want AFTER selection, not BEFORE */
				i = -1;
			if (!lbox_add(LBEXER_BOX1, i, buf))
				error("Error adding to list box.");
			if (!lbox_set_sel(LBEXER_BOX1, -1, FALSE) ||
			  !lbox_set_sel(LBEXER_BOX1, i == -1 ? count : i, TRUE))
				error("Error selecting.");
		}
		select_item_text(LBEXER_ADD_TEXT, 0, SHRT_MAX);
		break;
	case LBEXER_DELETE:
		switch (state->source) {
		case LBEXER_NONE:
			break;
		case LBEXER_ALL:
			if (!lbox_clear(LBEXER_BOX1))
				error("Error clearing.");
			break;
		case LBEXER_BYINDEX:
			if ((i = cvt_index()) == -1)
				error("Invalid index.");
			else if (!lbox_delete(LBEXER_BOX1, i))
				error("Error deleting.");
			break;
		case LBEXER_FIRST:
		case LBEXER_SELECTED:
			if ((x = lbox_get_sel(LBEXER_BOX1)) == NULL)
				error("Error getting selections.");
			else {
				lbox_suspend(LBEXER_BOX1);
				adjust = 0;
				for (e = slist_first(x); e != NULL; e = slist_next(x, e)) {
					if (adjust > 0)
						wait_cursor();
					if (slist_get(x, e, &index) == NULL) {
						error("Error getting SLIST element.");
						break;
					}
					if (!lbox_delete(LBEXER_BOX1, (int)index - adjust++)) {
						error("Error deleting.");
						break;
					}
					if (state->source == LBEXER_FIRST)
						break;
				}
				slist_dispose(x);
				lbox_resume(LBEXER_BOX1);
			}
		}
		break;
	case LBEXER_GET:
		if (!lbox_clear(LBEXER_BOX2)) {
			error("Error clearing.");
			break;
		}
		switch (state->source) {
		case LBEXER_NONE:
			break;
		case LBEXER_ALL:
			wait_cursor();
			if ((x = lbox_get_all(LBEXER_BOX1)) == NULL)
				error("Error getting.");
			else {
				if (!lbox_add(LBEXER_BOX2, -1, (char *)x))
					error("Error adding.");
			}
			slist_dispose(x);
			break;
		case LBEXER_BYINDEX:
			if ((i = cvt_index()) == -1)
				note("Invalid index.");
			else if (!lbox_get_elt(LBEXER_BOX1, i, buf, sizeof(buf)) || 
			  !lbox_add(LBEXER_BOX2, -1, buf))
				error("Error getting or adding.");
			break;
		case LBEXER_FIRST:
			if (!lbox_get_first_sel(LBEXER_BOX1, buf, sizeof(buf)) ||
			  !lbox_add(LBEXER_BOX2, -1, buf))
				error("Error getting or adding.");
			break;
		case LBEXER_SELECTED:
			wait_cursor();
			if ((x = lbox_get_sel(LBEXER_BOX1)) == NULL)
				error("Error getting.");
			else {
				if (!lbox_add(LBEXER_BOX2, -1, (char *)x))
					error("Error adding.");
				slist_dispose(x);
			}
		}
		break;
	case LBEXER_SELECT:
		switch (state->source) {
		case LBEXER_NONE:
			if (!lbox_set_sel(LBEXER_BOX1, -1, FALSE))
				error("Error selecting.");
			break;
		case LBEXER_ALL:
			if (!lbox_set_sel(LBEXER_BOX1, -1, TRUE))
				error("Error selecting.");
			break;
		case LBEXER_BYINDEX:
			if ((i = cvt_index()) == -1)
				note("Invalid index.");
			else if (!lbox_set_sel(LBEXER_BOX1, -1, FALSE) ||
			  !lbox_set_sel(LBEXER_BOX1, i, TRUE))
				error("Error selecting.");
		}
		break;
	case LBEXER_COUNT:
		switch (state->source) {
		case LBEXER_NONE:
			state->count = 0;
			break;
		case LBEXER_ALL:
			state->count = lbox_count_all(LBEXER_BOX1);
			break;
		case LBEXER_BYINDEX:
			state->count = cvt_index() == -1 ? 0 : 1;
			break;
		case LBEXER_FIRST:
			state->count = lbox_count_sel(LBEXER_BOX1) > 0 ? 1 : 0;
			break;
		case LBEXER_SELECTED:
			state->count = lbox_count_sel(LBEXER_BOX1);
			break;
		}
		break;
	case LBEXER_INSERT:
		i = lbox_get_sel_index(LBEXER_BOX1);
		if (i != -1)
			i++;
		if (i >= lbox_count_all(LBEXER_BOX1))
			i = -1;
		switch (state->contents) {
		case LBEXER_FACES:
			wait_cursor();
			if ((x = list_faces()) == NULL)
				error("Can't list faces.");
			else {
				if (!lbox_add(LBEXER_BOX1, i, (char *)x))
					error("Can't add to list box.");
				slist_dispose(x);
			}
			break;
		case LBEXER_FILES:
			wait_cursor();
			if ((x = list_files("", "", TRUE)) == NULL)
				error("Can't list files.");
			else {
				if (!lbox_add(LBEXER_BOX1, i, (char *)x))
					error("Can't add to list box.");
				slist_dispose(x);
			}
			break;
		case LBEXER_STATES:
			wait_cursor();
			if ((x = list_res_str(STR_STATES1, STR_STATES2)) == NULL)
				error("Can't list states.");
			else {
				if (!lbox_add(LBEXER_BOX1, i, (char *)x))
					error("Can't add to list box.");
				slist_dispose(x);
			}
			break;
		case LBEXER_WINDOWS:
			wait_cursor();
			if ((x = list_windows()) == NULL)
				error("Can't list windows.");
			else {
				if (!lbox_add(LBEXER_BOX1, i, (char *)x))
					error("Can't add to list box.");
				slist_dispose(x);
			}
			break;
		default:
			error("Invalid state->contents.");
		}
		break;
	case LBEXER_FACES:
		state->contents = LBEXER_FACES;
		break;
	case LBEXER_FILES:
		state->contents = LBEXER_FILES;
		break;
	case LBEXER_STATES:
		state->contents = LBEXER_STATES;
		break;
	case LBEXER_WINDOWS:
		state->contents = LBEXER_WINDOWS;
		break;
	case LBEXER_NONE:
		state->source = LBEXER_NONE;
		break;
	case LBEXER_ALL:
		state->source = LBEXER_ALL;
		break;
	case LBEXER_BYINDEX:
		state->source = LBEXER_BYINDEX;
		select_item_text(LBEXER_INDEX_TEXT, 0, SHRT_MAX);
		break;
	case LBEXER_FIRST:
		state->source = LBEXER_FIRST;
		break;
	case LBEXER_SELECTED:
		state->source = LBEXER_SELECTED;
		break;
	case DLG_CLOSE:
		free((char *)state);
		return(FALSE);
	default:
		return(TRUE);
	}
	check_radio_button(state->contents, LBEXER_FACES, LBEXER_WINDOWS);
	check_radio_button(state->source, LBEXER_NONE, LBEXER_BYINDEX);
	if (state->count == -1)
		buf[0] = '\0';
	else
		sprintf(buf, "%d", state->count);
	set_item_text(LBEXER_COUNT_RESULT, buf);
	return(TRUE);
}

/* Function to put up list box dialog box. */
#ifdef FPROTO
static void lbexer_dialog(void)
#else
static void lbexer_dialog()
#endif
{
	if (!new_dialog(DLG_MODELESS, LBEXER_DLG, cb_lbexer, 0L))
		error("Can't put up list box exerciser dialog.");
}

/*STATICFCN void opt_dialog(WIN_TYPE);*/ /* needed for mutual recursion */
/*
	Callback function for options dialog box. While the box is up, the
	options structure passed in reflects the current status of the
	check boxes. With every call, including the first one, all of the
	checkboxes are set to conform to that structure, whether they need
	setting or not. Initially, the passed-in structure is set from the global
	structure, win_opt. The options are made permanent by storing them back into
	win_opt only when and if the OK button is clicked. At that time we also
	build the window according to the user's specs.

	Note that this callback function is used for both modal and modeless
	dialogs. A TRUE return from this function means that XVT leaves the box up; a
	FALSE return makes it go away. The return value depends on the type
	member of the CONTROL_INFO structure.
*/
#ifdef FPROTO
static BOOLEAN XVTENTRY cb_opt BTCENTRY(int control_id, CONTROL_INFO *cip)
#else
static BOOLEAN XVTENTRY cb_opt BTCENTRY(control_id, cip)
int control_id;						/* ID of control that got activated */
CONTROL_INFO *cip;					/* pointer to control info */
#endif
{
	struct s_opt *opt = (struct s_opt *)get_app_data(cip->win);

	switch (control_id) {
	case DLG_INIT:
		if ((opt = (struct s_opt *)malloc(sizeof(struct s_opt))) == NULL) {
			error("Out of memory");
			return(FALSE);
		}
		set_app_data(cip->win, PTR_LONG(opt));
		*opt = win_opt;
		break;
	case DLG_OK:
		win_opt = *opt;
		if (cip->type == DLG_MODELESS) {
			build_window(W_DOC);
			return(TRUE);
		}
		win_opt.build_win = TRUE;
		return(FALSE);
	case DLG_CANCEL:
		*opt = win_opt;
		return(FALSE);
		break;
	case DLG_CLOSE:
		return (FALSE);
	case OPT_CLOSE:
		opt->close = !opt->close;
		break;
	case OPT_SIZE:
		note("A note box in a very strange place (for testing).");
		opt->size = !opt->size;
		break;
	case OPT_HSCROLL:
		opt->hscroll = !opt->hscroll;
		break;
	case OPT_VSCROLL:
		opt->vscroll = !opt->vscroll;
		break;
	case OPT_DIALOG:
		win_opt = *opt;
		opt_dialog(win_opt.type);
		return(cip->type == DLG_MODELESS);
	case OPT_MODAL:
		opt->type = DLG_MODAL;
		break;
	case OPT_MODELESS:
		opt->type = DLG_MODELESS;
		break;
	}
	check_box(OPT_CLOSE, opt->close);
	check_box(OPT_SIZE, opt->size);
	check_box(OPT_HSCROLL, opt->hscroll);
	check_box(OPT_VSCROLL, opt->vscroll);
	check_radio_button(opt->type == DLG_MODAL ? OPT_MODAL : OPT_MODELESS,
	  OPT_MODELESS, OPT_MODAL);
	return(TRUE);
}

/* Function to put up the options dialog box. */
#ifdef FPROTO
static void opt_dialog(WIN_TYPE type)
#else
static void opt_dialog(type)
WIN_TYPE type;					/* Type of dialog to put up. */
#endif
{
	if (!new_dialog(type, type == DLG_MODAL ? OPT_DLG_MODAL : OPT_DLG_MODELESS,
	  cb_opt, 0L))
		error("Can't put up dialog (xvterrno = %d)", xvterrno);
	if (win_opt.build_win) {
		win_opt.build_win = FALSE;
		build_window(W_DOC);
	}
}

/* Callback function for scroll bar dialog box. */
#ifdef FPROTO 
static BOOLEAN XVTENTRY cb_scroll BTCENTRY(int control_id, CONTROL_INFO *cip)
#else
static BOOLEAN XVTENTRY cb_scroll BTCENTRY(control_id, cip)
int control_id;						/* ID of control that got activated */
CONTROL_INFO *cip;					/* pointer to control info */
#endif
{
	int start_h, start_v;
	struct scl_state {
		struct bar_info {
			int val;
			int tmp_val;
			int mn;
			int mx;
		} h, v;
	} *state = (struct scl_state *)get_app_data(cip->win);
	struct bar_info *b = NULL;
	char buf[20];

	if (control_id != DLG_INIT) {
		start_h = state->h.tmp_val;
		start_v = state->v.tmp_val;
	}
	switch (control_id) {
	case DLG_INIT:
		assert2("app_data", get_app_data(cip->win) == 1234);
		if ((state = (struct scl_state *)malloc(sizeof(struct scl_state))) ==
		  NULL) {
			error("Out of memory");
			return(FALSE);
		}
		set_app_data(cip->win, PTR_LONG(state));
		start_h = start_v = -9999; /* ensure full update */
		state->h.val = state->h.tmp_val = 0;
		state->h.mn = -25;
		state->h.mx = 50;
		set_scroll_range((WINDOW)SCL_HORZ_BAR, HVSCROLL, state->h.mn,
		  state->h.mx);
		state->v.val = state->v.tmp_val = 100;
		state->v.mn = 0;
		state->v.mx = 100;
		set_scroll_range((WINDOW)SCL_VERT_BAR, HVSCROLL, state->v.mn,
		  state->v.mx);
		break;
	case DLG_CLOSE:
		note("Dialog closing...");
		if (state != NULL) {
			free((char *)state);
			state = NULL;
		}
		return (FALSE); /* NEW */
		/* break; OLD */
	case DLG_OK:
		state->h.val = state->h.tmp_val; /* should also store values globally */
		state->v.val = state->v.tmp_val;
		break;
	case DLG_CANCEL:
		state->h.tmp_val = state->h.val;
		state->v.tmp_val = state->v.val;
		break;
	case SCL_HORZ_BAR:
		b = &state->h;
		/* fall through */
	case SCL_VERT_BAR:
		if (b == NULL)
			b = &state->v;
		switch (cip->v.scroll.what) {
		case SC_LINE_UP:
			b->tmp_val -= 1;
			break;
		case SC_LINE_DOWN:
			b->tmp_val += 1;
			break;
		case SC_PAGE_UP:
			b->tmp_val -= (b->mx - b->mn) / 10;
			break;
		case SC_PAGE_DOWN:
			b->tmp_val += (b->mx - b->mn) / 10;
			break;
		case SC_THUMBTRACK:
		case SC_THUMB:
			b->tmp_val = cip->v.scroll.pos;
		}
		b->tmp_val = max(b->tmp_val, b->mn);
		b->tmp_val = min(b->tmp_val, b->mx);
		break;
	default:
		return(TRUE);
	}
	if (state != NULL && start_h != state->h.tmp_val) {
		set_scroll_pos((WINDOW)SCL_HORZ_BAR, HVSCROLL, state->h.tmp_val);
		sprintf(buf, "%d", state->h.tmp_val);
		set_item_text(SCL_HORZ_VAL, buf);
	}
	if (state != NULL && start_v != state->v.tmp_val) {
		set_scroll_pos((WINDOW)SCL_VERT_BAR, HVSCROLL, state->v.tmp_val);
		sprintf(buf, "%d", state->v.tmp_val);
		set_item_text(SCL_VERT_VAL, buf);
	}
	return(TRUE);
}

/* Function to put up scroll bar dialog. */
#ifdef FPROTO 
static void scroll_dialog(void)
#else
static void scroll_dialog()
#endif
{
	if (!new_dialog(DLG_MODELESS, SCL_DLG, cb_scroll, 1234L))
		error("Can't put up dialog (xvterrno = %d)", xvterrno);
}

/* Function to close a window. */
#ifdef FPROTO
static void do_close(WINDOW win)
#else
static void do_close(win)
WINDOW win;
#endif
{
	close_window(win);
	if (win == std_win)
		std_win = NULL_WIN;
	change_num_wins(FALSE);
}

#ifdef PROTO
static void close_dialog_window(void)
#else
static void close_dialog_window()
#endif
{
	WINDOW win;
	SLIST s;
	SLIST_ELT e;
	long l;
	
	win = get_front_window();
	if (win == NULL_WIN || get_window_type(win) == DLG_NONE) {
		s = list_windows();
		if (s == NULL) {
			error("Cannot list windows");
			return;
		}
		win = NULL_WIN;
		for (e = slist_first(s); e != NULL; e = slist_next(s, e)) {
			slist_get(s, e, &l);
			if (get_window_type((WINDOW)l) != DLG_NONE) {
				win = (WINDOW)l;
				break;
			}
		}
		slist_dispose(s);
	}
	if (win != NULL_WIN)
		close_window(win);
	else
		error("No dialog to close.");
}

/* Function to handle all menu commands. */
#ifdef FPROTO
static void do_menu(MENU_TAG cmd, BOOLEAN shift, BOOLEAN control)
#else
static void do_menu(cmd, shift, control)
MENU_TAG cmd;						/* menu tag */
BOOLEAN shift;						/* was shift key down? */
BOOLEAN control;					/* was control key down? */
#endif
{
	WINDOW win;
	FILE_SPEC fs;
	char s[11];

	NOREF(shift);
	NOREF(control);
	switch (cmd) {
	case M_FILE_NEW:
		build_window(W_DOC);
		break;
	case M_FILE_OPEN:
		fs.name[0] = fs.type[0] = '\0';
		get_dir(&fs.dir);
		switch (open_file_dlg(&fs, "Open something...")) {
		case FL_OK:
			note("FL_OK -- file = '%s'", fs.name);
			break;
		case FL_CANCEL:
			note("FL_CANCEL");
			break;
		case FL_BAD:
			note("FL_BAD");
			break;
		default:
			note("Unknown return.");
		}
		break;
	case M_FILE_SAVE:
		fs.type[0] = fs.name[0] = '\0';
		get_dir(&fs.dir);
		(void)save_file_dlg(&fs, "Save...");
		break;
	case M_FILE_CLOSE:
		if ((win = get_front_window()) != NULL_WIN)
			do_close(win);
		break;
	case M_FILE_QUIT:
		terminate();
		break;
	case M_DLG_DOC_MODAL:
	case M_DLG_DOC_MODELESS:
		opt_dialog(cmd == M_DLG_DOC_MODAL ? DLG_MODAL : DLG_MODELESS);
		break;
	case M_DLG_SCROLL:
		scroll_dialog();
		break;
	case M_DLG_CLOSE:
		close_dialog_window();
		break;
	case M_DLG_SEARCH:
		do_search();
		break;
	case M_DLG_RESPONSE:
		strcpy(s, "Default.");
		if (get_str_response("Type something...", s, sizeof(s)) == NULL)
			note("You cancelled.");
		else
			note("You typed '%s' (up to 10 chars were read).", s);
		break;
	case M_DLG_LIST:
		lbox_dialog();
		break;
	case M_DLG_LBEXER:
		lbexer_dialog();
		break;
	case M_DLG_TASK:
		multitask();
		break;
	default:
		error("Unknown menu item: %d", cmd);
	}
}

/*
	Application initialization.  Set font menu and record the existence of
	XVT's initial window, created from the appl_setup structure (std_win).
*/
BOOLEAN XVTENTRY appl_init()
{
	char buf[100];

	set_font_menu(&normal_font);
	change_num_wins(TRUE);
	caret_on(std_win, 10, 20);
	menu_enable(M_FILE_NEW, TRUE);
	menu_enable(M_FILE_OPEN, TRUE);
	menu_enable(M_FILE_SAVE, TRUE);
	if (get_res_str(STR_TEST, buf, sizeof(buf)) != NULL)
		note("%s", buf);
	else
		error("Can't read string resource %d.", STR_TEST);
	return(TRUE);
}

#ifdef FPROTO 
static void do_update(WINDOW win)
#else
static void do_update(win)
WINDOW win;
#endif
{
	RCT rct;
#if (XVTWS == WMWS)
	CBRUSH cbrush;
#endif

	get_client_rect(win, &rct);
	set_cpen(&hollow_cpen);
	set_cbrush(&white_cbrush);
	draw_rect(&rct);
	draw_text(4, 30, "Hello World!", -1);
#if (XVTWS == WMWS)
	set_cpen(&hollow_cpen);
	cbrush.pat = PAT_SOLID;
	set_rect(&rct, 16, 48, 48, 64);
	cbrush.color = COLOR_RED;
	set_cbrush(&cbrush);
	draw_rect(&rct);
	set_rect(&rct, 16, 64, 48, 80);
	cbrush.color = COLOR_BLUE;
	set_cbrush(&cbrush);
	draw_rect(&rct);
	set_rect(&rct, 16, 80, 48, 96);
	cbrush.color = COLOR_GREEN;
	set_cbrush(&cbrush);
	draw_rect(&rct);
	set_rect(&rct, 48, 48, 80, 64);
	cbrush.color = COLOR_CYAN;
	set_cbrush(&cbrush);
	draw_rect(&rct);
	set_rect(&rct, 48, 64, 80, 80);
	cbrush.color = COLOR_MAGENTA;
	set_cbrush(&cbrush);
	draw_rect(&rct);
	set_rect(&rct, 48, 80, 80, 96);
	cbrush.color = COLOR_YELLOW;
	set_cbrush(&cbrush);
	draw_rect(&rct);
	set_rect(&rct, 80, 48, 112, 64);
	cbrush.color = COLOR_WHITE;
	set_cbrush(&cbrush);
	draw_rect(&rct);
#else
	draw_icon(10, 40, ICON_RID);
#endif
}

/* Main application entry point. */
void XVTENTRY main_event(win, ep)
WINDOW win;							/* window */
EVENT *ep;							/* event */
{
	DRAW_CTOOLS tools;
	static int act_counter = 0;
	RCT r, q;
	static RCT orig_client;
	WINDOW w;
	static int count = 0;

	switch (ep->type) {
	case E_R_MOUSE_DOWN: /* not an XVT event -- generated by event_hook */
		note("Right button down!");
		break;
	case E_MOUSE_DBL:
		switch (count++ % 3) {
		case 0:
			get_client_rect(w = get_front_window(), &orig_client);
			local_global(w, (PNT *)&orig_client, 2);
			set_rect(&r, 100, 75, 300, 200);
			move_window(w, &r);
			get_client_rect(w, &q);
			if (q.right != r.right - r.left || q.bottom != r.bottom - r.top)
				note("You didn't exactly get what you asked for!");
			break;
		case 1:
			set_rect(&r, 20, 200, 450, 300);
			move_window(w = get_front_window(), &r);
			get_client_rect(w, &q);
			if (q.right != r.right - r.left || q.bottom != r.bottom - r.top)
				note("You didn't exactly get what you asked for!");
			break;
		case 2:
			set_rect(&r, 50, 100, 500, 130);
			move_window(w = get_front_window(), &r);
			get_client_rect(w, &q);
			if (q.right != r.right - r.left || q.bottom != r.bottom - r.top)
				note("You didn't exactly get what you asked for!");
			break;
		}
		break;
	case E_CHAR:
		note("shift = %d; ctl = %d; char = %d %c", ep->v.chr.shift,
		  ep->v.chr.control, ep->v.chr.ch, ep->v.chr.ch);
		break;
	case E_UPDATE:
		do_update(win);
		break;
	case E_ACTIVATE:
		if (ep->v.active) {
			set_cur_window(win); /* NEW */
			get_draw_ctools(&tools);
			set_font_menu(&tools.font);
		}
		break;
	case E_COMMAND:
		do_menu(ep->v.cmd.tag, ep->v.cmd.shift, ep->v.cmd.control);
		break;
	case E_CLOSE:
		do_close(win);
		break;
	case E_FONT:
		if ((win = get_front_window()) != NULL_WIN) {
			set_cur_window(win);
			set_font_menu(&ep->v.font.font);
			set_font(&ep->v.font.font, FALSE);
			invalidate_rect(win, NULL);
		}
		break;
	case E_QUIT:
		if (ep->v.query)
			quit_OK();
		else
			terminate();
	}
}

/* Application cleanup.  Nothing to do. */
void XVTENTRY appl_cleanup()
{
}
