/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  volume.c                                                                 */
/*                                                                           */
/*  Functions to access CTOS disk volumes and volume home blocks.            */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  07/10/92  122K.08  D. Gilson     /  P15511893 Suppress Confirm msg should*/
/*                                      not be tied to non-interactive       */
/*  11/07/91  130D.07  D. Gilson     /  Volume Verification message now      */
/*                                   /  happens whether                      */
/*                                   /  [Suppress Verification?] is set/reset /
/*  06/24/91  121J.06  P. Johansson  /  'BadBlk.Sys' entries ended by zero   */
/*                                      sector number, not zero cylinder;    */
/*                                      compute SMD spiral factor correctly. */
/*  06/20/91  121J.05  P. Johansson  /  Release bad spots correctly for new  */
/*                                      style VHB, non-LBA disks.            */
/*  05/01/91  121J.04  P. Johansson  /  "Processing file header n of N ... " */
/*  04/18/91  121J.04  V. Muppidi    /  GetCtosPartitionInfo changed to      */
/*                                      GetCTOSDiskPartition.                */
/*  03/28/91  121H.03  P. Johansson  /  Support multiple volume backup to    */
/*                                      [XXX]+ datasets on tape.             */
/*  03/19/91  121H.02  P. Johansson  /  Use GetCtosPartitionInfo on SuperGen.*/
/*  01/09/91  121F.01  P. Johansson  /  GetFileStatus for node doesn't work  */
/*                                      under real-mode CTOS.                */
/*  12/09/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Sequential Access service that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define BuildFileSpec
#define CSubparams
#define GetCTOSDiskPartition
#define GetFileStatus
#define NlsULCmpB
#define OpenFile
#define QueryDCB
#define QueryDiskGeometry
#define ParseFileSpec
#define Read
#define RgParam
#define Send
#define SetDiskGeometry

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define dcb_type
#define disk_geometry_type
#define FhbType
#define sdType
#define vhb_type

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "archive.h"
#include "archive_msgs.h"
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern iob_type *allocate_iob(unsigned buffer_size);
extern void exit_with_msg(unsigned erc, unsigned msg_num);
extern unsigned fetch_mfd(sdType *sd_filename);
extern void hex_dump(void *buffer, unsigned length, unsigned blackouts,
                     unsigned blackout_floor[], unsigned blackout_ceiling[]);
extern log_msg(unsigned nls_msg_index, unsigned signature, sdType nls_parms[],
               unsigned nls_parms_len);
extern Boolean proceed(void);
extern void release_disk_run(unsigned long vda, unsigned long length);
extern sdType *standard_msg(unsigned msg_num);

/* Error return codes used by this module */

#define FsErc
#define RqErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

extern archive_type archive;
extern path_type path;
extern summary_type summary;
extern vlpb_type vlpb;

/* Global variables exported by this manuscript */

filesys_type filesys;
vhb_type vhb;

/* Function prototypes defined before the functions themselves are declared */

void display_vhb(vhb_type *vhb, unsigned nls_msg, unsigned long vda);
unsigned read_disk_configuration(disk_geometry_type *disk_geometry);
unsigned read_pages(void *buffer, unsigned long vda,
                    unsigned page_offset, unsigned pages);
unsigned read_vhb(void);
unsigned valid_vhb(unsigned long vda, vhb_type *vhb, Boolean initial_vhb);

pragma Page(1);
/*-----------------------------------------------------------------------------
 Given the volume name supplied by the user, attempt to locate it, retrieve
 DCB and disk geometry information and then open the volume for read access.
 Then try to locate a VHB: either in the first 14 pages of track zero or in
 the middle of the disk (as determined by a conversation with the user).  If
 the user provided a device name, update the volume name field with the actual
 volume name from the VHB. */

unsigned open_volume(unsigned volume_index) {

   old_dcb_type dcb;
   unsigned erc, i, node_len, passwords, password_len, volume_len,
      volume_spec_len;
   iob_type *iob;
   char node[MAX_NODE_LENGTH + 1], volume_spec[MAX_FILE_SPECIFICATION_LENGTH];
   unsigned long partition_size;
   sdType *sd_local, *sd_mstr, sd_password, sd_volume;
   sdType nls_parms[] = {(void *) volume_spec, 0,
                                  NULL, 0,
                         (void *) &vlpb.backup_after,
                                  sizeof(vlpb.backup_after)};

   memset(&filesys, 0, sizeof(filesys));
   RgParam(VOLUME_PARAM, volume_index, &sd_volume);
   passwords = CSubparams(VOLUME_PASSWORD_PARAM);
   if (passwords > 1 && volume_index >= passwords) {
      sd_password.pb = NULL;
      sd_password.cb = 0;
   } else
      RgParam(VOLUME_PASSWORD_PARAM,
              (volume_index < passwords) ? volume_index : 0, &sd_password);
   ParseFileSpec(0, sd_volume.pb, sd_volume.cb, FALSE, &path.node[1],
                 &node_len, &path.volume[1], &volume_len, NULL, 0, NULL, 0,
                 &path.volume_password[1], &password_len, FALSE, 1);
   path.node[0] = node_len;
   path.volume[0] = volume_len;
   if (password_len == 0)
      memcpy(&path.volume_password[1], sd_password.pb,
             path.volume_password[0] = sd_password.cb);
   else
      path.volume_password[0] = password_len;
   memset(volume_spec, 0, sizeof(volume_spec));
   BuildFileSpec(0, volume_spec, &volume_spec_len, &nls_parms[0].cb,
                 sizeof(volume_spec), FALSE, &path.node[1], path.node[0],
                 &path.volume[1], path.volume[0], NULL, 0, NULL, 0, FALSE,
                 NULL, 0, FALSE, 0);
   if ((erc = QueryDCB(volume_spec, volume_spec_len, &dcb,
                       sizeof(dcb))) != ercOK) {
      log_msg((erc == ercNoSuchVolume || erc == ercBadFileSpec)
               ? NLS_NO_SUCH_VOLUME : NLS_CANT_ACCESS,
              1, nls_parms, sizeof(nls_parms));
      exit_with_msg(erc, 0);
   }
   memcpy(path.device, dcb.devName, dcb.devName[0] + 1);
   memcpy(path.device_password, dcb.devPassword, dcb.devPassword[0] + 1);
   if ((erc = OpenFile(&filesys.handle, volume_spec, volume_spec_len,
                       &path.volume_password[1], path.volume_password[0],
                       modeRead)) != ercOK) {
      log_msg(NLS_CANT_OPEN_VOLUME, 1, nls_parms, sizeof(nls_parms));
      exit_with_msg(erc, 0);
   }
   sd_local = standard_msg(NLS_LOCAL);
   sd_mstr = standard_msg(NLS_MASTER);
   if (     GetFileStatus(filesys.handle, FILE_STATUS_NODE, node,
                          sizeof(node)) == ercOK) {
      if (     node[0] == sd_mstr->cb
            && NlsULCmpB(NULL, &node[1], sd_mstr->pb, node[0], &i) == ercOK
            && i == 0xFFFF)
         node[0] = 0;		/* Don't clutter display if CT-Net unused */
      else if (   node[0] == sd_local->cb
               && NlsULCmpB(NULL, &node[1], sd_local->pb, node[0], &i) == ercOK
               && i == 0xFFFF) {
         filesys.local = TRUE;
         node[0] = 0;		/* Don't clutter display if CT-Net unused */
      }
      memcpy(path.node, node, node[0] + 1);	/* Update node name */
   }
   GetCTOSDiskPartition(filesys.handle, &filesys.vda_bom, &partition_size);
   if ((erc = read_vhb()) != ercOK)
      exit_with_msg(erc, NLS_NO_VHB);
   memcpy(path.volume, &vhb.volName[0], vhb.volName[0] + 1);
   path.volume_encrypted =    (vhb.magicWd == MAGIC_PC)
                           || (   (vhb.magicWd & (MAGIC_VM | MAGIC_ENCRYPT))
                               == (MAGIC_VM | MAGIC_ENCRYPT));
   memset(volume_spec, 0, sizeof(volume_spec));	/* Rebuild for display use */
   BuildFileSpec(0, volume_spec, &volume_spec_len, &nls_parms[0].cb,
                 sizeof(volume_spec), FALSE, &path.node[1], path.node[0],
                 &path.volume[1], path.volume[0], NULL, 0, NULL, 0, FALSE,
                 NULL, 0, FALSE, 0);
   if (vhb.allocPageCnt == 0)
      exit_with_msg(ercInconsistency, NLS_NO_BIT_MAP);
   if (vhb.cPagesMfd == 0)
      exit_with_msg(ercInconsistency, NLS_NO_MFD);
   if (vhb.cPagesFileHeaders == 0)
      exit_with_msg(ercInconsistency, NLS_NO_FILE_HEADERS);
   log_msg(NLS_ARCHIVE_OF, 1, nls_parms, sizeof(nls_parms));
   if (vlpb.backup_after != 0)
      log_msg(NLS_INCREMENTAL_FROM, 1, nls_parms, sizeof(nls_parms));
   if (dcb.userCount > 1 && !vlpb.suppress_confirmation_msg) {/* m07 & m08 */
      log_msg(NLS_VERIFICATION_SUSPECT, 1, NULL, 0);
      if (!vlpb.noninteractive && !proceed())
         exit_with_msg(ercOperatorIntervention, 0);
   }
   summary.count[TOTAL_FILE_HEADERS] = (vhb.altFileHeaderPageOffset)
                                        ? (vhb.cPagesFileHeaders / 2) - 1
                                        : vhb.cPagesFileHeaders - 1;
   summary.count[TOTAL_FILES] = summary.count[TOTAL_FILE_HEADERS]
                                 - vhb.cFreeFileHeaders;
   if (!vlpb.suppress_backup) {
      iob = allocate_iob(0);
      iob->ctrl = MODE;
      iob->mode.write = TRUE;
      iob->mode.sequential = archive.sequential;
      Send(archive.msg_exch, iob);
      path.volume_may_have_changed = TRUE;
   }
   fetch_mfd(NULL);				/* Get MFD for this volume */
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 */

private unsigned read_vhb(void) {

   char buffer[PAGE_SIZE];
   unsigned erc, page;
   old_dcb_type dcb;
   disk_geometry_type disk_geometry;
   unsigned long vda_vhb = 0, middle_vda;
   vhb_type *maybe_vhb = (vhb_type *) buffer;

   for (page = 0; page < 14; page++)
      if ((erc = valid_vhb(filesys.vda_bom + (page * PAGE_SIZE),
                           (vhb_type *) buffer, TRUE)) == ercOK) {
         memcpy(&vhb, buffer, sizeof(vhb));	/* Use initial VHB for now */
         break;
      } else if (erc == ercIoError)
         continue;
      else if (erc != ercNoValidHomeBlock)
         return(erc);
   if (erc == ercOK) {
      if ((erc = valid_vhb(vhb.lfaVhb, (vhb_type *) buffer, FALSE)) == ercOK)
         memcpy(&vhb, buffer, sizeof(vhb));	/* Use working VHB copy */
      return(ercOK);		/* Either initial or working is acceptable */
   }
   log_msg(NLS_NO_INITIAL_VHB, 1, NULL, 0);
   if (QueryDiskGeometry(&path.device[1], path.device[0],
                         &path.device_password[1], path.device_password[0],
                         &disk_geometry, sizeof(disk_geometry)) == ercOK)
      if (disk_geometry.blocksPerDisk == 0)
         middle_vda = (disk_geometry.cylindersPerDisk / 2)
                       * (disk_geometry.tracksPerCylinder
                          * ((unsigned long) (disk_geometry.sectorsPerTrack
                                            * disk_geometry.bytesPerSector)));
      else
         middle_vda = (disk_geometry.blocksPerDisk / 2) * PAGE_SIZE;
   else if (QueryDCB(&path.device[1], path.device[0],
                     &dcb, sizeof(dcb)) == ercOK)
      middle_vda = (dcb.cylindersPerDisk / 2) * (dcb.tracksPerCylinder
                    * ((unsigned long) (dcb.sectorsPerTrack
                                      * dcb.bytesPerSector)));
   else
      return(erc);
   for (page = 0; page < 14; page++)
      if ((erc = valid_vhb(middle_vda + (page * PAGE_SIZE),
                           (vhb_type *) buffer, FALSE)) == ercOK) {
         memcpy(&vhb, buffer, sizeof(vhb));
         return(erc);		/* Found a valid working Volume Home Block */
      } else if (erc == ercIoError)
         continue;
      else if (erc != ercNoValidHomeBlock)
         return(erc);
   if (     vlpb.noninteractive
         || (erc = read_disk_configuration(&disk_geometry)) == ercOK)
      return(erc);
   if (disk_geometry.blocksPerDisk == 0)
      middle_vda = (disk_geometry.cylindersPerDisk / 2)
                    * (disk_geometry.tracksPerCylinder
                       * ((unsigned long) (disk_geometry.sectorsPerTrack
                                         * disk_geometry.bytesPerSector)));
   else
      middle_vda = (disk_geometry.blocksPerDisk / 2) * PAGE_SIZE;
   for (page = 0; page < 14; page++)
      if ((erc = valid_vhb(middle_vda + (page * PAGE_SIZE),
                           (vhb_type *) buffer, FALSE)) == ercOK) {
         memcpy(&vhb, buffer, sizeof(vhb));
         return(erc);		/* Found a valid working Volume Home Block */
      } else if (erc == ercIoError)
         continue;
      else if (erc != ercNoValidHomeBlock)
         return(erc);
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 */

private unsigned valid_vhb(unsigned long vda, vhb_type *vhb,
                           Boolean initial_vhb) {

   unsigned checksum, erc, i;

   if ((erc = read_pages(vhb, vda, 0, 1)) != ercOK) {
      if (!initial_vhb) {
         sdType nls_parms[] = {(void *) &vda, sizeof(vda)};

         log_msg(NLS_VHB_IO_ERROR, 1, nls_parms, sizeof(nls_parms));
      }
      return(erc);
   }
   if (     (   vhb->magicWd == MAGIC || vhb->magicWd == MAGIC_PC
             || (vhb->magicWd & MAGIC_VM) != 0)
         && ((initial_vhb) ? vhb->lfaInitialVhb : vhb->lfaVhb) == vda) {
      checksum = -MAGIC;
      for (i = 0; i < sizeof(vhb_type) / 2; i++)
         checksum += ((unsigned *) vhb)[i];
      if (checksum == 0)
         return(ercOK);		/* Found a valid initial Volume Home Block! */
      filesys.invalid = TRUE;
      display_vhb(vhb, (initial_vhb) ? NLS_INVALID_INITIAL_VHB 
                                     : NLS_INVALID_WORKING_VHB, vda);
   }
   return(ercNoValidHomeBlock);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 */

private unsigned read_disk_configuration(disk_geometry_type *disk_geometry) {

   return(ercNoValidHomeBlock);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Display the VHB, in both a hex dump with offset addresses and a showing of
 the printable interpretation of the hex values at the right hand side of the
 screen.  Note that the volume password is not displayed (yes, even though the
 user knew the password to run Volume Archive, we don't show it to him!). */

private void display_vhb(vhb_type *vhb, unsigned nls_msg, unsigned long vda) {

   unsigned blackout_floor[] = {_offsetof(vhb_type, volPassword)};
   unsigned blackout_ceiling[] =
                   {_offsetof(vhb_type, volPassword) + last(vhb->volPassword)};
   sdType nls_parms[] = {(void *) &vda, sizeof(vda),
                         (void *) &vhb->volName[1], vhb->volName[0],
                         (void *) &vhb->creationDT, sizeof(vhb->creationDT),
                         (void *) &vhb->modificationDT,
                          sizeof(vhb->modificationDT),
                         (void *) &vhb->cFreePages, sizeof(vhb->cFreePages),
                         (void *) &vhb->cFreeFileHeaders,
                          sizeof(vhb->cFreeFileHeaders)};

   log_msg(nls_msg, 1, nls_parms, sizeof(nls_parms));
   log_msg(NLS_VHB_HEADER, 1, nls_parms, sizeof(nls_parms));
   hex_dump(vhb, sizeof(*vhb), last(blackout_floor) + 1, blackout_floor,
            blackout_ceiling);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 For the old style VHB (with its disk geometry expressed in terms of cylinders
 per disk, tracks per cylinder, sectors per track and bytes per sector) there
 is a file, <Sys>BadBlk.Sys, that contains the locations of the medium
 defects.  These defects have already been allocated in the bit map; here,
 mark them in use so that the bit map verification does not report errors for
 them. */

void release_bad_spots(void) {

#define DEFECTS_PER_PAGE 128

   struct {
      char sector[DEFECTS_PER_PAGE];
      char head[DEFECTS_PER_PAGE];
      unsigned cylinder[DEFECTS_PER_PAGE];
   } defect;
   unsigned long defect_vda;
   unsigned i, page;
   sdType nls_parms[] = {(void *) &defect_vda, sizeof(defect_vda)};

   if ((vhb.magicWd & (MAGIC_VM | MAGIC_LBA)) == (MAGIC_VM | MAGIC_LBA))
      return;			/* LBA devices don't have bad spot maps */
   for (page = 0; page < vhb.cPagesBadBlk; page++)
      if (read_pages(&defect, vhb.lfaBadBlkBase, page, 1) == ercOK)
         for (i = 0; i < DEFECTS_PER_PAGE && defect.sector[i] != 0; i++) {
            defect_vda = ( ( ( (defect.cylinder[i] * vhb.tracksPerCylinder)
                              + defect.head[i])
                            * (unsigned long) vhb.sectorsPerTrack)
                          + (defect.sector[i] - 1 + vhb.sectorsPerTrack
                             -   (defect.head[i] * vhb.spiralFactor)
                               % vhb.sectorsPerTrack)
                          % vhb.sectorsPerTrack)
                        * vhb.bytesPerSector;
            release_disk_run(defect_vda, PAGE_SIZE);
         } else
            log_msg(NLS_BAD_BLOCK_IO_ERROR, 1, nls_parms, sizeof(nls_parms));

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Generic function used throughout the archive utilities to read one or more
 pages from the disk volume.  There is no provision to indicate to the caller
 that part of a multiple page read has succeeded---it is all or nothing. */

unsigned read_pages(void *buffer, unsigned long vda,
                    unsigned page_offset, unsigned pages) {

   unsigned erc, xfer_count;

   vda += ((unsigned long) page_offset * PAGE_SIZE);
   do {
      xfer_count = (pages < PAGES_PER_SEGMENT)
                                ? pages * PAGE_SIZE : SEGMENT_SIZE - PAGE_SIZE;
      if ((erc = Read(filesys.handle, buffer, xfer_count,
                      vda, &xfer_count)) != ercOK) {
         sdType nls_parms[] = {(void *) &erc, sizeof(erc),
                               (void *) &vda, sizeof(vda)};

         log_msg(NLS_IO_ERROR, 1, nls_parms, sizeof(nls_parms));
      } else {
         offset_of(buffer) += xfer_count;
         vda += xfer_count;
         pages -= xfer_count / PAGE_SIZE;
      }
   } while (erc == ercOK && pages > 0);
   return(erc);

}
