/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  qic_alignment.c                                                          */
/*                                                                           */
/*  Perform head alignment tests on any QIC-24 or QIC-150 tape drive.        */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  05/25/91  010A.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <stdlib.h>
#include <string.h>
pragma Pop(List);

/* Suppress C run-time (only CTOS functionality needed) */

pragma Off(List);
#include <stub.h>
pragma Pop(List);

/* There are no procedures in the Archive utilities that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define Beep
#define CheckErc
#define ENLS_GetChar
#define ENLS_MapCharToStdValue
#define ErrorExit
#define Exit
#define ExpandLocalMsg
#define GetMsgUnexpanded
#define InitMsgFile
#define OpenByteStream
#define OutputBytesWithWrap
#define ParseFileSpec
#define RgParam

#define ENLS_KBD_DIRECT_MODE_0 0x00
#define ENLS_FINISH 0x04
#define ENLS_CANCEL 0x07
#define ENLS_GO 0x1B

extern unsigned SeqAccessControl(unsigned seq_handle, unsigned ctrl_function,
                                 signed ctrl_qualifier, signed long *residual);
extern unsigned SeqAccessOpen(unsigned *seq_handle_ret, void *device_name,
                              unsigned device_name_len, void *password,
                              unsigned password_len, unsigned access_mode,
                              void *mode_parameters,
                              unsigned mode_parameters_len);
extern unsigned SeqAccessRead(unsigned seq_handle, void *data_ret,
                              unsigned data_max, signed long *residual);
extern unsigned SeqAccessStatus(unsigned seq_handle, void *status_ret,
                                unsigned status_max, signed long *residual);

#define MODE_READ 0x6D72
#define CTRL_REWIND 1
#define CTRL_UNLOAD 2
#define CTRL_SCAN_FILEMARK 6
#define SYNCHRONIZE 0
#define IMMEDIATE 0xFFFF

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

/* Type definitions used by this module */

typedef unsigned char Boolean;
typedef struct {
   unsigned msg_num;
   char *msg;
   unsigned msg_len;
} msg_descriptor_type;

#define default_msg(INDEX, TEXT) INDEX, TEXT, last(TEXT)
#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define sdType

pragma Off(List);
#include <ctosTypes.h>
pragma Pop(List);

/* Error return codes used by this module */

#define RqErc
#define TapeErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

extern char bsVid[130];
extern char sbVerRun[];

/* Static variables global within this manuscript */

#define MSG_FILE "[Sys]<Sys>QicAlignmentMsg.Bin"

#define NLS_VID 1
#define NLS_COMMAND_VERSION 2
#define NLS_DEFAULT_DEVICE 3
#define NLS_NEED_SEQ_ACCESS_SERVICE 4
#define NLS_DEVICE_IN_USE 5
#define NLS_INSERT_CARTRIDGE 6
#define NLS_PROMPT 7
#define NLS_DEVICE_NOT_READY 8
#define NLS_POSITIONING_ERROR 9
#define NLS_REQUEST_INTERVENTION 10
#define NLS_READING_FILE 11
#define NLS_CENTERED 12
#define NLS_ABOVE_POSITIVE 13
#define NLS_ABOVE_NEGATIVE 14
#define NLS_BELOW_POSITIVE 15
#define NLS_BELOW_NEGATIVE 16
#define NLS_DONE 17
#define NLS_DATA_ERROR 18
#define NLS_TAPE_IO_ERROR 19
#define NLS_SKIPPING_FILE 20
#define NLS_UPDATE_LABEL 21
#define NLS_DEVICE_MISALIGNED 22
#define NLS_ALIGNMENT_OK 23

/* Function prototypes defined before the functions themselves are declared */

void insert_cartridge(char device_spec[], unsigned qic_handle, char cartridge);
void log_msg(unsigned msg_index, sdType nls_parms[], unsigned nls_parms_len);
Boolean proceed(void);
Boolean read_cartridge(unsigned qic_handle, char cartridge);
sdType *standard_msg(unsigned msg_num);

pragma Page(1);
/*-----------------------------------------------------------------------------
 Parse the device name supplied by the user and then try to open it.  If
 everything is OK, perform a read pass of BOTH alignment cartridges and report
 the results. */

void main(void) {

   Boolean alignment_error;
   char cartridge, device_spec[106], msg_buffer[1024], msg_cache[1024],
      position[12], vid_buffer[1024];
   sdType sd_qic, *sd_vid;
   unsigned erc, position_len, qic_handle;
   sdType nls_parms[] = {NULL, 0,
                         &sbVerRun[1], sbVerRun[0],
                         &device_spec[1], 0};

   sd_vid = standard_msg(NLS_VID);
   CheckErc(OpenByteStream(bsVid, sd_vid->pb, sd_vid->cb, NULL, 0, modeWrite,
                           vid_buffer, sizeof(vid_buffer)));
   InitMsgFile(MSG_FILE, last(MSG_FILE), NULL, 0, msg_buffer,
               sizeof(msg_buffer), msg_cache, sizeof(msg_cache));
   RgParam(0, 0, &nls_parms[0]);
   if (nls_parms[0].pb != NULL && nls_parms[0].cb != 0)
      log_msg(NLS_COMMAND_VERSION, nls_parms, sizeof(nls_parms));
   RgParam(1, 0, &sd_qic);
   if (sd_qic.pb == NULL || sd_qic.cb == 0)
      memcpy(&sd_qic, standard_msg(NLS_DEFAULT_DEVICE), sizeof(sd_qic));
   if (     ParseFileSpec(0, sd_qic.pb, sd_qic.cb, FALSE, NULL, NULL,
                          NULL, NULL, NULL, NULL, position, &position_len,
                          NULL, NULL, FALSE, 3) != ercOK
         || position_len != 0)
      ErrorExit(ercInvalidTapeSpecification);
   memcpy(&device_spec[1], sd_qic.pb,
          device_spec[0] = nls_parms[2].cb = sd_qic.cb);
   if ((erc = SeqAccessOpen(&qic_handle, sd_qic.pb, sd_qic.cb, NULL, 0,
                            MODE_READ, NULL, 0)) == ercServiceNotAvail)
      log_msg(NLS_NEED_SEQ_ACCESS_SERVICE, NULL, 0);
   else if (erc == ercTapeInUse)
      log_msg(NLS_DEVICE_IN_USE, nls_parms, sizeof(nls_parms));
   CheckErc(erc);
   for (cartridge = 'A'; cartridge <= 'B'; cartridge++) {
      insert_cartridge(device_spec, qic_handle, cartridge);
      alignment_error |= read_cartridge(qic_handle, cartridge);
   }
   log_msg((alignment_error) ? NLS_DEVICE_MISALIGNED : NLS_ALIGNMENT_OK,
           nls_parms, sizeof(nls_parms));
   Exit();

}


pragma Page(1);
/*-----------------------------------------------------------------------------
 Prompt the user to insert the next cartridge and allow a chance to correct
 any off-line or not ready conditions before we start reading any data. */

private void insert_cartridge(char device_spec[], unsigned qic_handle,
                              char cartridge) {

   Boolean retry;
   signed long residual;
   unsigned erc, i, seq_status;
   sdType nls_parms[] = {         &device_spec[1], device_spec[0],
                                  &cartridge, sizeof(cartridge),
                         (void *) &erc, sizeof(erc)};

   log_msg(NLS_INSERT_CARTRIDGE, nls_parms, sizeof(nls_parms));
   if (!proceed())
      ErrorExit(ercOperatorIntervention);
   do {
      for (i = 0; i < 10; i++)
         if ((erc = SeqAccessStatus(qic_handle, &seq_status,
                                    sizeof(seq_status),
                                    &residual)) == ercTapeStatusUnavailable)
            ErrorExit(erc);
         else if (erc != ercTapeReset)
            break;
      if (erc != ercTapeNotReady && erc != ercTapeBusy && erc != ercTapeReset)
         erc = SeqAccessControl(qic_handle, CTRL_REWIND, SYNCHRONIZE,
                                &residual);
      if (erc != ercOK) {
         if (     erc == ercTapeNotReady || erc == ercTapeBusy
               || erc == ercTapeReset)
            log_msg(NLS_DEVICE_NOT_READY, nls_parms, sizeof(nls_parms));
         else
            log_msg(NLS_POSITIONING_ERROR, nls_parms, sizeof(nls_parms));
         log_msg(NLS_REQUEST_INTERVENTION, NULL, 0);
         retry = proceed();
      }
   } while (erc != ercOK && retry);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Read all ten files on the specified cartridge and compare them, block by
 block, against the data pattern expected.  Note that if a data transfer error
 occurs, we skip to the next file instead of trying more read operations.
 This is done to minimize the number of error messages the user could get with
 a misaligned QIC drive. */

private Boolean read_cartridge(unsigned qic_handle, char cartridge) {

   Boolean cartridge_error = FALSE, data_compare_error;
   char qic_data[512];
   signed i;
   signed long residual;
   struct {
      char file_id[6];
      char data_id[16];
      char data_pattern[484];
      char block_id[6];
   } template;
   static unsigned file_position[] = {NLS_CENTERED,
                                      NLS_ABOVE_POSITIVE, NLS_ABOVE_POSITIVE,
                                      NLS_BELOW_NEGATIVE, NLS_BELOW_NEGATIVE,
                                      NLS_BELOW_POSITIVE, NLS_BELOW_POSITIVE,
                                      NLS_ABOVE_NEGATIVE, NLS_ABOVE_NEGATIVE,
                                      NLS_CENTERED, NLS_CENTERED,
                                      NLS_BELOW_POSITIVE, NLS_BELOW_POSITIVE,
                                      NLS_ABOVE_NEGATIVE, NLS_ABOVE_NEGATIVE,
                                      NLS_ABOVE_POSITIVE, NLS_ABOVE_POSITIVE,
                                      NLS_BELOW_NEGATIVE, NLS_BELOW_NEGATIVE,
                                      NLS_CENTERED};
   unsigned block, erc, file, j;
   sdType nls_parms[] = {(void *) &cartridge, sizeof(cartridge),
                         (void *) &file, sizeof(file),
                                  NULL, 0,
                         (void *) &erc, sizeof(erc)};

   memset(&template.file_id[0], 0xB0, sizeof(template.file_id));
   memset(&template.data_id[0], 0xA0, sizeof(template.data_id));
   memset(&template.data_pattern[0], 0x29, sizeof(template.data_pattern));
   for (file = 1; file <= 10; file++) {
      memcpy(&nls_parms[2],
             standard_msg(file_position[(10 * ((cartridge & 0x0F) - 1))
                                        + file - 1]), sizeof(nls_parms[2]));
      log_msg(NLS_READING_FILE, nls_parms, sizeof(nls_parms));
      template.file_id[last(template.file_id) - 1] = 0xB0 | (file / 10);
      template.file_id[last(template.file_id)] = 0xB0 | (file % 10);
      block = 0;
      data_compare_error = FALSE;
      while ((erc = SeqAccessRead(qic_handle, qic_data, sizeof(qic_data),
                                  &residual)) == ercOK) {
         j = ++block;
         for (i = last(template.block_id); i >= 0; i--) {
            template.block_id[i] = 0xB0 | (j % 10);
            j /= 10;
         }
         data_compare_error |= (memcmp(qic_data, &template,
                                       sizeof(qic_data)) != 0);
      }
      if (erc == ercTapeFileMark) {
         cartridge_error |= data_compare_error;
         log_msg((data_compare_error) ? NLS_DATA_ERROR : NLS_DONE, NULL, 0);
      } else {
         cartridge_error = TRUE;
         log_msg(NLS_TAPE_IO_ERROR, nls_parms, sizeof(nls_parms));
         if (erc == ercTapeIoError) {
            log_msg(NLS_SKIPPING_FILE, NULL, 0);
            erc = SeqAccessControl(qic_handle, CTRL_SCAN_FILEMARK, 1,
                                   &residual);
            log_msg((erc == ercOK) ? NLS_DONE : NLS_TAPE_IO_ERROR, nls_parms,
                    sizeof(nls_parms));
         }
         CheckErc(erc);
      }
   }
   CheckErc(SeqAccessControl(qic_handle, CTRL_REWIND, IMMEDIATE, &residual));
   log_msg(NLS_UPDATE_LABEL, NULL, 0);
   return(cartridge_error);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Hard-coded message texts, to be used if the nationalized message file cannot
 be opened by InitMsgFile. */

private msg_descriptor_type msg_descriptor[] = {
   default_msg(NLS_VID, "[Vid]"),
   default_msg(NLS_COMMAND_VERSION, "%0S %1S\n\n"),
   default_msg(NLS_DEFAULT_DEVICE, "[QIC]"),
   default_msg(NLS_NEED_SEQ_ACCESS_SERVICE, "The Sequential Access service "
                                            "must be installed before tape(s) "
                                            "can be used\n"),
   default_msg(NLS_DEVICE_IN_USE, "%2S is in use by another process\n"),
   default_msg(NLS_INSERT_CARTRIDGE, "Please insert Alignment Test Cartridge "
                                     "%1S in %0S"),
   default_msg(NLS_PROMPT, "\n    (Press GO to confirm, CANCEL to deny, or "
                           "FINISH to exit)\n"),
   default_msg(NLS_DEVICE_NOT_READY, "Device %0S is off line or not ready\n"),
   default_msg(NLS_POSITIONING_ERROR, "Error %2N while positioning medium on "
                                      "device %0S\n"),
   default_msg(NLS_REQUEST_INTERVENTION, "Please correct the problem.  Do you "
                                         "wish to continue?"),
   default_msg(NLS_READING_FILE, "Cartridge %0S, File %1N (%2S) ... "),
   default_msg(NLS_CENTERED, "centered"),
   default_msg(NLS_ABOVE_POSITIVE, "above centerline/positive azimuth offset"),
   default_msg(NLS_ABOVE_NEGATIVE, "above centerline/negative azimuth offset"),
   default_msg(NLS_BELOW_POSITIVE, "below centerline/positive azimuth offset"),
   default_msg(NLS_BELOW_NEGATIVE, "below centerline/negative azimuth offset"),
   default_msg(NLS_DONE, "done\n"),
   default_msg(NLS_DATA_ERROR, "\n    ***** Data comparison error\n"),
   default_msg(NLS_TAPE_IO_ERROR, "\n    ***** Data transfer (%3N) error\n"),
   default_msg(NLS_SKIPPING_FILE, "Skipping to next file on cartridge ... "),
   default_msg(NLS_UPDATE_LABEL, "\nAfter the cartridge is rewound, remove it "
                                 "and\n    update the Tape Certification "
                                 "Record\n\n"),
   default_msg(NLS_DEVICE_MISALIGNED, "\n***** ALIGNMENT ERROR(S) DETECTED "
                                      "FOR %2S\n"),
   default_msg(NLS_ALIGNMENT_OK, "\nCORRECT ALIGNMENT VERIFIED FOR %2S\n"),
   default_msg(0, "")};

pragma Page(1);
/*-----------------------------------------------------------------------------
 Descriptor information for the above-defined messages may be returned by
 calling the function below with a message number.  If found, the address of
 the sdType that describes the message is returned.  Callers should be careful
 that their code can handle a null sdType (in case they have asked for a
 message not in the table). */

private sdType *standard_msg(unsigned msg_num) {

   static char msg[256];
   static sdType sd_msg = {msg};
   unsigned i;

   for (i = 0; i < last(msg_descriptor); i++)
      if (msg_num == msg_descriptor[i].msg_num)
         if (GetMsgUnexpanded(msg_num, msg, sizeof(msg), &sd_msg.cb) == ercOK)
            return(&sd_msg);
         else
            return((sdType *) &msg_descriptor[i].msg);
   return((sdType *) &msg_descriptor[i].msg);

}

/*-----------------------------------------------------------------------------
 Expand the requested message and display it on [Vid]. */

private void log_msg(unsigned msg_index, sdType nls_parms[],
                     unsigned nls_parms_len) {

   char msg[256];
   unsigned msg_len;
   sdType *sd_msg;

   sd_msg = standard_msg(msg_index);
   CheckErc(ExpandLocalMsg(nls_parms, nls_parms_len, sd_msg->pb, sd_msg->cb,
                           msg, sizeof(msg), &msg_len, FALSE));
   CheckErc(OutputBytesWithWrap(msg, msg_len));

}

/*-----------------------------------------------------------------------------
 Request keyboard input and wait for user response (one of CANCEL, FINISH or
 GO).  FINISH requests a clean exit here (with a zero error code) while CANCEL
 may cuase program termination or other actions depending upon the context. */

private Boolean proceed(void) {

   char cmd, reply[4];
   unsigned reply_len;

   log_msg(NLS_PROMPT, NULL, 0);
   while (ENLS_GetChar(NULL, 0, reply, &reply_len,
                       ENLS_KBD_DIRECT_MODE_0) == ercOK) {
      if (ENLS_MapCharToStdValue(reply, &cmd) == ercOK)
         switch (cmd) {
            case ENLS_GO:
               return(TRUE);

            case ENLS_CANCEL:
               return(FALSE);

            case ENLS_FINISH:
               Exit();
         }
      Beep();
   }
   return(FALSE);			/* Unexpected ENLS problem... */

}
