/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  bit_map.c                                                                */
/*                                                                           */
/*                                                                           */
/*  Functions to retrieve and verify CTOS disk volume structures.            */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  10/01/91  130D.05  D. Gilson     /  Fix bogus Dup sector messages        */
/*                                      UCF # 42912654                       */
/*  06/13/91  121J.04  P. Johansson  /  Fix GP fault when bit map length is  */
/*                                      exactly the right size for the disk  */
/*                                      volume.                              */
/*  04/18/91  121J.03  P. Johansson  /  If disk size was not a multiple of   */
/*                                      8192, erroneous bit map errors were  */
/*                                      reported past the disk size.         */
/*  03/25/91  121H.02  P. Johansson  /  Enhance to support multiple volumes. */
/*  12/19/90  121F.01  P. Johansson  /  Fix bug when invalid VHB has volume  */
/*                                      capacity greater than allocation     */
/*                                      bit map range.                       */
/*  12/09/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Sequential Access service that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define AllocMemorySL
#define DeallocMemorySL
#define QueryBigMemAvail
#define QueryMemAvail

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define FhbType
#define sdType
#define vhb_type

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "archive.h"
#include "archive_msgs.h"
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern display_filename_once(void);
extern exit_with_msg(unsigned erc, unsigned nls_msg_index);
extern log_msg(unsigned nls_msg_index, unsigned signature, sdType nls_parms[],
               unsigned nls_parms_len);
extern unsigned read_pages(void *buffer, unsigned long vda,
                           unsigned page_offset, unsigned pages);

/* Error return codes used by this module */

#define FsErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

extern filesys_type filesys;
extern vhb_type vhb;

/* Static variables global within this manuscript */

private unsigned *bit_map[8];
private unsigned bit_map_alloc_length[8];
private unsigned long bit_map_length;

pragma Page(1);
/*-----------------------------------------------------------------------------
 At the outset of the verification phase of the backup, fetch the allocation
 bit map into memory so that utilized sectors may be marked as File Header
 blocks are processed for all the files on the disk.  Note that for large
 disks (i.e. formatted storage capacities greater than 256 Mb) the bit map
 does not fit in one 64 Kb segment and must be handled accordingly. */

void fetch_bit_map(void) {

   unsigned bits, bit_mask, erc, i, pages;
   unsigned long available_memory = 0;
   unsigned *bit_map_segment;


   if (vhb.magicWd == MAGIC || vhb.magicWd == MAGIC_PC)
      filesys.vda_eom =
                 (unsigned long) (vhb.cylindersPerDisk * vhb.tracksPerCylinder)
               * (unsigned long) (vhb.sectorsPerTrack * vhb.bytesPerSector);
   else
      filesys.vda_eom = vhb.volumeCapacity;
   bit_map_length = ((unsigned long) vhb.allocPageCnt) * PAGE_SIZE;
   if (filesys.vda_eom / (PAGE_SIZE * 8) > bit_map_length) {
      sdType nls_parms[] = {(void *) &filesys.vda_eom,
                                     sizeof(filesys.vda_eom)};
      
      filesys.invalid = TRUE;
      log_msg(NLS_BIT_MAP_TOO_SMALL, 1, nls_parms, sizeof(nls_parms));
   }
   memset(bit_map, 0, sizeof(bit_map));
   if (QueryBigMemAvail(&available_memory) != ercOK)
      if (QueryMemAvail(&available_memory) == ercOK)
         available_memory *= PARAGRAPH_SIZE;
   if (bit_map_length > available_memory) {
      log_msg(NLS_CANT_VERIFY_BIT_MAP, 1, NULL, 0);
      return;
   }
   for (i = 0; i <= bit_map_length / SEGMENT_SIZE; i++) {
      if (bit_map_length - (i * SEGMENT_SIZE) >= SEGMENT_SIZE) {
         bit_map_alloc_length[i] = 0xFFFF;	/* Actually forces 64 Kb */
         pages = PAGES_PER_SEGMENT;
      } else {
         bit_map_alloc_length[i] = bit_map_length - (i * SEGMENT_SIZE);
         pages = bit_map_alloc_length[i] / PAGE_SIZE;
      }
      if ((erc = AllocMemorySL(bit_map_alloc_length[i], &bit_map[i])) != ercOK)
         exit_with_msg(erc, 0);
      if ((erc = read_pages(bit_map[i], vhb.lfaAllocBase,
                            i * PAGES_PER_SEGMENT, pages)) != ercOK) {
         filesys.invalid = TRUE;
         log_msg(NLS_BIT_MAP_IO_ERROR, 1, NULL, 0);
      }
   }
   for (i = 0; i <= bit_map_length / SEGMENT_SIZE; i++) {
      bit_map_segment = bit_map[i];
      if (i == bit_map_length / SEGMENT_SIZE)
         if (filesys.vda_eom / (PAGE_SIZE * 8) > bit_map_length)
            offset_of(bit_map_segment) = bit_map_alloc_length[i];
         else
            offset_of(bit_map_segment) = (filesys.vda_eom / (PAGE_SIZE * 8))
                                          & 0xFFFE;
      do {
         bits = *--bit_map_segment;
         for (bit_mask = 1; bit_mask != 0; bit_mask <<= 1)
            if (bits & bit_mask)
               filesys.free_pages++;
      } while (offset_of(bit_map_segment));
   }
   if (filesys.vda_eom / (PAGE_SIZE * 8) < bit_map_length) {
      bits = bit_map_segment[(filesys.vda_eom / (PAGE_SIZE * 8)) / 2];
      for (bit_mask = 1 << ((filesys.vda_eom / PAGE_SIZE & 0x000F) - 1);
            bit_mask != 0; bit_mask >>= 1)
         if (bits & bit_mask)
            filesys.free_pages++;
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 After all explicit files (for which there are File Header Blocks), hidden
 files (such as the directory pages) and pages with medium defects have been
 accounted for, the bit map in memory should be a uniform collection of ones!
 Scan for any zeros: these represent pages in limbo that are either allocated
 to a file but not marked in the bit map or are actually free but are
 erroneously marked in the bit map.  After the results are summarized, release
 the memory used for the bit map (it is going to come in handy for sort space
 to process filenames in directories). */

void verify_bit_map(void) {

   unsigned bits, *bit_map_segment, bit_mask, i;
   unsigned long vda = 0, bit_map_eom = bit_map_length * (PAGE_SIZE * 8);
   sdType nls_parms[] = {(void *) &filesys.free_pages,
                                  sizeof(filesys.free_pages),
                         (void *) &vhb.cFreePages, sizeof(vhb.cFreePages),
                         (void *) &vda, sizeof(vda)};

   if (bit_map[0] == NULL)
      return;			/* Bit map was never fetched from the disk */
   log_msg(NLS_VERIFYING_BIT_MAP, 1, NULL, 0);
   for (i = 0; i <= bit_map_length / SEGMENT_SIZE; i++) {
      bit_map_segment = bit_map[i];
      do
         if (vda >= bit_map_eom) {
            filesys.invalid = TRUE;
            log_msg(NLS_PAGE_NOT_IN_BIT_MAP, 1, nls_parms, sizeof(nls_parms));
            vda += PAGE_SIZE;
         } else {
            bits = *bit_map_segment++;
            for (bit_mask = 1; bit_mask != 0; bit_mask <<= 1) {
               if ((bits & bit_mask) == 0) {
                  filesys.invalid = TRUE;
                  log_msg(NLS_PAGE_IN_LIMBO, 1, nls_parms, sizeof(nls_parms));
               }
               if ((vda += PAGE_SIZE) >= filesys.vda_eom)
                  break;
            }
         }
      while (vda < filesys.vda_eom);
   }
   log_msg(NLS_FREE_PAGES, 1, nls_parms, sizeof(nls_parms));
   if (filesys.free_pages != vhb.cFreePages) {
      filesys.invalid = TRUE;
      log_msg(NLS_FREE_PAGES_DISAGREEMENT, 1, nls_parms, sizeof(nls_parms));
   }
   for (i = last(bit_map); i != 0xFFFF; i--)	/* Return bit map memory */
      if (bit_map[i] != NULL)
         DeallocMemorySL(bit_map[i], bit_map_alloc_length[i]);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Each time an allocated disk run is encountered in a File Header Block, this
 function is called to set the corresponding bits in the allocation bit map to
 one (i.e. to release them, mark them as if they were free).  If any of the
 bits are already one, this is interpreted as a duplicate allocation error
 (even though it could have been caused by the bit map update error at the
 time the file was created). */

void release_disk_run(unsigned long vda, unsigned long length) {

   unsigned *bit_map_segment;
   unsigned bit, segment;
   unsigned long page, pages;

   if (vda + length > filesys.vda_eom) {
      sdType nls_parms[] = {(void *) &vda, sizeof(vda),
                            (void *) &length, sizeof(length)};

      display_filename_once();
      filesys.invalid = TRUE;
      log_msg(NLS_RUN_EXCEEDS_EOM, 1, nls_parms, sizeof(nls_parms));
      length = filesys.vda_eom - vda;
   }
   page = vda / PAGE_SIZE;
   if ((pages = length / PAGE_SIZE) == 0) {
      sdType nls_parms[] = {(void *) &vda, sizeof(vda),
                            (void *) &length, sizeof(length)};

      display_filename_once();
      filesys.invalid = TRUE;
      log_msg(NLS_ZERO_LENGTH_RUN, 1, nls_parms, sizeof(nls_parms));
      return;
   }
   if (        (bit_map_segment = bit_map[segment = page / BITS_PER_SEGMENT])
            == NULL
         ||    (offset_of(bit_map_segment) = page / 8 & 0xFFFE)
            >= bit_map_alloc_length[segment])
      return;		/* This VDA falls outside of the bit map in memory */
   bit = 1 << (page & 0x0F);
   while (pages--) {
      if (((*bit_map_segment ^= bit) & bit) == 0) {
         sdType nls_parms[] = {(void *) &vda, sizeof(vda)};

         *bit_map_segment |= bit;
         display_filename_once();
         filesys.invalid = TRUE;
         log_msg(NLS_DUPLICATE_ALLOCATION, 1, nls_parms, sizeof(nls_parms));
      }
      vda += PAGE_SIZE;
      if ((bit <<= 1) == 0) {
         bit = 1;
         bit_map_segment++;
         if (offset_of(bit_map_segment) >= bit_map_alloc_length[segment])
            return;	/* We have exceeded the range of the bit map */
         if (offset_of(bit_map_segment) == 0)
            bit_map_segment = bit_map[++segment]; /* m05 chgd from segment++ */
      }
   }

}
