/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  Arguments.c                                                              */
/*                                                                           */
/*                                                                           */
/*  Functions to retrieve and verify Executive command form parameters.      */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  12/19/91  130E.01  D. Gilson     /  Added multiple drive support to be   */
/*                                      able to backup to more than 1 drive. */
/*  12/14/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#define debug

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <ctype.h>
#include <intel80X86.h>
#include <stdlib.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Archive utilities that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define BuildFileSpec
#define CompactDateTime
#define CSubparams
#define NlsYesNoOrBlank
#define NlsYesOrNo
#define ExpandDateTime
#define GetDateTime
#define NlsParseTime
#define ParseFileSpec
#define RgParam

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define FhbType;
#define sdType

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "archive.h"
#include "archive_msgs.h"
pragma Pop(List);

/* External variables imported by this module */

extern archive_type archive;

/* Other external functions in this application invoked by this module */

extern log_msg(unsigned nls_msg_index, unsigned signature, sdType nls_parms[],
               unsigned nls_parms_len);
extern sdType *standard_msg(unsigned msg_num);

/* Error return codes used by this module */

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

/* Static variables global within this manuscript */

/* Function prototypes defined before the functions themselves are declared */

unsigned convert_date_time(unsigned i, unsigned n, void *destination);
void get_archive_datasets(unsigned param_num); 		/* m01 */
unsigned verify_file_specifications(unsigned i, unsigned n, char *file_spec,
                                    unsigned file_spec_max, Boolean path_only);

/* m01... */
pragma Page(1);
/*-----------------------------------------------------------------------------
 This function determines the number of archive datasets and placed them in 
 archive structure.
 */

void get_archive_datasets(unsigned param_num) {

   unsigned erc = ercOK, i;
   sdType sd;
   
   archive.device_num = 0;
   if ((archive.device_num_last = CSubparams(param_num)) != 0)
      for (i = 0; i < archive.device_num_last; i++)
         if (RgParam(param_num, i, &sd) == ercOK) {
            memcpy(&archive.spec[i][1], sd.pb, sd.cb);
            archive.spec[i][0] = sd.cb;
         }
}
/* ...m01 */
pragma Page(1);
/*-----------------------------------------------------------------------------
 This function is driven by a table that specifies for the ith parameter on
 the Executive command form a) it's data type and conversion procedure,
 b) it's storage destination once it is in internal form and c) appropriate
 NLS message number information to display error messages if required.  Note
 that the list of files is a special case that is not copied to an internal
 variable (since it may require more space than is at hand, it is kept in the
 VLPB). */

unsigned parse_parameters(param_descr_type *param_descr, unsigned n) {

   unsigned erc = ercOK, i, j, k, msg;
   sdType sd;
   char string[6];

   for (i = 1; i < n; i++)
      if ((j = CSubparams(i)) != 0 && RgParam(i, 0, &sd) == ercOK) {
         msg = 0;				/* No news is good news! */
         switch (param_descr[i].type) {
            case hex_param:
               break;
   
            case decimal_param:
               if (sd.cb > last(string))
                  msg = NLS_NUMBER_INVALID;
               else
                  for (k = 0; k < sd.cb; k++)
                     if (isdigit(sd.pb[k]))
                        string[k] = sd.pb[k];
                     else {
                        msg = NLS_NUMBER_INVALID;
                        break;
                     }
               if (msg == 0) {
                  string[k] = 0;
                  k = atoi(string);
                  if (param_descr[i].size == 1)
                     *((char *) param_descr[i].destination) = k;
                  else
                     *((unsigned *) param_descr[i].destination) = k;
               }
               break;
   
            case ascii_param:
               memcpy((char *) param_descr[i].destination + 1, sd.pb,
                      *((char *) param_descr[i].destination) =
                                             _min(param_descr[i].size, sd.cb));
               break;
   
            case Boolean_param:
               if (sd.cb) {
                  if (     NlsYesOrNo(NULL, &sd, param_descr[i].destination)
                        != ercOK)
                     msg = NLS_YES_OR_NO_EXPECTED;
               } else
                  *((Boolean *) param_descr[i].destination) = FALSE;
               break;
   
            case fuzzy_Boolean_param:
               if (NlsYesNoOrBlank(NULL, &sd,
                                   param_descr[i].destination) != ercOK)
                  msg = NLS_YES_OR_NO_EXPECTED;
               break;
   
            case date_param:
               msg = convert_date_time(i, j, param_descr[i].destination);
               break;

            case path_param:
               msg = verify_file_specifications(i, 1,
                                                param_descr[i].destination,
                                                param_descr[i].size, TRUE);
               break;

            case file_param:
               msg = verify_file_specifications(i, 1,
                                                param_descr[i].destination,
                                                param_descr[i].size, FALSE);
               break;

            case files_param:
               *((unsigned *) param_descr[i].destination) = j;
               msg = verify_file_specifications(i, j, NULL, 0, FALSE);
               break;

            default:
               break;
         }
         if (msg) {
            sdType *nls_parms;

            erc = ercSyntax;
            nls_parms = standard_msg(param_descr[i].nls_msg);
            log_msg(NLS_SYNTAX_ERROR, 1, nls_parms, sizeof(*nls_parms));
            log_msg(msg, 1, NULL, 0);
         }
   }
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Take the user supplied date and time and expand it in conjunction with
 today's date and time.  If there are no errors, the standard internal
 date/time format is returned to the destination variable specified. */

unsigned convert_date_time(unsigned i, unsigned n, void *destination) {

   char ascii_time[128] = {0};
   unsigned erc, j;
   exp_date_time_type expanded_time;
   sdType sd;
   unsigned long system_time;

   for (j = 0; j < n; j++) {
      if (RgParam(i, j, &sd) != ercOK)
         return(NLS_DATE_TIME_INVALID);
      if (sd.cb) {
         if (ascii_time[0])
            ascii_time[++ascii_time[0]] = ' ';
         if (ascii_time[0] + sd.cb > last(ascii_time))
            return(NLS_WRONG_NUMBER_ARGUMENTS);
         memcpy(&ascii_time[ascii_time[0] + 1], sd.pb, sd.cb);
         ascii_time[0] += sd.cb;
      }
   }
   if (     (erc = GetDateTime(&system_time)) == ercOK
         && (erc = ExpandDateTime(system_time, &expanded_time)) == ercOK) {
      expanded_time.hour = 0;
      expanded_time.minute = 0;
      expanded_time.second = 0;
      expanded_time.day_of_week = 0xFE;		/* Absent day of week is OK */
      if ((erc = NlsParseTime(NULL, ascii_time, &expanded_time)) == ercOK) {
         if (expanded_time.day_of_week == 0xFE)
            expanded_time.day_of_week = 0xFF;
         erc = CompactDateTime(&expanded_time, destination);
      }
   }
   if (erc == ercOK)
      return(0);
   else if (erc >= ercYearOutOfRange && erc <= ercInvalidTimeOfDay)
      return(NLS_YEAR_INVALID + (erc  - ercYearOutOfRange));
   else
      return(NLS_DATE_TIME_INVALID);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 As a preliminary sanity check, all the file specifications provided are
 scanned to insure that they are in one of the forms:

	{Node}[Volume]<Directory>Filename
	[Volume]<Directory>Filename
	<Directory>Filename
	Filename

 If the 'path_only' parameter is TRUE, the filename may optionally be omitted
 from any of the above specifications unless a null specification is the
 result.

 The optional ^password may be appended to any of the above.  If the form of
 the file specification is OK, the lengths of the individual elements are also
 checked for maxima and minima as appropriate.  One bad apple spoils the whole
 lot and verification of the file specifications stops as soon as it is
 encountered. */
 
unsigned verify_file_specifications(unsigned i, unsigned n, char *file_spec,
                                    unsigned file_spec_max,
                                    Boolean path_only) {

   char node[MAX_NODE_LENGTH], volume[MAX_VOLUME_LENGTH],
        directory[MAX_DIRECTORY_LENGTH], filename[MAX_FILENAME_LENGTH],
        password[MAX_PASSWORD_LENGTH];
   unsigned node_len, volume_len, directory_len, filename_len, password_len;
   unsigned file_spec_len, j, k, p;
   sdType sd;

   for (j = 0; j < n; j++) {
      if (RgParam(i, j, &sd) != ercOK)
         return(NLS_RGPARAM_ERROR);
      k = ParseFileSpec(0, sd.pb, sd.cb, TRUE, node, &node_len, volume,
                        &volume_len, directory, &directory_len, filename,
                        &filename_len, password, &password_len, FALSE, 0);
      if (path_only)	/* Interested in {Node}[Volume]<Directory> only? */
         k &= ~FILENAME_PARSE_ERROR;	/* Yes, ignore filename errors */
      if (k == 0) {
         if (file_spec != NULL) {
            memset(file_spec, 0, file_spec_max);
            BuildFileSpec(0, &file_spec[1], &file_spec_len, NULL,
                          file_spec_max, FALSE, node, node_len, volume,
                          volume_len, directory, directory_len, filename,
                          filename_len, FALSE, password, password_len, TRUE,
                          0);
            file_spec[0] = file_spec_len;
         }
      } else {
         for (p = 0; p < 6; p++)
            if ((k & (1 << p)) != 0)
               switch (p) {
                  case 0:  return(NLS_NODE_INVALID);
      
                  case 1:  return(NLS_VOLUME_INVALID);
      
                  case 2:  return(NLS_DIRECTORY_INVALID);
      
                  case 3:  return(NLS_FILENAME_INVALID);
      
                  case 4:  return(NLS_PASSWORD_INVALID);
      
                  case 5:  return(NLS_FILE_SPEC_INVALID);
               }
         return(NLS_FILE_SPEC_INVALID);
      }
   }
   return(0);

}
