/*
*	Program Name:	BOOTER
*
*	Filename:	flash.c
*
*	$Log:   /b/gregs/i960/util/flash.c_v  $
 * 
 *    Rev 1.2   12 Oct 1993 10:38:20   franks
 * No change.
 * 
 *    Rev 1.1   29 Sep 1993 10:27:02   franks
 * No change.
 * 
 *    Rev 1.0   14 Jul 1993 10:24:56   gregs
 * Initial revision.
 * 
 *    Rev 1.1   14 May 1992 11:03:18   kwok
 * Do not modify fault_cnt.
 * 
 *    Rev 1.0   30 Mar 1992 17:08:58   pvcs
 * Initial revision.
*
*	Creation Date:	N/A
*
*	Date:		9.16.91
*
*	Version:	1.3
*
*	Programmers:	K Kong
*
*	Modifications:	5.17.91	K Kong
*			We use 10us instead of 6us for the 
*			write erase verify and write verify delay such that
*			we only have to maintain 10ms and 10us delays.
*
*			6.13.91	K Kong
*			Adding the function "CheckFlash" for diagnostic.
*			
*			9.16.91 K Kong
*			The data bus is now 16 bits rather than 32 bits.
*
*	Comments:	This file contains modules to program the
*			flash eprom.  It is based on the source
*			from nindy.  It has been modified to handle
*			AMD flash eproms as well as inter flash eproms.
*
*			It is assumed that:=
*			1	there are 4 banks of flash
*				eproms and each bank contains only one flash
*				eprom.  In other words, the data bus width is
*				32 bit wide and there are 4 flash eproms.
*			2	All Eproms are of the same type and from
*				the same manufacturer.
*			3	All addresses are properly aligned.
*				i.e. all addresses will be divisible by 4.
*			
*
*	Copyright (c) 1991 by Hughes LAN Systems
*
**/

/******************************************************************/
/* 		Copyright (c) 1989, Intel Corporation

   Intel hereby grants you permission to copy, modify, and 
   distribute this software and its documentation.  Intel grants
   this permission provided that the above copyright notice 
   appears in all copies and that both the copyright notice and
   this permission notice appear in supporting documentation.  In
   addition, Intel grants this permission provided that you
   prominently mark as not part of the original any modifications
   made to this software or documentation, and that the name of 
   Intel Corporation not be used in advertising or publicity 
   pertaining to distribution of the software or the documentation 
   without specific, written prior permission.  

   Intel Corporation does not warrant, guarantee or make any 
   representations regarding the use of, or the results of the use
   of, the software and documentation in terms of correctness, 
   accuracy, reliability, currentness, or otherwise; and you rely
   on the software, documentation and results solely at your own 
   risk.							  */
/******************************************************************/
#include <types.h>
#include <krnl.h>
#include "flash.h"

extern	int	TimeTicksInt();
extern	int	TimeSliceInt();
extern void	init_flash_timers(void);
extern void	TimeTicksSliceInt();
extern void	usecInt();
extern void	FlashDelay10Us(void);
extern void	FlashDelay10Ms(void);
extern int	fault_cnt;


static int	FlashSize;	/* total size of flash in bytes	*/
static int	RetriesErase;	/* number of retries for erase the flash */

static int	InitFlash(void *FlashAddress);
static int	EraseFlash(byte *start_addr, int size, int retries);
static int	program_zero(byte *start_addr, int size);
static int	WriteFlash(volatile byte* address, byte data);
static int	ProgOneBankFlash(void *feprom, byte *DataPtr, int size);

static word	Flash_Start = 0xe0000000;	/* start addr for flash */
static int	Bank_Size = 0x40000;		/* size of each bank */
static int	Bank_Number = 4;		/* number of banks */


ProgramFlash(void *feprom, byte *DataPtr, int size)
{
	int result = 0;
	int count = 0;
	volatile int delay;

	if ((word)feprom < Flash_Start ||
		Flash_Start + Bank_Number*Bank_Size < (word)feprom + size)
	{
		printf("Error: Program out of flash range\n");
		return 1;
	}

	if ((word)feprom != Flash_Start &&
		(word)feprom != Flash_Start + Bank_Size &&
		(word)feprom != Flash_Start + 2*Bank_Size)
	{
		printf("Error: Program not aligned in flash\n");
		return 1;
	}

	while (result == 0)
	{
		if (size <= Bank_Size)
		{
			printf("(%d) Program %xH bytes into %lxH\n",
				++count, size, feprom);
			for (delay=0; delay<0xff; delay++)
				ReSchedule();
			result = ProgOneBankFlash(feprom, DataPtr, size);
			break;
		}
		else
		{
			printf("(%d) Program %xH bytes into %lxH\n",
				++count, Bank_Size, feprom);
			for (delay=0; delay<0xff; delay++)
				ReSchedule();
			ReSchedule();
			ReSchedule();
			result = ProgOneBankFlash(feprom, DataPtr, Bank_Size);
			feprom += Bank_Size;
			DataPtr += Bank_Size;
			size -= Bank_Size;
		}
	}
	return result;
}


/*---------------------------------------------------------------
* It burns the data starting at address "DataPtr" size of 
* which is "size" bytes  into the flash eprom. The flash
* eprom address is "FlashStart".
* It will read the manufacturer's id and device id from 
* the flash to decide if we can handle the flash and 
* get the size of the flash.
* To program the flash:-
* 1	Program all bytes of the flash to 0
* 2	Erase the flash such that all bytes are 0xff
* 3	Program the flash to the appropriate value.
*		
* It is assumed that all addresses will be properly
* aligned and be divisible by 4 such that we can
* program/erase all 4 flash eproms at the same time.
*
* Returns:
*	0 if ok
*	1 otherwise
*---------------------------------------------------------------*/
ProgOneBankFlash(void *feprom, byte *DataPtr, int size)
{
	volatile byte	*write_data;
	int		i;
	int		ReturnCode = 0;
	word		ticks;
	int		InitStatus;

	putchar('\n');
	disable_wdt();
	VppUp();

	InitStatus = InitFlash((void *)feprom);
	if (InitStatus || size > FlashSize ||
		EraseFlash((void *)feprom, FlashSize, RetriesErase))
	{
		VppDown();
		Di();
		stop_timers();
		init_timers(TimeTicksInt, TimeSliceInt,
			1, krnl.krnl_slice_time);
		fault_cnt = 0;
		Ei();
		if (InitStatus == 0 && size > FlashSize)
		{
			printf("Error: File (%d) too big for flash (%d)\n",
				size, FlashSize);
		}
		return 1;
	}

	/* begin writing */
	write_data = (byte *)feprom;
	ticks = RealTimeTicks();
	for (i=0; i<size; i++)
	{
		/*
		*	Printf a '*' every second to 
		*	indicate that we are still alive.
		*/
		if ((RealTimeTicks() - ticks) > 100)
		{
			ticks = RealTimeTicks();
			putchar('*');
			ReSchedule();
		}
		if (WriteFlash(write_data, *DataPtr))
		{
			printf("Error: Cannot Program flash eprom at %lx",
				write_data);
			ReturnCode = 1;
			break;
		}
		else
		{
			write_data++;
			DataPtr++;
		}
	}

	printf("\n");
	ReSchedule();
	/* reset for reading */
	write_data = (byte *)feprom;
	*write_data = MCMD_RESET;	/* reset */
	*write_data = MCMD_RESET;	/* reset */
	*write_data = MCMD_SET_READ;	/* set for reading */
	VppDown();
	Di();
	stop_timers();
	init_timers(TimeTicksInt, TimeSliceInt, 1, krnl.krnl_slice_time);
	fault_cnt = 0;
	Ei();
	return ReturnCode;
}


/*---------------------------------------------------------------
* It reads the manufacturer's id and device id to check
* if we can handle the flash eproms.
* Right now, we only support Intel 28F020
*
* Return:
*	0 if ok
* 	1 otherwise
*---------------------------------------------------------------*/
CheckFlash(void *FlashAddress, int size)
{
	volatile byte *check;	/* points to the flash */
	byte	manu;
	byte	DeviceType;

	VppUp();
	/*
	*	Write to the flash command port to get
	*	the manufacture code and id
	*/
	check = FlashAddress;
	*check = SCMD_MANU_ID;

	/*	 
	*	find out if devices are really Flash
	*	read from address 0 for manufacture code and
	*	address 1 for type of device
	*/
	manu = *check;
	DeviceType = *(check + 1);

	/*
	*	Reset the flash to normal working mode
	*/
	*check = SCMD_RESET;
	*check = SCMD_RESET;
	*check = SCMD_SET_READ;
	VppDown();

	if (manu != INTEL)
	{
		printf("FLASH error: unknown manufacturer ID %X\n", manu);
		return 1;
	}
	if (DeviceType != INTEL_2048)
	{
		printf("FLASH error: unknown device ID %X\n", DeviceType);
		return 1;
	}
	return 0;
}


/*---------------------------------------------------------------
*
*---------------------------------------------------------------*/
TestProgramFlash(void *feprom)
{
	volatile byte	*write_data;
	int		i;
	int		ReturnCode = 0;
	word		ticks;
	byte		data = 0x5a;

	VppUp();

	if (InitFlash(feprom))
	{
		VppDown();
		Di();
		stop_timers();
		init_timers(TimeTicksInt, TimeSliceInt,
			1, krnl.krnl_slice_time);
		fault_cnt = 0;
		Ei();
		printf("\nFlash Error: cannot init flash\n");
		return 1;
	}

	if (EraseFlash((void *)feprom, FlashSize, RetriesErase))
	{
		VppDown();
		Di();
		stop_timers();
		init_timers(TimeTicksInt, TimeSliceInt,
			1, krnl.krnl_slice_time);
		fault_cnt = 0;
		Ei();
		printf("\nFlash Error: cannot erase flash\n");
		return 1;
	}

	/* begin writing */
	write_data = (byte *)feprom;
	ticks = RealTimeTicks();
	for (i=0; i<FlashSize; i++)
	{
		/*
		*	Printf a '.' every second to 
		*	indicate that we are still alive.
		*/
		if ((RealTimeTicks() - ticks) > 100)
		{
			ticks = RealTimeTicks();
			putchar('+');
			ReSchedule();
		}
		if (WriteFlash(write_data, data))
		{
			printf("Error: Cannot Program flash at %lx\n",
				write_data);
			ReturnCode = 1;
			break;
		}
		else
		{
			write_data++;
			data++;
		}
	}

	printf("\n");
	ReSchedule();
	/* reset for reading */
	write_data = (byte *)feprom;
	*write_data = MCMD_RESET;	/* reset */
	*write_data = MCMD_RESET;	/* reset */
	*write_data = MCMD_SET_READ;	/* set for reading */
	VppDown();
	Di();
	stop_timers();
	init_timers(TimeTicksInt, TimeSliceInt, 1, krnl.krnl_slice_time);
	fault_cnt = 0;
	Ei();
	return ReturnCode;
}


/*---------------------------------------------------------------
* It reads the manufacturer's id and device id to check
* if we can handle the flash eproms.
* It then inits "FlashSize" and "RetriesErase".
* Returns:
*	0 if ok
*	1 otherwise
*---------------------------------------------------------------*/
static InitFlash(void *FlashAddress)
{
	volatile byte	*check;		/* points to the flash	*/
	unsigned	*TypeTable;	/* to Intel Type or AmdType	*/
	byte		manufacture;	/* intel or amd ?		*/
	byte		type;		/* what sort of flash is it	*/
	int		ReturnCode;	/* true or false		*/

	check = FlashAddress;

	/*
	*	Write to the flash command port to get
	*	the manufacture code and id
	*/
	*check = SCMD_MANU_ID;

	/*	 
	*	find out if devices are really Flash
	*	read from address 0 for manufacture code and
	*	address 1 for type of device
	*/
	manufacture = *check;
/*	type = *(check + 1);    changed due to a wiring error on the board */
	type = *(check + 2);    /* addr lines a0 and a1 are reversed */


	/*
	*	Reset the flash to normal working mode
	*/
	*check = SCMD_RESET;
	*check = SCMD_RESET;
	*check = SCMD_SET_READ;

	if (manufacture != INTEL)
	{
		printf("FLASH error: unknown manufacturer ID %X\n",
			manufacture);
		return 1;
	}

	if (type != INTEL_2048)
	{
		printf("FLASH error: unknown device ID %X\n", type);
		return 1;
	}

	FlashSize = 0x40000;
	RetriesErase = MAX_RETRIES_ERASE_INTEL;

	Di();
	fault_cnt = 1;
	stop_timers();
	init_timers(TimeTicksInt, usecInt, 1, 1);
	Ei();
	return 0;
}


/*---------------------------------------------------------------
* It erases the flash eproms at address "start_addr" and
* of size "size" bytes.  
* It first programs the flash eproms to have the value of
* 0x00 before starting the flash erase algorithm.
* After erasure, the contents of the flash eproms will
* have the value of 0xff.
*
* Returns
*	0 if ok
*	1 otherwise
*---------------------------------------------------------------*/
static EraseFlash(byte *start_addr, int size, int retries)
{
	volatile byte	*erase_data;	/* location to be erased */
	register byte	data;
	byte		*end_data;	/* last address of flash eprom	*/
	word		ticks;

	if (program_zero(start_addr, size))
	{
		return 1;
	}

	printf("\n");
	ReSchedule();

	/* begin erasing */
	erase_data = start_addr;
	end_data = start_addr + size;
	ticks = RealTimeTicks();

	while (erase_data < end_data)
	{
		if ((RealTimeTicks() - ticks) > 100)
		{
			putchar('-');
			ReSchedule();
			ticks = RealTimeTicks();
		}
		*erase_data = MCMD_ERASE;	/* send erase cmd	*/
		*erase_data = MCMD_ERASE;	/* send erase cmd	*/
		FlashDelay10Ms();		/* standby for erasure	*/
		*erase_data = MCMD_ERASE_STOP;	/* send stop cmd	*/
		FlashDelay10Us();		/* wait 10 uS		*/
		FlashDelay10Us();		/* wait 10 uS		*/

		while ((data = *erase_data) == ERASED_PATTERN)
		{
			if ((RealTimeTicks() - ticks) > 100)
			{
				putchar('-');
				ReSchedule();
				ticks = RealTimeTicks();
			}
			erase_data++;
			/*
			*	If we reach the end of flash eprom,
			*	then we are done.
			*/
			if (erase_data >= end_data)
				break;
			*erase_data = MCMD_ERASE_STOP;	/* send stop cmd*/
			FlashDelay10Us();		/* wait 10 uS */
		}

		if (erase_data < end_data)
		{
			/*
			* flash not been erased properly,
			*/
			if (retries-- == 0)
			{
				/*
				* Timeout, we cannot erase one flash
				* reset the flash and return error
				*/
				*erase_data = MCMD_RESET;
				*erase_data = MCMD_RESET;
				*erase_data = MCMD_SET_READ;
				printf("Error: Cannot erase flash at %lx\n",
					erase_data);
				return 1;
			}
			/*putchar('=');*/
		}
	}

	printf("\n");
	ReSchedule();
	/* reset for reading */
	erase_data = start_addr;
	*erase_data = MCMD_RESET;	/* reset */
	*erase_data = MCMD_RESET;	/* reset */
	*erase_data = MCMD_SET_READ;	/* set for reading */
	return 0;
}


/*---------------------------------------------------------------
* It programs the flash eproms to the value of 0 such that
* they can be erased.
* Returns
*	0 if ok
*	1 otherwise
*---------------------------------------------------------------*/
static program_zero(start_addr, size)
byte  	*start_addr;
int	size;
{
	volatile byte	*write_data;
	int		i;
	word		ticks = RealTimeTicks();


	/* begin writing */
	write_data = start_addr;
	for (i=0; i<size; i++)
	{
		/*
		*	Printf a '.' every second to 
		*	indicate that we are still alive.
		*/
		if ((RealTimeTicks() - ticks) > 100)
		{
			ticks = RealTimeTicks();
			putchar('.');
			ReSchedule();
		}

		if(WriteFlash(write_data, 0))
		{
			/*
			*	Reset the flash eproms
			*/
			*start_addr = MCMD_RESET;
			*start_addr = MCMD_RESET;
			*start_addr = MCMD_SET_READ;
			printf("Error: Cannot clear flash at %lx",
				write_data);
			return 1;
		}
		else
			write_data++;
	}

	/* reset for reading */
	write_data = start_addr;
	*write_data = MCMD_RESET;	/* reset */
	*write_data = MCMD_RESET;	/* reset */
	*write_data = MCMD_SET_READ;	/* set for reading */
	return 0;
}


/*---------------------------------------------------------------
* It burns the "data" into the flash eprom at the address
* "address".  It is assumed that "address" is properly
* aligned.
* Return
*	0 if ok
*	1 otherwise
*---------------------------------------------------------------*/
static int WriteFlash(volatile byte *address, byte data)
{
	int	retries = MAX_RETRIES_WRITE;

	while (retries--)
	{
		*address = MCMD_WRITE; 		/* send write cmd */
		*address = data;		/* write data */
		FlashDelay10Us();		/* standby for program	*/
		*address = MCMD_WRITE_STOP;	/* send stop cmd */
		FlashDelay10Us();		/* wait 10 uS */

		if (*address == data)	/* verify data	*/
			return 0;
	}
	return 1;
}


ConfigFlash(start, size, banks)
word start;
int size;
int banks;
{
	Flash_Start = start;
	Bank_Size = size;
	Bank_Number = banks;
}
