
/*
 *	Program Name:	SNMP Program
 *
 *	Filename:	encode.c
 *
 *	$Log:   /b/gregs/i960/tcpip/snmp/encode.c_v  $
 * 
 *    Rev 1.2   12 Oct 1993 10:43:22   franks
 * No change.
 * 
 *    Rev 1.1   29 Sep 1993 10:36:40   franks
 * No change.
 * 
 *    Rev 1.0   14 Jul 1993 10:16:56   gregs
 * Initial revision.
 * 
 *    Rev 1.0   16 Apr 1992 18:26:38   pvcs
 * Initial revision.
 *
 *	Comments:	Initial version for the 960 platform
 *
 *	Copyright (c) 1992 by Hughes LAN Systems
 */

/****************************************************************************
 *     Copyright (c) 1986, 1988  Epilogue Technology Corporation
 *     All rights reserved.
 *
 *     This is unpublished proprietary source code of Epilogue Technology
 *     Corporation.
 *
 *     The copyright notice above does not evidence any actual or intended
 *     publication of such source code.
 ****************************************************************************/

#include <libfuncs.h>
#include <asn1.h>
#include <buffer.h>
#include <encode.h>
#include <snmp.h>
#include <syteksnm.h>


static	ALENGTH_T	A_SizeOfSubId(unsigned int);
static	void		A_EncodeSubId(unsigned int, ALENGTH_T (*)(),
OCTET_P );

/****************************************************************************
A_SizeOfInt -- Return total size that an integer would occupy when
               ASN.1 encoded (tag and length fields are not included).

Parameters:
        INT_32_T    The integer (signed 32 bit)

Returns: ALENGTH_T  Number of octets the integer would occupy if
		    in ASN.1 encoding
****************************************************************************/
ALENGTH_T
A_SizeOfInt(i)
INT_32_T i;
{
	if (i >= 0L)
		return (i <= 127 ? 1 : (i <= 32767 ? 2 : (i <= 8388607 ? 3 : 4)));
	else return (i >= -128 ? 1 : (i >= -32768 ? 2 : (i >= -8388608 ? 3 : 4)));
}

/****************************************************************************
A_SizeOfUnsignedInt -- Return total size that an unsigned integer would
		       occupy when ASN.1 encoded (tag and length fields
		       are not included).

Parameters:
        UINT_32_T    The integer (unsigned 32 bit)

Returns: ALENGTH_T  Number of octets the integer would occupy if
		    in ASN.1 encoding
****************************************************************************/
ALENGTH_T
A_SizeOfUnsignedInt(i)
UINT_32_T i;
{
	return (i <= 127 ? 1 : (i <= 32767 ? 2 : (i <= 8388607 ? 3 :
	    (i <= 0x7FFFFFFF ? 4 : 5))));
}

/****************************************************************************
A_SizeOfSubId -- Compute the number of bytes required to hold a
		 subidentifier from an object id.
		 ASN.1 encoded (tag and length fields are not included)

Parameters:
        unsigned int

Returns: ALENGTH_T  Number of octets needed in ASN.1 encoding
****************************************************************************/
static
ALENGTH_T
A_SizeOfSubId(i)
unsigned int i;
{
	return (i <= 127 ? 1 : (i <= 16383 ? 2 : 3));
}

/****************************************************************************
A_SizeOfObjectId -- Return total size that an object ID would occupy when
                ASN.1 encoded (tag and length fields are not included)

Parameters:
	OBJ_ID_T *	Pointer to the internal object Id structure

Returns: ALENGTH_T  Number of octets the object ID would occupy if
		    in ASN.1 encoding

Note: It is assumed by this routine that the object identifier has at least
two components.
****************************************************************************/
ALENGTH_T
A_SizeOfObjectId(objp)
OBJ_ID_T  *objp;
{
	ALENGTH_T leng;
	unsigned int  *cp = objp->component_list;
	int i;

	/* Compute the value of the first subidentifier from the values of the	*/
	/* first two components.				*/
	{
		unsigned int  x;
		x = *cp++;
		x = x * 40 + *cp++;
		leng = A_SizeOfSubId(x);
	}

	for (i = 2; i < objp->num_components; i++)
	{
		leng += A_SizeOfSubId(*cp++);
	}
	return leng;
}

/***************************************************************************
A_SizeOfPduType --  Return total size that an pdu type would occupy when
                    ASN.1 encoded 
Parameter:
      ATVALUE_T --  pdu_type

Returns: ALENGTH_T
****************************************************************************/
ALENGTH_T
A_SizeOfPduType(type_val)
ATVALUE_T type_val;
{
	ALENGTH_T count = 1;

	if ( type_val > 30 )
	{
		for( ; ;  )
		{
			if ( (type_val = type_val/128) > 0 )
			{
				count++;
				continue;
			}
			else
				break;
		}
		return count;
	}
	return 0;
}

/****************************************************************************
A_EncodeType -- Encode an ASN.1 type field into buffer.

Parameters:
        ATVALUE_T        The type value
        OCTET_T          A_IDCF_MASK flag values
        ALENGTH_T (*f()) Function to be called to take generated data
	OCTET_T *	 Parameter to be passed unchanged to the function.

Notes:  The function whose address is passed as a parameter is called zero
or more times to take away some accumulated data.  The function is called
with these parameters:
        OCTET_T *   The parameter (funcparm) passed to this routine
        OCTET_T *   The buffer where the data resides
        ALENGTH_T   The number of octets in the buffer.

The function should return the number of octets consumed, type ALENGTH_T.
The function should return a zero if it has taken all the data it wants.

Returns: nothing
****************************************************************************/
void
A_EncodeType(id, flags, func, funcparm)
ATVALUE_T	id;
ALENGTH_T	flags;
ALENGTH_T   (*func)();
OCTET_P     funcparm;
{
	flags &= A_IDCF_MASK;
	if (id <= (ATVALUE_T)30)
		{
			ALENGTH_T c;
			c = flags | id;
			(void)(*func)((OCTET_P)funcparm,(OCTET_P)&c,(ALENGTH_T)sizeof(OCTET_T));
		}
	else {
		/* Build a partial reverse order version of the result and then */
		/* reverse it again back to correct order */
		OCTET_T buff[5], reverse[4];  /* Can't handle more than 4 octets */
		OCTET_T *bp = buff;
		OCTET_T *rp = reverse;
		unsigned short int count = 0;      /* Should never exceed 4 */
		ALENGTH_T cnt;
		*bp++ = (flags & A_IDCF_MASK) | 0x1F;
		while (id > (ATVALUE_T)0)
		{
			*rp++ = (OCTET_T) id & 0x7F;
			id >>= 7;
			count++;
		}
		cnt = count + 1;
		while ((count--) > 1)
		{
			*bp++ = *(--rp) | 0x80;
		}
		*bp++ = *(--rp);
		(void)(*func)((OCTET_P)funcparm, (OCTET_P)buff, cnt);
	}
}

/****************************************************************************
A_EncodeLength -- Encode an ASN.1 definite form length field into buffer.

Parameters:
        ALENGTH_T        Length to be encoded
        ALENGTH_T (*f()) Function to be called to take generated data
	OCTET_T *	 Parameter to be passed unchanged to the function.

Notes:  The function whose address is passed as a parameter is called zero
or more times to take away some accumulated data.  The function is called
with these parameters:
        OCTET_T *   The parameter (funcparm) passed to this routine
        OCTET_T *   The buffer where the data resides
        ALENGTH_T   The number of octets in the buffer.

The function should return the number of octets consumed, type ALENGTH_T.
The function should return a zero if it has taken all the data it wants.

Returns: nothing
****************************************************************************/
void
A_EncodeLength(leng, func, funcparm)
ALENGTH_T	leng;
ALENGTH_T   (*func)();
OCTET_P     funcparm;
{
	if (leng <= (ALENGTH_T)127)
		{
			OCTET_T c;
			c = (OCTET_T) leng;
			(void)(*func)((OCTET_P)funcparm,(OCTET_P)&c,(ALENGTH_T)sizeof(OCTET_T));
		}
	else {
		OCTET_T buff[OCTETS_PER_INT32+1];
		OCTET_T reverse[OCTETS_PER_INT32];
		OCTET_T *bp = buff;
		OCTET_T *rp = reverse;
		unsigned short int count = 0; /* Never exceeds OCTETS_PER_INT32 */
		ALENGTH_T cnt;
		while (leng > 0)
		{
			*rp++ = (OCTET_T) leng;
			leng >>= 8;
			count++;
		}
		*bp++ = (OCTET_T)(((OCTET_T) count) | (OCTET_T) 0x80);
		cnt = count + 1;
		while ((count--) > 0)
		{
			*bp++ = *(--rp);
		}
		(void)(*func)((OCTET_P)funcparm, (OCTET_P)buff, cnt);
	}
}

/****************************************************************************
A_EncodeInt -- generate ASN.1 format of integer (WITH TYPE & LENGTH)

Parameters:
    ATVALUE_T        The type value
    OCTET_T          A_IDC_MASK flag values
    INT_32_T	     The integer to convert (signed 32 bit)
    ALENGTH_T (*f()) Function to be called to take generated data
    OCTET_T *	     Parameter to be passed unchanged to the function.

Notes:  The function whose address is passed as a parameter is called zero
or more times to take away some accumulated data.  The function is called
with these parameters:
        OCTET_T *   The parameter (funcparm) passed to this routine
        OCTET_T *   The buffer where the data resides
        ALENGTH_T   The number of octets in the buffer.

The function should return the number of octets consumed, type ALENGTH_T.
The function should return a zero if it has taken all the data it wants.

Returns: nothing

****************************************************************************/
void
A_EncodeInt(id, flags, value, func, funcparm)
ATVALUE_T	id;
ALENGTH_T	flags;
INT_32_T	value;
ALENGTH_T   (*func)();
OCTET_P     funcparm;
{
	ALENGTH_T leng;
	OCTET_T   *rp;
	OCTET_T	  buff[OCTETS_PER_INT32];

	leng = A_SizeOfInt(value);
	A_EncodeType(id, flags & A_IDC_MASK, func,(OCTET_P)funcparm);
	A_EncodeLength(leng, func,(OCTET_P)funcparm);
	rp = buff + (unsigned short int)leng;
	for(;;)
	{
		*(--rp) = (OCTET_T) value;
		if (rp == buff) break;
		value >>= 8;
	}

	(void)(*func)((OCTET_P)funcparm, (OCTET_P)buff, leng);
}

/****************************************************************************
A_EncodeUnsignedInt -- generate ASN.1 format of integer (WITH TYPE & LENGTH)
		       where the local form of the integer is unsigned.

Parameters:
    ATVALUE_T        The type value
    OCTET_T          A_IDC_MASK flag values
    UINT_32_T	     The integer to convert (unsigned 32 bit)
    ALENGTH_T (*f()) Function to be called to take generated data
    OCTET_T *	     Parameter to be passed unchanged to the function.

Notes:  The function whose address is passed as a parameter is called zero
or more times to take away some accumulated data.  The function is called
with these parameters:
        OCTET_T *   The parameter (funcparm) passed to this routine
        OCTET_T *   The buffer where the data resides
        ALENGTH_T   The number of octets in the buffer.

The function should return the number of octets consumed, type ALENGTH_T.
The function should return a zero if it has taken all the data it wants.

Returns: nothing

   Bug fix : Anthony Chung

     unsigned int32 need 5 bytes to do the encoding

****************************************************************************/
void
A_EncodeUnsignedInt(id, flags, value, func, funcparm)
ATVALUE_T	id;
ALENGTH_T	flags;
UINT_32_T	value;
ALENGTH_T   (*func)();
OCTET_P     funcparm;
{
	ALENGTH_T leng;
	OCTET_T   *rp;
	OCTET_T	  buff[OCTETS_PER_UINT32];

	leng = A_SizeOfUnsignedInt(value);
	A_EncodeType(id, flags & A_IDC_MASK, func,(OCTET_P)funcparm);
	A_EncodeLength(leng, func,(OCTET_P)funcparm);
	rp = buff + (unsigned short int)leng;
	/***Anthony Chung 6/1/89*/
	for(;;)
	{
		*(--rp) = (OCTET_T) value;
		if (rp == buff) break;
		value >>= 8;
	}

	(void)(*func)((OCTET_P)funcparm, (OCTET_P)buff, leng);
}

/****************************************************************************
A_EncodeOctetString -- Generate ASN.1 format of octet string (WITH TYPE &
		       LENGTH)

Parameters:
    ATVALUE_T        The type value
    OCTET_T          A_IDC_MASK flag values
    OCTET_T *	     Address of the string
    ALENGTH_T	     Length of the string
    ALENGTH_T (*f()) Function to be called to take generated data
    OCTET_T *	     Parameter to be passed unchanged to the function.

Notes:  The function whose address is passed as a parameter is called zero
or more times to take away some accumulated data.  The function is called
with these parameters:
        OCTET_T *   The parameter (funcparm) passed to this routine
        OCTET_T *   The buffer where the data resides
        ALENGTH_T   The number of octets in the buffer.

The function should return the number of octets consumed, type ALENGTH_T.
The function should return a zero if it has taken all the data it wants.

****************************************************************************/
void
A_EncodeOctetString(id, flags, osp, oslen, func, funcparm)
ATVALUE_T	id;
ALENGTH_T	flags;
OCTET_P	osp;
ALENGTH_T	oslen;
ALENGTH_T	(*func)();
OCTET_P	funcparm;
{
	/* Do a primitive encoding */
	A_EncodeType(id, flags & A_IDC_MASK, func,(OCTET_P)funcparm);
	A_EncodeLength(oslen, func, (OCTET_P)funcparm);
	if (oslen != 0) 
		(void)(*func)((OCTET_P)funcparm, (OCTET_P)osp, oslen);
}

/****************************************************************************
A_EncodeSubId -- generate ASN.1 format of a subidentifier from an
		 object identifier

Parameters:
    unsigned int	The subidentifier to encode
    ALENGTH_T (*f())	Function to be called to take generated data
    OCTET_T *	    	Parameter to be passed unchanged to the function.

Notes:  The function whose address is passed as a parameter is called zero
or more times to take away some accumulated data.  The function is called
with these parameters:
        OCTET_T *   The parameter (funcparm) passed to this routine
        OCTET_T *   The buffer where the data resides
        ALENGTH_T   The number of octets in the buffer.

The function should return the number of octets consumed, type ALENGTH_T.
The function should return a zero if it has taken all the data it wants.

Returns: nothing

****************************************************************************/
static
void
A_EncodeSubId(value, func, funcparm)
unsigned int	value;
ALENGTH_T		(*func)();
OCTET_P    		funcparm;
{
	ALENGTH_T leng;
	OCTET_T   *rp;
	OCTET_T	  buff[OCTETS_PER_INT32];
	OCTET_T   last;

	leng = A_SizeOfSubId(value);

	for(rp = buff + leng, last = 0x00; rp != buff;)
	{
		*(--rp) = (OCTET_T)(value & 0x007F) | last;
		value >>= 7;
		last = 0x80;
	}

	(void)(*func)((OCTET_P)funcparm, (OCTET_P)buff, leng);
}

/****************************************************************************
A_EncodeObjectId -- generate ASN.1 format of Object ID (WITH TYPE & LENGTH)

Parameters:
    ATVALUE_T        The type value
    OCTET_T          A_IDC_MASK flag values
    OBJ_ID_T *	     Pointer to the internal object Id structure
    ALENGTH_T (*f()) Function to be called to take generated data
    OCTET_T *	     Parameter to be passed unchanged to the function.

Notes:  The function whose address is passed as a parameter is called zero
or more times to take away some accumulated data.  The function is called
with these parameters:
        OCTET_T *   The parameter (funcparm) passed to this routine
        OCTET_T *   The buffer where the data resides
        ALENGTH_T   The number of octets in the buffer.

The function should return the number of octets consumed, type ALENGTH_T.
The function should return a zero if it has taken all the data it wants.

Returns: nothing

****************************************************************************/
void
A_EncodeObjectId(id, flags, objp, func, funcparm)
ATVALUE_T	id;
ALENGTH_T	flags;
OBJ_ID_T  *objp;
ALENGTH_T   (*func)();
OCTET_P     funcparm;
{
	ALENGTH_T leng;
	unsigned int  *cp = objp->component_list;
	int i;

	leng = A_SizeOfObjectId((OBJ_ID_T  *)objp);
	A_EncodeType(id, flags & A_IDC_MASK, func, (OCTET_P)funcparm);
	A_EncodeLength(leng, func, (OCTET_P)funcparm);

	/* Merge the first two components of the object identifier to form the	*/
	/* first subidentifier.							*/
	{
		unsigned int x;
		x = *cp++;
		x = x * 40 + *cp++;
		A_EncodeSubId(x, func, (OCTET_P)funcparm);
	}

	/* Handle the remaining components, each as its own subidentifier.	*/
	for (i = 2; i < objp->num_components; i++)
	{
		A_EncodeSubId(*cp++, func, (OCTET_P)funcparm);
	}
}

/****************************************************************************

NAME:  A_EncodeHelper

PURPOSE:  Collect encoded data from the ASN.1 encoding routines and
	  place it into a buffer.

PAREMETERS:
        EBUFFER_T * The "opaque" parameter (funcparm) passed to the encoding
		    routines.
        OCTET_T *   The buffer where the encoded data resides
        ALENGTH_T   The number of encoded octets in the buffer.

RETURNS:  Returns the number of octets consumed, zero is returned if
          no more data is desired.  (This may not, however, prevent
	  subsequent calls.)

RESTRICTIONS:  
	Can not handle length > 64K

BUGS:  

****************************************************************************/

ALENGTH_T
A_EncodeHelper(ebp, bp, leng)
EBUFFER_P ebp;
OCTET_P   bp;
ALENGTH_T leng;
{
	ALENGTH_T actual = 0;
	actual = min(leng, ebp->remaining);
	if (actual != 0)
		{
			memcpy((OCTET_P)ebp->next_bp, (OCTET_P)bp, (unsigned short)actual);
			ebp->remaining -= actual;
			ebp->next_bp += (unsigned short)actual;
		}
	return (actual);
}
