/**			       
*
*	Program Name:	nim960 bootp client
*
*	Filename:	bootp.c
*
*	$Log:   /b/gregs/i960/tcpip/bootp/bootp.c_v  $
 * 
 *    Rev 1.2   12 Oct 1993 10:39:18   franks
 * No change.
 * 
 *    Rev 1.1   29 Sep 1993 10:27:46   franks
 * No change.
 * 
 *    Rev 1.0   14 Jul 1993 10:13:26   gregs
 * Initial revision.
 * 
 *    Rev 1.5   16 Jun 1992 17:28:28   vinay
 * Changed back to rev 1.3
 * 
 *    Rev 1.4   16 Jun 1992 14:48:08   vinay
 * Changed printf to include Errorlog Function
 * 
 *    Rev 1.3   04 May 1992 12:16:52   kwok
 * break out of the while loop if udp_send returns ok.
 * 
 *    Rev 1.2   27 Apr 1992 19:25:54   pvcs
 * Retry if upd_send returns the error code of no resource error.
*
*	Creation Date:	4.23.91
*
*	Date:		11.14.91
*
*	Version:	1.1
*
*	Programmers:	K Kong
*
*	Modifications:	K Kong	1.1	11.14.91
*			Adding support for the BOOTP server gateway.
*
*	Comments:	It sends bootp packets until we get a response
*			or timeout.
*
*	Copyright (c) 1991 by Hughes LAN Systems
*
**/


#include <types.h>
#include <krnl.h>
#include <sockets.h>
#include <netbuf.h>
#include <ip.h>
#include <udp.h>
#include <tcpip.h>
#include <error.h>
#include <bootp.h>


#define	LongCopy(to, from)	ipcopy(to, from)



struct	bootpinfo	bootpinfo;


static	volatile int	BootpDone = 0;
static	volatile int	BootpTimeout = 0;
static	ulong	bp_xid;
static	TIMER	BootpTimer;
static	byte	vm_rfc1048[4] = VM_RFC1048;	/* magic for rfc1048	*/
static void	bp_timeout();

static void	BootpReceive();
NID	*MyNid();


/*
 * name		updboot		get config info via bootp
 *
 * synopsis	updboot(my_ip, ServerIp, filename)
 *		in_name	my_ip; <<	my ip address if known, 
 *					otherwise 0
 *		in_name	ServerIp; <<	Bootp server ip address if known,
 *					otherwise 0
 *		char	*filename; <<	generic boot file name
 *					or "" if unknown
 *
 * description	It gets my configuration parameters from the bootp server.
 *		Some of the following parameters may be returned from
 *		the bootp server:-
 *		1	my ip address
 *		2	bootp server ip address
 *		3	subset mask
 *		4	up to "BP_MAX_IP" default router ip addresses
 *		5	up to "BP_MAX_IP" Domain name server ip addresses
 *		6	up to "BP_MAX_IP" Time servers ip addresses
 *		7	Time offset in seconds of the local subnet from UTC
 *		8	The boot file name
 *		9	My host name
 *
 *		It should be noted that SOME or ALL of the above parameters
 *		MAY be returned from the bootp server.
 *		All the parameters returned will be stored in "bootpinfo".
 *
 * returns	TRUE	parameters have been copied into "bootpinfo"
 *		FALSE	failed
 */

udpboot(in_name my_ip, in_name ServerIp, char *filename)

	{
	struct	bootp	bootp;
	int	retries = 0;
	ulong	xid;
	int	err;
	int	timeout = INIT_TIMEOUT;
	UDPCONN	con;	/* the bootp connection	*/
	int	noresrc_retries = MAX_RETRIES;  /* no resource retries */

	memset(&bootp, 0, UNPADDED_BOOTP_SIZE);
	BootpDone = 0;
	BootpTimeout = 0;
	if ((con = udp_open(UDP_BOOTPC, BootpReceive, 0)) == NULL)
		{
		printf("Error: cannot open bootp connection\n");
		udp_close(con);
		return 0;
		}
	/*
	 *	Fill in the request packet
	 */
	bootp.bp_op = BOOTREQUEST;	/*	this is a request	*/
	bootp.bp_htype = MyHtype();	/*	my hardware addr type	*/
	bootp.bp_hlen = 6;		/*	hardware addr length	*/
	bootp.bp_xid = rand();
	bp_xid = bootp.bp_xid;
	bootp.bp_ciaddr = my_ip;	/*	client ip address	*/
	bootp.bp_siaddr = ServerIp == 0 ? 0xffffffff : 0;
	bootp.bp_secs = htons(60);
	memcpy(bootp.bp_vend, vm_rfc1048, 4);
	ncopy(bootp.bp_chaddr, MyNid(0));
	strcpy(bootp.bp_file, filename);
	CreatTimer(&BootpTimer);

	while(noresrc_retries--)
		{
		if ((err = udp_send(bootp.bp_siaddr, UDP_BOOTPS, UDP_BOOTPC, 
			&bootp, UNPADDED_BOOTP_SIZE, _initp->in_me)) != 0)
			{
			if (err == ERR_NORSRC)
				{
				ReSchedule();
				}
			else
				break;
			}
		else
			break;
		}
	if (err != 0)
		{
		printf("Error: Cannot send bootp packet (%d)\n", err);
		udp_close(con);
		return 0;
		}
	StartTimerCall(&BootpTimer, timeout, bp_timeout, 0);
	while (!BootpDone)
		{
		if (BootpTimeout)
			{
			if (++retries >= MAX_RETRIES)
				{
				udp_close(con);
				return 0;
				}
			if ((err = udp_send(bootp.bp_siaddr, UDP_BOOTPS,
				UDP_BOOTPC, &bootp, UNPADDED_BOOTP_SIZE, 
				_initp->in_me)) != 0)
				{
				/*
				 *	Retry later if the error
				 *	is no resource.
				 */
				if (err == ERR_NORSRC)
					{
					ReSchedule();
					BootpTimeout++;
					continue;
					}
				printf("Error: Cannot send bootp packet(retries = %d)\n", retries);
				udp_close(con);
				return 0;
				}
			BootpTimeout = 0;
			timeout += 100;		/* add another second	*/
			if (timeout > MAX_TIMEOUT)
				timeout = MAX_TIMEOUT;
			StartTimerCall(&BootpTimer, timeout, bp_timeout, NULL);
			}
		ReSchedule();
		}
	/*
	 *	We are done, return my ip, server ip and filename
	 */
	BootpDone = 0;
	udp_close(con);
	return 1;
	}

/*
 * name		bp_timeout	bootp timeout
 *
 * synopsis	bp_timeout(void)
 *
 * description	This function is called when there is no valid response
 *		to our bootp request. It sets the timeout flag
 *		BootpTimeout.  The main task will detect this and
 *		re-transmit the request.
 *
 * returns	nothing
 */

static void bp_timeout()

	{
	BootpTimeout++;
	}

/*
 * name		BootpReceive	bootp receive upcall
 *
 * synopsis	BootpReceive(pkt, length, fhost, sport, dport, tag, lhost)
 *		PACKET	pkt; <<		incoming packet
 *		int	length; <<	of bootp data in "pkt".
 *		in_name	fhost; <<	"pkt" is from this host, the 
 *					bootp server
 *		uint	sport; <<	"pkt" is from this port number
 *					this should be UDP_BOOTPS
 *		uint	dport; <<	"pkt" is to this port number
 *					this should be UDP_BOOTPC
 *		void	tag; <<		unused
 *		in_name	lhost; <<	packet is for this host.
 *
 * description	This is called by the udp module when a bootp reply
 *		packet is received.  It checks if the packet is valid
 *		and if it contains rfc1048 style vendor information.
 *		It then copies the data into "bootpinfo" for later use.
 *
 * returns	nothing
 */

static void BootpReceive(pkt, length, fhost, sport, dport, tag, lhost)
PACKET	pkt;		/*	receiving packet		*/
int	length;		/*	of data in number of bytes	*/
in_name	fhost;		/*	ip address of the sender	*/
uint	sport;		/*	source port			*/
uint	dport;		/*	destination port		*/
void	*tag;		/* 	not used			*/
in_name	lhost;
	{
	struct bootp	*data = (struct bootp *)pkt->nb_prot;
	ulong	xid;


	LongCopy((short *)&xid, (short *)&data->bp_xid);
	if (sport != UDP_BOOTPS 	|| 
		dport != UDP_BOOTPC	|| 
		data->bp_op != BOOTREPLY ||
		xid != bp_xid	||
		length < UNPADDED_BOOTP_SIZE ||
		ncompare(data->bp_chaddr, MyNid(0)) == FALSE)
		{
		/*
		 *	Bad reply, ignore the packet
		 */
		in_free(pkt);
		return;
		}
	BootpDone++;
	ipcopy(&bootpinfo.yiaddr, &data->bp_yiaddr);
	ipcopy(&bootpinfo.siaddr, &data->bp_siaddr);
	strcpy(bootpinfo.file, data->bp_file);
	if (memcmp(data->bp_vend, vm_rfc1048, 4) == 0)
		Rfc1048Vender((byte *)data->bp_vend);
	else
		printf("unkown magic %X %X %X %X\n", data->bp_vend[0],
			data->bp_vend[1], data->bp_vend[2],
			data->bp_vend[3]);
	in_free(pkt);
	}
		
/*
 * Process a RFC 1048 style vendor field.
 */
/*
 * name		Rfc1048Vender	process a rfc 1048 style vendor field.
 *
 * synopsis	Rfc1048Vender(data)
 *		byte	*data; <<	the vendor field, 
 *
 * description	It extracts the information from the vendor field and 
 *		put it "bootpinfo".  It supports the following tags:
 *		
 *		-	subnet mask
 *		-	up to "BP_MAX_IP" default router ip addresses
 *		-	up to "BP_MAX_IP" Domain name server ip addresses
 *		-	up to "BP_MAX_IP" Time servers ip addresses
 *		-	Time offset in seconds of the local subnet from UTC
 *		-	My host name
 *		All other informations will be ignored.
 *		It doesn't check the vendor magic number.  The caller of
 *		this function must check "data" is in rfc 1048 style.
 *
 * returns	nothing
 */

Rfc1048Vender(data)
byte	*data;

	{
	byte	*EndOfData;	/*	end of vender data address	*/	
	int	tag;		/*	vender field tag		*/
	int	length;		/*	Length of tag data		*/

	/*
	 * Skip over magic number.
	 */
	data += 4;
	/*
	 * Parse options.
	 */
	for (EndOfData = data + BP_VENDOR_LENGTH - 4; data < EndOfData;)
		{
		tag = (int)*data++;
		/*
		 *	Skip any paddings and
		 *	if we see TAG_END, then we have
		 *	finished.
		 */
		if (tag == TAG_PAD)
			continue;
		else if (tag == TAG_END)
			return;

		length = *data++;
		if (length > EndOfData - data)
            		length = EndOfData - data;
		switch (tag)
			{
		case TAG_SUBNET_MASK:
			if (length == sizeof(in_name))
				{
				ipcopy(&bootpinfo.NetMask, data);
				}
			break;

		case TAG_TIME_OFFSET:
			if (length == 4)
				{
				LongCopy((short *)&bootpinfo.TimeOffset, (short *)data);
				bootpinfo.TimeOffset = ntohl(bootpinfo.TimeOffset);
				}
			break;

		case TAG_GATEWAY:
			AddServers(length / 4, bootpinfo.Router, data);
			break;

		case TAG_TIME_SERVER:
			AddServers(length / 4, bootpinfo.TimeServer, data);
			break;

		case TAG_NAME_SERVER:
			break;

		case TAG_DOMAIN_SERVER:
			AddServers(length / 4, bootpinfo.DomainNameServer, data);
			break;
		/*
		 *	These tags are not supported 
		 *	right now. We ignore them.
		 */
		case TAG_LOG_SERVER:
		case TAG_COOKIE_SERVER:
		case TAG_LPR_SERVER:
		case TAG_IMPRESS_SERVER:
			break;

		case TAG_HOSTNAME:
			/*
			 * The hostname tag combines both the hostname
			 * and the domain name.  It is not necessarily
			 * NULL terminated, so we have to check its length
			 * very carefully.
			 * First, copy the hostname part until we run out of
			 * tag, we run out of hostname space or we run into
			 * the domain portion (a period).
			 * Second, eat any part of hostname which was longer
			 * than our buffer.
			 * Third, eat any periods separating hostname and 
			 * domain.
			 * Fourth, copy the domain portion of the tag if there
			 * is any, again being carefull about string lengths.
			 */
			memcpy(bootpinfo.MyHostName, data, length);
			if (length >= BP_MYNAME_LENGTH)
				bootpinfo.MyHostName[BP_MYNAME_LENGTH - 1] = '\0';
			else
				bootpinfo.MyHostName[length] = '\0';
			break;

		case TAG_BOOTSIZE:
			/*
			 *	We don't care about the boot file size
			 *	ignore it.
			 */
		default:
			/*
			 * Don't understand this one so skip it.
			 */
			break;
			}
		data += length;
		}
	}

/*
 * name		AddServers	copy ip addresses
 *
 * synopsis	AddServers(entries, to, from)
 *		int	entries; <<	number of valid ip addresses in
 *					"from"
 *		in_name	to[]; >>	copy "entries" number of ip addresses
 *					to here
 *		in_name	from[]; <<	copy "entries" number of ip addresses
 *					from here
 *
 * description	It copies up to "entries" or BP_MAX_IP number,  which ever is 
 *		smaller, of ip address from "from" to "to".
 *
 * returns	nothing.
 */

AddServers(int entries, in_name to[], in_name from[])

	{
	/*
	 *	we can handle this many entries
	 */
	entries = entries < BP_MAX_IP ? entries : BP_MAX_IP;
	if (entries == 0)
		return;
	while (entries--)
		{
		/*
		 *	we call ipcopy to copy the ip address instead of
		 *	doing the assignment of "*to = *from" because
		 *	from may not be properly aligned on a 32 bit 
		 *	boundry.
		 */
		ipcopy(to++, from++);
		}
	}

MyHtype()
	{
	return 1;
	}


/*
 * name		BootpServer	bootp server receive upcall
 *
 * synopsis	BootpServer(pkt, length, fhost, sport, dport, tag)
 *		PACKET	pkt; <<		incoming packet
 *		int	length; <<	of bootp data in "pkt".
 *		in_name	fhost; <<	"pkt" is from this host, the 
 *					bootp server
 *		uint	sport; <<	"pkt" is from this port number
 *					this should be UDP_BOOTPS
 *		uint	dport; <<	"pkt" is to this port number
 *					this should be UDP_BOOTPC
 *		void	tag; <<		unused
 *		in_name	lhost; <<	my ip
 *
 * description	This is called by the udp module when a bootp server request 
 *		is received. If It is coming from the admin. bus, we
 *		have to put our ip address in the bp_giaddr and re-broadcast
 *		the packet to the network.
 *
 * returns	nothing
 */

BootpServer(pkt, length, fhost, sport, dport, tag, lhost)
PACKET	pkt;		/*	receiving packet		*/
int	length;		/*	of data in number of bytes	*/
in_name	fhost;		/*	ip address of the sender	*/
uint	sport;		/*	source port			*/
uint	dport;		/*	destination port		*/
void	*tag;		/* 	not used			*/
in_name	lhost;
	{
	struct bootp	*data = (struct bootp *)pkt->nb_prot;
	in_name		toip;
	in_name		giaddr;

	ipcopy(&giaddr, &data->bp_giaddr);
	switch (data->bp_op) 
		{
	case BOOTREQUEST:
		/*
		 *	We handle bootp request from the admin. bus only.
		 */
		if ((pkt->nb_flags & PKT_ADMIN) == 0 || giaddr != 0)
			{
			in_free(pkt);
			return;
			}
		/*	This is a request from the
		 *	admin. bus. we have to
		 *	put my ip address in the
		 * 	bp_giaddr field and send it
		 *	down to the network.
		 */
		ipcopy(&data->bp_giaddr, &_initp->in_me);
		sport = UDP_BOOTPS;
		ipcopy(&toip, &data->bp_siaddr);
		break;

	case BOOTREPLY:
		/*	This is a reply from a bootp
		 *	server for the SYSCARD if the
		 *	"giaddr" is me.. 
		 *	We redirect it to the admin.
		 * 	bus.
		 */
		ipcopy(&toip, &data->bp_yiaddr);
		if (giaddr != _initp->in_me)
			{
			in_free(pkt);
			return;
			}
		dport = UDP_BOOTPC;
		break;
	
	default:
		in_free(pkt);	/* bad opcode */
		return;
		}

	udp_send(toip, dport, sport, data, UNPADDED_BOOTP_SIZE, _initp->in_me);
	in_free(pkt);
	}
