/*--------------------------------------------------------------------------
 * File name:   brip.c
 *
 * Description: Contains send & receive routines for the BRIP protocol.
 *
 * Author:      Shaun Hui
 *
 * Module History:
 *
 * who  when            what
 * -------------------------------------------------------------------------
 * lsh  Aug-10-92       Module creation.
 * lsh	Mar-09-93	Added code for FDDI Con support.
 *--------------------------------------------------------------------------
 */
#include <types.h>
#include <krnl.h>
#include <defines.h>
#include <uart.h>
#include <memory.h>
#ifdef __FDDI_CON
#include "nvrecs.h"
#endif

#include "scc_defs.h"
#include "scc_buff.h"
#include <brip.h>
#include "member.h"
#define  SEGMENTS_OFF 	0x11
extern MBOX SccRcvMbox;
extern MBOX SccFreeMbox;
extern byte *get_scc_tx_buff();
extern struct Ring_Member_Table RingMemberTbl[];

extern int scc_rcv_count;
extern int scc_frame_count;
int prev_scc_rcv_count = 0;
extern int rcv_overrun_error;
extern int rcv_buff_gap;
extern int backplane_p2_missing;
BRIP_INFO 	BripInfo[MAX_MAC_CHANNEL];
BRIP_TIMER_INFO	BpTmrInfo[MAX_XMT_LINE+1];

TIMER	RspTimer;
TIMER	RetryTimer;
TIMER	KaTimer[MAX_MAC_CHANNEL];

int neighbor_err = 0;
int no_match_err = 0;
int brip_inited  = 0;

int brip_neighbor_err( short );

#ifdef yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy
byte get_device_type()
{
	return (1);
}
byte get_media_type()
{
	return (1);
}
/*--------------------------------------------------------------------------
 * get_media_speed()
 * 	returns the speed of the media.
 *	0 -	TR 	4Mbps
 *	1 -	TR 	16Mbps
 *	2 -	FDDI	100Mbps
 *--------------------------------------------------------------------------
 */
byte get_media_speed()
{
	return ( (byte) get_token_ring_speed());
}
#endif /*yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy*/


/*--------------------------------------------------------------------------
 * get_xmt_line()
 *	TR concentrator uses its own slot no. for transmit line.
 *--------------------------------------------------------------------------
 */
byte get_xmt_line()
{
	return ( GetSlotID() );
}


/*--------------------------------------------------------------------------
 * init_brip( mac_chnl )
 * Initialize the structure BripInfo of the specified mac channel.
 *--------------------------------------------------------------------------
 */
 
void init_brip( short mac_chnl )
{
        BripInfo[mac_chnl].state 		= BRIP_INIT_STATE;
        BripInfo[mac_chnl].device_type  	= get_device_type();
        BripInfo[mac_chnl].media_type		= get_media_type();
        BripInfo[mac_chnl].media_speed  	= get_media_speed();
        BripInfo[mac_chnl].hub_ring_nbr 	= 0;
        BripInfo[mac_chnl].xmt_line_nbr 	= get_xmt_line();
        BripInfo[mac_chnl].usn_line_nbr 	= 0;
        BripInfo[mac_chnl].dsn_line_nbr 	= 0;
        BripInfo[mac_chnl].rcv_req_count        = 0;
        BripInfo[mac_chnl].rcv_rsp_count        = 0;
        BripInfo[mac_chnl].rcv_rmv_count        = 0;
        BripInfo[mac_chnl].rcv_ins_count        = 0;
        BripInfo[mac_chnl].rcv_ka_count 	= 0;
        BripInfo[mac_chnl].rcv_unknown_count 	= 0;
        BripInfo[mac_chnl].rcv_byte_count	= 0;
        BripInfo[mac_chnl].xmt_req_count        = 0;
        BripInfo[mac_chnl].xmt_rsp_count        = 0;
        BripInfo[mac_chnl].xmt_rmv_count        = 0;
        BripInfo[mac_chnl].xmt_ins_count        = 0;
        BripInfo[mac_chnl].xmt_ka_count 	= 0;
        BripInfo[mac_chnl].xmt_byte_count	= 0;
        BripInfo[mac_chnl].xmt_req_retries 	= MAX_REQ_RETRY;
        BripInfo[mac_chnl].xmt_failure  	= 0;
        BripInfo[mac_chnl].send_ka_flag 	= 0;
        
}

/*--------------------------------------------------------------------------
 * init_brip_timer(  )
 * Initialize the timer field in the structure BpTmrInfo for all backplan 
 * transmit lines.
 *--------------------------------------------------------------------------
 */
 
void init_brip_timer( )
{
int	i;

	CreatTimer( &RspTimer );
	CreatTimer( &RetryTimer );
	for (i=0; i <= MAX_XMT_LINE; i++)
	{
		CreatTimer( &(BpTmrInfo[i].tmr) );
	}
}


/*--------------------------------------------------------------------------
 * init_ka_timer(  )
 * Initialize the keep_alive timer for each mac channel. 
 *--------------------------------------------------------------------------
 */
 
void init_ka_timer( )
{
int	i;

	for (i=0; i < MAX_MAC_CHANNEL; i++)
	{
		CreatTimer( &KaTimer[i] );
	}
}


/*--------------------------------------------------------------------------
 * insert_hub_ring( mac_chnl)
 * Insert into the hub ring for the given mac channel.
 *--------------------------------------------------------------------------
 */

void insert_hub_ring( short mac_chnl, byte ring_nbr )
{
	set_seven_segment_display( ring_nbr );

	printf("\nInserting into backplane ring %d\n", ring_nbr);
        /*------------------------------------
         * Remove ourself from current back
         * plane ring if we are inserted.
         *------------------------------------
         */
        if ( BripInfo[mac_chnl].state == BRIP_INSERTED_STATE )
        {
                send_rmv_frame( mac_chnl, 0xFF, 0xFF );
        	/*----------------------------------------------
		 * Switch our receive line to internal by-pass
		 * until the receive_rsp_timeout routine switches
		 * to our new USN's xmt line. This prevents us from
		 * receiving frames from the previous receive line
		 * which can cause problem in our MAC frame
		 * processing tasks.
        	 *----------------------------------------------
		 */

#ifdef __TRC
		*((volatile byte *)BP_RCV_LINE) = 0;
#endif

#ifdef __FDDI_CON
		notify_fddi_smt(EVENT_ISOLATE_AB, XMT_QUIET_LINE );
#endif
        }

        /*------------------------------------
         * Set our state to INIT and hub ring
         * to ring_nbr. Initialize our member
         * table.
         *------------------------------------
         */
        BripInfo[mac_chnl].state = BRIP_INIT_STATE;
        BripInfo[mac_chnl].hub_ring_nbr = ring_nbr;
        init_ring_member_table(mac_chnl);


        /*------------------------------------
         * If we are not stand-alone (hub ring 0)
         * start the response timer and send a
         * request frame. Insert ourselves when
         * timer expires.
         * If we are stand-alone, change to
         * INSERTED state, and switch receive
         * line to 0 for internal by-pass.
         *------------------------------------
         */
        if ( BripInfo[mac_chnl].hub_ring_nbr != 0 )
        {

        	/*------------------------------------
       		 * Set up response timeout call.
        	 *------------------------------------
         	 */
                StartTimerCall(&RspTimer,MAX_RSP_TIMEOUT,receive_rsp_timeout,mac_chnl);

        	/*------------------------------------
       		 * Send a REQ_RING_MEMBER broadcast.
        	 *------------------------------------
         	 */
                send_req_frame( mac_chnl, 0xFF, 0xFF );

        	/*------------------------------------
       		 * Set up send REQ_RING_MEMBER retry.
        	 *------------------------------------
         	 */
        	BripInfo[mac_chnl].xmt_req_retries 	= MAX_REQ_RETRY;
		StartTimerCall(&RetryTimer,MAX_RETRY_TIMEOUT,send_req_retry,(int)mac_chnl);
        }
        else
        {   
                BripInfo[mac_chnl].state = BRIP_INSERTED_STATE;
                BripInfo[mac_chnl].usn_line_nbr = 0;
                BripInfo[mac_chnl].dsn_line_nbr = 0;

#ifdef __TRC
                switch_rcv_line( mac_chnl, 0 );
                 
#endif

#ifdef __FDDI_CON
                /*------------------------------------
                 * If we are on FDDI media, notify SMT
		 * to isolate A & B ports if M ports 
		 * should be on the backplane.
                 *------------------------------------
                 */
		if (nvr_fddi_rec.UI_Defaults.Port1_PC_Type == PC_Type_M)
		{
		   notify_fddi_smt(EVENT_INSERT_A, 
				   BripInfo[mac_chnl].xmt_line_nbr );
		}

		if (nvr_fddi_rec.UI_Defaults.Port6_PC_Type == PC_Type_M)
		{
		   notify_fddi_smt(EVENT_INSERT_B, 
				  BripInfo[mac_chnl].xmt_line_nbr );
		}
#endif
		brip_inited = 1;

		StartTimerCall(&KaTimer[mac_chnl],MAX_KA_TIMEOUT,ka_monitor,(int) mac_chnl);

        }

}

                 
void brip_free_task()
{

SCC_BD 		*tx_bd;

	for (;;)
	{
                /*------------------------------------
                 * The task will be blocked when there
                 * is no message to receive.
                 *------------------------------------
                 */
		tx_bd = (SCC_BD *) RcvMessage(&SccFreeMbox);

		if ( tx_bd->bd_type == BD_XMT )
		{
			free_scc_bd ( tx_bd );
		}
		ReSchedule();
	}
}

void brip_rcv_task()
{
SCC_BD 		*rx_bd;
BRIP_FRAME 	*frame;
short		mac_chnl;
short		ring_nbr;
int		NoMatch;
u_short brip_ftype, brip_ver;
int	i;

	/* enter_debug();	 zzz gjs */
	init_brip( 0 );
	init_brip_timer();
	init_ka_timer();

	ring_nbr = (short)get_ring_number();
	insert_hub_ring( 0,ring_nbr);

	for (;;)
	{
                /*------------------------------------
                 * The task will be blocked when there
                 * is no message/frames to receive.
                 *------------------------------------
                 */
		rx_bd = (SCC_BD *) RcvMessage(&SccRcvMbox);

                frame = (BRIP_FRAME *)rx_bd->buff_ptr;

                NoMatch = 1;

		/*
		* intercept all frames for SCC monitor
		*/
		PrintSccFrame(frame);

                /*------------------------------------
                 * Check if hub ring number and media
                 * type belong to anyone of the mac
                 * channels.
		 * We only process BRIP frames.
                 *------------------------------------
                 */

                for (mac_chnl=0;mac_chnl<MAX_MAC_CHANNEL && NoMatch;mac_chnl++)
                {
                        if ((frame->hub_ring!=0) &&
			    (frame->frame_type_hi == BRIP_FRAME_TYPE) &&
			    (frame->hub_ring==BripInfo[mac_chnl].hub_ring_nbr)&&
                            (frame->media_type==BripInfo[mac_chnl].media_type))
                        {
                        /*------------------------------------
                         * Check if destination line number is
                         * a broadcast or to us.
                         *------------------------------------
                         */
                                if ((frame->dst_line == 0xFF ) ||
                                    (frame->dst_line == BripInfo[mac_chnl].xmt_line_nbr))
				{
                                	NoMatch = 0;
					break;
				}
                        }
                }

                /*------------------------------------
                 * Process frame if a match is found.
                 *------------------------------------
                 */
 
                if ( !NoMatch )
                {
                        /*------------------------------------
                         * Send to corresponding receive routine
                         * based on frame type.
                         *------------------------------------
                         */
                        switch ( frame->frame_type_lo )
                        {
                                case    REQ_RING_MEMBER:
                                        receive_req_frame(mac_chnl, (REQ_FRAME *)frame );
                                        break;
 
                                case    RSP_RING_MEMBER:
                                        receive_rsp_frame(mac_chnl, (RSP_FRAME *)frame );
					break;
	 
                                case    RMV_RING_MEMBER:
                                        receive_rmv_frame(mac_chnl, (RMV_FRAME *)frame );
                                        break;
 
                                case    INS_RING_MEMBER:
                                        receive_ins_frame(mac_chnl, (INS_FRAME *)frame );
                                        break;
 
                                case    KEEP_ALIVE:
                                        receive_ka_frame( mac_chnl, (KA_FRAME *)frame );
                                        break;
 
                                /*---------------------------------------
                                 * Receive an unknown frame type.
                                 *---------------------------------------
                                 */
                                default:
#ifdef SCC_DEBUG
					printf("\nRxUnknown\n");
#endif
                                        BripInfo[mac_chnl].rcv_unknown_count++;
					break;
			}
                }
		else
		{
			if ((frame->hub_ring != 0 )   &&
			    (frame->dst_slot != 0xFF) && 
			    (frame->dst_slot != GetSlotID() ))
			{
				rcv_overrun_error = 1;
#ifdef SCC_DEBUG
	printf("\nNO MATCH %d\t %x", frame->src_line, frame);
#endif
			}
		}
                /*------------------------------------
                 * Free the receive buffer descriptor.
                 *------------------------------------
                 */
		free_scc_bd ( rx_bd );
		ReSchedule();
	}
}


/*---------------------------------------------------------------------------
 * receive_req_frame
 *      process a request frame
 *
 *---------------------------------------------------------------------------
 */
void receive_req_frame( short mac_chnl, REQ_FRAME *frame )
{
BRIP_TIMER_INFO	*BpTmrPtr;
byte	rsp_bitmap[4], new_usn_line, new_dsn_line;
int	i;

#ifdef SCC_DEBUG
	printf("\nRxREQ %d\t %x", frame->src_line, frame);
	putchar('r');
#endif

        BripInfo[mac_chnl].rcv_req_count++;

	for(i=0; i< 4; i++)
	{
		rsp_bitmap[i] = frame->rsp_bmap[i];
	}
        /*----------------------------------------------
         * See if our xmt line is set in the requestor's
         * response bit map. If our bit is not set, send
         * a response frame to requestor, else requestor
         * has already seen our response.
         *----------------------------------------------
         */
        if ( IsRspNeeded( mac_chnl, (word*) &rsp_bitmap ) )
        {
                BripInfo[mac_chnl].xmt_req_retries = 3;
                send_rsp_frame( mac_chnl, frame->src_slot, frame->src_line );
        }

        /*----------------------------------------------
         * See if requestor is not in our member table.
         * If so, add it to the member table.
         *----------------------------------------------
         */
	if (!IsRingMember( mac_chnl, frame->src_line))
	{
		add_ring_member( mac_chnl, (BRIP_FRAME *)frame );

                /*----------------------------------------------
                 * Stop the timer on this line and start a timer
                 * call to delete this entry from the member tbl
                 * if no INS_RING_MEMBER frame is received when
                 * the timer expires.
                 *----------------------------------------------
                 */

                BpTmrPtr	= &BpTmrInfo[frame->src_line];
                BpTmrPtr->line	= frame->src_line;
                BpTmrPtr->channel= mac_chnl;

                StopTimer( &BpTmrPtr->tmr );
                StartTimerCall( &BpTmrPtr->tmr, MAX_INS_TIMEOUT,
			receive_ins_timeout, (int) BpTmrPtr);

	}
	else
	{

                /*----------------------------------------------
                 * See if requestor is in our member table and
                 * it has changed from INSERTED to INIT state. If
                 * so, change its state in the member table to
                 * INIT and check for any changes in the USN or
                 * DSN as a result. Possibly, the sender has been
                 * reset without removing itself first.
                 *----------------------------------------------
                 */
                if (IsStateChanged( mac_chnl, frame->src_line, BRIP_INIT_STATE))
                {
                        add_ring_member( mac_chnl, (BRIP_FRAME *)frame );
 
                        /*----------------------------------------------
                         * Stop the timer on this line and start a timer
                         * call to delete this entry from the member tbl
                         * if no INS_RING_MEMBER frame is received when
                         * the timer expires.
                         *----------------------------------------------
                         */
                        BpTmrPtr = &BpTmrInfo[frame->src_line];
                        BpTmrPtr->line = frame->src_line;
                        BpTmrPtr->channel = mac_chnl;
        
                        StopTimer( &BpTmrPtr->tmr );
                        StartTimerCall( &BpTmrPtr->tmr, MAX_INS_TIMEOUT,
				receive_ins_timeout, (int) BpTmrPtr);
        
                        /*----------------------------------------------
                         * If we are already inserted, we may need to
                         * update our DSN and/or USN lines if sender is
                         * our current USN and/or DSN. If we are in INIT
                         * state, we will determine our USN & DSN when
                         * our RSP_TIMER expires.
                         *----------------------------------------------
                         */
                        if (BripInfo[mac_chnl].state == BRIP_INSERTED_STATE )
                        {
                                if ((new_usn_line=get_usn(mac_chnl)) !=
					BripInfo[mac_chnl].usn_line_nbr)
                                {
                                        BripInfo[mac_chnl].usn_line_nbr = new_usn_line; 
#ifdef __TRC
                                        switch_rcv_line(mac_chnl, new_usn_line);
#endif

#ifdef __FDDI_CON
                                        /*------------------------------------
                                         * If we are on FDDI media, check if we
					 * need to isolate or insert A port. 
                                         *------------------------------------
					 */
					if ( new_usn_line == XMT_QUIET_LINE)
						notify_fddi_smt(EVENT_ISOLATE_A,
							XMT_QUIET_LINE);
					else
						notify_fddi_smt(EVENT_INSERT_A, 
							new_usn_line);
#endif
                                }
 
                                if ((new_dsn_line=get_dsn(mac_chnl)) !=
                                         BripInfo[mac_chnl].dsn_line_nbr)
                                {
                                        BripInfo[mac_chnl].dsn_line_nbr = new_dsn_line; 
#ifdef __FDDI_CON
                                        /*------------------------------------
                                         * If we are on FDDI media, check if we
					 * need to isolate or insert B port. 
                                         *------------------------------------
					 */
					if ( new_dsn_line == XMT_QUIET_LINE)
						notify_fddi_smt(EVENT_ISOLATE_B,
							XMT_QUIET_LINE);
					else
						notify_fddi_smt(EVENT_INSERT_B, 
							new_dsn_line);
#endif
				}
			}
		}
                                         
	
	}
}

/*---------------------------------------------------------------------------
 * receive_rsp_frame
 *
 *---------------------------------------------------------------------------
 */
void receive_rsp_frame( short mac_chnl, RSP_FRAME *frame )
{

BRIP_TIMER_INFO  *BpTmrPtr;
 
#ifdef SCC_DEBUG
	printf("\nRxRSP %d\t %x", frame->src_line, frame);
	putchar('p');
#endif
        switch( BripInfo[mac_chnl].state )
        {
                /*---------------------------------------
                 * Received a response. Add the responding
                 * slot and xmt line to our member table
                 * and reflect whether sender is in INIT
                 * state or INSERTED state.
                 *---------------------------------------
                 */
                case    BRIP_INIT_STATE:
			BripInfo[mac_chnl].rcv_rsp_count++;
                        add_ring_member( mac_chnl, (BRIP_FRAME *)frame );
                        if ( frame->state == BRIP_INIT_STATE )
                        {
                        	BpTmrPtr 	= &BpTmrInfo[frame->src_line];
                                BpTmrPtr->line	= frame->src_line;
                                BpTmrPtr->channel = mac_chnl;
                                StopTimer( &BpTmrPtr->tmr );
                                StartTimerCall( &BpTmrPtr->tmr, MAX_INS_TIMEOUT,
                                	receive_ins_timeout, (int) BpTmrPtr);
 
			}
                        else
                        {   
			/*--------------------------------------- 
                         * Received a response from an inserted
                         * device. Check if speed is the same. If
			 * not, we have a speed conflict and the
                         * inserted card has higher priority.
                         * Remove our card from the backplane ring.
                         * Change our ring nubmer to 0 (standalone)
                         * and notify upper layer.
                         *--------------------------------------- 
                         */
                         	if (frame->media_speed != BripInfo[mac_chnl].media_speed)
                                {
                                	send_rmv_frame( mac_chnl, 0xFF, 0xFF);
                                        insert_hub_ring( mac_chnl, 0 );
                                        /* notify_system( ); */
                                 }
                                 else
                                 	update_ring_member( mac_chnl, frame->src_line );
                         }
                         break;
                /*---------------------------------------
                 * Ignore the late response frame. Sender
                 * must be busy,since it should have
                 * responded before our response timer exp.
                 *---------------------------------------
                 */
                case    BRIP_INSERTED_STATE:
                        BripInfo[mac_chnl].rcv_rsp_count++;
                        break;
 
        }
}




/*---------------------------------------------------------------------------
 * receive_ins_frame
 *
 *---------------------------------------------------------------------------
 */
void receive_ins_frame( short mac_chnl, INS_FRAME *frame )
{
byte	usn_line, dsn_line;

#ifdef SCC_DEBUG
	printf("\nRxINS %d\t %x", frame->src_line, frame);
#endif

	BripInfo[mac_chnl].rcv_ins_count++;
	switch( BripInfo[mac_chnl].state )
	{
		/*---------------------------------------
		 * Received an insert frame. Update the
		 * the state of the sender to INSERTED
		 * in the member table. Do not perform
		 * any switching of receive line yet.
		 * Will switch when we are ready to insert.
		 *---------------------------------------
		 */
		case	BRIP_INIT_STATE:
			StopTimer( &BpTmrInfo[frame->src_line].tmr );
			update_ring_member( mac_chnl, frame->src_line );
			break;

		/*---------------------------------------
		 * Update the state of the sender to 
		 * INSERTED. Check if it becomes the new
		 * up or down stream neighbour. If USN,
		 * switch our receive line, reset the
		 * USN Keep alive timer. If DSN, ???
		 *---------------------------------------
		 */
		case	BRIP_INSERTED_STATE:
			StopTimer( &BpTmrInfo[frame->src_line].tmr );
			update_ring_member( mac_chnl, frame->src_line );
			if ((usn_line=get_usn(mac_chnl)) !=
				BripInfo[mac_chnl].usn_line_nbr)
			{
				BripInfo[mac_chnl].usn_line_nbr = usn_line;
#ifdef __TRC
				switch_rcv_line( mac_chnl, usn_line );
#endif
#ifdef __FDDI_CON
                                /*------------------------------------
                                 * If we are on FDDI media, check if we
				 * need to isolate or insert A port. 
                                 *------------------------------------
				 */
				if ( usn_line == XMT_QUIET_LINE)
					notify_fddi_smt(EVENT_ISOLATE_A, XMT_QUIET_LINE);
				else
					notify_fddi_smt(EVENT_INSERT_A, usn_line);
#endif
			}
			if ((dsn_line=get_dsn(mac_chnl)) != BripInfo[mac_chnl].dsn_line_nbr)
			{
				BripInfo[mac_chnl].dsn_line_nbr = dsn_line;
#ifdef __FDDI_CON
                                /*------------------------------------
                                 * If we are on FDDI media, check if we
				 * need to isolate or insert B port. 
                                 *------------------------------------
				 */
				if ( dsn_line == XMT_QUIET_LINE)
					notify_fddi_smt(EVENT_ISOLATE_B, XMT_QUIET_LINE);
				else
					notify_fddi_smt(EVENT_INSERT_B, dsn_line);
#endif
			}
			break;

	}
}


/*---------------------------------------------------------------------------
 * receive_ka_frame
 *
 *---------------------------------------------------------------------------
 */
void receive_ka_frame( short mac_chnl, KA_FRAME *frame )
{

byte	ring_nbr;

#ifdef SCC_DEBUG
	printf("\nRxKA %d\t %x", frame->src_line, frame);
#endif

	switch( BripInfo[mac_chnl].state )
	{
		/*---------------------------------------
		 * Received a keep alive. Ignore it. We
		 * are still initializing and have not
		 * sent the INS_RING_MEMBER frame yet.
		 * If a card has us in its member table as
		 * INSERTED state, it will be updated to
		 * INIT when we send the REQ_RING_MEMBER
		 * frame.
		 *---------------------------------------
		 */
		case	BRIP_INIT_STATE:
			BripInfo[mac_chnl].rcv_ka_count++;
			break;

		/*---------------------------------------
		 * 
		 *---------------------------------------
		 */
		case	BRIP_INSERTED_STATE:
			BripInfo[mac_chnl].rcv_ka_count++;

			/*---------------------------------------
			 * Check if sender is our USN. If so, see
			 * if its DSN is us.
			 *---------------------------------------
			 */
			if (frame->src_line == BripInfo[mac_chnl].usn_line_nbr)
			{	
#ifdef __FDDI_CON
			        if ((frame->src_line > BripInfo[mac_chnl].xmt_line_nbr) && (frame->dsn_line == XMT_QUIET_LINE))
				{
				}
				else
#endif

			        if (frame->dsn_line != BripInfo[mac_chnl].xmt_line_nbr)
				{
					if ( neighbor_err++ > MAX_NEIGHBOR_ERR )
					{
#ifdef SCC_DEBUG
	printf("\n***BRIP ERROR src %d !=dsn line %d\n",
	BripInfo[mac_chnl].xmt_line_nbr, frame->dsn_line);
#endif
						brip_neighbor_err( mac_chnl );
					}
					break;
				/* 
				 * discrepency, insert hub ring
				 */
				}
			}

			/*---------------------------------------
			 * Check if sender is our DSN. If so, see
			 * if its USN is us.
			 *---------------------------------------
			*/
			if (frame->src_line == BripInfo[mac_chnl].dsn_line_nbr)
			{
#ifdef __FDDI_CON
			        if ((frame->src_line < BripInfo[mac_chnl].xmt_line_nbr) && (frame->usn_line == XMT_QUIET_LINE))
				{
				}
				else
#endif
				if (frame->usn_line != BripInfo[mac_chnl].xmt_line_nbr)
				{
					if ( neighbor_err++ > MAX_NEIGHBOR_ERR )
					{
#ifdef SCC_DEBUG
	printf("\n***BRIP ERROR src %d !=usn line %d\n",
	BripInfo[mac_chnl].xmt_line_nbr, frame->usn_line);
#endif
						brip_neighbor_err( mac_chnl );
					}
					break;
					/* 
					 * discrepency, insert hub ring
					 */
				}
			}

			/*---------------------------------------
			 * Check if sender's USN is us. If so, see
			 * if it matches our DSN.
			 *---------------------------------------
		 	 */
			if (frame->usn_line == BripInfo[mac_chnl].xmt_line_nbr)
			{
#ifdef __FDDI_CON
			        if ((frame->src_line < BripInfo[mac_chnl].xmt_line_nbr) && (BripInfo[mac_chnl].dsn_line_nbr == XMT_QUIET_LINE))
				{
				}
				else
#endif
				if (frame->src_line != BripInfo[mac_chnl].dsn_line_nbr)
				{
					if ( neighbor_err++ >  MAX_NEIGHBOR_ERR)
					{
#ifdef SCC_DEBUG
	printf("\n***BRIP ERROR src usn !=xmt line\n");
#endif
						brip_neighbor_err( mac_chnl );
					}
					break;
					/* 
					 * discrepency, insert hub ring
					 */
				}
			}

			/*---------------------------------------
			 * Check if sender's DSN is us. If so, see
			 * if it matches our USN.
			 *---------------------------------------
			 */
			if (frame->dsn_line == BripInfo[mac_chnl].xmt_line_nbr)
			{
#ifdef __FDDI_CON
			        if ((frame->src_line > BripInfo[mac_chnl].xmt_line_nbr) && (BripInfo[mac_chnl].usn_line_nbr == XMT_QUIET_LINE))
				{
				}
				else
#endif
				if (frame->src_line != BripInfo[mac_chnl].usn_line_nbr)
				{
					if ( neighbor_err++ > MAX_NEIGHBOR_ERR)
					{
#ifdef SCC_DEBUG
	printf("\n***BRIP ERROR src dsn !=xmt line\n");
#endif
						brip_neighbor_err( mac_chnl );
					}
					break;
					/* 
					 * discrepency, insert hub ring
					 */
				}
			}


			/*------------------------------------------
			 * if we are standalone and still receive
			 * keep alive frames of the same ring,
			 * we are in err.
			 *------------------------------------------
			 */
			if ( BripInfo[mac_chnl].xmt_line_nbr == 
				BripInfo[mac_chnl].usn_line_nbr )
			{
				if (neighbor_err++ > MAX_NEIGHBOR_ERR)
				{
#ifdef SCC_DEBUG
	printf("\nStandalone ERR\n");	
#endif
					brip_neighbor_err( mac_chnl );
				}
				break;
			}
			/*---------------------------------------
			 * Check if sender is in member table. If
			 * so, reset the corresponding keep alive 
			 * counter to 1 whenever a ka frame is 
			 * received. Ka counters are decremented
					brip_neighbor_err( mac_chnl );
			 * whenever the ka timer expires. Else,
			 * ignore it.
			 *---------------------------------------
			 */

			if ( IsRingMember( mac_chnl, frame->src_line ))
			{
				neighbor_err = 0;
				reset_ka_counter( mac_chnl, frame->src_line );
				RingMemberTbl[mac_chnl].ka_expire[frame->src_line] = 0;
			}

			break;

	}
}


/*---------------------------------------------------------------------------
 * receive_rmv_frame
 *
 *---------------------------------------------------------------------------
 */
void receive_rmv_frame( short mac_chnl, RMV_FRAME *frame )
{
byte	usn_line, dsn_line;

#ifdef SCC_DEBUG
	printf("\nRxRMV %d\t %x", frame->src_line, frame);
	putchar('m');
#endif

	BripInfo[mac_chnl].rcv_rmv_count++;
	
	/*---------------------------------------
	 * Received a remove frame. Check if 
	 * the sender is in our member table.
	 * If so, delete it from member table.
	 * If not, ignore the frame.
	 *---------------------------------------
	 */
	if ( IsRingMember( mac_chnl, frame->src_line ))
	{
		switch( BripInfo[mac_chnl].state )
		{
			/*---------------------------------------
			 * Received a remove frame. Delete the
			 * the sender from our member table.
			 * Do not perform any switching of receive 
			 * line yet. Will switch when we are ready 
			 * to insert.
			 *---------------------------------------
			 */
			case	BRIP_INIT_STATE:
				delete_ring_member( mac_chnl, frame->src_line );
				break;

			/*---------------------------------------
			 * Delete the sender from member table.
			 * Check if deletion changes our up or 
			 * down stream neighbour. If USN changed,
			 * switch our receive line, reset the
			 * USN Keep alive timer. If DSN, ???
			 *---------------------------------------
			 */
			case	BRIP_INSERTED_STATE:
				delete_ring_member( mac_chnl, frame->src_line );
				if ((usn_line=get_usn(mac_chnl)) != 
					BripInfo[mac_chnl].usn_line_nbr)
				{
					BripInfo[mac_chnl].usn_line_nbr = usn_line;
#ifdef __TRC
					switch_rcv_line( mac_chnl, usn_line);
#endif

#ifdef __FDDI_CON
                                	/*------------------------------------
	                                 * If we are on FDDI media, check if we
					 * need to isolate or insert A port. 
       		                         *------------------------------------
					 */
					if ( usn_line == XMT_QUIET_LINE)
						notify_fddi_smt(EVENT_ISOLATE_A,
							XMT_QUIET_LINE);
					else
						notify_fddi_smt(EVENT_INSERT_A,
							usn_line);
#endif
				}
				if ((dsn_line=get_dsn(mac_chnl)) != 
					BripInfo[mac_chnl].dsn_line_nbr)
				{
					BripInfo[mac_chnl].dsn_line_nbr = dsn_line;
#ifdef __FDDI_CON
                                	/*------------------------------------
	                                 * If we are on FDDI media, check if we
					 * need to isolate or insert A port. 
       		                         *------------------------------------
					 */
					if ( dsn_line == XMT_QUIET_LINE)
						notify_fddi_smt(EVENT_ISOLATE_B,
							XMT_QUIET_LINE);
					else
						notify_fddi_smt(EVENT_INSERT_B, 
							dsn_line);
#endif
				}
				break;
		}
	}
}



#ifdef SCC_DEBUG
print_frame( frame )
KA_FRAME	*frame;
{

	printf("DST SLOT: %x\n",frame->dst_slot);
	printf("CRTL    : %x\n",frame->crtl);
	printf("SRC SLOT: %x\n",frame->src_slot);
	printf("FRAME   : %x %x\n",frame->frame_type_hi, frame->frame_type_lo);
	printf("VERSION : %x %x\n",frame->version_major, frame->version_minor);
	printf("DST LINE: %x\n",frame->dst_line);
	printf("SRC LINE: %x\n",frame->src_line);
	printf("HUB RING: %x\n",frame->hub_ring);
	printf("MEDIA TYPE: %x\n",frame->media_type);
	printf("MEDIA SPEED: %x\n",frame->media_speed);
	printf("HW TYPE: %x\n",frame->hw_type);
	printf("USN LINE: %x\n",frame->usn_line);
	printf("DSN LINE: %x\n",frame->dsn_line);
}
#endif


/*--------------------------------------------------------------------------
 * send_req_frame( mac_chnl, d_slot, d_line )
 *
 *--------------------------------------------------------------------------
 */
void send_req_frame( short mac_chnl, byte d_slot, byte d_line )
{
SCC_BD    *tx_bd;
REQ_FRAME *req_frame;
word	i, bitmap;
byte	*bmap_ptr;
 
        BripInfo[mac_chnl].xmt_req_count++;
 
	req_frame 	= (REQ_FRAME *)get_scc_tx_buff( REQ_FRAME_SIZE );
        while ( (tx_bd= get_scc_bd( BD_XMT )) == NULL)
        {
#ifdef SCC_DEBUG
                printf(" NULL TX BD\n");
#endif
                ReSchedule();
        }
        tx_bd->length	= REQ_FRAME_SIZE;
	tx_bd->buff_ptr	= (byte *)req_frame; 
        tx_bd->bd_type	= BD_XMT;
        
        BripInfo[mac_chnl].xmt_byte_count += tx_bd->length;
 
 
	
        req_frame->dst_slot     = d_slot;
	req_frame->crtl		= 0;
        req_frame->src_slot     = BripInfo[mac_chnl].xmt_line_nbr;
	req_frame->frame_type_hi= BRIP_FRAME_TYPE;
	req_frame->frame_type_lo= REQ_RING_MEMBER;
        req_frame->version_major= BRIP_MAJOR_VER;
        req_frame->version_minor= BRIP_MINOR_VER;
        req_frame->dst_line     = d_line;
        req_frame->src_line     = BripInfo[mac_chnl].xmt_line_nbr;
        req_frame->hub_ring	= BripInfo[mac_chnl].hub_ring_nbr;
	req_frame->media_type	= BripInfo[mac_chnl].media_type;
	req_frame->media_speed	= BripInfo[mac_chnl].media_speed;
	req_frame->hw_type	= BripInfo[mac_chnl].device_type;

#ifdef SCC_DEBUG
	printf("\tTxREQ\n");
	putchar('R');
#endif
	bitmap = get_rsp_bmap( mac_chnl );
	bmap_ptr = (byte *) &bitmap;
	for ( i=0; i < 4; i++ )
	{
		req_frame->rsp_bmap[i]	= *(bmap_ptr+i);
#ifdef SCC_DEBUG
		printf(" %x ", req_frame->rsp_bmap[i]);
#endif
	}

	scc_send( tx_bd );
}
 

/*--------------------------------------------------------------------------
 * send_rsp_frame( mac_chnl, d_slot, d_line )
 *
 *--------------------------------------------------------------------------
 */
void send_rsp_frame( short mac_chnl, byte d_slot, byte d_line )
{
SCC_BD    *tx_bd;
RSP_FRAME *rsp_frame;
 
        BripInfo[mac_chnl].xmt_rsp_count++;
 
	rsp_frame 	= (RSP_FRAME *)get_scc_tx_buff( RSP_FRAME_SIZE );
        while ( (tx_bd= get_scc_bd( BD_XMT )) == NULL)
        {
#ifdef SCC_DEBUG
                printf(" NULL TX BD\n");
#endif
                ReSchedule();
        }
        tx_bd->length	= RSP_FRAME_SIZE;
	tx_bd->buff_ptr	= (byte *)rsp_frame; 
        tx_bd->bd_type	= BD_XMT;
        
        BripInfo[mac_chnl].xmt_byte_count += tx_bd->length;
 
 
	
        rsp_frame->dst_slot     = d_slot;
	rsp_frame->crtl		= 0;
        rsp_frame->src_slot     = BripInfo[mac_chnl].xmt_line_nbr;
	rsp_frame->frame_type_hi= BRIP_FRAME_TYPE;
	rsp_frame->frame_type_lo= RSP_RING_MEMBER;
        rsp_frame->version_major= BRIP_MAJOR_VER;
        rsp_frame->version_minor= BRIP_MINOR_VER;
        rsp_frame->dst_line     = d_line;
        rsp_frame->src_line     = BripInfo[mac_chnl].xmt_line_nbr;
        rsp_frame->hub_ring	= BripInfo[mac_chnl].hub_ring_nbr;
	rsp_frame->media_type	= BripInfo[mac_chnl].media_type;
	rsp_frame->media_speed	= BripInfo[mac_chnl].media_speed;
	rsp_frame->hw_type	= BripInfo[mac_chnl].device_type;
	rsp_frame->state	= BripInfo[mac_chnl].state;

#ifdef SCC_DEBUG
	printf("\tTxRSP\n");
#endif
	scc_send( tx_bd );
}
 
/*--------------------------------------------------------------------------
 * send_ins_frame( mac_chnl, d_slot, d_line )
 *
 *--------------------------------------------------------------------------
 */
void send_ins_frame( short mac_chnl, byte d_slot, byte d_line )
{
SCC_BD    *tx_bd;
INS_FRAME *ins_frame;
 
        BripInfo[mac_chnl].xmt_ins_count++;
 
	ins_frame 	= (INS_FRAME *)get_scc_tx_buff( INS_FRAME_SIZE );
        while ( (tx_bd= get_scc_bd( BD_XMT )) == NULL)
        {
#ifdef SCC_DEBUG
                printf(" NULL TX BD\n");
#endif
                ReSchedule();
        }
        tx_bd->length	= INS_FRAME_SIZE;
	tx_bd->buff_ptr	= (byte *)ins_frame; 
        tx_bd->bd_type	= BD_XMT;

        BripInfo[mac_chnl].xmt_byte_count += tx_bd->length;
 
        ins_frame->dst_slot     = d_slot;
	ins_frame->crtl		= 0;
        ins_frame->src_slot     = BripInfo[mac_chnl].xmt_line_nbr;
	ins_frame->frame_type_hi= BRIP_FRAME_TYPE;
	ins_frame->frame_type_lo= INS_RING_MEMBER;
        ins_frame->version_major= BRIP_MAJOR_VER;
        ins_frame->version_minor= BRIP_MINOR_VER;
        ins_frame->dst_line     = d_line;
        ins_frame->src_line     = BripInfo[mac_chnl].xmt_line_nbr;
        ins_frame->hub_ring	= BripInfo[mac_chnl].hub_ring_nbr;
	ins_frame->media_type	= BripInfo[mac_chnl].media_type;
	ins_frame->media_speed	= BripInfo[mac_chnl].media_speed;
	ins_frame->hw_type	= BripInfo[mac_chnl].device_type;
        ins_frame->usn_line     = BripInfo[mac_chnl].usn_line_nbr;
        ins_frame->dsn_line     = BripInfo[mac_chnl].dsn_line_nbr;

#ifdef SCC_DEBUG
	printf("\tTxINS\n");
	putchar('I');
#endif
	scc_send( tx_bd );
}
/*--------------------------------------------------------------------------
 * send_ka_frame( mac_chnl, d_slot, d_line )
 *
 *--------------------------------------------------------------------------
 */
void send_ka_frame( short mac_chnl, byte d_slot, byte d_line )
{
SCC_BD	 *tx_bd;
KA_FRAME *ka_frame;
 
        BripInfo[mac_chnl].xmt_ka_count++;
 
	ka_frame 	= (KA_FRAME *)get_scc_tx_buff( KA_FRAME_SIZE );
        while ( (tx_bd= get_scc_bd( BD_XMT )) == NULL)
        {
#ifdef SCC_DEBUG
                printf(" NULL TX BD\n");
#endif
                ReSchedule();
        }
        tx_bd->length	= KA_FRAME_SIZE;
	tx_bd->buff_ptr	= (byte *)ka_frame; 
        tx_bd->bd_type	= BD_XMT;

        BripInfo[mac_chnl].xmt_byte_count += tx_bd->length;
 
        ka_frame->dst_slot      = d_slot;
        ka_frame->crtl		= 0;
        ka_frame->src_slot      = BripInfo[mac_chnl].xmt_line_nbr;
	ka_frame->frame_type_hi= BRIP_FRAME_TYPE;
	ka_frame->frame_type_lo= KEEP_ALIVE;
        ka_frame->version_major= BRIP_MAJOR_VER;
        ka_frame->version_minor= BRIP_MINOR_VER;
        ka_frame->dst_line      = d_line;
        ka_frame->src_line      = BripInfo[mac_chnl].xmt_line_nbr;
        ka_frame->hub_ring	= BripInfo[mac_chnl].hub_ring_nbr;
	ka_frame->media_type	= BripInfo[mac_chnl].media_type;
	ka_frame->media_speed	= BripInfo[mac_chnl].media_speed;
	ka_frame->hw_type	= BripInfo[mac_chnl].device_type;
        ka_frame->hub_ring      = BripInfo[mac_chnl].hub_ring_nbr;
        ka_frame->usn_line      = BripInfo[mac_chnl].usn_line_nbr;
        ka_frame->dsn_line      = BripInfo[mac_chnl].dsn_line_nbr;
 
#ifdef SCC_DEBUG
	{
		printf("\nTxKA %d",BripInfo[mac_chnl].xmt_ka_count);
		printf("\tDSN: %x USN: %x\n",ka_frame->dsn_line, ka_frame->usn_line);
	}	
#endif
        /*----------------------------------------------
	 * Reset the keep alive couter for our xmt line.
         *----------------------------------------------
         */
	reset_ka_counter( mac_chnl, ka_frame->src_line );
	RingMemberTbl[mac_chnl].ka_expire[ka_frame->src_line] = 0;
	scc_send( tx_bd );
}
 

/*--------------------------------------------------------------------------
 * send_rmv_frame( mac_chnl, d_slot, d_line )
 *
 *--------------------------------------------------------------------------
 */
void send_rmv_frame( short mac_chnl, byte d_slot, byte d_line )
{
SCC_BD    *tx_bd;
RMV_FRAME *rmv_frame;
 
        BripInfo[mac_chnl].xmt_rmv_count++;
 
        rmv_frame       = (RMV_FRAME *)get_scc_tx_buff( RMV_FRAME_SIZE );
        while ( (tx_bd= get_scc_bd( BD_XMT )) == NULL)
        {
#ifdef SCC_DEBUG
                printf(" NULL TX BD\n");
#endif
                ReSchedule();
        }
        tx_bd->length   = RMV_FRAME_SIZE;
        tx_bd->buff_ptr = (byte *)rmv_frame;
        tx_bd->bd_type  = BD_XMT;
 
        BripInfo[mac_chnl].xmt_byte_count += tx_bd->length;
 
        rmv_frame->dst_slot     = d_slot;
        rmv_frame->crtl         = 0;
        rmv_frame->src_slot     = BripInfo[mac_chnl].xmt_line_nbr;
        rmv_frame->frame_type_hi= BRIP_FRAME_TYPE;
        rmv_frame->frame_type_lo= RMV_RING_MEMBER;
        rmv_frame->version_major= BRIP_MAJOR_VER;
        rmv_frame->version_minor= BRIP_MINOR_VER;
        rmv_frame->dst_line     = d_line;
        rmv_frame->src_line     = BripInfo[mac_chnl].xmt_line_nbr;
        rmv_frame->hub_ring     = BripInfo[mac_chnl].hub_ring_nbr;
        rmv_frame->media_type   = BripInfo[mac_chnl].media_type;
        rmv_frame->media_speed  = BripInfo[mac_chnl].media_speed;
        rmv_frame->hw_type      = BripInfo[mac_chnl].device_type;
        rmv_frame->usn_line     = BripInfo[mac_chnl].usn_line_nbr;
        rmv_frame->dsn_line     = BripInfo[mac_chnl].dsn_line_nbr;

#ifdef SCC_DEBUG 
	printf("\tTxRMV\n");
	putchar('M');
#endif
	scc_send( tx_bd );
}
/*--------------------------------------------------------------------------
 * IsRspNeeded(mac_chnl, bmap_ptr )
 *
 * Verify if we need to send a response frame by checking if our line number
 * is already in the sender's member bit map.
 *--------------------------------------------------------------------------
 */
int IsRspNeeded(short mac_chnl, word *bmap_ptr)
{

register word bmask=0x01;

                /*------------------------------------------
                 * No need to send response if our transmit
                 * line is already in sender's bit map.
                 *------------------------------------------
                 */
                if (*bmap_ptr & ( bmask << BripInfo[mac_chnl].xmt_line_nbr))
			return FALSE;
                else
                        return TRUE;
}

/*---------------------------------------------------------------------------
 * receive_ins_timeout
 *      This routine is called when  MAX_INS_TIMEOUT is expired. A transmit
 * line has sent a REQ_RING_MEMBER frame or a RSP_RING_MEMBER frame of INIT
 * state, but no INS_RING_MEMBER has been received when timer expires. This
 * routine removes the corresponding entry in the member table.
 *---------------------------------------------------------------------------
 */
void receive_ins_timeout( BRIP_TIMER_INFO *BpTmrPtr )
{
#ifdef SCC_DEBUG
	printf("\tInsTO\t");
#endif
        delete_ring_member(BpTmrPtr->channel, BpTmrPtr->line );

}


/*---------------------------------------------------------------------------
 * receive_rsp_timeout
 *	This routine is called when MAX_RSP_TIMEOUT is expired. We have sent a
 * REQ_RING_MEMBER frame and we are waiting for RSP_RING_MEMBER frames in the
 * INIT state. When this timer expires, this routine performs the followings:
 *		- changes our state to INSERTED,
 *		- determines the DSN & USN from our member table,
 *		- switches our receive line to the USN transmit line,
 *		- sends a INS_RING_MEMBER frame, 
 *		- sends a KEEP_ALIVE frame, and
 *		- starts the keep alive monitor.
 *---------------------------------------------------------------------------
 */
void receive_rsp_timeout( int mac_channel )
{
short	mac_chnl;

	mac_chnl = (byte) mac_channel;

	/*---------------------------------------
	 * Check if we have any speed conflict
	 * with other cards, or P2 backplane is
	 * bad.
	 *---------------------------------------
	 */
#ifdef SCC_DEBUG
	printf("\tRspTO\t");
	putchar('o');
#endif
	if ( IsSameSpeed( mac_chnl )  && !backplane_p2_missing )
	{
		update_ring_member( mac_chnl, BripInfo[mac_chnl].xmt_line_nbr );
		BripInfo[mac_chnl].state = BRIP_INSERTED_STATE;

		BripInfo[mac_chnl].usn_line_nbr = get_usn( mac_chnl );
		BripInfo[mac_chnl].dsn_line_nbr = get_dsn( mac_chnl );

#ifdef __TRC
		switch_rcv_line( mac_chnl, BripInfo[mac_chnl].usn_line_nbr);

		printf("Up stream card %d	Down stream card %d\n",
			BripInfo[mac_chnl].usn_line_nbr, 
			BripInfo[mac_chnl].dsn_line_nbr);

		printf("\nInserted into ring %d\n", 
			BripInfo[mac_chnl].hub_ring_nbr);
#endif

#ifdef __FDDI_CON
		/*------------------------------------
                 * If we are on FDDI media, check if we
		 * need to isolate or insert A & B ports. 
                 *------------------------------------
		 */
		if ( BripInfo[mac_chnl].usn_line_nbr == XMT_QUIET_LINE)
			notify_fddi_smt(EVENT_ISOLATE_A, XMT_QUIET_LINE);
		else
			notify_fddi_smt(EVENT_INSERT_A, 
				BripInfo[mac_chnl].usn_line_nbr);

		if ( BripInfo[mac_chnl].dsn_line_nbr == XMT_QUIET_LINE)
			notify_fddi_smt(EVENT_ISOLATE_B, XMT_QUIET_LINE);
		else
			notify_fddi_smt(EVENT_INSERT_B, 
				BripInfo[mac_chnl].dsn_line_nbr);
#endif

		send_ins_frame(mac_chnl, 0xFF, 0xFF);
		send_ka_frame(mac_chnl, 0xFF, 0xFF);

		brip_inited = 1;

		StartTimerCall(&KaTimer[mac_chnl],MAX_KA_TIMEOUT,ka_monitor,(int) mac_chnl);
	}
	else
	{
		/*---------------------------------------
		 * We have a speed conflict and some other
		 * card has higher priority.
		 * Remove our card from the backplane ring.
		 * Change our ring nubmer to 0 (standalone)
		 * and notify upper layer.
		 *---------------------------------------
		 */ 
		send_rmv_frame( mac_chnl,0xFF, 0xFF );
		insert_hub_ring(mac_chnl, 0);
	}

}


/*---------------------------------------------------------------------------
 * ka_monitor()
 * 
 * Monitors the keep alive counters
 *---------------------------------------------------------------------------
 */
void ka_monitor( int mac_channel )
{

	register byte i;
	short mac_chnl = (short) mac_channel;
	
	BripInfo[mac_chnl].send_ka_flag = TRUE;

#ifdef SCC_DEBUG
printf("\nKaTO\tin rcv %d\trcv frame %d\trcv gap %d",
	scc_rcv_count - prev_scc_rcv_count, scc_frame_count, rcv_buff_gap);
prev_scc_rcv_count = scc_rcv_count;
#endif
	for (i=0; i <= MAX_XMT_LINE; i++)
	{
		if ((RingMemberTbl[mac_chnl].slot_nbr[i] !=0 ) &&
		 (RingMemberTbl[mac_chnl].state[i] == BRIP_INSERTED_STATE ))
		{
			/*-------------------------------------------
			 * Check if keep alive count is 0 for each 
			 * member in the backplane ring.
			 *-------------------------------------------
			 */
			if ( --RingMemberTbl[mac_chnl].ka_count[i] == 0 )
			{
				/*-------------------------------------------
				 * To ensure that we do not delete a member 
				 * due to corrupted received buffer, we allow 
				 * this member's keep alive to be expired once.
				 * And we reset the receive dma buffer by
				 * setting rcv_overrun_error flag, and the 
				 * keep alive count.
				 * If this keep alive count still expires, we
				 * delete this member and switch receive line
				 * if needed.
				 *-------------------------------------------
				 */
				if (++RingMemberTbl[mac_chnl].ka_expire[i] > 1 )
				{
					delete_ring_member( mac_chnl, i );

					if (i==BripInfo[mac_chnl].usn_line_nbr)
					{
						BripInfo[mac_chnl].usn_line_nbr=get_usn(mac_chnl);
#ifdef __TRC
						switch_rcv_line(mac_chnl,BripInfo[mac_chnl].usn_line_nbr);
#endif

#ifdef __FDDI_CON
					/*------------------------------------
			                 * If we are on FDDI media, check if we
					 * need to isolate or insert A  port. 
			                 *------------------------------------
					 */
					if ( BripInfo[mac_chnl].usn_line_nbr ==
						 XMT_QUIET_LINE)
						notify_fddi_smt(EVENT_ISOLATE_A, 								XMT_QUIET_LINE);
					else
						notify_fddi_smt(EVENT_INSERT_A, 
							BripInfo[mac_chnl].usn_line_nbr);

#endif
						BripInfo[mac_chnl].send_ka_flag = FALSE;
					}

					if (i==BripInfo[mac_chnl].dsn_line_nbr)
					{
						BripInfo[mac_chnl].dsn_line_nbr=get_dsn(mac_chnl);
						BripInfo[mac_chnl].send_ka_flag = FALSE;
#ifdef __FDDI_CON
					/*------------------------------------
			                 * If we are on FDDI media, check if we
					 * need to isolate or insert B  port. 
			                 *------------------------------------
					 */
					if ( BripInfo[mac_chnl].dsn_line_nbr ==
						 XMT_QUIET_LINE)
						notify_fddi_smt(EVENT_ISOLATE_B, 								XMT_QUIET_LINE);
					else
						notify_fddi_smt(EVENT_INSERT_B, 
							BripInfo[mac_chnl].dsn_line_nbr);

#endif
					}
				}
				else
				{
					rcv_overrun_error = 1;
					reset_ka_counter( mac_chnl, i);
				}

			}
		}
	}

/*
	if (BripInfo[mac_chnl].send_ka_flag) 
*/
		send_ka_frame(mac_chnl, 0xFF, 0xFF);

	if (BripInfo[mac_chnl].state == BRIP_INSERTED_STATE)
		StartTimerCall( &KaTimer[mac_chnl], MAX_KA_TIMEOUT, ka_monitor, mac_channel);
}


/*---------------------------------------------------------------------------
 * switch_rcv_line( mac_chnl, line_nbr )
 * 
 *---------------------------------------------------------------------------
 */
#ifdef yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy
void switch_rcv_line( short mac_chnl, byte line_nbr )
{

	*((volatile byte *)BP_RCV_LINE) = line_nbr;
	printf("\nBackplane receive line ");
	if ( line_nbr )	
		printf("switched to line %d\n", line_nbr);
	else
		printf("bypassed, module is in standalone mode\n");
}
#endif /*yyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyyy*/

/*---------------------------------------------------------------------------
 * send_req_retry( mac_chnl ) 
 * 
 *---------------------------------------------------------------------------
 */
void send_req_retry( int mac_chnl )
{

	if (--BripInfo[mac_chnl].xmt_req_retries > 0  )
	{
		send_req_frame( (short) mac_chnl, 0xFF, 0xFF );
		StartTimerCall(&RetryTimer,MAX_RETRY_TIMEOUT,send_req_retry,mac_chnl);
	}
}

/*---------------------------------------------------------------------------
 * brip_neighbor_err( mac_chnl ) 
 * 
 *---------------------------------------------------------------------------
 */
int
brip_neighbor_err( short mac_chnl )
{
byte ring_nbr;
	ring_nbr = (short)get_ring_number();

	StopTimer(&KaTimer[mac_chnl] );
/*
	set_rcv_dma();
*/
	rcv_overrun_error = 1;
	init_brip( mac_chnl );
	insert_hub_ring( mac_chnl,ring_nbr);
	neighbor_err = 0;
}


/*---------------------------------------------------------------------------
 * brip_stop( ) 
 *	Terminates BRIP for all mac chnl by inserting into ring 0 and stopping
 *	the keep alive timers and dissabling scc tx and rx.  
 *	Inserting into ring 0 will cause us to send a rmv frame and be removed
 *	from the backplane ring gracefully. Also, we will receive from line 0
 *	(internal bypass), and therefore keeps our ring intact.	
 *---------------------------------------------------------------------------
 */
brip_stop()
{
short	mac_chnl;

	for (mac_chnl = 0; mac_chnl < MAX_MAC_CHANNEL; mac_chnl++)
	{
		insert_hub_ring( mac_chnl, 0);
		StopTimer(&KaTimer[mac_chnl]);
#ifdef __FDDI_CON
#   if 0
		fddi_brip_stop();
#   endif
#endif
	}
	
	scc_stop();
}

/*---------------------------------------------------------------------------
 * print_brip( mac_chnl ) 
 *	print out the back plane config info. Note that the receive line 
 *	and the up stream card may not be the same for TR or FDDI bridges.
 *---------------------------------------------------------------------------
 */
print_brip( int mac_chnl )
{

	printf("\nBackplane Ring Information:\n");
	printf("\tBackplane Ring: %d\t\tReceive Line: %d\n",
		BripInfo[mac_chnl].hub_ring_nbr, 
		BripInfo[mac_chnl].usn_line_nbr) ;
	printf("\tUp Stream Card: %d\t\tDown Stream Card: %d\n",
		get_usn_slot( mac_chnl ), 
		get_dsn_slot( mac_chnl ) );
}


/*--------------------------------------------------------------------
* This routine return the current brip port state
* return:
*	1 - port is inserted
*	0 - port is not inserted yet
*--------------------------------------------------------------------*/
BripState(port)
int port;
{
	return (BripInfo[port].state == BRIP_INSERTED_STATE);
}
