#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) mibsup.c V6.2.3:cs.911022:6:6 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/**********************************************************************
	Management Information Base Module

	MIB support module.

	File:		mibsup.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.911022	10/28/91
	
	Copyright 1990,1991 XLNT Designs, Inc.

	This module contains functions used to calculate various
	MIB attributes when they are requested.

	Modification History:

	*** Updated to SMT 6.2 ***

	910318-001	LJP
		Corrected processing in BuildConnectedResources(). It
		was not traversing the objects in the proper order.
		Now, it goes from the last MAC to PHY B; from PHY B
		to PHY A; and from PHY A to the first M PORT or first
		MAC if no M PORTs are in the station.
*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"mibdefs.h"
#include	"mibtypes.h"
#include	"mibglbl.h"


/*********************************************************************
	External function definitions.
*********************************************************************/


/*********************************************************************
	MIB Functions.
*********************************************************************/

uChar
BuildTopology ()
/*********************************************************************
Function:	Build the topology bit string for NIFs.
Parameters:	None.
Input:		Uses attributes from MIB.
Output:		None.
Return:		Returns the 8-bit vector containing the current station
		topology.
Modification History:
*********************************************************************/
{
uChar		topology;		/* station's topology */
int		i;			/* loop counter */
Flag		unrooted;		/* track unrooted concentrator */
Flag		rooted = SET;

	/*
	*	Initialize topology.
	*/
	topology = Topology_Thru;
	unrooted = (mib->SMTStationConfigGrp.Master_Ct > 0);

	/*
	*	Set wrapped state.
	*/
	if (mib->SMTStatusGrp.CF_State == CF_WRAP_A 
			|| mib->SMTStatusGrp.CF_State == CF_WRAP_B 
			|| mib->SMTStatusGrp.CF_State == CF_WRAP_AB)
		topology |= Topology_Wrapped;

	/*
	*	Check connection for each PHY.
	*/
	for (i = 0; i < mib->SMTStationConfigGrp.NonMaster_Ct; i++)
	{
		/*
		*	Check for twisted rings.
		*/
		if (mib->PORTConfigGrp[i].PC_Type == PC_Type_A
			&& mib->PORTConfigGrp[i].PC_Neighbor == PC_Type_A)
			topology |= Topology_Twisted_AA;

		else if (mib->PORTConfigGrp[i].PC_Type == PC_Type_B
			&& mib->PORTConfigGrp[i].PC_Neighbor == PC_Type_B)
			topology |= Topology_Twisted_BB;

		/*
		*	Check for wrapped ring on single attaches.
		*/
		else if (mib->PORTConfigGrp[i].PC_Type == PC_Type_S
			&& (mib->PORTConfigGrp[i].PC_Neighbor == PC_Type_A
				|| mib->PORTConfigGrp[i].PC_Neighbor 
					== PC_Type_B))
			topology |= Topology_Wrapped;

		/*
		*	Check if concentrator is still unrooted.
		*/
		if (mib->SMTStationConfigGrp.Master_Ct > 0)
		{
			if (mib->PORTConfigGrp[i].PC_Neighbor != PC_Type_None)
				unrooted = CLEAR;
		}

		if (mib->PORTConfigGrp[i].PC_Neighbor == PC_Type_M)
			rooted = CLEAR;
	}

	if (unrooted)
		topology |= Topology_Unrooted;

	if (rooted)
		topology |= 0x10;

	return (topology);
}

uInt16
BuildConnectedResources (resIndex)
	uInt16	resIndex;
/*********************************************************************
Function:	Determine connection resource ID for a given resource.
Parameters:	resIndex	= resource ID for determining connection
				resource ID.
Input:		Current station state from MIB.
Output:		None.
Return:		ID of resource on output of given resource.
*********************************************************************/
{
Flag	foundit = FALSE;
uInt16	entity;
uInt16	type;
uInt16	path = PA_UNKNOWN;

	/*
	*	Convert resIndex into an ID & type.
	*/
	if (resIndex > mib->XDISMTGrp.Port_Ct)
	{
		type = MAC_TYPE;
		entity = resIndex - mib->XDISMTGrp.Port_Ct - 1;
	}
	else
	{
		type = PORT_TYPE;
		entity = resIndex - 1;
	}

	/*
	*	Set current path to follow.
	*/
	if (type == MAC_TYPE)
	{
		/* first check if MAC is connected to anything */
		if ((mib->MACConfigGrp[entity].CurrentPath == PA_UNKNOWN)
			|| (mib->MACConfigGrp[entity].CurrentPath ==
				PA_ISOLATED))
		{
			/* object is connected to itself */
			foundit = TRUE;
			/* adjust back to resource index */
			entity = resIndex - 1;
		}
		else
			path = mib->MACConfigGrp[entity].CurrentPath;
	}
	else
	{
		/*
		*	Select ouput path based on configuration.
		*/
		switch (mib->PORTConfigGrp[entity].CE_State)
		{
		case CE_INSERT_P:
			path = PA_PRIMARY;
			break;

		case CE_INSERT_S:
		case CE_INSERT_X:
			path = PA_SECONDARY;
			break;

		case CE_LOCAL:
			path = PA_LOCAL;
			break;

		default:
			/* object must be isolated */
			foundit = TRUE;
			break;
		}
	}

	/*
	*	Find downstream entity.
	*/
	while (!foundit)
	{
		/*
		*	Move to next entity in station.
		*/
		switch (type)
		{
		case MAC_TYPE:
			/*
			*	If last MAC, then skip to attach port.
			*/
			if ((entity + 1) == mib->SMTStationConfigGrp.MAC_Ct)
			{
				if (mib->SMTStationConfigGrp.NonMaster_Ct > 1)
					entity = PHY_B;
				else
					/*
					*	910318-001	LJP
					*	Use proper define.
					*/
					entity = PHY_S;
				type = PORT_TYPE;
			}
			else
				/*
				*	Otherwise just move to next MAC.
				*/
				entity++;
			break;

		case PORT_TYPE:
			/*
			*	910318-001	LJP
			*	From PHY A or S, go to first M port
			*	or first MAC.
			*	From PHY B, go to PHY A.
			*	From M port, go to next M port.
			*/
			switch (mib->PORTConfigGrp[entity].PC_Type)
			{
			case PC_Type_A:
			case PC_Type_S:
				/* if M ports in station */
				if (mib->SMTStationConfigGrp.Master_Ct > 0)
				{
					/* use first M port */
					entity = 
					mib->SMTStationConfigGrp.NonMaster_Ct;
				}

				/* else skip to first MAC */
				else
				{
					entity = MAC_P;
					type = MAC_TYPE;
				}
				break;

			case PC_Type_B:
				/* move to A port */
				entity = PHY_A;
				break;

			case PC_Type_M:
				/* move to next M port or MAC if last port */
				if ((entity + 1) == mib->XDISMTGrp.Port_Ct)
				{
					entity = MAC_P;
					type = MAC_TYPE;
				}
				else
					entity++;
				break;
			}
			break;
		}

		/*
		*	Check if this entity is connected on this path.
		*/
		switch (type)
		{
		case MAC_TYPE:
			if (mib->MACConfigGrp[entity].CurrentPath == path)
			{
				/* this is it, exit loop */
				foundit = TRUE;
				/* adjust to resource index */
				entity += mib->XDISMTGrp.Port_Ct;
			}
			break;

		case PORT_TYPE:
			/*
			*	Check configuration to see if PHY
			*	is on the path.
			*/
			switch (mib->PORTConfigGrp[entity].CE_State)
			{
			case CE_INSERT_P:
				foundit = (path == PA_PRIMARY);
				break;

			case CE_INSERT_S:
				foundit = (path == PA_SECONDARY);
				break;

			case CE_INSERT_X:
				if (path == PA_PRIMARY)
					foundit = SET;
				else
					/*
					*	If this is not it,
					*	then the path switches.
					*/
					path = PA_PRIMARY;
				break;

			case CE_LOCAL:
				foundit = (path == PA_LOCAL);
				break;
			}
		}
	}

	/* adjust to resource index (base 1) */
	return (entity + 1);
}

