#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) fbmframe.c V6.2.3:cs.910920:6:6 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Frame Based Management Module

	Build Frame Module

	File:		fbmframe.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.910920	09/20/91
	
	Copyright 1990,1991 XLNT Designs, Inc.

	Frame Based Management for SMT.  This module contains routines
	for building NIF, SIF, ECHO, RAF, RDF, ESF, and SRF frames.

	Modification History:

	*** Updated to SMT 6.2 ***

	910529-001	LJP
		Corrected calls to BuildSMTHeader() so that parameter
		types match function definition.
	910920-002	LJP
		Changed BuildSMTHeader() to NOT automatically generate
		transaction IDs for response frames. Instead, it uses
		the exact value passed in transID.
*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"fbmhdr.h"
#include	"fbmframe.h"
#include	"fbmglbl.h"
#include	"mibdefs.h"


/*********************************************************************
	External Declarations
*********************************************************************/

extern	uInt32	FBMGetMIBAttr ();
extern	uInt16	htons ();
extern	uInt32	htonl ();
extern	uInt16	ntohs ();
extern	uInt32	ntohl ();
extern	uInt32	GenTransID ();
extern	uInt16	UNA_Param ();
extern	uInt16	StationDescriptor_Param ();
extern	uInt16	StationState_Param ();
extern	uInt16	Attribute_Param ();
extern	uInt16	MsgTimeStamp_Param ();
extern	uInt16	SupportedVersions_Param ();
extern	uInt16	StationPolicies_Param ();
extern	uInt16	PathLatency_Param ();
extern	uInt16	MACNeighbors_Param ();
extern	uInt16	MACStatus_Param ();
extern	uInt16	FrameCounters_Param ();
extern	uInt16	FrameNotCopied_Param ();
extern	uInt16	PriorityValues_Param ();
extern	uInt16	PathDescriptor_Param ();
extern	uInt16	LERStatus_Param ();
extern	uInt16	EBStatus_Param ();
extern	uInt16	Manufacturer_Param ();
extern	uInt16	User_Param ();
extern	uInt16	Echo_Param ();
extern	uInt16	ReasonCode_Param ();
extern	uInt16	RejectedFrame_Param ();
extern	uInt16	ESFID_Param ();


/*********************************************************************
	Send Frame Routines
*********************************************************************/

uInt32
BuildSMTHeader (frameHdr, fcField, destAddr, outputMAC,
		frameClass, frameType, transID)
	SMTFrameHdrType	*frameHdr;
	uChar		fcField;
	MACAddr48	*destAddr;
	uInt16		outputMAC;
	uChar		frameClass;
	uChar		frameType;
	uInt32		transID;
/*********************************************************************
Function:	This function creates the MAC and SMT header portion of 
		a request or announcement frame.
Parameters:	frameHdr	= buffer to use to build the frame. This
				must point to the start of the three pad 
				bytes at the start of the frame buffer.
		fcField		= value to use for FC field (X'41' or X'4F')
		destAddr	= buffer containing destination address.
		outputMAC	= index of MAC to send frame out on.
		frameClass	= SMT frame class identifier.
		frameType	= SMT frame type identifier.
		transID		= transaction ID to use.
input:		frameHdr	= MAC and SMT frame header location in
				the frame.
		destAddr	= 48-bit station ID of destination.
		transID		= If 0, then a new transaction ID is
				generated. Otherwise the value given is
				used. This value must be in host order.
Output:		frameHdr	= contains frame's MAC and SMT headers set
				except for the InfoField_Length.
Return:		Transaction ID is returned if frame was built. Otherwise
		0 is returned.
Modification History:
	910920-002	LJP
*********************************************************************/
{
TLVParamType	mibData;

	/*
	*	Set MAC header.
	*/
	frameHdr->macHdr.FC = fcField;
	MCopyAddress (frameHdr->macHdr.DA, destAddr);
	mibData.paramType = fddiMACSMTAddress;
	mibData.paramLen = 4;
	mibData.MACINDEX = outputMAC + 1;
	FBMGetMIBAttr (sizeof (mibData), (uChar *) &mibData,
		(SetCountType *) NULL);
	MCopyAddress (frameHdr->macHdr.SA, mibData.MACADDR);

	/*
	*	Set SMT header.
	*/
	frameHdr->smtHdr.Frame_Class = frameClass;
	frameHdr->smtHdr.Frame_Type = frameType;
	frameHdr->smtHdr.Version_ID = htons (fbmStationData.opVersionID);
	/*
	*	910920-002	LJP
	*	Generate transaction ID when none is given and not
	*	a response frame.
	*/
	if ((transID == 0) && (frameType != SMTRESPONSE))
		transID = GenTransID ();
	frameHdr->smtHdr.Transaction_ID = htonl (transID);
	mibData.paramType = fddiSMTStationId;
	mibData.paramLen = 0;
	FBMGetMIBAttr (sizeof (mibData), (uChar *) &mibData,
		(SetCountType *) NULL);
	MEMCOPY (&frameHdr->smtHdr.Station_ID, &mibData.SMTOTHER,
		sizeof (SMTStationIdType));

	return (transID);
}

/*********************************************************************
	Neighborhood Information Frames
*********************************************************************/

uInt32
BuildNIFAnnouncement (frameBuffer, outputMAC, transID)
	uChar		*frameBuffer;
	uInt16		outputMAC;
	uInt32		transID;
/*********************************************************************
Function:	This function builds an NIF Announcement.
Parameters:	frameBuffer	= buffer to use to build the frame
		outputMAC	= index of MAC to send frame out on.
		transID		= transaction ID to use in frame.
Input:		None.
Output:		frameBuffer	= contains complete NIF announcment frame.
Return:		Transaction ID is returned if frame was built. Otherwise
		0 is returned.
Modification History:
*********************************************************************/
{
uChar		*memPtr;	/* Buffer used to build frame */
SMTFrameHdrType	*frameHdr;	/* MAC header */
uInt16		frameLen;	/* Total bytes in info field */	
uInt16		paramLen;	/* Total bytes of a frame parameter */
uInt32		result;		/* value to return */
TLVParamType	mibData;	/* for MAC frame status capabilities */

	/*
	*	Initialize values.
	*/
	frameLen = 0;
	result = 0;
	frameHdr = (SMTFrameHdrType *) frameBuffer;

	/*
	*	Initialize frame header.
	*/
	/* 910529-001 LJP */
	result = BuildSMTHeader (frameHdr, SMT_NSA_FC,
		(MACAddr48 *) BROADCAST_ADDRESS,
		outputMAC, NIF_CLASS, SMTANNOUNCE, transID);

	/*
	*	Get the start of the info portion.
	*/
	memPtr = (uChar *) (frameHdr + 1);

	/* set upstream neighbor address */
	paramLen = UNA_Param (memPtr, outputMAC);
	frameLen += paramLen;
	memPtr += paramLen;

	/* set station descriptor */
	paramLen = StationDescriptor_Param (memPtr);
	frameLen += paramLen;
	memPtr += paramLen;

	/* set station state */
	paramLen = StationState_Param (memPtr, outputMAC);
	frameLen += paramLen;
	memPtr += paramLen;

	/* get fddiMACFrameStatusCapabilities */
	mibData.paramType = fddiMACFrameStatusCapabilities;
	mibData.paramLen = 4;
	mibData.MACINDEX = outputMAC + 1;
	FBMGetMIBAttr (sizeof (TLVParamType), (uChar *) &mibData,
		(SetCountType *) NULL);
	paramLen = Attribute_Param (memPtr, &mibData);
	frameLen += paramLen;

	/*
	*	Set frame length.
	*/
	frameHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

uInt32
BuildNIFRequest (frameBuffer, fcField, destAddr, outputMAC, transID)
	uChar		*frameBuffer;
	uChar		fcField;
	MACAddr48	*destAddr;
	uInt16		outputMAC;
	uInt32		transID;
/*********************************************************************
Function:	This function builds an NIF Request.
Parameters:	frameBuffer	= buffer to use to build the frame
		fcField		= value to use for FC field (X'41' or X'4F')
		destAddr	= buffer containing destination address.
		outputMAC	= index of MAC to send frame out on.
		transID		= transaction ID to use in frame.
Input:		destAddr	= 48-bit station ID of destination.
Output:		frameBuffer	= contains completed frame.
Return:		Transaction ID is returned if frame was built. Otherwise
		0 is returned.
Modification History:
*********************************************************************/
{
uChar		*memPtr;	/* Buffer used to build frame */
SMTFrameHdrType	*frameHdr;	/* SMT header */
uInt16		frameLen;	/* Total bytes in info field */
uInt16		paramLen;	/* Total bytes of a frame parameter */
uInt32		result;		/* value to return */
TLVParamType	mibData;	/* for MAC frame status capabilities */

	/*
	*	Initialize values.
	*/
	frameLen = 0;
	result = 0;
	frameHdr = (SMTFrameHdrType *) frameBuffer;

	/*
	*	Initialize frame header.
	*/
	/* 910529-001 LJP */
	result = BuildSMTHeader (frameHdr, fcField, destAddr,
		outputMAC, NIF_CLASS, SMTREQUEST, transID);

	/*
	*	Get the start of the info portion.
	*/
	memPtr = (uChar *) (frameHdr + 1);

	/* set upstream neighbor address */
	paramLen = UNA_Param (memPtr, outputMAC);
	frameLen += paramLen;
	memPtr += paramLen;

	/* set station descriptor */
	paramLen = StationDescriptor_Param (memPtr);
	frameLen += paramLen;
	memPtr += paramLen;

	/* set station state */
	paramLen = StationState_Param (memPtr, outputMAC);
	frameLen += paramLen;
	memPtr += paramLen;

	/* get fddiMACFrameStatusCapabilities */
	mibData.paramType = fddiMACFrameStatusCapabilities;
	mibData.paramLen = 4;
	mibData.MACINDEX = outputMAC + 1;
	FBMGetMIBAttr (sizeof (TLVParamType), (uChar *) &mibData,
		(SetCountType *) NULL);
	paramLen = Attribute_Param (memPtr, &mibData);
	frameLen += paramLen;

	/*
	*	Set frame length.
	*/
	frameHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

uInt32
BuildNIFResponse (requestFrame, responseBuffer, outputMAC)
	uChar	*requestFrame;
	uChar	*responseBuffer;
	uInt16	outputMAC;
/*********************************************************************
Function:	This function builds an NIF Response.
Parameters:	requestFrame	= pointer to NIF Request buffer.
		responseBuffer	= pointer to buffer for response frame.
		outputMAC	= index of MAC to send frame out on.
Input:		requestFrame	= contains NIF Request Frame to respond to.
Output:		responseBuffer	= contains NIF Response Frame.
Return:		No value returned.
Modification History:
*********************************************************************/
{
SMTFrameHdrType	*reqSMTHdr;	/* SMT header in request frame */
SMTFrameHdrType	*rspSMTHdr;	/* SMT header in response frame */
uChar		*memPtr;	/* parameter pointer in frame */
uInt16		frameLen;	/* Total bytes in info field */	
uInt16		paramLen;	/* Total bytes of a frame parameter */
uInt32		result;		/* value to return */
TLVParamType	mibData;	/* for MAC frame status capabilities */

	/*
	*	Get frame headers.
	*/
	reqSMTHdr = (SMTFrameHdrType *) requestFrame;
	rspSMTHdr = (SMTFrameHdrType *) responseBuffer;

	/*
	*	Initialize frame header.
	*/
	result = ntohl (reqSMTHdr->smtHdr.Transaction_ID);
	/* 910529-001 LJP */
	result = BuildSMTHeader (rspSMTHdr, SMT_INFO_FC,
		reqSMTHdr->macHdr.SA,
		outputMAC, NIF_CLASS, SMTRESPONSE, result);

	/* get start of info field */
	memPtr = (uChar *) (rspSMTHdr + 1);
	frameLen = 0;

	/* set upstream neighbor address */
	paramLen = UNA_Param (memPtr, outputMAC);
	frameLen += paramLen;
	memPtr += paramLen;

	/* set station descriptor */
	paramLen = StationDescriptor_Param (memPtr);
	frameLen += paramLen;
	memPtr += paramLen;

	/* set station state */
	paramLen = StationState_Param (memPtr, outputMAC);
	frameLen += paramLen;
	memPtr += paramLen;

	/* get fddiMACFrameStatusCapabilities */
	mibData.paramType = fddiMACFrameStatusCapabilities;
	mibData.paramLen = 4;
	mibData.MACINDEX = outputMAC + 1;
	FBMGetMIBAttr (sizeof (TLVParamType), (uChar *) &mibData,
		(SetCountType *) NULL);
	paramLen = Attribute_Param (memPtr, &mibData);
	frameLen += paramLen;

	/*
	*	Set frame length.
	*/
	rspSMTHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

/*********************************************************************
	Station Information Configuration Frames
*********************************************************************/

uInt32
BuildSIFConfigRequest (frameBuffer, fcField, destAddr, outputMAC)
	uChar		*frameBuffer;
	uChar		fcField;
	MACAddr48	*destAddr;
	uInt16		outputMAC;
/*********************************************************************
Function:	This function builds an SIF Configuration Request.
Parameters:	frameBuffer	= buffer to use to build the frame
		fcField		= value to use for FC field (X'41' or X'4F')
		destAddr	= buffer containing destination address.
		outputMAC	= index of MAC to send frame out on.
Input:		destAddr	= 48-bit station ID of destination.
Output:		frameBuffer	= contains completed frame.
Return:		Transaction ID is returned if frame was built. Otherwise
		0 is returned.
Modification History:
*********************************************************************/
{
#if 0
uChar		*memPtr;	/* Buffer used to build frame */
#endif

SMTFrameHdrType	*frameHdr;	/* SMT header */

#if 0
uInt16		paramLen;	/* Total bytes of a frame parameter */
#endif

uInt32		result;		/* value to return */

	/*
	*	Initialize values.
	*/
	result = 0;
	frameHdr = (SMTFrameHdrType *) frameBuffer;

	/*
	*	Initialize frame header.
	*/
	/* 910529-001 LJP */
	result = BuildSMTHeader (frameHdr, fcField, destAddr,
		outputMAC, SIFCONFIG_CLASS, SMTREQUEST, (uInt32) 0);

	/*
	*	Set frame length.
	*/
	frameHdr->smtHdr.InfoField_Length = 0;

	/*
	*	Return transaction ID.
	*/
	return (result);
}

uInt32
BuildSIFConfigResponse (requestFrame, responseBuffer, outputMAC)
	uChar	*requestFrame;
	uChar	*responseBuffer;
	uInt16	outputMAC;
/*********************************************************************
Function:	This function builds an SIF Configuration Response.
Parameters:	requestFrame	= pointer to SIF Request buffer.
		responseBuffer	= pointer to buffer for response frame.
		outputMAC	= index of MAC to send frame out on.
Input:		requestFrame	= contains SIF Request Frame to respond to.
Output:		responseBuffer	= contains SIF Response Frame.
Return:		No value returned.
Modification History:
*********************************************************************/
{
SMTFrameHdrType	*reqSMTHdr;	/* SMT header in request frame */
SMTFrameHdrType	*rspSMTHdr;	/* SMT header in response frame */
uChar		*memPtr;	/* parameter pointer in frame */
uInt16		frameLen;	/* Total bytes in info field */
uInt16		paramLen;	/* Total bytes of a frame parameter */
uInt32		result;		/* value to return */
TLVParamType	mibData;	/* for Set Count */
uInt16		i;		/* loop counter */
uInt16          spaceNeeded;
uInt32          BuildRDFResponse();    /* forward decl */
	
	/*
	*	Get frame headers.
	*/
	reqSMTHdr = (SMTFrameHdrType *) requestFrame;
	rspSMTHdr = (SMTFrameHdrType *) responseBuffer;

	/*
	*	Initialize frame header.
	*/
	result = ntohl (reqSMTHdr->smtHdr.Transaction_ID);
	/* 910529-001 LJP */
	result = BuildSMTHeader (rspSMTHdr, SMT_INFO_FC,
		reqSMTHdr->macHdr.SA,
		outputMAC, SIFCONFIG_CLASS, SMTRESPONSE, result);

	/* get start of info field */
	memPtr = (uChar *) (rspSMTHdr + 1);
	frameLen = 0;

	/* time stamp */
	paramLen = MsgTimeStamp_Param (memPtr);
	frameLen += paramLen;
	memPtr += paramLen;

	/* station descriptor */
	paramLen = StationDescriptor_Param (memPtr);
	frameLen += paramLen;
	memPtr += paramLen;

	/* SMT supported versions */
	paramLen = SupportedVersions_Param (memPtr);
	frameLen += paramLen;
	memPtr += paramLen;

	/* station state */
	paramLen = StationState_Param (memPtr, outputMAC);
	frameLen += paramLen;
	memPtr += paramLen;

	/* station policies */
	paramLen = StationPolicies_Param (memPtr);
	frameLen += paramLen;
	memPtr += paramLen;

	/* path latency */
	paramLen = PathLatency_Param (memPtr);
	frameLen += paramLen;
	memPtr += paramLen;

	/* set count */
	mibData.paramType = fddiSMTSetCount;
	mibData.paramLen = 0;
	if (FBMGetMIBAttr (sizeof (TLVParamType), (uChar *) &mibData,
		(SetCountType *) NULL) == RC_SUCCESS)
	{
		paramLen = Attribute_Param (memPtr, &mibData);
		frameLen += paramLen;
		memPtr += paramLen;
	}

        spaceNeeded = (fbmStationData.macCount *
		       (sizeof(MACNeighborsParamType) +
			sizeof(PathMACRecordType)))
	    + (fbmStationData.phyCount * sizeof (PathPORTRecordType))
	    + sizeof(TLVHdrType);

        if ((MAX_SMT_INFO_LEN - frameLen) > spaceNeeded)
	{
	   /* MAC neighbors */
	   for (i = 0; i < fbmStationData.macCount; i++)
	   {
		paramLen = MACNeighbors_Param (memPtr, i);
		frameLen += paramLen;
		memPtr += paramLen;
	   }

	   /* path descriptor */
	   paramLen = PathDescriptor_Param (memPtr);
	   frameLen += paramLen;
	   memPtr += paramLen;

	   /*
	    *	Set frame length.
	    */
	   rspSMTHdr->smtHdr.InfoField_Length = htons (frameLen);
	}
	else
	{
	   /* issue RDF for frame too long */
	   ClearFrameBuffer (responseBuffer);
	   BuildRDFResponse (requestFrame, responseBuffer, 
			     outputMAC, RC_FRAME_TOO_LONG, 
			     ntohs (reqSMTHdr->smtHdr.InfoField_Length)
			     + sizeof(SMTFrameHdrType) - FC_PAD_SIZE);
	   result = 0;
	}
	/*
	*	Return transaction ID.
	*/
	return (result);
}

/*********************************************************************
	Station Information Operation Frames
*********************************************************************/

uInt32
BuildSIFOpRequest (frameBuffer, fcField, destAddr, outputMAC)
	uChar		*frameBuffer;
	uChar		fcField;
	MACAddr48	*destAddr;
	uInt16		outputMAC;
/*********************************************************************
Function:	This function builds an SIF Operation Request.
Parameters:	frameBuffer	= buffer to use to build the frame
		fcField		= value to use for FC field (X'41' or X'4F')
		destAddr	= buffer containing destination address.
		outputMAC	= index of MAC to send frame out on.
Input:		destAddr	= 48-bit station ID of destination.
Output:		frameBuffer	= contains complete SIF Operation frame.
Return:		Transaction ID is returned if frame was built. Otherwise
		0 is returned.
Modification History:
*********************************************************************/
{
#if 0
uChar		*memPtr;	/* Buffer used to build frame */
#endif

SMTFrameHdrType	*frameHdr;	/* SMT header */

#if 0
uInt16		paramLen;	/* Total bytes of a frame parameter */
#endif

uInt32		result;		/* value to return */

	/*
	*	Initialize values.
	*/
	result = 0;
	frameHdr = (SMTFrameHdrType *) frameBuffer;

	/*
	*	Initialize frame header.
	*/
	/* 910529-001 LJP */
	result = BuildSMTHeader (frameHdr, fcField, destAddr,
		outputMAC, SIFOP_CLASS, SMTREQUEST, (uInt32) 0);

	/*
	*	Set frame length.
	*/
	frameHdr->smtHdr.InfoField_Length = 0;

	/*
	*	Return transaction ID.
	*/
	return (result);
}

uInt32
BuildSIFOpResponse (requestFrame, responseBuffer, outputMAC)
	uChar	*requestFrame;
	uChar	*responseBuffer;
	uInt16	outputMAC;
/*********************************************************************
Function:	This function builds an SIF Operation Response.
Parameters:	requestFrame	= pointer to SIF Request buffer.
		responseBuffer	= pointer to buffer for response frame.
		outputMAC	= index of MAC to send frame out on.
Input:		requestFrame	= contains SIF Request Frame to respond to.
Output:		responseBuffer	= contains SIF Response Frame.
Return:		No value returned.
Modification History:
*********************************************************************/
{
SMTFrameHdrType	*reqSMTHdr;	/* SMT header in request frame */
SMTFrameHdrType	*rspSMTHdr;	/* SMT header in response frame */
uChar		*memPtr;	/* parameter pointer in frame */
uInt16		frameLen;	/* Total bytes in info field */
uInt16		paramLen;	/* Total bytes of a frame parameter */
uInt32		result;		/* value to return */
TLVParamType	mibData;	/* for Set Count */
uInt16		i;		/* loop counter */
uInt32          BuildRDFResponse();    /* forward decl */

	/*
	*	Get frame headers.
	*/
	reqSMTHdr = (SMTFrameHdrType *) requestFrame;
	rspSMTHdr = (SMTFrameHdrType *) responseBuffer;

	/*
	*	Initialize frame header.
	*/
	result = ntohl (reqSMTHdr->smtHdr.Transaction_ID);
	/* 910529-001 LJP */
	result = BuildSMTHeader (rspSMTHdr, SMT_INFO_FC,
		reqSMTHdr->macHdr.SA,
		outputMAC, SIFOP_CLASS, SMTRESPONSE, result);

	/* get start of info field */
	memPtr = (uChar *) (rspSMTHdr + 1);
	frameLen = 0;

	/* time stamp */
	paramLen = MsgTimeStamp_Param (memPtr);
	frameLen += paramLen;
	memPtr += paramLen;

	/* manufacturer field */
	paramLen = Manufacturer_Param (memPtr);
	frameLen += paramLen;
	memPtr += paramLen;

	/* user field */
	paramLen = User_Param (memPtr);
	frameLen += paramLen;
	memPtr += paramLen;

	/* set count */
	mibData.paramType = fddiSMTSetCount;
	mibData.paramLen = 0;
	if (FBMGetMIBAttr (sizeof (TLVParamType), (uChar *) &mibData,
		(SetCountType *) NULL) == RC_SUCCESS)
	{
		paramLen = Attribute_Param (memPtr, &mibData);
		frameLen += paramLen;
		memPtr += paramLen;
	}

	/* per MAC parameters */
	for (i = 0; 
	     (frameLen < MAX_SMT_INFO_LEN) && (i < fbmStationData.macCount)
	     ; i++)
	{
		/* MAC status */
		paramLen = MACStatus_Param (memPtr, i);
		frameLen += paramLen;
		memPtr += paramLen;

		/* MAC frame counters */
		paramLen = FrameCounters_Param (memPtr, i);
		frameLen += paramLen;
		memPtr += paramLen;

		/* MAC frame not copied */
		paramLen = FrameNotCopied_Param (memPtr, i);
		frameLen += paramLen;
		memPtr += paramLen;

		/* MAC priority values */
		paramLen = PriorityValues_Param (memPtr, i);
		frameLen += paramLen;
		memPtr += paramLen;
	}

	/* per PORT parameters */
	for (i = 0; 
	     (frameLen < MAX_SMT_INFO_LEN) && (i < fbmStationData.phyCount)
	     ; i++)
	{
		/* PORT LEM status */
		paramLen = LERStatus_Param (memPtr, i);
		frameLen += paramLen;
		memPtr += paramLen;

		/* PORT EB status */
		paramLen = EBStatus_Param (memPtr, i);
		frameLen += paramLen;
		memPtr += paramLen;
	}


        if (frameLen <= MAX_SMT_INFO_LEN)
	{
	   /*
	    *	Set frame length.
	    */
	   rspSMTHdr->smtHdr.InfoField_Length = htons (frameLen);
	}
        else
	{
	   /* issue RDF for frame too long */
	   ClearFrameBuffer (responseBuffer);
	   BuildRDFResponse (requestFrame, responseBuffer,
			     outputMAC, RC_FRAME_TOO_LONG,
			     ntohs (reqSMTHdr->smtHdr.InfoField_Length)
			     + sizeof(SMTFrameHdrType) - FC_PAD_SIZE);
	   result = 0;
	}

	/*
	*	Return transaction ID.
	*/
	return (result);
}

/*********************************************************************
	Echo Frames
*********************************************************************/

uInt32
BuildEchoRequest (frameBuffer, fcField, destAddr, outputMAC, 
		info, infoLen)
	uChar		*frameBuffer;
	uChar		fcField;
	MACAddr48	*destAddr;
	uInt16		outputMAC;
	uChar		*info;
	uInt16		infoLen;
/*********************************************************************
Function:	This function builds an Echo Request Frame.
Parameters:	frameBuffer	= buffer to use to build the frame
		fcField		= value to use for FC field (X'41' or X'4F')
		destAddr	= buffer containing destination address.
		outputMAC	= index of MAC to send frame out on.
		info		= buffer containing data for echo frame 
				info field.
		infoLen		= The number of bytes in the information field.
Input:		destAddr	= 48-bit station ID of destination.
		info		= info field of echo frame.
Output:		frameBuffer	= contains completed frame.
Return:		Transaction ID is returned if frame was built. Otherwise
		0 is returned.
Modification History:
*********************************************************************/
{
uChar		*memPtr;	/* Buffer used to build frame */
SMTFrameHdrType	*frameHdr;	/* Used for building SMT header */
uInt16		frameLen;	/* Total bytes in outgoing frame */	
uInt32		result;		/* value to return */

	/*
	*	Initialize values.
	*/
	frameLen = 0;
	result = 0;
	frameHdr = (SMTFrameHdrType *) frameBuffer;

	/*
	*	Initialize frame header.
	*/
	/* 910529-001 LJP */
	result = BuildSMTHeader (frameHdr, fcField, destAddr,
		outputMAC, ECHO_CLASS, SMTREQUEST, (uInt32) 0);

	/*
	*	Get the start of the info portion.
	*/
	memPtr = (uChar *) (frameHdr + 1);

	/*
	*	Fill in echo data
	*/
	frameLen = Echo_Param (memPtr, info, infoLen);

	/*
	*	Set frame length.
	*/
	frameHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

uInt32
BuildEchoResponse (requestFrame, responseBuffer, outputMAC)
	uChar	*requestFrame;
	uChar	*responseBuffer;
	uInt16	outputMAC;
/*********************************************************************
Function:	This function builds an Echo Response Frame.
Parameters:	requestFrame	= pointer to Echo Request Frame buffer.
		responseBuffer	= pointer to buffer for response frame.
		outputMAC	= index of MAC to send frame out on.
Input:		requestFrame	= contains Echo Request Frame to respond to.
Output:		responseBuffer	= contains Echo Response Frame.
Return:		No value returned.
Modification History:
*********************************************************************/
{
SMTFrameHdrType	*reqframeHdr;	/* SMT header in request frame */
SMTFrameHdrType	*rspframeHdr;	/* SMT header in response frame */
uInt16		frameLen;
uInt32		result;
uChar		*responsePtr;
uChar		*requestPtr;

	/*
	*	Get frame headers.
	*/
	reqframeHdr = (SMTFrameHdrType *) requestFrame;
	rspframeHdr = (SMTFrameHdrType *) responseBuffer;

	/*
	*	Initialize frame header.
	*/
	result = ntohl (reqframeHdr->smtHdr.Transaction_ID);
	/* 910529-001 */
	result = BuildSMTHeader (rspframeHdr, SMT_INFO_FC,
		reqframeHdr->macHdr.SA, outputMAC, ECHO_CLASS,
		SMTRESPONSE, result);

	/*
	*	Get and set info length.
	*/
	frameLen = ntohs (reqframeHdr->smtHdr.InfoField_Length);

        if (frameLen > MAX_SMT_INFO_LEN)    /* jlin: 06/18/93 */
	{
	   frameLen = MAX_SMT_INFO_LEN;
	}

	rspframeHdr->smtHdr.InfoField_Length
		= reqframeHdr->smtHdr.InfoField_Length;

	/*
	*	Get the start of the info portion.
	*/
	responsePtr = (uChar *) (rspframeHdr + 1);
	requestPtr = (uChar *) (reqframeHdr + 1);

	/*
	*	Copy the echo parameter.
	*/
	MEMCOPY (responsePtr, requestPtr, frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

/*********************************************************************
	Request Denied Frames
*********************************************************************/

uInt32
BuildRDFResponse (requestFrame, responseBuffer, outputMAC, reasonCode,
		  reqFrameLen)
	uChar	*requestFrame;
	uChar	*responseBuffer;
	uInt16	outputMAC;
	uInt32	reasonCode;
        uInt16  reqFrameLen;
/*********************************************************************
Function:	This function builds a Request Denied Frame Response.
Parameters:	requestFrame	= pointer to frame being denied.
		responseBuffer	= pointer to buffer for response frame.
		outputMAC	= index of MAC to send frame out on.
		reqFrameLen     = denied frame's actual received length.
Input:		requestFrame	= contains frame being denied service.
Output:		responseBuffer	= contains RDF.
Return:		Transaction ID of request frame, or 0 if frame not
		built.
Modification History:
*********************************************************************/
{
SMTFrameHdrType	*reqframeHdr;	/* SMT header in request frame */
SMTFrameHdrType	*rspframeHdr;	/* SMT header in response frame */
uChar		*memPtr;	/* pointer to frame parameters */
uInt16		frameLen;	/* length of response frame INFO */
uInt16		paramLen;	/* length of frame parameters */
uInt32		result;
	
	/*
	*	Get frame headers.
	*/
	reqframeHdr = (SMTFrameHdrType *) requestFrame;
	rspframeHdr = (SMTFrameHdrType *) responseBuffer;

	/*
	*	Initialize frame header.
	*/
	result = ntohl (reqframeHdr->smtHdr.Transaction_ID);
	/* 910529-001 */
	result = BuildSMTHeader (rspframeHdr, SMT_INFO_FC, 
		reqframeHdr->macHdr.SA, outputMAC, RDF_CLASS,
		SMTRESPONSE, result);

	/*
	*	Get the start of the info portion.
	*/
	memPtr = (uChar *) (rspframeHdr + 1);
	frameLen = 0;

	/* reason code */
	paramLen = ReasonCode_Param (memPtr, reasonCode);
	frameLen += paramLen;
	memPtr += paramLen;

	/* SMT supported versions */
	paramLen = SupportedVersions_Param (memPtr);
	frameLen += paramLen;
	memPtr += paramLen;

	/* refused frame fragment */
	paramLen = RejectedFrame_Param (memPtr, reqframeHdr,
		(MAX_SMT_INFO_LEN - frameLen), reqFrameLen);
	frameLen += paramLen;
	memPtr += paramLen;

	/*
	*	Set info length.
	*/
	rspframeHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

/*********************************************************************
	Resource Allocation Frames
*********************************************************************/

uInt32
BuildRAFAnnouncement (frameBuffer, outputMAC)
	uChar		*frameBuffer;
	uInt16		outputMAC;
/*********************************************************************
Function:	This function builds a Resource Allocation 
		Announcement frame.
Parameters:	frameBuffer	= buffer to use to build the frame
		outputMAC	= index of MAC to send frame out on.
Input:		None.
Output:		frameBuffer	= contains completed frame.
Return:		Transaction ID is returned if frame was built. Otherwise
		0 is returned.
Notes:		As of SMT 6.2, no protocol has been defined for RAFs.
		This routine always returns 0.
Modification History:
*********************************************************************/
{
	return (0);
}

uInt32
BuildRAFRequest (frameBuffer, outputMAC)
	uChar		*frameBuffer;
	uInt16		outputMAC;
/*********************************************************************
Function:	This function builds a Resource Allocation 
		Request frame.
Parameters:	frameBuffer	= buffer to use to build the frame
		outputMAC	= index of MAC to send frame out on.
Input:		None.
Output:		frameBuffer	= contains completed frame.
Return:		Transaction ID is returned if frame was built. Otherwise
		0 is returned.
Notes:		As of SMT 6.2, no protocol has been defined for RAFs.
		This routine always returns 0.
Modification History:
*********************************************************************/
{
	return (0);
}

uInt32
BuildRAFResponse (requestFrame, responseBuffer, outputMAC)
	uChar	*requestFrame;
	uChar	*responseBuffer;
	uInt16	outputMAC;
/*********************************************************************
Function:	This function builds a Resource Allocation Response frame.
Parameters:	requestFrame	= pointer to RAF Request buffer.
		responseBuffer	= pointer to buffer for response frame.
		outputMAC	= index of MAC to send frame out on.
Input:		requestFrame	= contains frame to respond to.
Output:		responseBuffer	= contains response Frame.
Return:		No value returned.
Notes:		As of SMT 6.2, there is not protocol defined for RAFs.
		Therefore, this routine always generates an RDF.
Modification History:
*********************************************************************/
{
   SMTFrameHdrType *reqframeHdr = (SMTFrameHdrType *) requestFrame;

	return (BuildRDFResponse (requestFrame, responseBuffer,
		outputMAC, RC_FRAME_CLASS),
		ntohs (reqframeHdr->smtHdr.InfoField_Length)
		+ sizeof(SMTFrameHdrType) - FC_PAD_SIZE);
}

/*********************************************************************
	Extended Service Frames
*********************************************************************/

uInt32
BuildESFAnnouncement (frameBuffer, destAddr, outputMAC, 
		esfID, info, infoLen)
	uChar		*frameBuffer;
	MACAddr48	*destAddr;
	uInt16		outputMAC;
	uChar		*esfID;
	uChar		*info;
	uInt16		infoLen;
/*********************************************************************
Function:	This function builds an Extended Service Announcement frame.
Parameters:	frameBuffer	= buffer to use to build the frame
		destAddr	= buffer containing destination address.
		outputMAC	= index of MAC to send frame out on.
		esfID		= buffer containing ESF identifier.
		info		= buffer containing data for ESF
				info field.
		infoLen		= The number of bytes in the information field.
Input:		destAddr	= 48-bit station ID of destination.
		esfID		= 48-bit, IEEE-assigned ESF ID.
		info		= info field of the ESF. This field must
				be properly formatted in network byte order.
Output:		frameBuffer	= contains complete echo frame.
Return:		Transaction ID is returned if frame was built. Otherwise
		0 is returned.
Modification History:
*********************************************************************/
{
uChar		*memPtr;	/* Buffer used to build frame */
SMTFrameHdrType	*frameHdr;	/* Used for building SMT header */
uInt16		frameLen;	/* Total bytes in outgoing frame */	
uInt16		paramLen;	/* Length of frame parameter */
uInt32		result;		/* value to return */

	/*
	*	Initialize values.
	*/
	frameLen = 0;
	result = 0;
	frameHdr = (SMTFrameHdrType *) frameBuffer;

	/*
	*	Initialize frame header.
	*/
	/* 910529-001 LJP */
	result = BuildSMTHeader (frameHdr, SMT_NSA_FC, destAddr,
		outputMAC, ESF_CLASS, SMTANNOUNCE, (uInt32) 0);

	/*
	*	Get the start of the info portion.
	*/
	memPtr = (uChar *) (frameHdr + 1);

	/* ESF ID parameter */
	paramLen = ESFID_Param (memPtr, esfID);
	frameLen += paramLen;
	memPtr += paramLen;

	/* Fill in ESF data */
	if (info)
	{
		MEMCOPY (memPtr, info, infoLen);
		frameLen += infoLen;
		memPtr += infoLen;
	}

	/*
	*	Set frame length.
	*/
	frameHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

uInt32
BuildESFRequest (frameBuffer, fcField, destAddr, outputMAC,
		esfID, info, infoLen)
	uChar		*frameBuffer;
	uChar		fcField;
	MACAddr48	*destAddr;
	uInt16		outputMAC;
	uChar		*esfID;
	uChar		*info;
	uInt16		infoLen;
/*********************************************************************
Function:	This function builds an Extended Service Request frame.
Parameters:	frameBuffer	= buffer to use to build the frame
		fcField		= value to use for FC field (X'41' or X'4F')
		destAddr	= buffer containing destination address.
		outputMAC	= index of MAC to send frame out on.
		esfID		= buffer containing ESF identifier.
		info		= buffer containing data for ESF
				info field.
		infoLen		= The number of bytes in the information field.
Input:		destAddr	= 48-bit station ID of destination.
		esfID		= 48-bit, IEEE-assigned ESF ID.
		info		= info field of the ESF. This field must
				be properly formatted in network byte order.
Output:		frameBuffer	= contains completed frame.
Return:		Transaction ID is returned if frame was built. Otherwise
		0 is returned.
Modification History:
*********************************************************************/
{
uChar		*memPtr;	/* Buffer used to build frame */
SMTFrameHdrType	*frameHdr;	/* Used for building SMT header */
uInt16		frameLen;	/* Total bytes in outgoing frame */	
uInt16		paramLen;	/* Length of frame parameter */
uInt32		result;		/* value to return */

	/*
	*	Initialize values.
	*/
	frameLen = 0;
	result = 0;
	frameHdr = (SMTFrameHdrType *) frameBuffer;

	/*
	*	Initialize frame header.
	*/
	/* 910529-001 LJP */
	result = BuildSMTHeader (frameHdr, fcField, destAddr,
		outputMAC, ESF_CLASS, SMTREQUEST, (uInt32) 0);

	/*
	*	Get the start of the info portion.
	*/
	memPtr = (uChar *) (frameHdr + 1);

	/* ESF ID parameter */
	paramLen = ESFID_Param (memPtr, esfID);
	frameLen += paramLen;
	memPtr += paramLen;

	/* Fill in ESF data */
	if (info)
	{
		MEMCOPY (memPtr, info, infoLen);
		frameLen += infoLen;
		memPtr += infoLen;
	}

	/*
	*	Set frame length.
	*/
	frameHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

uInt32
BuildESFResponse (responseBuffer, destAddr, outputMAC, transID,
		esfID, info, infoLen)
	uChar	*responseBuffer;
        MACAddr48 *destAddr;
	uInt16	outputMAC;
        uInt32  transID;
	uChar	*esfID;
	uChar	*info;
	uInt16	infoLen;
/*********************************************************************
Function:	This function builds an ESF Response.
Parameters:	requestFrame	= pointer to ESF Request buffer.
		responseBuffer	= pointer to buffer for response frame.
		outputMAC	= index of MAC to send frame out on.
		esfID		= buffer containing ESF identifier.
		info		= buffer containing data for ESF
				info field.
		infoLen		= The number of bytes in the information field.
Input:		requestFrame	= contains ESF Request Frame to respond to.
		esfID		= 48-bit, IEEE-assigned ESF ID.
		info		= info field of the ESF. This field must
				be properly formatted in network byte order.
Output:		responseBuffer	= contains ESF Response Frame.
Return:		No value returned.
Modification History:
*********************************************************************/
{
SMTFrameHdrType	*rspframeHdr;	/* SMT header in response frame */
uInt16		frameLen;
uInt16		paramLen;
uInt32		result;
uChar		*memPtr;
	
	/*
	*	Get frame headers.
	*/
	rspframeHdr = (SMTFrameHdrType *) responseBuffer;

	/*
	*	Initialize frame header.
	*/
	result = transID;
	/* 910529-001 LJP */
	result = BuildSMTHeader (rspframeHdr, SMT_INFO_FC, 
		destAddr, outputMAC, ESF_CLASS, 
		SMTRESPONSE, result);

	/*
	*	Get the start of the info portion.
	*/
	memPtr = (uChar *) (rspframeHdr + 1);
	frameLen = 0;

	/* ESF ID */
	paramLen = ESFID_Param (memPtr, esfID);
	frameLen += paramLen;
	memPtr += paramLen;

	/* set remainder of info field */
	if (info)
	{
		MEMCOPY (memPtr, info, infoLen);
		frameLen += infoLen;
		memPtr += paramLen;
	}

	/*
	*	Set info length.
	*/
	rspframeHdr->smtHdr.InfoField_Length = htons (frameLen);

	/*
	*	Return transaction ID.
	*/
	return (result);
}

/*********************************************************************
	Status Report Frames
*********************************************************************/

uInt32
BuildSRFAnnouncement (frameBuffer, outputMAC, infoFieldLen)
	uChar		*frameBuffer;
	uInt16		outputMAC;
	uInt16		infoFieldLen;
/*********************************************************************
Function:	This function builds an SRF Announcement.
Parameters:	frameBuffer	= buffer to use to build the frame
		outputMAC	= index of MAC to send frame out on.
		infoFieldLen	= length of info field (in bytes)
Input:
Output:		frameBuffer	= contains complete SRF announcment frame.
Return:		Transaction ID is returned if frame was built. Otherwise
		0 is returned.
Modification History:
*********************************************************************/
{
SMTFrameHdrType	*frameHdr;	/* SMT header */
uInt32		result;		/* value to return */

	/*
	*	Initialize values.
	*/
	result = 0;
	frameHdr = (SMTFrameHdrType *) frameBuffer;

	/*
	*	Initialize frame header.
	*/
	/* 910529-001 LJP */
	result = BuildSMTHeader (frameHdr, SMT_INFO_FC,
		SRF_MULTICAST_ADDRESS,
		outputMAC, SRF_CLASS, SMTANNOUNCE, (uInt32) 0);

	/*
	*	Set frame length.
	*/
	frameHdr->smtHdr.InfoField_Length = htons (infoFieldLen);

	return (result);
}

