#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) fbmsrp.c V6.2.3:cs.622c:5:5 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Frame Based Management Module

	Status Reporting Protocol Module

	File:		fbmsrp.c
	Created:	09/05/90

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.622c	08/02/91
	
	Copyright 1990,1991 XLNT Designs, Inc.

	This module is responsible for processing the Status Reporting
	protocol specified by the SMT standard. Every FBM_CLOCK_PULSE
	seconds, ProcessSRPTick() is executed. Based on the current
	station status, the appropriate SRF is generated if necessary.

	Modification History:
	910122-005	LJP
		Removed debugging code.
	910404-003	LJP
		Reworked element manipulation in DeleteECParam() and
		AddECParam() to correct faulty logic.
	910503-002	LJP
		Added code to limit the number of EC entries in the
		list (see comments below).
*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"fbmhdr.h"
#include	"fbmframe.h"
#include	"fbmglbl.h"
#include	"smtmsg.h"
#include	"mibdefs.h"
#include        "msgutil.h"


/*********************************************************************
	Defined Values
*********************************************************************/

/*
*	910503-002	LJP
*
*	NOTICE
*
*	The SRF protocol in the SMT 6.2 standard had several problems
*	associated with it. One of the most significant was the
*	assumption of unlimited memory to hold MIB events and
*	conditions awaiting an SRF. There was no defined action when
*	enough events/conditions have occurred to fill one SRF. The
*	standard indicates holding all events/conditions until the
*	older ones age out. This is unrealistic. If all memory is
*	consumed to hold events/conditions, there may be no memory
*	to use for transmitting the SRF, thus nothing would get aged
*	and the node could not communicate.
*
*	To avoid this lock condition, XDI has imposed an artificial
*	maximum on the number of events/conditions in the list. This
*	simulates the same result with regard to the SRF as if the
*	node had run out of memory, but does not use all of memory.
*	The maximum value is based on the largest number of the
*	average-sized events/conditions. In this implementation,
*	the average size of an event is 24 bytes. The SRF
*	can hold 0x116A (4458) bytes in the info field. Thus a
*	maximum of 185 average-sized events could be held in an SRF. A
*	list consisting of this many events would use 10,360 bytes.
*
*	Future SMT versions will address this problem.
*/

/*********************************************************************
	Global Data
*********************************************************************/

/*
*	This module maintains a list of event/condition parameters to
*	include in the current SRF. The list is a doubly linked list
*	with each element defined as the structure listed below.
*/
struct	ECListElement	*ecListHead;
struct	ECListElement	*ecListTail;

/* 910503-002	LJP */
static	uInt16	ecListCount;


/*********************************************************************
	External Functions
*********************************************************************/

extern	uInt32	BuildSRFAnnouncement ();
extern	uInt32	BuildSMTHeader ();
extern	void	ClearFrameBuffer ();
extern	void	AttrNetOrder ();
extern	uInt32	FBMGetMIBAttr ();
extern	void	SendSMTFrame ();
extern	uInt16	MsgTimeStamp_Param ();
extern	uInt16	Attribute_Param ();
extern 	uInt16	htons ();
extern	void	SendFBMMessage ();


/*********************************************************************
	FBM Status Reporting Protocol Routines
*********************************************************************/

void
DeleteECParam (ecParam)
	struct ECListElement	*ecParam;
/*********************************************************************
Function:	This routine removes a parameter from the event/condition
		list.
Parameters:	ecParam		= address of element to remove.
Input:		Uses event/condition list.
Output:		Updates event/condition list and head/tail pointers.
Return:		None.
Modification History:
910404-003	LJP
	Rework as in AddECParam().
*********************************************************************/
{
	/*
	*	If currently at the end,
	*/
	if (ecParam->nextEC == NULL)
	{
		/* if there is nothing before it */
		if (ecParam->lastEC == NULL)
			/* then the list is now empty */
			ecListHead = ecListTail = NULL;

		else
		{
			/* set new last element */
			ecListTail = ecParam->lastEC;
			ecListTail->nextEC = NULL;
		}
	}

	/*
	*	If at the head of the list,
	*/
	else if (ecParam->lastEC == NULL)
	{
		/* set new head element */
		ecListHead = ecParam->nextEC;
		ecListHead->lastEC = NULL;
	}

	/*
	*	Otherwise, in the middle of the list.
	*/
	else
	{
		/* remove condition element from list */
		ecParam->lastEC->nextEC
			= ecParam->nextEC;
		ecParam->nextEC->lastEC
			= ecParam->lastEC;
	}

	/* remove element */
#if 0
	FREEMEMORY (ecParam);
#else
	FreeSRFMsgBuf(ecParam);   /* jlin */
#endif

	/* 910503-002	LJP */
	ecListCount--;

	return;
}

struct ECListElement *
FindECParam (paramHdr, startPtr)
	TLVParamType		*paramHdr;
	struct ECListElement	*startPtr;
/*********************************************************************
Function:	This routine locates a particular event/condition parameter
		in the event/condition list.
Parameters:	paramHdr	= start of the parameter buffer.
		startPtr	= element in list to start with.
Input:		Uses the event/condition list.
Output:		None.
Return:		Address of EC list element or NULL if not found.
Modification History:
*********************************************************************/
{
struct ECListElement	*p;
TLVParamType		*hdr;

	p = startPtr;
	while (p != NULL)
	{
		/*
		*	If this type matches,
		*/
		hdr = (TLVParamType *) (&p->data);
		if (hdr->paramType == paramHdr->paramType)
		{
			/*
			*	If not an SMT or PATH event/condition,
			*/
			if ((hdr->paramType != fddiSMTConfigurationChgEvent)
				&& (hdr->paramType != fddiPATHTraceStatusEvent))
			{
				/*
				*	If index matches, exit loop.
				*/
				if (hdr->MACINDEX == paramHdr->MACINDEX)
					break;
			}

			else
				/*
				*	Else exit and return this element.
				*/
				break;
		}

		/*
		*	Continue to next element.
		*/
		p = p->nextEC;
	}

	return (p);
}

void
AddECParam (paramHdr)
	TLVHdrType	*paramHdr;
/*********************************************************************
Function:	This routine adds a new parameter to the end of the
		event/condition list.
Parameters:	paramHdr	= start of the parameter buffer.
Input:		Uses ecListHead and ecList Tail.
		paramHdr	= points to the parameter header in the
				buffer holding the parameter to add.
Output:		Updates event/condition list.
Return:		None.
Modification History:
910404-003	LJP
	Complete rework of section that moves existing conditions
	to the end of the list.
*********************************************************************/
{
struct ECListElement	*newElement;

	/*
	*	Add parameter to list.
	*/
	switch (paramHdr->paramType)
	{
	/*
	*	For conditions, only one instance of the condition
	*	is kept in the list. If this is a condition to be
	*	added, first try to find if it already exists in the list.
	*	If it does, then move it to the end of the list.
	*	Otherwise, add a new element like adding an event.
	*/
	case fddiMACDuplicateAddressCondition:
	case fddiMACFrameErrorConditionEvent:
	case fddiMACNotCopiedConditionEvent:
	case fddiPORTLerConditionEvent:
	case fddiPORTEBErrorConditionEvent:
		/*
		*	If this condition is already in the list,
		*	condition is moved to end to maintain
		*	sequence of occurrence.
		*/
		newElement = FindECParam (paramHdr, ecListHead);
		if (newElement)
		{
			/*
			*	If currently at the end,
			*/
			if (newElement->nextEC == NULL)
				/*
				*	Don't touch it.
				*/
				;

			/*
			*	If at the head of the list,
			*/
			else if (newElement->lastEC == NULL)
			{
				/* set new head element */
				ecListHead = newElement->nextEC;

				/* set new head element's last ptr */
				ecListHead->lastEC = NULL;

				/* set last element's new next ptr */
				ecListTail->nextEC = newElement;

				/* set new last element's ptr */
				newElement->lastEC = ecListTail;
				newElement->nextEC = NULL;

				/* set new last element */
				ecListTail = newElement;
			}

			/*
			*	Otherwise, in the middle of the list.
			*/
			else
			{
				/* remove condition element from list */
				newElement->lastEC->nextEC
					= newElement->nextEC;
				newElement->nextEC->lastEC
					= newElement->lastEC;

				/* set last element's new next ptr */
				ecListTail->nextEC = newElement;

				/* set new last element's ptr */
				newElement->lastEC = ecListTail;
				newElement->nextEC = NULL;

				/* set new last element */
				ecListTail = newElement;
			}

			/* exit processing of condition element */
			break;
		}

		/*
		*	Else add a new element like adding an event.
		*/


	/*
	*	Events are always added.
	*/
	case fddiSMTConfigurationChgEvent:
	case fddiMACNeighborChangeEvent:
	case fddiPATHTraceStatusEvent:
	case fddiPORTUndesiredConnectionAttempt:
		/*
		*	910503-002	LJP
		*	Check if max count is reached.
		*/
		if (ecListCount >= MAX_EC_ELEMENTS)
			return;

		/*
		*	Allocate an element.
		*/
		newElement = (struct ECListElement *)
#if 0
			GETMEMORY (sizeof (struct ECListElement));
#else
		        AcptSRFMsgBuf();   /* jlin */
#endif
		if (!newElement)
			return;

		/* 910503-002	LJP */
		ecListCount++;

		/*
		*	Clear buffer.
		*/
		MEMZERO (newElement, sizeof (struct ECListElement));

		/*
		*	Add to list.
		*/
		newElement->nextEC = NULL;
		if (ecListHead == NULL)
			ecListHead = newElement;

		if (ecListTail == NULL)
		{
			ecListTail = newElement;
			newElement->lastEC = NULL;
		}
		else
		{
			ecListTail->nextEC = newElement;
			newElement->lastEC = ecListTail;
			ecListTail = newElement;
		}

		break;
	default:
		return; /* events can not be found */

	}

	/*
	*	Set report limit countdown.
	*/
	newElement->limitCount = fbmStationData.reportLimit;

	/*
	*	Fill in the event data.
	*/
	switch (paramHdr->paramType)
	{
	case fddiSMTConfigurationChgEvent:
		MEMCOPY (&newElement->data.evtCfgChg,
			paramHdr, sizeof (EvtCfgChgType));
		break;

	case fddiMACNeighborChangeEvent:
		MEMCOPY (&newElement->data.evtNbrChg,
			paramHdr, sizeof (EvtNbrChgType));
		break;

	case fddiPATHTraceStatusEvent:
		MEMCOPY (&newElement->data.evtTrStat,
			paramHdr, sizeof (EvtTrStatType));
		break;

	case fddiPORTUndesiredConnectionAttempt:
		MEMCOPY (&newElement->data.evtConnect,
			paramHdr, sizeof (EvtConnectType));
		break;

	case fddiMACDuplicateAddressCondition:
		MEMCOPY (&newElement->data.condDA,
			paramHdr, sizeof (CondDAType));
		break;

	case fddiMACFrameErrorConditionEvent:
		MEMCOPY (&newElement->data.condFrErr,
			paramHdr, sizeof (CondFrErrType));
		break;

	case fddiMACNotCopiedConditionEvent:
		MEMCOPY (&newElement->data.condNotCopied,
			paramHdr, sizeof (CondNotCopiedType));
		break;

	case fddiPORTLerConditionEvent:
		MEMCOPY (&newElement->data.condLer,
			paramHdr, sizeof (CondLerType));
		break;

	case fddiPORTEBErrorConditionEvent:
		MEMCOPY (&newElement->data.condEBError,
			paramHdr, sizeof (CondEBErrType));
		break;
	}

	return;
}


/*********************************************************************
	FBM Status Reporting Protocol Routines
*********************************************************************/

static uInt16
BuildSRFInfo (buffer)
	uChar	*buffer;
/*********************************************************************
Function:	This routine builds the SRF Info field.
		functions of the Status Reporting protocol.
Parameters:	buffer	= pointer to start of SMT Info field in frame buffer.
Input:		Uses fbmStationData, ecList, and conditionList.
Output:		buffer	= Info field for SRF.
Return:		Length of SMT Info field.
Modification History:
*********************************************************************/
{
uInt16			infoFieldLen;	/* length of SMT info */
struct ECListElement	*p;		/* pointer through EC list */
uInt16			paramLen;	/* length of each parameter */
uChar			*memPtr;	/* pointer within info field */
TLVParamType		*mibPtr;	/* MIB parameter */

	/*
	*	Initialize values.
	*/
	infoFieldLen = 0;
	memPtr = buffer;

	/*
	*	Get MsgTimeStamp parameter.
	*/
	paramLen = MsgTimeStamp_Param (memPtr);
	memPtr += paramLen;
	infoFieldLen += paramLen;

	/*
	*	Get TransitionTimeStamp.
	*/
	mibPtr = (TLVParamType *) memPtr;
	mibPtr->paramType = fddiSMTTransitionTimeStamp;
	mibPtr->paramLen = 4;
	mibPtr->MACINDEX = 0;
	FBMGetMIBAttr (sizeof (TLVParamType),
		(uChar *) mibPtr, (SetCountType *) NULL);
	paramLen = Attribute_Param (memPtr, mibPtr);
	memPtr += paramLen;
	infoFieldLen += paramLen;

	/*
	*	Loop through event/condition list.
	*/
	p = ecListHead;
	paramLen = 0;
	while (p)
	{
		paramLen = 0;
		switch (p->data.hdr.paramType)
		{
		case fddiSMTConfigurationChgEvent:
#ifdef OPTIONAL_PARAMETER
			paramLen = sizeof (TLVHdrType)
				+ sizeof (TLVHdrType)
				+ sizeof (TLV8BitType)
				+ p->data.evtCfgChg.PathListHdr.paramLen;
#else
			paramLen = sizeof (TLVHdrType)
				+ sizeof (TLVHdrType)
				+ sizeof (TLV8BitType);
#endif
			break;

		case fddiMACNeighborChangeEvent:
			paramLen = sizeof (EvtNbrChgType);
			break;

		case fddiPATHTraceStatusEvent:
			paramLen = sizeof (EvtTrStatType);
			break;

		case fddiPORTUndesiredConnectionAttempt:
			paramLen = sizeof (EvtConnectType);
			break;

		case fddiMACDuplicateAddressCondition:
			paramLen = sizeof (CondDAType);
			break;

		case fddiMACFrameErrorConditionEvent:
			paramLen = sizeof (CondFrErrType);
			break;

		case fddiMACNotCopiedConditionEvent:
			paramLen = sizeof (CondNotCopiedType);
			break;

		case fddiPORTLerConditionEvent:
			paramLen = sizeof (CondLerType);
			break;

		case fddiPORTEBErrorConditionEvent:
			paramLen = sizeof (CondEBErrType);
			break;
		}

		/*
		*	If valid parameter and there is room in the frame,
		*	then add the parameter to the frame.
		*/
		if (paramLen &&
			((infoFieldLen + paramLen) <= MAX_SMT_INFO_LEN))
		{
			/* copy parameter to info field */
			MEMCOPY (memPtr, &p->data, paramLen);

			/* convert to network order */
			AttrNetOrder (memPtr, NET_ORDER);

			/* accumulate length */
			infoFieldLen += paramLen;

			/* move pointer */
			memPtr += paramLen;
		}
/* 910122-005 LJP */

		/* move to next element */
		p = p->nextEC;
	}

	return (infoFieldLen);
}

static void
UpdateSRP ()
/*********************************************************************
Function:	This routine updates the EC list and sets the next time
		for an SRF.
Parameters:	None.
Input:		Uses fbmStationData, ecList, and conditionList.
Output:		Updates fbmStationData, ecList, and conditionList.
Return:		None.
Modification History:
*********************************************************************/
{
struct ECListElement	*p;		/* pointer through EC list */

	p = ecListHead;
	while (p != NULL)
	{
		/* decrement if not active condition */
		switch (p->data.hdr.paramType)
		{
		case fddiSMTConfigurationChgEvent:
		case fddiMACNeighborChangeEvent:
		case fddiPATHTraceStatusEvent:
		case fddiPORTUndesiredConnectionAttempt:
			p->limitCount--;
			break;

		case fddiMACDuplicateAddressCondition:
			if (p->data.condDA.Condition == 0)
				p->limitCount--;
			break;

		case fddiMACFrameErrorConditionEvent:
			if (p->data.condFrErr.Condition_State == 0)
				p->limitCount--;
			break;

		case fddiMACNotCopiedConditionEvent:
			if (p->data.condNotCopied.Condition_State == 0)
				p->limitCount--;
			break;

		case fddiPORTLerConditionEvent:
			if (p->data.condLer.ConditionState == 0)
				p->limitCount--;
			break;

		case fddiPORTEBErrorConditionEvent:
			if (p->data.condEBError.ConditionState == 0)
				p->limitCount--;
			break;
		}

		/* if count is 0 */
		if (p->limitCount == 0)
		{
			/* delete it from the list */
			if (p->nextEC != NULL)
			{
				/* not last in list */
				p = p->nextEC;
				DeleteECParam (p->lastEC);
			}
			else
			{
				/* last in list */
				DeleteECParam (p);
				p = NULL;
			}
		}
		else
			p = p->nextEC;
	}

}

void
ProcessSRPTick ()
/*********************************************************************
Function:	This routine is called to process the periodic
		functions of the Status Reporting protocol.
Parameters:	None.
Input:		Uses fbmStationData, ecList, and conditionList.
Output:		Updates fbmStationData, ecList, and conditionList.
Return:		None.
Modification History:
*********************************************************************/
{
uInt16		len,			/* length of frame */
		infoLen;		/* length of info field */
uChar		*buffer;		/* buffer to build info field */
Int16		i;			/* loop counter */
TLVParamType	mibData;		/* MIB parameter */
Flag		frameQueued;		/* flag set if a frame is queued */
uInt32		transID;		/* transaction ID for frame */
SMTMessage	smtmsg;			/* message for MIB */

	/*
	*	Send message to MIB to check for conditions.
	*/
	smtmsg.destination = MIB_MSG_ID;
	smtmsg.source = FBM_MSG_ID;
	smtmsg.type = MIB_ACTION_DETECT_CONDITIONS;
	smtmsg.typeInfo = 0;
	smtmsg.localID = 0;
	smtmsg.len1 = 0;
	smtmsg.len2 = 0;
	smtmsg.entity = 0;
	SendFBMMessage (&smtmsg);

	/*
	*	Check if timer has expired.
	*/
	fbmStationData.TSR += FBM_CLOCK_TICK;
	if (fbmStationData.TSR < fbmStationData.nextSRF)
		/* not time to send SRF yet */
		return;

	/*
	*	If nothing in the list, then return.
	*/
	if (ecListHead == NULL)
		return;

        frameQueued = CLEAR;  /* jlin */
        transID = 0;         /* jlin */

	for (i = 0; i < fbmStationData.macCount; i++)   /* jlin */
	{                               /* build the SRF for each buffer */
	   /*
	    *	Get address in buffer of the info field.
	    */
	   buffer = (uChar *) (fbmFrameHeader + 1);

	   /*
	    *	Clear frame buffer.
	    */
	   ClearFrameBuffer (fbmFrameBuffer);

	   /*
	    *	Build frame info field.
	    */
	   infoLen = BuildSRFInfo (buffer);

	   /*
	    *	For each MAC with RING_OP, send an SRF.
	    */
	   len = SMT_FRAME_HDR_SIZE + infoLen;

		/*
		*	Skip invalid MACs.
		*/
		if (!fbmMACData[i].operational)
			continue;

		/*
		*	Get RMT state.
		*/
		mibData.paramType = fddiMACRMTState;
		mibData.paramLen = 4;
		mibData.MACINDEX = i + 1;
		FBMGetMIBAttr (sizeof (TLVParamType),
			(uChar *) &mibData, (SetCountType *) NULL);

		/*
		*	Ring is up if in RM2 or RM5.
		*/
		if (mibData.MACPARAM8 == RM_RING_OP
			|| mibData.MACPARAM8 == RM_RING_OP_DUP)
		{
			/*
			*	Info field is already set,
			*	fill in MAC and SMT headers.
			*/
			transID = BuildSMTHeader (fbmFrameBuffer,
				SMT_INFO_FC, SRF_MULTICAST_ADDRESS,
				i, SRF_CLASS, SMTANNOUNCE, transID);

			/*
			*	Set frame length.
			*/
			fbmFrameHeader->smtHdr.InfoField_Length
				= htons (infoLen);

			/* send frame */
			SendSMTFrame (fbmFrameBuffer, len, i);

			/*
			*	Set flag indicating at least one
			*	frame was queued for transmission
			*/
			frameQueued = SET;
		}
	}

	/*
	*	If at least one frame was queued,
	*	remove all elements in list that have reached the
	*	report limit.
	*/
	if (frameQueued)
		UpdateSRP ();
	/*
	*	Update time to next SRF.
	*/
	if (fbmStationData.nextSRF < MAX_SRF_INTERVAL)
		/* double the time */
		fbmStationData.nextSRF <<= 1;

	/*
	*	Reset timer for next SRF. Note that if no frames were
	*	sent, then the time until the next SRF is sent is not
	*	changed.
	*/
	fbmStationData.TSR = 0;

	return;
}

uInt32
InitSRP ()
/*********************************************************************
Function:	Initialize the Status Reporting Protocol.
Parameters:	None.
Input:		None.
Output:		Initializes the event/condition list to be empty.
Return:		0 if successful, otherwise an error code is returned.
Modification History:
*********************************************************************/
{
	ecListHead = NULL;
	ecListTail = NULL;

	/* 910503-002 LJP */
	ecListCount = 0;

	return (0);
}

