/*********************************************************************
	Connection Services Process Module

	Message Interface Routines

	THIS FILE IS TO BE PORTED BY THE IMPLEMENTOR.

	File:		cspmsgif.c
	Created:	12/01/89

	SID:		1.3
	Last Modified:	1/28/91

	Copyright 1990,1991 XLNT Designs Inc.

	This module provides the interface routines to handle message
	passing between CSP and the other processes in the SMT system.

	Modification History:

	*** Updated to SMT 6.2 ***

*********************************************************************/
#define __CSP_DEBUG

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include        "csphdr.h"
#include	"fddihdr.h"
#include	"smtmsg.h"
#include	"fbmhdr.h"
#include        "krnl.h"
#include        "msgutil.h"
#include        "cmtflag.h"
#include        "dips.h"

/*********************************************************************
	Message Interface External Functions
*********************************************************************/
extern SMTStationIdType *ThisStationId;

/*********************************************************************
	Message Interface Defined Values
*********************************************************************/


/*********************************************************************
	Message Interface Global Variables
*********************************************************************/
SEM Sema_CSP;
uInt32 SigData[MAX_PORT_COUNT+1];
uInt16 SigRepeatCount[MAX_PORT_COUNT+1];

/*********************************************************************
	Message Interface Routines
*********************************************************************/
uInt32 ReceiveCSPMessage (msg)
SMTMessage *msg;
{
   ProcessCSPMessage (msg);
   SendSignal  ( &Sema_CSP );       /* send a signal trigger in case
				       there's a signal triggered by 
				       ProcessCSPMessage()
				     */
}


/*
 *   Function provided to FDDI DRIVER's ISR
 */

void CSP_SMTSendSignal (signalType, entity, data)
uInt16 signalType;
uInt16  entity;
uInt32 data;
{
   extern int fault_cnt;
   extern uChar Port2LedMap[];

   switch (signalType)
     {
     case SIG_PC_Start:
     case SIG_PC_Stop:
       PushSignal(signalType, entity);
       break;

     case SIG_Line_State:
	  if (data == SigData[entity])
	  {
	     if (++SigRepeatCount[entity] > 100)
	     {
	        if (get_debug() == DEBUG_ON)
		{
		   fault_cnt++;
		   printf("Port%d: Intensive PHY interrupt pattern %x detected.\n", entity, SigData[entity]);
		   fault_cnt--;
		}
	        if (entity == PHY_B)
		{
		   SetBPBQuietLine();
		}
		else if (entity == PHY_A)
	        {
		   SetBPAQuietLine();
		}
		SigRepeatCount[entity] = 0;
		strobe_wdt();
	     }
	  }
	  else
	  {
	     SigRepeatCount[entity] = 0;
	     SigData[entity] = data;
	  }
          /* fall through */

     default:
       SMTSendSignal ( signalType, entity, data );        /* to CSP */
       break;
     }
   SendSignal  ( &Sema_CSP );       /* send a signal trigger */
}


void
SendCSPMessage (msg)
	SMTMessage	*msg;
/*********************************************************************
Function:	Send a message from CSP to another process.
Parameters:	msg	= address of buffer containing message.
Input:		msg	= contains SMT message structure that defines the
				message to send.
Output:		None.
Return:		None.
Notes:		In systems, where all Manager tasks (CSP, FBM, MIB and
		possibly MAP), this routine may call the destination
		task's ProcessXXXMessage() function directly.
Modification History:
*********************************************************************/
{
   switch (msg->destination) {
   case MIB_MSG_ID:
      ProcessMIBMessage (msg);         /* Call directly without allocate
					  extra memory and copy the data 
				        */
      break;

   case MAP_MSG_ID:
      ProcessMAPMessage (msg);         /* Call directly without allocate
					  extra memory and copy the data 
				        */
      break;


   case FBM_MSG_ID:
#ifdef __CMT_ONLY
      return;
#endif
      CSPDPT("$CSP: Send message to FBM$...");
      ProcessFBMMessage (msg);         /* Call directly without allocate
					  extra memory and copy the data 
				        */
      break;
   }

   
	return;
}

uInt32
InitCSPMessage ()
/*********************************************************************
Function:	Initialize system for sending and receiving SMT messages.
Parameters:	None.
Input:		None.
Output:		Set system message processing.
Return:		0 if successful, error number if failed.
*********************************************************************/
{
   int i;

   CreatSemaphore (&Sema_CSP, 0);      /* Semaphore for signal */
   for (i=0; i < MAX_PORT_COUNT; i++)
   {
      SigData[i] = SigRepeatCount[i] = 0;
   }
   return (0);
}

/*********************************************************************
	MIB Interface Routines

	The MIB interface routines are placed with the message
	interface routines because the MIB interface may require
	strong support from the message passing facilities provided
	by the system. In cases where this process is separate and
	unique from the MIB process, the message passing facilities
	will implement some form of interprocess communication.
		The strong support will be required when this process 
	is waiting for a response from the MIB. In this case, any other
	messages will need to be held off or enqueued for storage
	until the MIB response is received.
		CSP does not set MIB attributes through the standard
	change interface. Therefore, only the get interface needs to
	be implemented.
*********************************************************************/

uInt32
CSPGetMIBAttr (bufSize, bufData, setCount)
	uInt16		bufSize;
	uChar		*bufData;
	SetCountType	*setCount;
/*********************************************************************
Function:	Get information from the MIB. This function provides
		CSP with access to the MIB. In cases where the MIB and
		CSP reside in the same executable image, this function
		can simply call GetMIBAttr(). If CSP and MIB are separated,
		this function will perform the necessary communication
		functions to access the MIB and wait for a reply.
Parameter:	bufSize		= maximum length of buffer bufData
					measured in chars.
		bufData		= pointer to buffer containing the
					requested attribute information
					and used to hold attribute value
					returned.
		setCount	= pointer to buffer to hold current value
					of the fddiSMTSetCountAttribute
					(may be NULL if current SetCount
					is not required).
Input:		bufData		= contains the requested attribute ID and
					object index (if required) using
					the MIB TLV encoding for a Get
					Request (see SMT standard).
Output:		bufData		= holds the attribute value requested
					using the MIB TLV encoding specified
					in the SMT standard.
		setCount	= if not NULL, this buffer gets set to the
					current value of fddiSMTSetCount,
					also using the MIB TLV encoding.
Return:		Returns the same values as specified by GetMIBAttr().
Note:		The operation of this function must cause CSP to block
		(or effectively stop processing) until the response
		is received from the MIB. This routine is only called
		during initialization when the station is disconnected
		from the ring.
*********************************************************************/
{
extern	uInt32	GetMIBAttr ();

	return (GetMIBAttr (bufSize, bufData, setCount));
}


uInt32
CSPChangeMIBAttr (bufSize, bufData, setCount)
	uInt16		bufSize;
	uChar		*bufData;
	SetCountType	*setCount;
{
extern	uInt32	ChangeMIBAttr ();

	return (ChangeMIBAttr (bufSize, bufData, setCount, ThisStationId));
}



