#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) csppcode.c V6.2.3:cs.911022:7:7 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Connection Services Process Module
	
	PCM State Machine Pseudo Code
	
	File:		csppcode.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.911022	10/28/91
	
	Copyright 1990,1991 XLNT Designs, Inc.
	
	This module implements the PCM state machine pseudo code listed 
	in the ANSI X3T9.5 standard.
		
	This file consists of three functions. The first function 
	implements the Transmit Code and the second function implements
	the Receive Code. The third function test the station policies
	to see if a connection topology is allowed.

	Modification History:

	*** Updated to SMT 6.2 ***

	910125-002	LJP
		The MACActiveOutput() routine was not implemented 
		properly for dual-MAC stations.
	910829-001	LJP
		Force the A link to use the extended LCT when withholding
		the connection due to Tree mode connection.
	911022-003	LJP
		Changed PC_TCode_Actions() cases 4 & 9 to accomodate
		SMT-LBC-69 (avoid MAC-less wrap and rapid cycling of
		withheld A connection). Removes 910829-001.
*********************************************************************/
#define         REDUNDANT_LINK

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"smtmsg.h"
#include	"cspmacro.h"
#include	"csphdr.h"
#include	"cspglbl.h"
#include	"mibdefs.h"


/*********************************************************************
	PCM Pseudo Code External Functions
*********************************************************************/

extern	void	SendCSPEvent ();
extern	void	SMTSendSignal ();
extern	void	StartLCT ();
extern	void	StopLCT ();



/*********************************************************************
	PCM Connection Test
*********************************************************************/

Flag
Allow_Connection (phyID)
	uInt16	phyID;
/*********************************************************************
Function:	Determine if this connection topology is allowed by the
		PHY's connection policy.
Parameters:	phyID	= index of PHY to process.
Input:		Uses phyData and stationData.
Output:		None.
Return:		SET if connection is allowed, CLEAR if not allowed.
Modification History:
*********************************************************************/
{
uInt16	connectionType;			/* type of connection */
Flag	result;				/* return value */

	/*
	*	The connection type is determined by setting the
	*	bit corresponding to the connection policy. The way
	*	the bits are ordered in fddiSMTConnectionPolicy so that
	*	the policies for A ports are in the rightmost four bits,
	*	the B port policies are in the next four bits, etc. Each
	*	of the four bits represent neighbors of type A, B, S and M
	*	moving right to left.
	*
	*	To select the proper bit representing the current connection,
	*	set the neighbors bit, then shift it to the proper nibble.
	*/

	/* Set neighbor bit */
	connectionType = 1 << phyData[phyID].PC_Neighbor;
	
	/* Shift 4 bits per port type */
	connectionType <<= (phyData[phyID].PC_Type << 2);

	/*
	*	If bit is set in connection policy, then
	*	this link is not acceptable to this station.
	*/
	result = !(stationData.connectionPolicy & connectionType);

	return (result);
}

Flag
MACActiveOutput (phyID)
	uInt16	phyID;
/*********************************************************************
Function:	Determine if MAC output is connected to this PORT when
		the PORT is in the ACTIVE state.
Parameters:	phyID	= index of PHY to process.
Input:		Uses phyData and stationData.
Output:		None.
Return:		SET if MAC connected, otherwise CLEAR.
Modification History:
	910125-002	LJP
		This routine did not properly handle dual-MAC concentrator
		situations. It used to report Remote MAC on M ports.
*********************************************************************/
{
Flag	result=CLEAR;			/* return value */
uInt16	i;				/* loop counter */

	/*
	*	If there are no operational MACs, then this is a
	*	MACless station and no ports have MAC output.
	*/
	if (!macData[PRIMARY_MAC].operational 
			&& !macData[SECONDARY_MAC].operational)
		return (CLEAR);

	/*
	*	Select operation based on number of attachment ports.
	*/
	switch (stationData.attachCount)
	{
	/*
	*	For single attachment stations, the S port always has
	*	the MAC output.
	*/
	case 1:
		result = (phyData[phyID].PC_Type == PC_Type_S) ? SET : CLEAR;
		break;

	/*
	*	For dual attachment stations, if there are two MACs, then
	*	the A and B ports each have MAC outputs. Otherwise, the 
	*	B port always gets the MAC output in Tree mode.
	*/
	/*
	*	910125-002	LJP
	*	Added support for dual attachment concentrators by testing
	*	if this is an M port.
	*/
	case 2:
		if (phyData[phyID].PC_Type == PC_Type_M)
			result = CLEAR;
		else
		{
			if (stationData.macCount == 2)
				result = SET;
			else 
				result = (phyData[phyID].PC_Type == PC_Type_B)
					? SET : CLEAR;
		}
		break;

	/*
	*	In NULL attachment stations, then if there is only one
	*	MAC, the first M port will have the MAC output. If there are
	*	two MACs, the first M port on this path has the MAC output.
	*/
	case 0:
		if (stationData.macCount == 1)
			/* first M port in station has output */
			result = (phyID == PHY_M) ? SET : CLEAR;

		else
		{
			/* assume this is the first M port on the path */
			result = SET;

			/* find first M port on this path */
			for (i = phyID - 1; i >= PHY_M; i--)
				if (phyData[i].operational &&
					(phyData[i].pathsRequested ==
					phyData[phyID].pathsRequested))
				{
					/* found a port before this one */
					result = CLEAR;
					break;
				}

			/* 
			* if this is the first M port on the path,
			* then check if there is a MAC on this path.
			*/
			if (result)
			{
				if (phyData[phyID].pathsRequested & PA_PRIMARY)
				{
					if (!macData[PRIMARY_MAC].operational)
						result = CLEAR;
				}
				else
				{
					if (!macData[SECONDARY_MAC].operational)
						result = CLEAR;
				}
			}
		}
		break;
				
	}

	return (result);
}

Flag
MACThruOutput (phyID)
	uInt16	phyID;
/*********************************************************************
Function:	Determine if MAC output is connected to this PORT when
		the station is in the CFM Thru state.
Parameters:	phyID	= index of PHY to process.
Input:		Uses phyData and stationData.
Output:		None.
Return:		SET if MAC connected, otherwise CLEAR.
Notes:		Being able to enter a THRU state implies that this
		station has attachment ports and cannot be a NULL
		attachment concentrator.
Modification History:
*********************************************************************/
{
Flag	result;				/* return value */

	/*
	*	If there are no operational MACs, then this is a
	*	MACless station and no ports have MAC output.
	*/
	if (!macData[PRIMARY_MAC].operational 
			&& !macData[SECONDARY_MAC].operational)
		return (CLEAR);

	/*
	*	Master ports do not have MACs in output path.
	*/
	if (phyData[phyID].PC_Type == PC_Type_M)
		result = CLEAR;

	else 
	{
		/*
		*	Dual-MAC or single attach always 
		*	have MAC in output path.
		*/
		if ((stationData.macCount > 1) 
				|| (stationData.attachCount == 1))
			result = SET;

		else
		{
			/*
			*	If MAC goes on primary path.
			*/
			if (macData[PRIMARY_MAC].pathsRequested & PA_PRIMARY)
				/* B port has MAC */
				result = (phyData[phyID].PC_Type == PC_Type_B);
			else
				/* A Port has MAC */
				result  = (phyData[phyID].PC_Type == PC_Type_A);
		}
	}

	return (result);
}

/*********************************************************************
	PCM Transmit Code
*********************************************************************/

void
PC_TCode_Actions (phyID)
	uInt16	phyID;
/*********************************************************************
Function:	PCM Transmit Code.
Parameters:	phyID	= index of PHY to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/*
	*	Select transmit bit.
	*/
	switch (pcmData[phyID].n)
	{
	/*
	*	T_Val(0)
	*	Escape Bit. Reserved for future use. Set to 0.
	*/
	case 0:
		MSetBit (pcmData[phyID].T_Val, pcmData[phyID].n, CLEAR);
		SMTSendSignal (SIG_PC_Signal, phyID, (uInt32) 0);
		break;
		
	/*
	*	T_Val(1)
	*	First bit for PC_Type. Bits 1 and 2 are set as follows:
	*		PC_Type		Bit 1	Bit 2
	*		A		0	0
	*		B		0	1
	*		S		1	0
	*		M		1	1
	*/
	case 1:
		if (phyData[phyID].PC_Type == PC_Type_A
				|| phyData[phyID].PC_Type == PC_Type_B)
			MSetBit (pcmData[phyID].T_Val, pcmData[phyID].n, CLEAR);
		else
			MSetBit (pcmData[phyID].T_Val, pcmData[phyID].n, SET);
			
		SMTSendSignal (SIG_PC_Signal, phyID, (uInt32) 0);
		break;

	/*
	*	T_Val(2)
	*	Second bit for PC_Type. See T_Val(1).
	*/
	case 2:
		if (phyData[phyID].PC_Type == PC_Type_A
				|| phyData[phyID].PC_Type == PC_Type_S)
			MSetBit (pcmData[phyID].T_Val, pcmData[phyID].n, CLEAR);
		else
			MSetBit (pcmData[phyID].T_Val, pcmData[phyID].n, SET);
			
		SMTSendSignal (SIG_PC_Signal, phyID, (uInt32) 0);
		break;

	/*
	*	T_Val(3)
	*	PHY Compatibility. Set this bit if this station allows this
	*	topology connection.
	*/
	case 3:
		/*
		*	Set bit based on topology rules.
		*/
		MSetBit (pcmData[phyID].T_Val, pcmData[phyID].n,
			Allow_Connection (phyID));
		SMTSendSignal (SIG_PC_Signal, phyID, (uInt32) 0);
		break;

	/*
	*	T_Val(4)
	*	Bits 4 and 5 determine the duration of the Link Confidence
	*	Test using the following encoding:
	*		Length		Bit 4	Bit 5
	*		Short		0	0
	*		Medium		0	1
	*		Long		1	0
	*		Extended	1	1
	*/
	case 4:
		/*
		*	911022-003	LJP
		*	Changed if() condition for SMT-LBC-69.
		*/
		if ((phyData[phyID].PC_Withhold == PC_WH_None)
#ifdef REDUNDANT_LINK
			&& !(((phyData[phyID].PC_Type == PC_Type_A) ||
			      ((phyData[phyID].PC_Type == PC_Type_S &&
				phyID == PHY_A)))
#else
			&& !((phyData[phyID].PC_Type == PC_Type_A)
#endif
				&& (stationData.WA_Flag
				    /* phyID must be PHY A here */
				    || (stationData.WAT_Flag &&
				      phyData[phyID].PC_Mode == PC_Mode_Tree))))
		{
			if (lemData[phyID].LEM_Fail)
			{
				MSetBit (pcmData[phyID].T_Val,
					pcmData[phyID].n, SET);
				MSetBit (pcmData[phyID].T_Val,
					pcmData[phyID].n + 1, CLEAR);
			}
			else
			{
				MSetBit (pcmData[phyID].T_Val,
					pcmData[phyID].n, CLEAR);
				if (phyData[phyID].PC_LCT_Fail > 0)
					MSetBit (pcmData[phyID].T_Val, 
						pcmData[phyID].n + 1, SET);
				else
					MSetBit (pcmData[phyID].T_Val, 
						pcmData[phyID].n + 1, CLEAR);
			}
		}
		else
		{
			MSetBit (pcmData[phyID].T_Val, pcmData[phyID].n, SET);
			MSetBit (pcmData[phyID].T_Val, 
				pcmData[phyID].n + 1, SET);
		}

		SMTSendSignal (SIG_PC_Signal, phyID, (uInt32) 0);
		break;

	/*
	*	T_Val(5)
	*	Bit 5 is set in T_Val(4) so just send PC_Signal.
	*/
	case 5:
		SMTSendSignal (SIG_PC_Signal, phyID, (uInt32) 0);
		break;

	/*
	*	T_Val(6)
	*	SET if MAC is used at this end for LCT.
	*/
	case 6:
		/*
		*	Check station policies if MAC should be used.
		*/
		if ((phyData[phyID].connectionPolicies & PC_MAC_LCT)
				&& stationData.CF_MAC)
			MSetBit (pcmData[phyID].T_Val, pcmData[phyID].n, SET);
		else
			MSetBit (pcmData[phyID].T_Val, pcmData[phyID].n, CLEAR);
		SMTSendSignal (SIG_PC_Signal, phyID, (uInt32) 0);
		break;

	/*
	*	T_Val(7)
	*	SET if this end failed the LCT.
	*/
	case 7:
		StopLCT (phyID);
		SMTSendSignal (SIG_CF_Loop, phyID, (uInt32) 0);
		if (phyData[phyID].PC_LCT_Fail > 0)
			MSetBit (pcmData[phyID].T_Val, pcmData[phyID].n, SET);
		else
			MSetBit (pcmData[phyID].T_Val, pcmData[phyID].n, CLEAR);
		SMTSendSignal (SIG_PC_Signal, phyID, (uInt32) 0);
		break;

	/*
	*	T_Val(8)
	*	SET if this end will provide a MAC for the local loop.
	*/
	case 8:
		if ((phyData[phyID].connectionPolicies & PC_MAC_Loop)
				&& stationData.CF_MAC)
			MSetBit (pcmData[phyID].T_Val, pcmData[phyID].n, SET);
		else
			MSetBit (pcmData[phyID].T_Val, pcmData[phyID].n, CLEAR);
		SMTSendSignal (SIG_PC_Signal, phyID, (uInt32) 0);
		break;

	/*
	*	T_Val(9)
	*	SET if this end will place a MAC on the PHY output.
	*/
	case 9:
		SMTSendSignal (SIG_CF_Loop, phyID, (uInt32) CLEAR);
		/*
		*	911022-003	LJP
		*	Changed if() condition for SMT-LBC-69.
		*/
		if ((phyData[phyID].PC_Withhold == PC_WH_None)
#ifdef REDUNDANT_LINK
			&& !(((phyData[phyID].PC_Type == PC_Type_A) ||
			      ((phyData[phyID].PC_Type == PC_Type_S &&
				phyID == PHY_A)))
#else
			&& !((phyData[phyID].PC_Type == PC_Type_A)
#endif
				&& (stationData.WA_Flag
				    /* phyID must be PHY A here */
				    || (stationData.WAT_Flag &&
				      phyData[phyID].PC_Mode == PC_Mode_Tree))))
		{
			if (phyData[phyID].PC_Mode == PC_Mode_Tree)
				MSetBit (pcmData[phyID].T_Val, 
					pcmData[phyID].n, 
					MACActiveOutput (phyID));
			else
				MSetBit (pcmData[phyID].T_Val, 
					pcmData[phyID].n, 
					MACThruOutput (phyID));
			SMTSendSignal (SIG_PC_Signal, phyID, (uInt32) 0);
		}
		else
			SMTSendSignal (SIG_PC_Start, phyID, (uInt32) 0);

		break;
	}
	
	return;
}


/*********************************************************************
	PCM Receive Code
*********************************************************************/

void
PC_RCode_Actions (phyID)
	uInt16	phyID;
/*********************************************************************
Function:	PCM Receive Code.
Parameters:	phyID	= index of PHY to process.
Input:		Uses pcmData.
Output:		Changes pcmData.
Return:		No value returned.
Modification History:
*********************************************************************/
{
	/*
	*	Select receive bit.
	*/
	switch (pcmData[phyID].n)
	{
	/*
	*	R_Val(0)
	*	No receive actions, just start transmit code.
	*/
	case 0:
		PC_TCode_Actions (phyID);
		break;
		
	/*
	*	R_Val(1)
	*	Escape Bit. No receive action.
	*/
	case 1:
		PC_TCode_Actions (phyID);
		break;
		
	/*
	*	R_Val(2)
	*	First bit of PC_Type. No receive action.
	*/
	case 2:
		PC_TCode_Actions (phyID);
		break;
		
	/*
	*	R_Val(3)
	*	Second bit of PC_Type. Set PC_Neighbor.
	*/
	case 3:
		if (MReadBit (pcmData[phyID].R_Val, 1))
		{
			if (MReadBit (pcmData[phyID].R_Val, 2))
				phyData[phyID].PC_Neighbor = PC_Type_M;
			else
				phyData[phyID].PC_Neighbor = PC_Type_S;
		}
		else
		{
			if (MReadBit (pcmData[phyID].R_Val, 2))
				phyData[phyID].PC_Neighbor = PC_Type_B;
			else
				phyData[phyID].PC_Neighbor = PC_Type_A;
		}
		SendCSPEvent (fddiPORTPC_Neighbor, phyID);
		PC_TCode_Actions (phyID);
		break;
		
	/*
	*	R_Val(4)
	*	Set PC_Withhold and PC_Mode
	*/
	case 4:
		if (phyData[phyID].PC_Type == PC_Type_M
				&& phyData[phyID].PC_Neighbor == PC_Type_M)
		{
			phyData[phyID].PC_Withhold = PC_WH_M_to_M;
		}
			
		else
		{
			if (MReadBit (pcmData[phyID].T_Val, 3)
				|| MReadBit (pcmData[phyID].R_Val, 3))
			{
				phyData[phyID].PC_Withhold = PC_WH_None;
				if (phyData[phyID].PC_Type == PC_Type_M
				|| phyData[phyID].PC_Neighbor == PC_Type_M)
					phyData[phyID].PC_Mode = PC_Mode_Tree;
				else
					phyData[phyID].PC_Mode = PC_Mode_Peer;
			}
			else
			{
				phyData[phyID].PC_Withhold = PC_WH_Other;
			}
		}
		
		SendCSPEvent (fddiPORTPC_Withhold, phyID);
		PC_TCode_Actions (phyID);
		break;
		
	/*
	*	R_Val(5)
	*	First bit of LCT duration. No receive action.
	*/
	case 5:
		PC_TCode_Actions (phyID);
		break;
		
	/*
	*	R_Val(6)
	*	Determine LCT duration.
	*/
	case 6:
		if (MReadBit (pcmData[phyID].T_Val, 4)
				|| MReadBit (pcmData[phyID].R_Val, 4))
		{
			if ((MReadBit (pcmData[phyID].T_Val, 4)
				&& MReadBit (pcmData[phyID].T_Val, 5))
				|| (MReadBit (pcmData[phyID].R_Val, 4)
					&& MReadBit (pcmData[phyID].R_Val, 5)))
				pcmData[phyID].LC_Test = LC_Extended;
			else
				pcmData[phyID].LC_Test = LC_Long;
		}
		
		else
		{
			if (MReadBit (pcmData[phyID].T_Val, 5)
					|| MReadBit (pcmData[phyID].R_Val, 5))
				pcmData[phyID].LC_Test = LC_Medium;
			else
				pcmData[phyID].LC_Test = LC_Short;
		}

		PC_TCode_Actions (phyID);
		break;
		
	/*
	*	R_Val(7)
	*	Check if other end will use MAC for LCT.
	*/
	case 7:
		/*
		*	If my end is to place a MAC on the link,
		*	then do so if MAC is not already in use.
		*/
		if (MReadBit (pcmData[phyID].T_Val, 6)
				&& (rmtData[LOCAL_MAC].RM_Loop == 0))
			SMTSendSignal (SIG_CF_Loop, phyID, (uInt32) SET);

		/* turn on LCT */
		StartLCT (phyID);

		/* Enable PHY to source Idle and repeat symbols. */
		SMTSendSignal (SIG_PC_PDR, phyID, (uInt32) 0);

		pcmData[phyID].TD_Flag = SET;
		break;

	/*
	*	R_Val(8)
	*	Check if LCT passed.
	*/
	case 8:
		if (MReadBit (pcmData[phyID].T_Val, 7)
				|| MReadBit (pcmData[phyID].R_Val, 7))
			SMTSendSignal (SIG_PC_Start, phyID, (uInt32) 0);
		else
			PC_TCode_Actions (phyID);
		break;
		
	/*
	*	R_Val(9)
	*	Check for local loop.
	*/
	case 9:
		if (MReadBit (pcmData[phyID].T_Val, 8)
			|| MReadBit (pcmData[phyID].R_Val, 8))
		{
			SMTSendSignal (SIG_PC_PDR, phyID, (uInt32) 0);
			/* put MAC on local loop if not already in use */
			if (MReadBit (pcmData[phyID].T_Val, 8)
					&& (rmtData[LOCAL_MAC].RM_Loop == 0))
				SMTSendSignal (SIG_CF_Loop, phyID, (uInt32) SET);
			pcmData[phyID].TD_Flag = SET;
		}
		else
			PC_TCode_Actions (phyID);
		break;

	/*
	*	R_Val(10)
	*	Set topology.
	*/
	case 10:
		SendCSPEvent (fddiPORTRemoteMACIndicated, phyID);
		SMTSendSignal (SIG_PC_Join, phyID, (uInt32) 0);
		break;
	}

	return;
}
