#ifdef USE_WHAT_STRING
static char xdi_id[] = "@(#) csplem.c V6.2.3:cs.911111:7:7 Mon Nov 11 16:39:36 1991 Copyright 1990,1991 XLNT Designs, Inc.";
#endif
/*********************************************************************
	Connection Services Process Module
	
	LEM State Machine
	
	File:		csplem.c
	Created:	12/01/89

	Version:	V6.2.3	Mon Nov 11 16:39:36 1991
	Last Modified:	cs.911111	11/11/91
	
	Copyright 1990,1991 XLNT Designs, Inc.
	
	This module implements the LEM state machine discussed in
	the XDI SMT documentation.
		
	This file consists of two parts. The first part consists of
	a set of functions that perform the various actions associated
	with a transition into a state. The second part implements the
	LEM state machine.

	Modification History:

	*** Updated to SMT 6.2 ***

	910807-001	LJP
		Corrected logic for accumulating LEM error count in
		sliding window rateEst[][].
	911111-002	LJP
		Corrected change 910807-001. Sliding window now counts
		up to a max time value in each cell as long as no errors
		occur in that cell. The index value is incremented if
		an error occurs and the total time in the table is 
		greater than 512 seconds. Index also increments when
		the cutoff rate changes.
*********************************************************************/

#include	"smtdefs.h"
#include	"smttypes.h"
#include	"smterror.h"
#include	"smtmacro.h"
#include	"fddihdr.h"
#include	"smtmsg.h"
#include	"cspmacro.h"
#include	"csphdr.h"
#include	"cspglbl.h"
#include	"mibdefs.h"


/*********************************************************************
	LEM External Functions
*********************************************************************/

extern	void	SendCSPEvent ();
extern	void	SMTSendSignal ();
extern	void	SetCSPTimer ();
extern	void	SetLEMCounter ();
extern	uInt16	ReadLEMCounter ();
extern	void	SetLEMState ();


/*********************************************************************
	LEM Defined Values
*********************************************************************/

/*
*	This value is the amount of time in a sample interval for LEM.
*	This is measured in seconds and microseconds.
*/
#define	LEM_INTERVAL		4
#define	LEM_INTERVAL_USECS	((uTime) 4000000)


/*********************************************************************
	LEM Global Variables
*********************************************************************/

/*
*	This structure is uses to associate error counts with
*	time values. The count member indicates the number of
*	errors counted by the LEM hardware. The time member is the
*	amount of time in seconds.
*/
struct	TimeCount {
	uInt16	count;
	uInt32	time;
};

/*
*	The errorRate array consists of TimeCount entries. Each entry
*	indicates the sample period and the threshhold error count
*	for cutoff at the given error rate. The first four entries
*	are not used because error rates of 10E0 to 10E-3 are not
*	measured.
*/
struct	TimeCount	errorRate[16] =
{
	{ 0, (uInt32) 0 },		/* skip first error rates 0 - 3 */
	{ 0, (uInt32) 0 },
	{ 0, (uInt32) 0 },
	{ 0, (uInt32) 0 },
	{ 50000, (uInt32) 4 },		/* 10E-4 */
	{ 5000, (uInt32) 4 },		/* 10E-5 */
	{ 500, (uInt32) 4 },		/* 10E-6 */
	{ 50, (uInt32) 4 },		/* 10E-7 */
	{ 50, (uInt32) 40 },		/* 10E-8 */
	{ 5, (uInt32) 40 },		/* 10E-9 */
	{ 5, (uInt32) 400 },		/* 10E-10 */
	{ 5, (uInt32) 4000 },		/* 10E-11 */
	{ 5, (uInt32) 40000 }, 		/* 10E-12 */
	{ 5, (uInt32) 400000 },		/* 10E-13 */
	{ 5, (uInt32) 4000000 },	/* 10E-14 */
	{ 5, (uInt32) 40000000 }	/* 10E-15 */
};

/*
*	The rateEst array accumulates error rate information for
*	determining the current error rate estimate.
*/
#define	MAX_EST		5
struct	TimeCount	rateEst[MAX_PORT_COUNT][MAX_EST];

/*
*	The logTime array determines the logarithm (base 10) for
*	log (125 * sampleTime). Since 125 is a constant, only sampleTime
*	is used to determine the log. To determine a log, compare
*	sampleTime to each entry (starting at 0) until the table entry
*	exceeds sampleTime. The index of this entry is the log.
*	Note: the first two logs (0 and 1) are not used.
*/
#define	MAX_LOG_TABLE	11
static	uInt32	logTime[MAX_LOG_TABLE] = {
		0, 0, 8, 80, 800, 8000,
		80000, 800000, 8000000, 80000000, 800000000
	};

/*
*	The logTable array determine straight base 10 logarithms using
*	the same method described above.
*/
static	uInt32	logTable[MAX_LOG_TABLE] = {
		10, 100, 1000, 10000, 100000, 1000000, 10000000,
		100000000, 1000000000, -1, -1
	};

/*********************************************************************
	Support Functions
*********************************************************************/

static	uInt16
log (n, table)
	uInt32	n;
	uInt32	table[];
/*********************************************************************
Function:	Estimate base 10 logarithm of a number.
Parameters:	n	= number to find log of.
		table	= base of log table to use.
Input:		None.
Output:		None.
Return:		Base 10 logarithm of number. This function only estimates
		the log value. The log returned is just the exponent.
		For example: log (5) = 0, log (12) = 1, log (3456) = 3.
*********************************************************************/
{
uInt16	l = 0;

	while ((l < MAX_LOG_TABLE) && (n >= table[l]))
		l++;

	return (l);
}


/*********************************************************************
	State Transition Functions
*********************************************************************/

static void
Off_Actions (phyID)
	uInt16	phyID;
/*********************************************************************
Function:	Process entry to LE0:OFF.
Parameters:	phyID	= index of PHY.
Input:		Uses lemData.
Output:		Changes lemData.
Return:		No value returned.
Notes:		None.
Modification History:
*********************************************************************/
{
uInt16	i;

	/* set current LEM state */
	lemData[phyID].leState = LE_OFF;

	/* disable LEM hardware */
	SetLEMState (phyID, DISABLE);

	/* clear timer */
	SetCSPTimer ((uInt32) 0, 0, 0, &lemData[phyID].TLE);

	/* clear estimator array */
	for (i = 0; i < MAX_EST; i++)
	{
		rateEst[phyID][i].count = 0;
		rateEst[phyID][i].time = (uInt32) 0;
	}
	lemData[phyID].rateIndex = 0;

	return;
}

static void
Start_Actions (phyID)
	uInt16	phyID;
/*********************************************************************
Function:	Process entry to LE1:MONITOR.
Parameters:	phyID	= index of PHY.
Input:		Uses lemData & errorRate array.
Output:		Changes lemData.
Return:		No value returned.
Notes:		None.
Modification History:
*********************************************************************/
{
	/* set current LEM state */
	lemData[phyID].leState = LE_MONITOR;

	/* set up LEM hardware */
	lemData[phyID].threshhold
		= errorRate[lemData[phyID].cutoffRate].count;
	SetLEMCounter (phyID, lemData[phyID].threshhold);
	SetLEMState (phyID, ENABLE);

	/* start interval timer in usecs */
	SetCSPTimer (LEM_INTERVAL_USECS, LEM_SIGNALS,
		phyID, &lemData[phyID].TLE);

	return;
}

static void
Monitor_Actions (phyID)
	uInt16	phyID;
/*********************************************************************
Function:	Process sample period expiration in LE1:MONITOR.
Parameters:	phyID	= index of PHY link.
Input:		Uses lemData.
Output:		Changes lemData.
Return:		No value returned.
Notes:		None.
Modification History:
*********************************************************************/
{
   uInt32 leCount;
   uInt32 totalTime;
   uInt32 totalError;
   uInt16 errLog;
   uInt16 timeLog;
   uInt16 i;
   uInt16 oldEst;
   struct TimeCount *windowPtr;

   leCount = ReadLEMCounter (phyID);
   leCount = lemData[phyID].threshhold - leCount;

   lemData[phyID].threshhold = errorRate[lemData[phyID].cutoffRate].count;
   SetLEMCounter (phyID, lemData[phyID].threshhold);
   SetCSPTimer (LEM_INTERVAL_USECS, LEM_SIGNALS, phyID, &lemData[phyID].TLE);

   totalError = 0;
   totalTime = 0;
   for (i=0; i < MAX_EST; i++)
   {
      totalError += rateEst[phyID][i].count;
      totalTime  += rateEst[phyID][i].time;
   }

   windowPtr = &rateEst[phyID][lemData[phyID].rateIndex];
   if ((windowPtr->count == 0) 
        && ((windowPtr->time >= 2000000)
             || ((leCount > 0) 
		 && ((totalTime > 512) || (windowPtr->time >= 64)))))
   {
      if (++lemData[phyID].rateIndex >= MAX_EST)
      {
	 lemData[phyID].rateIndex = 0;
      }
      windowPtr = &rateEst[phyID][lemData[phyID].rateIndex];
      windowPtr->time = 0;
      windowPtr->count = 0;
   }

   windowPtr->count += leCount;
   windowPtr->time += LEM_INTERVAL;
   totalError += leCount;
   totalTime += LEM_INTERVAL;

   oldEst = lemData[phyID].estimate;
   errLog = log ((uInt32) leCount, logTable);
   timeLog = log (totalTime, logTime);
   lemData[phyID].estimate = 6 + timeLog - errLog;

   if (totalError == 0)
   {
      lemData[phyID].estimate = 15;
   }

   if (leCount)
   {
      lemData[phyID].lemCount += leCount;
      SendCSPEvent (fddiPORTLem_Ct, phyID);
   }

   if (oldEst != lemData[phyID].estimate)
   {
      SendCSPEvent (fddiPORTLer_Estimate, phyID);
   }

   if (lemData[phyID].estimate <= lemData[phyID].cutoffRate)
   {
      lemData[phyID].LEM_Fail = SET;
      lemData[phyID].LEM_Reject_Ct++;
      SMTSendSignal (SIG_LEM_Stop, phyID, (uInt32) 0);
      SMTSendSignal (SIG_PC_LEM_Fail, phyID, (uInt32) 0);
      SendCSPEvent (fddiPORTLem_Reject_Ct, phyID);
   }
}

#if 0
{
uInt16	errCount;
uInt32	timeCount;
uInt16	errLog, timeLog;
uInt16	i;

	/*
	*	Accumulate estimator values.
	*
	*	Note: the count in the LEM counter is measured from
	*	the beginning of the sample period. Thus, at the end
	*	of a sub-sample period, this counter has the number
	*	of TOTAL errors so far in this sample. This means that
	*	errCount does not have to be added in to the count value
	*	for the sample; it is the count value.
	*/

	/* get count left till threshhold */
	errCount = ReadLEMCounter (phyID);

	/* get # of errs that occurred */
	errCount = lemData[phyID].threshhold - errCount;

	/* add the change to the aggregate count */
	lemData[phyID].lemCount +=
		(errCount - rateEst[phyID][lemData[phyID].rateIndex].count);

	/* set sample values so far */
	rateEst[phyID][lemData[phyID].rateIndex].count = errCount;
	rateEst[phyID][lemData[phyID].rateIndex].time += LEM_INTERVAL;

	/*
	*	Estimate error rate.
	*/
	errCount = 0;
	timeCount = 0;
	for (i = 0; i < MAX_EST; i++)
	{
		errCount += rateEst[phyID][i].count;
		timeCount += rateEst[phyID][i].time;
	}

	/*errLog = (errCount < 5) ? -1 : log ((uInt32) errCount, logTable);*/
	errLog = log ((uInt32) errCount, logTable);
	timeLog = log (timeCount, logTime);

        if (errCount > 1)          /* Advisory 6.2.3-016 */
	{
	   lemData[phyID].estimate = 6 + timeLog - errLog;
	}
        else
	{
           lemData[phyID].estimate = 15;
	}

	/*
	*	Report new error rate.
	*/
	SendCSPEvent (fddiPORTLem_Ct, phyID);
	SendCSPEvent (fddiPORTLer_Estimate, phyID);

	/*
	*	Restart monitoring.
	*
	*	If the time for this sub-sample period meets or
	*	exceeds the sample period, then the sample period is
	*	complete. Or, if the error threshold count has changed,
	*	then the cutoff rate has changed during the sub-sample
	*	period. In either case,  move to the next entry in the
	*	estimator table and restart the monitoring.
	*
	*	Else, just restart the sub-sample timer.
	*/
	/*
	*	910807-001	LJP
	*	Two changes: the first variable in the first half of the
	*	if() was rateEst[]...count. It is now rateEst...time.
	*	The second half is now != instead of ==.
	*/
	/*
	*	911111-002	LJP
	*	Three cases where index into estimate table is incremented:
	*	1)	lemData[].threshhold != errorRate[].count
	*	2)	rateEst[][].count > 0 &&
	*			timeCount > min table time
	*	3)	rateEst[][].count == 0 &&
	*			rateEst[][].time > max element time
	*	max element time is 1/4 time for 10E-15 (2,000,000 secs)
	*	min table time is 512 secs
	*/
	i = (lemData[phyID].threshhold !=
			errorRate[lemData[phyID].cutoffRate].count);
	if (!i)
	{
		if (rateEst[phyID][lemData[phyID].rateIndex].count)
			i = timeCount > 512;
		else
			i = (rateEst[phyID][lemData[phyID].rateIndex].time)
				> 2000000;
	}
	if (i)
	{
		/* Increment rateIndex with wrap around. */
		if (++lemData[phyID].rateIndex >= MAX_EST)
			lemData[phyID].rateIndex = 0;

		/*
		*	910807-001	LJP
		*	Clear array element in case we have wrapped
		*	around.
		*/
		rateEst[phyID][lemData[phyID].rateIndex].time = 0;
		rateEst[phyID][lemData[phyID].rateIndex].count = 0;

		/* restart monitoring */
		Start_Actions (phyID);
	}

	else
	{
		/* Restart interval. */
		SetCSPTimer (LEM_INTERVAL_USECS, LEM_SIGNALS,
			phyID, &lemData[phyID].TLE);
	}

	return;
}

#endif

static void
Cutoff_Actions (phyID)
	uInt16	phyID;
/*********************************************************************
Function:	Process threshhold exceeded.
Parameters:	phyID	= index of PHY to process.
Input:		Uses lemData.
Output:		Changes lemData.
Return:		No value returned.
Notes:		None.
Modification History:
*********************************************************************/
{

	/* Set LEM_Fail */
	lemData[phyID].LEM_Fail = SET;

	/* Increment reject count */
	lemData[phyID].LEM_Reject_Ct++;

	/* Send LEM_Stop */
	SMTSendSignal (SIG_LEM_Stop, phyID, (uInt32) 0);
		
	/* Send PC_LEM_Fail */
	SMTSendSignal (SIG_PC_LEM_Fail, phyID, (uInt32) 0);
		
	/* Report cutoff event */
	SendCSPEvent (fddiPORTLem_Reject_Ct, phyID);

	return;
}

static void
Start_LCT_Actions (phyID)
	uInt16	phyID;
/*********************************************************************
Function:	Process entry to LE2:LCT
Parameters:	phyID	= index of PHY to process.
Input:		Uses lemData, phyData, and stationData.
Output:		Changes lemData.
Return:		No value returned.
Notes:		None.
Modification History:
*********************************************************************/
{
	/* set current LEM state */
	lemData[phyID].leState = LE_LCT;

	/*
	*	Set LCT_Passed flag when starting LCT.
	*/
	phyData[phyID].LCT_Passed = SET;

	return;
}


/*********************************************************************
	LEM State Machine
*********************************************************************/

void
SignalLEM (sigType, sigEntity, sigData)
	uInt16	sigType;
	uInt16	sigEntity;
	uInt32	sigData;
/*********************************************************************
Function:	Process an LEM signal.
Parameters:	sigType		= the signal ID.
		sigEntity	= entity (PHY) to process.
		sigData		= any data associated with the signal.
Input:		Uses lemData.
Output:		Changes lemData.
Return:		No value returned.
Notes:		This state machine is NOT defined by the SMT committee.
		It is used by XDI to simplify SMT implementation.
Modification History:
*********************************************************************/
{
	/*
	*	If timer has expired, then verify this timer is the current
	*	event being timed.
	*/
	if (sigType == SIG_LE_Timer)
	{
		if (lemData[sigEntity].TLE != sigData)
			return;			/* not current timer */
		else
			lemData[sigEntity].TLE = 0;/* mark timer as expired */
	}
			
	/*
	*	Select LEM state for processing.
	*/
	switch (lemData[sigEntity].leState)
	{
	/*
	*	LE0:OFF
	*/
	case LE_OFF:
		/*
		*	Select action on signal.
		*/
		switch (sigType)
		{		
		/* LE(01) */
		case SIG_LEM_Start:
			Start_Actions (sigEntity);
			break;

		/* LE(02) */
		case SIG_LCT_Start:
			Start_LCT_Actions (sigEntity);
			break;
		}
		break;
		
	/*
	*	LE1:MONITOR
	*/
	case LE_MONITOR:
		/*
		*	Select action on signal.
		*/
		switch (sigType)
		{		
		/* Stop LEM */
		case SIG_LEM_Stop:
			Off_Actions (sigEntity);
			break;

		/* LE(10) */
		case SIG_LEM_Threshhold:
			Cutoff_Actions (sigEntity);
			break;

		/* LE(11) */
		case SIG_LE_Timer:
			Monitor_Actions (sigEntity);
			break;
		}
		break;
		
	/*
	*	LE2:LCT
	*/
	case LE_LCT:
		switch (sigType)
		{
		/* Stop LEM or Stop LCT LE(20a) */
		case SIG_LEM_Stop:
		case SIG_LCT_Stop:
			/* EC(22) */
			/*
			*	Indicate LCT passed. Move
			*	clear count to StopLCT().
			*/
			/* phyData[sigEntity].PC_LCT_Fail = 0; */
			phyData[sigEntity].LCT_Passed = SET;
			Off_Actions (sigEntity);
			break;
			
		/* LE(20b) */
		case SIG_LEM_Threshhold:
			/*
			*	Move count incrment to StopLCT().
			*	Clear passed condition.
			*/
			/* phyData[sigEntity].PC_LCT_Fail++; */
			phyData[sigEntity].LCT_Passed = CLEAR;
			Off_Actions (sigEntity);
			break;
		}
		break;
	}

	return;
}
