/**************************************************************************
 *  File    phy.c
 *
 *  Description
 *      This is the PLAYER device driver. The port numbering scheme is
 *      various for SMT, UI/SNMP, and hardware.
 *      from up to don -
 *          Hardware numbering: A,7,6,5,4,B
 *          SMT numbering:      0,2,3,4,5,1
 *          UI/SNMP numbering:  1,3,4,5,6,2
 *
 *      The Backplane ports are treated differently.
 *
 *  Copyright (c) 1992, 1993 Hughes Lan Systems
 *
 *  Author:     Jia-wei Jang
 *
 *  $Log:   /user/pvcs/fddicon/drv/src/phy.c_v  
 * 
 *    Rev 1.77   14 Oct 1993 15:33:10   gregs
 * Fixed the inialization of the offline B-Phy, and fixed the reconfiguration
 * of the A and B phys when switching the phys from front to back and visa versa
 * This should correct some of the BMAC beacon failures that have been occuring.
 * 
 *    Rev 1.76   12 Oct 1993 11:06:16   gregs
 * No change.
 * 
 *    Rev 1.75   02 Oct 1993 11:52:44   gregs
 * Fixed the Insert_X routine so that the bridge's B phy will be correctly 
 * configured in a Wrap_B configuration with A/B ports split between front and
 * back.
 * 
 *    Rev 1.74   29 Sep 1993 13:42:10   gregs
 * No change.
 * 
 *    Rev 1.73   13 Sep 1993 13:13:06   gregs
 * Modified ResetPlayerRegs so that the UserDefinedRegister will not get wiped out.
 * The way in which A_Port_Config_State and B_Port_Config_State variables
 * are initialized was changed so that they are now initialized in init.c
 * Initial SIT Release
 * 
 *    Rev 1.72   02 Aug 1993 15:32:38   gregs
 * Made some modifications to PHY_Reset, Insert_X, Br_Switch_A_Phys, Br_Switch_B_Phys
 * and Added three new routines to support the switching and connections of PHYS on
 * the backplane for the 836 bridge.
 * 
 *    Rev 1.71   13 Jul 1993 11:29:38   jlin
 * Allow the SMT to enable and disable SILS interrupt
 * 
 *    Rev 1.70   08 Jul 1993 15:21:14   jang
 * clean house to get rid of the compiler warning message
 * 
 *    Rev 1.69   08 Jul 1993 12:37:28   gregs
 * Fixed Insert_X routine for FEBRIDGE
 * 
 *    Rev 1.68   08 Jul 1993 10:04:02   gregs
 * Added ifdef __FEBRIDGE to GetPhyDefault().
 * 
 *    Rev 1.67   07 Jul 1993 17:15:08   gregs
 * Added some code to allow A/B ports to switch to the backplane on the FDDI
 * Bridge.
 * 
 *    Rev 1.66   25 Jun 1993 14:36:28   gregs
 * redefinition of INSERT_X for fddi-eth bridge.
 * 
 *    Rev 1.65   07 Jun 1993 17:19:22   jang
 * moved variables to sram
 * 
 *    Rev 1.64   10 May 1993 11:56:56   jang
 * clear CWI bit when polling the conditional write registers
 * 
 *    Rev 1.63   05 May 1993 16:46:20   jang
 * added limitation to the do-while loop
 * 
 *    Rev 1.63   05 May 1993 14:45:50   jang
 * added max count to WriteCtsr()
 * 
 *    Rev 1.62   29 Apr 1993 15:41:04   jang
 * called strobe_wdt() so wdt won't reset the system
 * 
 *    Rev 1.61   28 Apr 1993 10:29:28   gregs
 * Moved the 'setRR38' call to fix the 'control bus' test problem.
 * 
 *    Rev 1.60   27 Apr 1993 17:11:50   jang
 * commented CBusTest()
 * 
 *    Rev 1.59   24 Apr 1993 16:01:32   jang
 * added function to do chip self test
 * 
 *    Rev 1.58   22 Apr 1993 17:42:38   jang
 * added a new function PHY_SetRCMRx()
 * 
 *    Rev 1.57   17 Apr 1993 15:38:06   jang
 * added one var to indicate the current phy configuration
 * 
 *    Rev 1.57   16 Apr 1993 09:32:12   jang
 * set port_txflkcnt[] to 0 when port is isloated
 * 
 *    Rev 1.56   13 Apr 1993 13:36:28   jang
 * moved GetPortLockRec() out and moved PHY_PstatsMACPkts() in from monp.c
 * 
 *    Rev 1.55   10 Apr 1993 16:59:02   jang
 * added some parameters
 * 
 *    Rev 1.54   10 Apr 1993 16:18:08   jang
 * reenable port locking init
 * 
 *    Rev 1.53   09 Apr 1993 14:33:10   jang
 * load noise threshold register before enable interrupt
 * 
 *    Rev 1.51   05 Apr 1993 08:25:44   jang
 * took out PLAYER default table phy_master_default[]
 * 
 *    Rev 1.50   31 Mar 1993 11:26:18   jang
 * added code to call DRV_INitFddiCon() to get driver related MIBs
 * 
 *    Rev 1.49   31 Mar 1993 11:09:20   gregs
 * ifdefd the PHY_MASTER_PORTS chk in PHY_Wrap for the bridge
 * 
 *    Rev 1.48   29 Mar 1993 14:35:06   jang
 * modified RR38 so that only PLAYER+ rev. A will set the register
 * 
 *    Rev 1.47   24 Mar 1993 18:42:54   jlin
 * 
 *    Rev 1.46   23 Mar 1993 10:43:36   jang
 * took out port locking and changed #ifdef FDDI_BRIDGE to
 * #ifdef __FEBRIDGE
 * 
 *    Rev 1.45   19 Mar 1993 14:14:16   jang
 * added ifdef FDDI_BRIDGE to phy_test()
 * 
 *    Rev 1.44   19 Mar 1993 10:30:16   jang
 * added bridge init table
 * 
 *    Rev 1.43   18 Mar 1993 14:53:42   jang
 * changed back to write 1 to rr38 instead of 0xff
 * 
 *    Rev 1.42   18 Mar 1993 09:07:34   jang
 * deketed duplicate DRV_ResetLANChips()
 * 
 *    Rev 1.41   18 Mar 1993 08:00:50   jang
 * added PLAYER and PLAYER+ together until
 * rev 2 fully functioning
 * 
 *    Rev 1.39   09 Mar 1993 17:27:58   jang
 * added player+
 * 
 *    Rev 1.38   05 Mar 1993 10:07:42   jang
 * added one parameter to DRV_Init()
 * 
 *    Rev 1.37   16 Feb 1993 11:06:02   jang
 * added code to load port lock list from nvram
 * 
 *    Rev 1.36   12 Feb 1993 10:30:02   jlin
 * 
 *    Rev 1.35   11 Feb 1993 13:29:46   jang
 * deleted my_ports[]
 * 
 *    Rev 1.34   09 Feb 1993 16:08:56   jang
 * changed *my_ports to my_ports[]. Should change it back later
 * 
 *    Rev 1.33   09 Feb 1993 10:00:54   jang
 * added function to return elasity buffer error counts
 * 
 *    Rev 1.32   19 Jan 1993 15:05:58   jang
 * 
 *    Rev 1.31   12 Jan 1993 09:33:58   jang
 * added code to retrieve the port type from map module and added a
 * function PHY_ChangePortType()
 * 
 *    Rev 1.30   11 Jan 1993 09:57:28   shekhar
 * fddi_pstats_tbl is split into fddi_if_pstats_tbl and fddi_port_pstats_tbl.
 * 
 *    Rev 1.29   07 Jan 1993 18:56:52   jang
 * modified to distinguish the backplane ports and the normal ports
 ****************************************************************************/

#include <fddi.h>
#ifdef __FEBRIDGE
#include <krnl.h>
#endif
#include <drv.h>
#include <bsi.h>
#include <bmac.h>
#include <phy.h>
#include <monp.h>
#include <dips.h>
#include <nvrfddi.h>


#define PHY_DEBUG         0


/*
 *-------------------------------------------------------------------------
 *      exported variables
 *-------------------------------------------------------------------------
 */
/* the index of port_list is the port number based on the
   lable on the master fddi concentrator board and mapping to the SMT
   port numbering order */
#ifdef __FEBRIDGE
int port_list[] = {0, 1, 2, 3};
#else
int port_list[] = {
    PORT_A,
    PORT_MASTER_BA,
	PORT_M2, PORT_M3, PORT_M4, PORT_M5, 
    PORT_MASTER_BB, 
    PORT_B,
};
#endif





/*
 *-------------------------------------------------------------------------
 *      Imported variables
 *-------------------------------------------------------------------------
 */

extern PLAYER_TYPE phys[];
extern PHY_INIT_TYPE phy_br_default[];
extern PHY_INIT_TYPE phy_br_default_AB2BP[];
extern PHY_INIT_TYPE phy_br_default_A2BP[];
extern PHY_INIT_TYPE phy_br_default_B2BP[];
extern PHY_INIT_TYPE phy_masplus_default[];
extern uint16 previousLS[];
extern FDDI_PSTATS fddi_if_pstats_tbl[];
extern FDDI_PSTATS fddi_port_pstats_tbl[];
extern PHY_DEFAULT_TABLE phycon_deftbl, phybr_deftbl, phycon_bkp_deftbl;
extern byte port_tx_flkcnt[];
extern bool phy_init_flag;
extern uChar SILSmask;     /* jlin: 07/13/93 */

 /*
 *-------------------------------------------------------------------------
 *      Local variables
 *-------------------------------------------------------------------------
 */
static byte ntr_default, nptr_default;

/* These variables are for FDDI Bridge use only */
#ifdef __FEBRIDGE
int A_Port_Config_State ;
int B_Port_Config_State ;
#endif
/***********************************************/

/*
 *-------------------------------------------------------------------------
 *      function prototypes
 *-------------------------------------------------------------------------
 */
#ifdef DO_LINT
static byte ReadPhyRev (PHY_REG_TYPE *);
static void ResetPlayerRegs(uint32);
static void WriteCtsr (PHY_REG_TYPE *,byte);
static status_type GetPhyDefault(uint32,uint32);
#else
static byte ReadPhyRev ();
static void ResetPlayerRegs();
static void WriteCtsr ();
static status_type GetPhyDefault();
static int CBusTest();

#ifdef DRV_DEBUG
static int test_phy_regs();

#endif

#endif


/***************************************************************************
 *  Function PHY_Init
 *
 *  Description
 *      This function initializes all the phys in the board.
 *
 *  Parameters:
 *      uint32 num_phys - number of phys
 *      uint32 type - [FDDICON_MASTER | FDDICON_SLAVE]
 *
 *  Return: [OK | FDDI_ERROR]
 **************************************************************************/
status_type PHY_Init (num_phys,type)
uint32 num_phys;
uint32 type;
{
  PHY_CTRL_TYPE *ptr;
  int i;
/*  PORT_INFO *port_ptr;  */
  TLVParamType mibData;
/*  extern unsigned long ram_control_table[]; */
  extern void FDDIChipIsr();

  if (phy_init_flag)
    return (OK);

  for (i = 0; i < num_phys; i++)
    previousLS[i] = 0;

  for (i=0; i < num_phys; i++) {
    strobe_wdt();
    if (GetPhyDefault(i,type) != OK) {
#if PHY_DEBUG
      printf("ERROR: getting phy default\n");
#endif
      return FDDI_ERROR;
    }
    if (SelfTest(i) != OK) {
      return FDDI_ERROR;
    }
    if (i < (num_phys - PHY_BACKPLANE_PORTS)) {
      mibData.paramType = fddiPORTPC_Type;
      mibData.paramLen = 4;
      mibData.PORTINDEX = i + 1;
      MAPGetMIBAttr(sizeof(mibData),(uChar *)&mibData,NULL);
      phys[i].p_type = mibData.PORTPARAM8;
    }
    else
      phys[i].p_type = PC_TYPE_NONE;   /* backplane ports */
    
    if ((ptr=(PHY_CTRL_TYPE*)lmalloc(sizeof(PHY_CTRL_TYPE))) == NULL) {
      return (FDDI_ERROR);
    }
    phys[i].image_ptr = ptr;
    /* read revision number */
/*    phys[i].rev = (phys[i].base_addr)->idr;  */
    phys[i].lem_count = phys[i].state_count = phys[i].noise_count = 0;
    phys[i].LerCutoff = phys[i].BufErrCount = 0;
    ResetPlayerRegs(i);
#ifdef __FEBRIDGE
	ResetOfflineBPlayerRegs();
#endif
    PHY_Run(i);
  }       /* for */

  phy_init_flag = TRUE;
  return (OK);
}

/***************************************************************************
 *  Function PHY_Reset
 *
 *  Description
 *      This function resets all the phys to the default value.
 *
 *  Parameters:
 *      uint32 num_phys - number of phys
 *      uint32 type - [FDDICON_MASTER | FDDICON_SLAVE]
 *
 *  Return: [OK | FDDI_ERROR]
 **************************************************************************/
status_type PHY_Reset (num_phys,type)
uint32 num_phys;
uint32 type;
{
  uint32 i;
  PHY_REG_TYPE *r_ptr;
    
#ifdef __FEBRIDGE
  for (i=0; i < num_phys; i++) {
    GetPhyDefault(i,type);
    r_ptr = phys[i].base_addr;
    r_ptr->mode2 = PHY_MODE2_PHYRST;
    phys[i].lem_count = phys[i].state_count = phys[i].noise_count = 0;
    phys[i].LerCutoff = phys[i].BufErrCount = 0;
    ResetPlayerRegs(i);
	if( i < 2 )		/* only activate the first 2 phys */
    	PHY_Run(i);
  }
#else
  for (i=0; i < num_phys; i++) {
    r_ptr = phys[i].base_addr;
    r_ptr->mode2 = PHY_MODE2_PHYRST;
    GetPhyDefault(i,type);
    phys[i].lem_count = phys[i].state_count = phys[i].noise_count = 0;
    phys[i].LerCutoff = phys[i].BufErrCount = 0;
    ResetPlayerRegs(i);
    PHY_Run(i);
  }
#endif
  return (OK);
}

/***************************************************************************
 *  Function PHY_ReadUDR
 *
 *  Description
 *    This function reads UDR to find out whether or not the system
 *    card is presented.
 *
 *  Parameter:  none
 *
 *  Return: TRUE | FALSE
 **************************************************************************/
#define PHY_SYSCARD_NUM   7
bool PHY_ReadUDR ()
{
  volatile PHY_REG_TYPE *r_ptr =  phys[PHY_SYSCARD_NUM].base_addr;
    
  r_ptr->udr = 0;
  if (r_ptr->udr == 0x01)   /* no system card */
    return FALSE;
  return TRUE;
}

/***************************************************************************
 *  Function PHY_ClearPhys
 *
 *  Description
 *     This function set all the PLAYERs to stop mode, and releases all the
 *     memory allocated for PHYs.
 *
 *  Parameters:
 *      uint32 num_phys - number of phys
 *      uint32 type - [FDDICON_MASTER | FDDICON_SLAVE | FDDICON_BRIDGE]
 *
 *  Return: [OK | FDDI_ERROR]
***************************************************************************/
status_type PHY_ClearPhys (num_phys,type)
uint32 type;
uint32 num_phys;
{
  uint32 i;
  PHY_REG_TYPE *r_ptr;
    
  for (i=0; i < num_phys; i++) {
    r_ptr = phys[i].base_addr;
    PHY_Stop(i);
    r_ptr->icmr = 0;
    PHY_ClearIcr(r_ptr);
/*    free(phys[i].image_ptr);  Do we have free()? */
  }

  return OK;
}

/***************************************************************************
 *  Function   PHY_ChangePortType
 *
 *  Description
 *     This function changes the port type.
 *
 *  Parameter:
 *   uint32 port_num
 *   uint 32 type - [PC_TYPE_A | PC_TYPE_B | PC_TYPE_S | PC_TYPE_M]
 *
 *  Return:  void
 **************************************************************************/
void PHY_ChangePortType (phy_num,type)
uint32 phy_num;
uint32 type;
{
  if (phy_num < PHY_BACKPLANE_PORTS)
    phys[phy_num].p_type = (short)type;
}

/*************************************************************************
 *  Function  PHY_PstatsMACPkts
 *
 *  Description
 *    THis function records the Promiscuous Statistics for mac 0 and
 *    mac 1.
 *
 *  Parameter:
 *    int mac_num
 *
 *  Return: void
 ************************************************************************/
void PHY_PstatsMACPkts (mac_num)
uint32 mac_num;
{
  FDDI_PSTATS *pstats_ptr;

  if (phys[mac_num].p_type == PC_TYPE_M)
    pstats_ptr = &fddi_port_pstats_tbl[mac_num];
  else
    pstats_ptr = &fddi_if_pstats_tbl[mac_num];
  pstats_ptr->PStatsMacPkts++;
  pstats_ptr->PStatsMacOctets += 4;
} 

/***************************************************************************
 *  Function  PHY_SetRR38
 *
 *  Description
 *    This function writes 0x01 to reg 38 for the bug in player+. This bug 
 *    only exist for PLAYER+ rev A. which IDR is 0x10. The bug is related
 *    to phase locked loop operation.
 *
 *  Parameter:
 *
 *  Return: 
 **************************************************************************/
int PHY_SetRR38 (r_ptr)
volatile PHY_REG_TYPE *r_ptr;
{
   int i = 0;

   if (r_ptr->idr != 0x10)
     return OK;

   i = RealTimeTicks() + 1;  /* this is necessary wait period */
   while (RealTimeTicks() < i);
   r_ptr->rr38 = 0x01;
   return OK;
}

/***************************************************************************
 *  Function    PHY_GetPortType
 *
 *  Description
 *    This function returns the specified port type
 *
 *  Parameter:
 *    uint32 phy_num
 *
 *  Return: uint -[PC_TYPE_M | PC_TYPE_S | PC_TYPE_A | PC_TYPE_B
 *                 PC_TYPE_UNKNOWN]
 ***************************************************************************/
int PHY_GetPortType (phy_num)
uint32 phy_num;
{
  if (phy_num >= PHY_MAX_PORTS)
    return (PC_TYPE_UNKNOWN);

  return ((int)phys[phy_num].p_type);
}

/***************************************************************************
 *  Function    PHY_Run
 *
 *  Description
 *      This function sets the phy to RUN mode
 *
 *  Parameter:
 *      PLAYER_TYPE phy_num
 *
 *  Return: void
 **************************************************************************/
void PHY_Run (phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *ptr;
    
    ptr = phys[phy_num].base_addr;
    ptr->mr |= PHY_MR_RUN;
    if (phys[phy_num].id == PHY_ID_PLAYERPLUS) 
      PHY_SetRR38(ptr);
    phys[phy_num].status = DRV_MODE_RUN;
}

/***************************************************************************
 *  Function    PHY_Stop
 *
 *  Description
 *      This function sets the phy to Stop mode
 *
 *  Parameter:
 *      PLAYER_TYPE phy_num
 *
 *  Return: void
 **************************************************************************/
void PHY_Stop (phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *ptr;

    if (phys[phy_num].status == DRV_MODE_STOP)
        return;
    ptr = phys[phy_num].base_addr;
    ptr->mr &= ~PHY_MR_RUN;

    if (phys[phy_num].id == PHY_ID_PLAYERPLUS) 
      PHY_SetRR38(ptr);
    phys[phy_num].status = DRV_MODE_STOP;
}

/***************************************************************************
 *  Function    PHY_SetRCMRx
 *
 *  Description
 *    This function sets the RCMRA and/or RCMRB
 *
 *  Parameter:
 *    uint32 phy_num
 *    uint32 act
 *
 *  Return: void
 **************************************************************************/
void PHY_SetRCMRx (phy_num,act)
uint32 phy_num,act;
{
  volatile PHY_REG_TYPE *r_ptr = phys[phy_num].base_addr;
#if 0
  printf("PHY_SetRCMRx(), port %d, line_st %x\n",phy_num,act);
#endif  
  switch (act) {
  case SILS:
    r_ptr->rcmrb |= PHY_RCMRB_SILSM;
    break;

  case MLS:
    r_ptr->rcmra |= PHY_RCMRA_MLSM;
    break;

  case HLS:
  case QLS:
  default:

    break;
  }
}

/***************************************************************************
 *  Function    PHY_SetLerCutoff
 *
 *  Description
 *      This function sets the LEM threshold. THis function is not for
 *      backplane ports.
 *
 *  Parameter
 *      uint32 sphy_num
 *
 *  Return: void
 **************************************************************************/
void PHY_SetLerCutoff (phy_num,val)
uint32 phy_num;
uint32 val;
{
    volatile PHY_REG_TYPE *r_ptr;
    PHY_CTRL_TYPE *i_ptr;
    r_ptr = phys[phy_num].base_addr;
    i_ptr = phys[phy_num].image_ptr;
    /* threshold reg has only 8 bits */
    if (val > 0xff) {
        i_ptr->letr = 0xff;
        r_ptr->letr = 0xff;
        val -= 0xff;
    }
    else {
        i_ptr->letr = val;
        r_ptr->letr = val;
        val = 0;
    }
    phys[phy_num].LerCutoff = val;
}


/***************************************************************************
 *  Function    PHY_GetLerCutoff
 *
 *  Description
 *      This function gets the remaining LEM cutoff. THis function is not for
 *      backplane ports.
 *
 *  Parameter
 *      uint32 sphy_num
 *
 *  Return: void
 **************************************************************************/
void PHY_GetLerCutoff (phy_num,val)
uint32 phy_num;
uint32 *val;
{
    volatile PHY_REG_TYPE *r_ptr;
    uint32 num;

    r_ptr = phys[phy_num].base_addr;
    num = r_ptr->clecr;
    *val = phys[phy_num].LerCutoff + num;
}
 

/***************************************************************************
 * Function   PHY_GetElBufErr
 *
 * Description
 *   This function gets the elasticity buffer overflow/underflow
 *   error counts.
 *
 * Parameter:
 *   uint32 phy_num
 *
 * Return: # of error counts
 ***************************************************************************/
uint32 PHY_GetElBufErr (phy_num)
uint32 phy_num;
{
  return (phys[phy_num].BufErrCount);
}

/***************************************************************************
 *  Function    PHY_RestoreCR
 *
 *  Description
 *     This function restores the PHY configuration switch by writing back
 *     the image pointer.
 *
 *  Parameter: 
 *     uint32 phy_num
 *
 *  Return: void
 **************************************************************************/
void PHY_RestoreCR (phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *r_ptr;
    PHY_CTRL_TYPE *i_ptr;

    r_ptr = phys[phy_num].base_addr;
    i_ptr = phys[phy_num].image_ptr;
    r_ptr->cr = i_ptr->cr;
}

/***************************************************************************
 *  Function    PHY_Insert_S
 *
 *  Description
 *      THis function sets up the configuration register according to
 *      the following:
 *          connect a.indicate to b.req bus;
 *          connect b.indicate to rcv bus
 *          connect tr to a.req bus
 *
 *  Parameter:
 *      PHY_REG_TYPE *ptr - to control register base address
 *
 *  Return: void
 ****************************************************************************/
void PHY_Insert_S (phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *ptr;
    PHY_CTRL_TYPE *i_ptr;

    ptr = phys[phy_num].base_addr;
    i_ptr = phys[phy_num].image_ptr;
    i_ptr->cr = PHY_CR_AIND_BREQ | PHY_CR_BIND_RBUS | PHY_CR_TRS_AREQ |
                PHY_CR_AIE | PHY_CR_BIE;
    ptr->cr = i_ptr->cr;
    phys[phy_num].config_state = PHY_INSERT_S;
}


/***************************************************************************
 *  Function    PHY_Insert_P
 *
 *  Description
 *      THis function sets up the configuration register according to
 *      the following:
 *          connect a.indicate to rcv bus;
 *          connect b.indicate to a.req bus
 *          connect tr to b.req bus
 *
 *  Parameter:
 *      PHY_REG_TYPE *ptr - to control register base address
 *
 *  Return: void
 ****************************************************************************/
void PHY_Insert_P(phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *ptr;
    PHY_CTRL_TYPE *i_ptr;

    i_ptr = phys[phy_num].image_ptr;
    ptr = phys[phy_num].base_addr;
    i_ptr->cr = PHY_CR_AIND_RBUS | PHY_CR_BIND_AREQ | PHY_CR_TRS_BREQ |
                PHY_CR_AIE | PHY_CR_BIE;
    ptr->cr = i_ptr->cr;
    phys[phy_num].config_state = PHY_INSERT_P;
}

/***************************************************************************
 *  Function    PHY_Insert_X
 *
 *  Description
 *      THis function sets up the configuration register according to
 *      the following:
 *          connect a.indicate to a.request
 *          connect b.indicate to rcv bus
 *          connect tr to b.req bus
 *
 *  Parameter:
 *      PHY_REG_TYPE *ptr - to control register base address
 *
 *  Return: void
 ****************************************************************************/
void PHY_Insert_X(phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *ptr;
    PHY_CTRL_TYPE *i_ptr;

    i_ptr = phys[phy_num].image_ptr;
    ptr = phys[phy_num].base_addr;

#ifdef __FEBRIDGE
	/* re-definition of INSERT_X for the fddi-eth bridge */
 	/* A.IND to A.REQ	B.IND to RCVR	TR to B.REQ  */
	if( (A_Port_Config_State != B_Port_Config_State ) &&  /*** FSFIX ****/
		(GetFddiCFState() == CF_THRU)  &&
		(phy_num == 1)
	  ) { 
		i_ptr->cr = PHY_CR_AIND_AREQ | PHY_CR_BIND_RBUS | PHY_CR_TRS_BREQ |
					PHY_CR_AIE | PHY_CR_BIE;
	}
	else
	if( (A_Port_Config_State != B_Port_Config_State)   &&  /**** FSFIX ****/
		(GetFddiCFState() == CF_WRAP_B)  && 
		(phy_num == 1)
	  ) { 
		/**** Put the online phy in Insert_X configuration ****/
		i_ptr->cr = PHY_CR_AIND_AREQ | PHY_CR_BIND_RBUS | PHY_CR_TRS_BREQ |
					PHY_CR_AIE | PHY_CR_BIE;
	}
	else { /* A.IND to RCVR     B.IND to B.REQ     TR to A.REQ */
    	i_ptr->cr = PHY_CR_AIND_RBUS | PHY_CR_BIND_BREQ | PHY_CR_TRS_AREQ |
                	PHY_CR_AIE | PHY_CR_BIE;
	}
#else
    i_ptr->cr = PHY_CR_AIND_AREQ | PHY_CR_BIND_RBUS | PHY_CR_TRS_BREQ |
                PHY_CR_AIE | PHY_CR_BIE;
#endif
    ptr->cr = i_ptr->cr;
    phys[phy_num].config_state = PHY_INSERT_X;
}

/***************************************************************************
 *  Function    PHY_Isolate
 *
 *  Description
 *      THis function sets up the configuration register according to
 *      the following:
 *          connect a.indicate to b.req bus
 *          connect b.indicate to a.req bus
 *          connect tr to rcv bus
 *      This function is also called if the path is PATH_L.
 *
 *  Parameter:
 *      PHY_REG_TYPE *ptr - to control register base address
 *
 *  Return: void
 ****************************************************************************/
void PHY_Isolate (phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *ptr;
    PHY_CTRL_TYPE *i_ptr;

    port_tx_flkcnt[phy_num] = 0; 
    i_ptr = phys[phy_num].image_ptr;
    ptr = phys[phy_num].base_addr;
    i_ptr->cr = PHY_CR_AIND_BREQ | PHY_CR_BIND_AREQ | PHY_CR_TRS_RBUS |
                PHY_CR_AIE | PHY_CR_BIE;
    ptr->cr = i_ptr->cr;
    phys[phy_num].config_state = PHY_ISOLATE;
#ifdef __FEBRIDGE
	if( phy_num == 0 ) {
    	if( phys[1].config_state == PHY_INSERT_X) {
			PHY_Insert_X(1);
		}
	}
#endif

}

/***************************************************************************
 *  Function    PHY_Wrap
 *
 *  Description
 *      THis function sets up the configuration register according to
 *      the following:
 *          connect a.indicate to a.req bus
 *          connect b.indicate to b.req bus
 *          connect tr to rcv bus
 *      Note that the backplane ports should never be wrapped.
 *
 *  Parameter:
 *      PHY_REG_TYPE *ptr - to control register base address
 *
 *  Return: void
 ****************************************************************************/
void PHY_Wrap (phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *ptr;

#ifndef __FEBRIDGE
    if (phy_num >= PHY_MASTER_PORTS)
      return;
#endif

    ptr = phys[phy_num].base_addr;
    ptr->cr = PHY_CR_AIND_AREQ | PHY_CR_BIND_BREQ | PHY_CR_TRS_RBUS |
                PHY_CR_AIE | PHY_CR_BIE;
    phys[phy_num].config_state = PHY_WRAP;
}


/***************************************************************************
 *  Function    PHY_TxSymbols
 *
 *  Description
 *      This function indicates the PLAYER to transmit the request
 *      symbols.
 *
 *  Parameters:
 *      uint32 phy_num;
 *      uint32 line_st - [DRV_ACTIVE_LS | DRV_IDLE_LS | DRV_MASTER_LS
 *                  DRV_HALT_LS | DRV_QUIET_LS]
 *
 *  Return: [OK | FDDI_ERROR]
 ****************************************************************************/
status_type PHY_TxSymbols (phy_num,line_st)
uint32 phy_num;
uint32 line_st;
{
    byte symbol;

    switch (line_st) {
        case DRV_ACTIVE_LS:
            symbol = PHY_CTSR_ATM;
            break;
        case DRV_HALT_LS:
            symbol = PHY_CTSR_HTM;
            break;
        case DRV_QUIET_LS:
#if PHY_DEBUG
printf(".....port %d TX QUIET. LER %d, noise cnt %d\n",phy_num,phys[phy_num].lem_count,phys[phy_num].noise_count);
#endif
            symbol = PHY_CTSR_QTM;
            break;
        case DRV_IDLE_LS:
            symbol = PHY_CTSR_ITM;
            break;
        case DRV_MASTER_LS:
            symbol = PHY_CTSR_MTM;
            break;
	  default:
	    return FDDI_ERROR;
    }
    WriteCtsr(phys[phy_num].base_addr,symbol);
    return (OK);
}

/***************************************************************************
 *  Function    PHY_GetTxLs
 *
 *  Description
 *      This function reads CTSR to get Tx Line State
 *      symbols.
 *
 *  Parameters:
 *      uint32 phy_num;
 *
 *  Return: [DRV_ACTIVE_LS | DRV_IDLE_LS | DRV_MASTER_LS
 *                  DRV_HALT_LS | DRV_QUIET_LS]
 ****************************************************************************/
uint32 PHY_GetTxLs (phy_num)
uint32 phy_num;
{
    PHY_REG_TYPE *r_ptr;
    uint32 symbol;

    r_ptr = phys[phy_num].base_addr;
    symbol = r_ptr->ctsr & ~PHY_CTSR_TM_MASK;
    switch (symbol) {
        case PHY_CTSR_ATM:
            symbol = DRV_ACTIVE_LS;
            break;
        case PHY_CTSR_HTM:
            symbol = DRV_HALT_LS;
            break;
        case PHY_CTSR_QTM:
            symbol = DRV_QUIET_LS;
            break;
        case PHY_CTSR_ITM:
            symbol = DRV_IDLE_LS;
            break;
        case PHY_CTSR_MTM:
            symbol = DRV_MASTER_LS;
            break;
    }

    return (symbol);
}

/***************************************************************************
 *  Function    PHY_GetRxLs
 *
 *  Description
 *      This function reads CTSR to get Rx Line State
 *      symbols.
 *
 *  Parameters:
 *      uint32 phy_num;
 *
 *  Return: [DRV_ACTIVE_LS | DRV_IDLE_LS | DRV_MASTER_LS
 *                  DRV_HALT_LS | DRV_QUIET_LS]
 ****************************************************************************/
uint32 PHY_GetRxLs (phy_num)
uint32 phy_num;
{
    PHY_REG_TYPE *r_ptr;
    uint32 symbol;

    r_ptr = phys[phy_num].base_addr;
    symbol = r_ptr->crsr & PHY_CRSR_LS_MASK;
    switch (symbol) {
        case PHY_CRSR_ILS:
            symbol = previousLS[phy_num] | SILS;
            break;
            
        case PHY_CRSR_MLS:
            symbol = previousLS[phy_num] | MLS;
            break;
            
        case PHY_CRSR_HLS:
            symbol = previousLS[phy_num] | HLS;
            break;
            
        case PHY_CRSR_NSD:
        case PHY_CRSR_QLS:
            symbol = previousLS[phy_num] | QLS;
            break;

        default:
            symbol = previousLS[phy_num];
            break;
    }       /* switch */

    return (symbol);
}

/******************************************************************************
 *  Function PHY_ConfigEBOU
 *
 *  Description
 *      This function enable/disable the elasticity buffer overflow/underflow
 *      interrupt.
 *
 *  Parameters:
 *      uint32 phy_num
 *      uint32 act - [DRV_ENABLE | DRV_DISABLE]
 *
 *  Return:  void
 *****************************************************************************/
void PHY_ConfigEBOU (phy_num,act)
uint32 phy_num;
uint32 act;
{
    volatile PHY_REG_TYPE *r_ptr;

    r_ptr = phys[phy_num].base_addr;
    if (act == DRV_ENABLE)
        r_ptr->rcmrb |= PHY_RCMRB_EBOUM;
    else
        r_ptr->rcmrb &= (~PHY_RCMRB_EBOUM);
}

/***************************************************************************
 *  Function    StartLCTest
 *
 *  Description
 *      This function starts the Link Confidence Test. It first store the
 *      value of CTSR so that this value can be written back when stops
 *      LCT.
 *
 *  Parameter
 *      uint32 phy_num
 *
 *  Return: [OK | FDDI_ERROR]
 ***************************************************************************/
void PHY_StartLCTest (phy_num)
uint32 phy_num;
{
    PHY_REG_TYPE *r_ptr;
    PHY_CTRL_TYPE *i_ptr;

    if (phy_num >= PHY_MASTER_PORTS)  /* no LCT for backplane ports */
      return;

    r_ptr = phys[phy_num].base_addr;
    i_ptr = phys[phy_num].image_ptr;

    i_ptr->ctsr = r_ptr->ctsr & ~PHY_CTSR_TM_MASK;
    PHY_TxSymbols(phy_num,DRV_IDLE_LS);     /* transmitting idle symbols */
}

/***************************************************************************
 *  Function    StopLCTest
 *
 *  Description
 *      This function stops the Link Confidence Test and restores CTSR to
 *      its original transmit state.
 *
 *  Parameter
 *      uint32 phy_num
 *
 *  Return: [OK | ERROR]
 ***************************************************************************/
void PHY_StopLCTest (phy_num)
uint32 phy_num;
{
    PHY_REG_TYPE *r_ptr;
    PHY_CTRL_TYPE *i_ptr;
    byte ctsr;
    uint32 line_st;
    
    if (phy_num >= PHY_MASTER_PORTS)  /* no LCT for backplane ports */
      return;

    r_ptr = phys[phy_num].base_addr;
    i_ptr = phys[phy_num].image_ptr;

    ctsr = i_ptr->ctsr;
    switch (ctsr) {
        case PHY_CTSR_ATM:
            line_st = DRV_ACTIVE_LS;
            break;
        case PHY_CTSR_ITM:
            line_st = DRV_IDLE_LS;
            break;
        case PHY_CTSR_MTM:
            line_st = DRV_MASTER_LS;
            break;
        case PHY_CTSR_HTM:
            line_st = DRV_HALT_LS;
            break;
        case PHY_CTSR_QTM:
            line_st = DRV_QUIET_LS;
            break;
        case PHY_CTSR_OTM:  /* Don't touch this one. Chip has a bug */
        default:
            line_st = DRV_IDLE_LS;
            break;
    }       /* SWITCH */
    PHY_TxSymbols(phy_num,line_st);
}
 
/***************************************************************************
 *  Function    PHY_ClearInt
 *
 *  Description
 *      This function clears PLAYER ICMR. The phy icmr image will not be
 *      changed by this function. When PHY_RestoreInt() is called, the
 *      image will be reloaded to the ICMR.
 *
 *  Parameter:
 *      uint32 phy_num
 *
 *  Return: void
 **************************************************************************/
void PHY_ClearInt (phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *r_ptr;
    
    r_ptr = phys[phy_num].base_addr;
    r_ptr->icmr = 0;
}

/***************************************************************************
 *  Function    PHY_RestoreInt
 *
 *  Description
 *      This function restores the phy ICMR.
 *
 *  Parameter:
 *      uint32 phy_num
 *
 *  Return: void
 **************************************************************************/
void PHY_RestoreInt (phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *r_ptr;
    PHY_CTRL_TYPE *i_ptr;
    
    r_ptr = phys[phy_num].base_addr;
    i_ptr = phys[phy_num].image_ptr;
    r_ptr->icmr = i_ptr->icmr;
}

/***************************************************************************
 *  Function    PHY_ConfigLEM 
 *
 *  Description
 *      This function enable/disable Link Error Monitoring
 *
 *  Parameter:
 *      uint32 phy_num
 *      uint32 act - [DRV_ENABLE | DRV_DISABLE]
 *
 *  Return: void
 **************************************************************************/
void PHY_ConfigLEM (phy_num,act)
uint32 phy_num;
uint32 act;
{
    volatile PHY_REG_TYPE *r_ptr;
    PHY_CTRL_TYPE *i_ptr;
    
    r_ptr = phys[phy_num].base_addr;
    i_ptr = phys[phy_num].image_ptr;
    if (act == DRV_ENABLE) {
        i_ptr->icmr |= PHY_ICMR_LEMTM;
        r_ptr->icmr = i_ptr->icmr;
	phys[phy_num].lem_int = TRUE; /* so we'll chk it in ISR */
    }
    else {
        i_ptr->icmr &= ~PHY_ICMR_LEMTM;
        r_ptr->icmr = i_ptr->icmr;
	phys[phy_num].lem_int = FALSE;
    }
}
  
/***************************************************************************
 *  Function    PHY_SetExterLoopback
 *
 *  Description
 *      This function sets the phy in external loopback mode
 *
 *  Parameter:
 *      PHY_REG_TYPE *r_ptr
 *
 *  Return: void
 **************************************************************************/
void PHY_SetExterLoopback (phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *r_ptr;
    byte reg;

    r_ptr = phys[phy_num].base_addr;
    reg = r_ptr->mr;
    if (reg & PHY_MR_RUN) {
#if 0
      if (phys[phy_num].id == PHY_ID_PLAYERPLUS) 
#endif

      r_ptr->mr = reg & ~PHY_MR_RUN;      /* stop the phy first */
	PHY_SetRR38(r_ptr);
  }
    r_ptr->mr = reg | PHY_MR_EXLB;
}

/***************************************************************************
 *  Function    PHY_SetInterLoopback
 *
 *  Description
 *      This function sets the phy in internal loopback mode
 *
 *  Parameter:
 *      PHY_REG_TYPE *r_ptr
 *      [P[HY_CTRL_TYPE *i_ptr;
 *
 *  Return: void
 **************************************************************************/
void PHY_SetInterLoopback (phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *r_ptr;
    byte reg;
    
    r_ptr = phys[phy_num].base_addr;
    reg = r_ptr->mr;
    if (reg & PHY_MR_RUN) {
      r_ptr->mr = reg & ~PHY_MR_RUN;      /* stop the phy first */
      if (phys[phy_num].id == PHY_ID_PLAYERPLUS) 
	PHY_SetRR38(r_ptr);
    }
    r_ptr->mr = reg | PHY_MR_ILB;
}

/***************************************************************************
 *  Function    PHY_StopLoopback
 *
 *  Description
 *      This function clears the internal loopback and external loopback
 *      bits in the MR.
 *
 *  Parameters:
 *      PHY_REG_TYPE *r_ptr
 *
 *  Return: void
 *****************************************************************************/
void PHY_StopLoopback (phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *r_ptr;
    byte reg;
    
    r_ptr = phys[phy_num].base_addr;
    reg = r_ptr->mr;
    if (reg & PHY_MR_RUN) {
      r_ptr->mr = reg & ~PHY_MR_RUN;      /* stop the phy first */
      if (phys[phy_num].id == PHY_ID_PLAYERPLUS) 
	PHY_SetRR38(r_ptr);
    }
    r_ptr->mr = reg & ~PHY_MR_ILB & ~PHY_MR_EXLB;
}
 
/***************************************************************************
 *  Function    PHY_ClearIcr
 *
 *  Description
 *      This function clears the PLAYER Interrupt Condition Register
 *
 *  Parameters:
 *      PHY_REG_TYPE *r_ptr;
 *
 *  Return: void
 **************************************************************************/
void PHY_ClearIcr (r_ptr)
volatile PHY_REG_TYPE *r_ptr;
{
    byte i = 0;
    
    do {
        r_ptr->icr &= 0;
    } while ((i++ < CWI_REPEAT_MAX) && ((r_ptr->icr & PHY_ICR_CWI) != 0));
}

 
/***************************************************************************
 *  Function    PHY_SetIcr
 *
 *  Description
 *      This function resets some bits in the PLAYER Interrupt Condition
 *      Register. It is used to resets the bits by the isr.
 *
 *  Parameters:
 *      PHY_REG_TYPE *r_ptr;
 *
 *  Return: void
 **************************************************************************/
void PHY_SetIcr (r_ptr,mask)
volatile PHY_REG_TYPE *r_ptr;
byte mask;
{
    byte val, i = 0; 

    val = r_ptr->icr;
    do {
        r_ptr->icr = (val ^ mask) & ~PHY_ICR_CWI;
    }   while ((i++ < CWI_REPEAT_MAX) && (((val=r_ptr->icr) & PHY_ICR_CWI) != 0));
}

/***************************************************************************
 *  Function    PHY_ClearRCRA
 *
 *  Description
 *      This function clears RCRA except NT bit. The NT bit should be
 *      cleared through NTR and NPTR.
 *
 *  Parameter
 *      PHY_REG_TYPE *r_ptr
 *
 *  Return: void
 **************************************************************************/
void PHY_ClearRCRA (r_ptr)
volatile PHY_REG_TYPE *r_ptr;
{
  byte  i=0;

  do {
    r_ptr->icr &= ~PHY_ICR_CWI;
    r_ptr->rcra = 0;
  }   while ((i++ < CWI_REPEAT_MAX) && ((r_ptr->icr & PHY_ICR_CWI) != 0));
}

/***************************************************************************
 *  Function    PHY_ConfigSILS
 *
 *  Description
 *      This function sets SILS bit in the RCRB
 *
 *  Parameter
 *      PHY_REG_TYPE *r_ptr
 *      uint32       action - [DRV_SET | DRV_RESET]
 *
 *  Return: void
 **************************************************************************/
void PHY_ConfigSILS (phy_num,action)
uint32 phy_num;
uint32 action;
{
    volatile PHY_REG_TYPE *r_ptr;
    byte reg;
    
    r_ptr = phys[phy_num].base_addr;
    reg = r_ptr->rcmrb; 
    if (action == DRV_ENABLE)
    {
        r_ptr->rcmrb = reg | PHY_RCMRB_SILSM;
	SILSmask = PHY_RCMRB_SILSM;   /* jlin 07/13/93 */
    }
    else
    {
        r_ptr->rcmrb = reg & ~PHY_RCMRB_SILSM;
	SILSmask = 0;     /* jlin 07/13/93 */
    }
}


/***************************************************************************
 *  Function    PHY_ClearRcrb
 *
 *  Description
 *      This function clears RCRB
 *
 *  Parameter
 *      PHY_REG_TYPE *r_ptr;
 *
 *  Return: void
 *************************************************************************/
void PHY_ClearRCRB (r_ptr)
volatile PHY_REG_TYPE *r_ptr;
{
    byte  i=0;
    
    do {
      r_ptr->icr &= ~PHY_ICR_CWI;
      r_ptr->rcrb = 0;
    }   while ((r_ptr->icr & PHY_ICR_CWI != 0) && (i++ < CWI_REPEAT_MAX));
}

/***************************************************************************
 *  Function    PHY_SetNoiseThresh
 *
 *  Description
 *      This function sets the Noise Threshold Register
 *
 *  Parameters:
 *      PHY_REG_TYPE *r_ptr;
 *      byte val - the new value
 *
 *  Return: void
 **************************************************************************/
void PHY_SetNoiseThresh (r_ptr)
volatile PHY_REG_TYPE *r_ptr;
{
  r_ptr->nptr = nptr_default;
  r_ptr->ntr = ntr_default & PHY_BIT7_MASK;
}


/***************************************************************************
 *  Function    PHY_SetStr
 *
 *  Description
 *      This function sets the State Threshold Register
 *
 *  Parameters:
 *      PHY_REG_TYPE *r_ptr;
 *      byte val - the new value
 *
 *  Return: void
 **************************************************************************/
void PHY_SetStr (r_ptr,val)
volatile PHY_REG_TYPE *r_ptr;
byte val;
{
    r_ptr->str = val & PHY_BIT7_MASK;
}

/***************************************************************************
 *  Function    PHY_SetSptr
 *
 *  Description
 *      This function sets the State Prescale Threshold Register
 *
 *  Parameters:
 *      PHY_REG_TYPE *r_ptr;
 *      byte val - the new value
 *
 *  Return: void
 **************************************************************************/
void PHY_SetSptr (r_ptr,val)
volatile PHY_REG_TYPE *r_ptr;
byte val;
{
    r_ptr->sptr = val;
}


/***************************************************************************
 *  Function    PHY_ReadCncr
 *
 *  Description
 *      This function reads CNCR
 *
 *  parameter:
 *      PHY_REG_TYPE *r_ptr
 *
 *  return: byte - current noise counts
 **************************************************************************/
byte PHY_ReadCncr (r_ptr)
volatile PHY_REG_TYPE *r_ptr;
{
    return (r_ptr->cncr & 0x7f);    /* mask out the msb */
}

/***************************************************************************
 *  Function    PHY_ConfigTx
 *
 *  Description
 *      This function enable/disable Transmitter
 *
 *  parameter:
 *      uint32 phy_num
 *
 *  return: void
 **************************************************************************/
void PHY_ConfigTx (phy_num,act)
uint32 phy_num;
uint32 act;
{
    byte reg;
    volatile PHY_REG_TYPE *r_ptr;

    r_ptr = phys[phy_num].base_addr;
    reg = r_ptr->mr;
    if (act == DRV_ENABLE)
        r_ptr->mr = reg | PHY_MR_TE;
    else
        r_ptr->mr = reg & ~PHY_MR_TE;
}



/***************************************************************************
 *  Function    PHY_ReadCnpcr
 *
 *  Description
 *      This function reads CNPCR
 *
 *  parameter:
 *      PHY_REG_TYPE *r_ptr
 *
 *  return: byte - current noise counts
 **************************************************************************/
byte PHY_ReadCnpcr (r_ptr)
volatile PHY_REG_TYPE *r_ptr;
{
    return (r_ptr->cncr);
}
 
 
/*
 *-----------------------------------------------------------------------
 *          Local Functions
 *-----------------------------------------------------------------------
 */

/***************************************************************************
 *  Function    ResetPlayerRegs
 *
 *  Description
 *      This function resets all the registers in the PLAYER. NOTE that
 *      there is no register can be used to software reset the PLAYER.
 *
 *  Parameters:
 *      uint32 phy_num    
 *
 *  Return: void
 **************************************************************************/
static void ResetPlayerRegs (phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *r_ptr;
    PHY_CTRL_TYPE *i_ptr;
    byte i;

    PHY_DEFAULT_TABLE *tbl_ptr;

    if (fddi_board.type == FDDI_BRIDGE)
        tbl_ptr = &phybr_deftbl;
    else {
        tbl_ptr = &phycon_deftbl;
    }
        
    r_ptr = phys[phy_num].base_addr;
    i_ptr = phys[phy_num].image_ptr;
    memset(i_ptr,0,sizeof(PHY_CTRL_TYPE));
    
    /* h/w bug in PLAYER+ chip */
    if (phys[phy_num].id == PHY_ID_PLAYERPLUS) 
      PHY_SetRR38(r_ptr);

    /***
        setup mode register - TQL is always high so that when unconnected,
        the LED is off.
    ***/
    r_ptr->mr = tbl_ptr->mr;
    
    /***
        set configuration register - initial with isolated
    ***/
    PHY_Isolate(phy_num);

    /***
        clear interrupt condition register
    ***/
    PHY_ClearIcr(r_ptr);

    /***
        interrrupt condition mask register
    ***/
    r_ptr->icmr = i_ptr->icmr = tbl_ptr->icmr;

    /***
        write the h/w reset default value 0xa2 to CTSR
    ***/
    WriteCtsr(r_ptr,tbl_ptr->ctsr);

    /***
        write injection threshold register, default to 0. Not used.
    ***/
    i_ptr->ijtr = r_ptr->ijtr = tbl_ptr->ijtr;

    /***
        write injection symbol register  A, B, default to 0. Not used.
    ***/
    i_ptr->isra = r_ptr->isra = tbl_ptr->isra;
    i_ptr->isrb = r_ptr->isrb = tbl_ptr->isrb;

    /***
        config the RCRA and RCRB to the h/w reset value
    ***/
    i = 0;
    do {
      r_ptr->icr &= ~PHY_ICR_CWI;
      r_ptr->rcra = PHY_RCRA_NT;
    }   while ((r_ptr->icr & PHY_ICR_CWI != 0) && (i++ < CWI_REPEAT_MAX));
    
    i = 0;
    do {
      r_ptr->icr &= ~PHY_ICR_CWI;
        r_ptr->rcra = PHY_RCRB_ST;
    }   while ((r_ptr->icr & PHY_ICR_CWI != 0) && (i++ < CWI_REPEAT_MAX));

    /***
        set RCMRA
    ***/
    r_ptr->rcmra = tbl_ptr->rcmra;

    /***
        set RCMRB
    ***/
    r_ptr->rcmrb = tbl_ptr->rcmrb;

    /***
        reload NTR and NPTR, for 1 msec
    ***/
    ntr_default = tbl_ptr->ntr;
    nptr_default = tbl_ptr->nptr;
    PHY_SetNoiseThresh(r_ptr);

    /***
        reload STR ans SPTR.
     ***/
     PHY_SetStr(r_ptr,tbl_ptr->str);
     PHY_SetSptr(r_ptr,tbl_ptr->sptr);
     
    /***
        set Link Error Threshold Reg - to 0xff so we won't see
	LEMT bit in the ICR
    ***/
    PHY_SetLerCutoff(phy_num,0xff);
    phys[phy_num].lem_int = FALSE;    

    /***
        clear user define register
    ***/
#ifdef __FEBRIDGE
	r_ptr->udr &= 0x48;
#else
    r_ptr->udr = 0;
#endif
}

 
/***************************************************************************
 *  Function  CBusTest
 *
 *  Description
 *    THis function test the PLAYER+ CBus
 *
 *  Parameter:
 *
 *  Return: OK | FDDI_ERROR
 **************************************************************************/
static void delay ()   /* wait for 10 msec */
{
  uint32 ticks;
  ticks = RealTimeTicks() + 1;
  while (ticks > RealTimeTicks());
}

static int CBusTest (r_ptr)
volatile PHY_REG_TYPE *r_ptr;
{
  r_ptr->cr = 0xf9;
  delay();
  if (r_ptr->cr != 0xf9)
    return FDDI_ERROR;

  return OK;
}

/***************************************************************************
 *  Function  SelfTest
 *
 *  Description
 *    This function test each phy chip
 *
 *  Parameter:
 *    uint32 phy_num
 *
 *  Return: OK | FDDI_ERROR
 **************************************************************************/
static int SelfTest (phy_num)
uint32 phy_num;
{
  volatile PHY_REG_TYPE *r_ptr = phys[phy_num].base_addr;

  /* do this according to errata A011.1 */
  if (phys[phy_num].id == PHY_ID_PLAYERPLUS) 
  {
    PHY_SetRR38(r_ptr);	
	delay();
  }
  if (CBusTest(r_ptr) != OK) {
    printf("ERROR: PLAYER+ %d at %x Control Bus test fail\n",phy_num,r_ptr);
    return FDDI_ERROR;
  }
  r_ptr->mr = PHY_MR_RUN | PHY_MR_ILB;   /* set to internal loop back mode */

  PHY_TxSymbols(phy_num,DRV_HALT_LS);  /* send out halt symbols */
  delay();
  if (r_ptr->crsr != PHY_CRSR_HLS) {
    printf("ERROR: PLAYER+ %d at %x internal loopback test fail\n",phy_num,r_ptr);
    return FDDI_ERROR;
  }

  r_ptr->mr = 0;
  r_ptr->mode2 = PHY_MODE2_PHYRST;   /* reset chip */
  return OK;
}

/***************************************************************************
 *  Function    ReadRev
 *
 *  Description
 *      This function reads IDR
 *
 *  Parameter:
 *      PHY_REG_TYPE *r_ptr
 *
 *  Return: byte - device revisioin
 **************************************************************************/
static byte ReadPhyRev (r_ptr)
volatile PHY_REG_TYPE *r_ptr;
{
    return (r_ptr->idr);
}


/***************************************************************************
 *  Function    WriteCtsr
 *
 *  Description
 *      This function writes the CTSR Current Transmit State Register. The
 *      original register is overwritten.
 *
 *  Parameters
 *      PHY_REG_TYPE *r_ptr;
 *      PHY_CTRL_TYPE *i_ptr;
 *      byte        val - the value to OR with the original byte
 *
 **************************************************************************/
static void WriteCtsr (r_ptr,val)
volatile PHY_REG_TYPE *r_ptr;
byte val;
{
  byte ctsr, i=0;
  
  do {
    r_ptr->icr &= ~PHY_ICR_CWI;
    ctsr = r_ptr->ctsr & PHY_CTSR_TM_MASK;
    r_ptr->ctsr = val | ctsr;
  } while ((r_ptr->icr & PHY_ICR_CWI != 0) && (i++ < CWI_REPEAT_MAX));
}

/***************************************************************************
 *  Function  GetPhyID
 *
 *  Description
 *    This function reads IDR in PLAYER at b0000000 and returns PLAYER ID.
 *    THIS FUNCTION ASSUMES THAT THERE IS A PLAYER/PLAYER+ AT ADDRESS
 *    0xB0000000.
 *
 *  parameter:  none
 *
 *  Return: ID
 **************************************************************************/
static byte GetPhyID ()
{
  volatile PHY_REG_TYPE *r_ptr;

  r_ptr = (PHY_REG_TYPE *)0xb0000000;
  return (r_ptr->idr & PHY_IDR_ID_MASK);
}

/***************************************************************************
 *  Function    GetPhyDefault
 *
 *  Description
 *      This function copies phy default data from default table.
 *
 *  Parameter:
 *      uint32 phy_num
 *      uint32 board_type - FDDICON_MASTER | FDDICON_SLAVE
 *
 *  Return: void
 ***************************************************************************/
static status_type GetPhyDefault (phy_num,board_type)
uint32 phy_num;
uint32 board_type;
{
  PHY_INIT_TYPE *default_tbl;

#ifdef __FEBRIDGE
  if (board_type == FDDI_BRIDGE) {
	if( A_Port_Config_State == FRONT &&
		B_Port_Config_State == FRONT )  {
    	default_tbl = &(phy_br_default[phy_num]);
	}
	else
	if( A_Port_Config_State == BACK &&
		B_Port_Config_State == BACK )  {
    	default_tbl = &(phy_br_default_AB2BP[phy_num]);
	}
	else
	if( A_Port_Config_State == FRONT &&
		B_Port_Config_State == BACK )  {
    	default_tbl = &(phy_br_default_B2BP[phy_num]);
	}
	else
	if( A_Port_Config_State == BACK &&
		B_Port_Config_State == FRONT )  {
    	default_tbl = &(phy_br_default_A2BP[phy_num]);
	}
  }
  else
#endif
    default_tbl = &(phy_masplus_default[phy_num]);

  phys[phy_num].base_addr = default_tbl->base_addr;
  phys[phy_num].id = default_tbl->id;
  phys[phy_num].mac_num = default_tbl->mac_num;
  phys[phy_num].status = default_tbl->status;


  return OK;
}

 /*
 *------------------------------------------------------------------------
 *      Functions for unit test
 *------------------------------------------------------------------------
 */



/**************************************************************************
 *  Function   test_phy
 *
 *  Description
 *      entry point of phy test
 *
 *  Parameters:
 *      phy_num
 *      act
 *
 *  Return - OK (0), ERROR (1)
 ************************************************************************/
int test_phy (phy_num,act)
uint32 phy_num;
uint32 act;
{
    int i,j, err = 0;
    uint32 line_st;

    if (!phy_init_flag) {
        DRV_ResetLANChips();
		DRV_InitFddiCon();
	if (PHY_Init(fddi_board.phy_num,fddi_board.type) != OK) {
	  printf("ERROR: Player init fail\n");
	}
    }
#ifdef __FEBRIDGE
	for (i=0; i < fddi_board.phy_num; i++)
	  PHY_Stop(i);
	PHY_Run(phy_num);
#endif

    switch (act) {
        case 1:     /* wrap */
            PHY_Wrap(phy_num);
            return (0);

        case 2:     /* isolate */
            i = SM_PH_CTRL_CONFIG;
            j = CE_ISOLATED;
            DRV_ControlReq(phy_num,i,j);
            return (0);

        case 3:     /* insert P */
            i = SM_PH_CTRL_CONFIG;
            j = CE_INSERT_P;
            DRV_ControlReq(phy_num,i,j);
            return (0);

        case 4:     /* insert S */
            i = SM_PH_CTRL_CONFIG;
            j = CE_INSERT_S;
            DRV_ControlReq(phy_num,i,j);
            return (0);
                        
        case 5:     /* test init */
	    /* for test prupose only */
            if (PHY_Init(8,FDDICON_MASTER) != OK) {
                printf("PHY_Init() failed\n");
                err = 1;
            }
            else {
                printf("PHY_Init() successed\n");
                err = 0;
            }        
            return (err);

        case 6:     /* display all regs */
            return (test_phy_regs(phy_num));

        case 7:     /* tx quiet symbols */
            i = SM_PM_CTRL_TX_SYMBOL;
            j = QUIET_LINE_STATE;
	    DRV_ControlReq(phy_num,i,j);
	    line_st = PHY_GetTxLs(phy_num);
	    printf("the line state read from CTSR is %d\n",line_st);
	    return (0);

        case 8:     /* tx quiet symbols */
            i = SM_PM_CTRL_TX_SYMBOL;
            j = HALT_LINE_STATE;
	    DRV_ControlReq(phy_num,i,j);
	    line_st = PHY_GetTxLs(phy_num);
	    printf("the line state read from CTSR is %d\n",line_st);
	    return (0);

        case 9:     /* tx master symbols */
            i = SM_PM_CTRL_TX_SYMBOL;
            j = MASTER_LINE_STATE;
	    DRV_ControlReq(phy_num,i,j);
	    line_st = PHY_GetTxLs(phy_num);
	    printf("the line state read from CTSR is %d\n",line_st);
	    return (0);

                
        case 10:     /* tx idle symbols */
            i = SM_PM_CTRL_TX_SYMBOL;
            j = IDLE_LINE_STATE;
	    DRV_ControlReq(phy_num,i,j);
	    line_st = PHY_GetTxLs(phy_num);
	    printf("the line state read from CTSR is %d\n",line_st);
	    return (0);

        case 11:     /* tx active symbols */
            i = SM_PM_CTRL_TX_SYMBOL;
            j = ACTIVE_LINE_STATE;
	    DRV_ControlReq(phy_num,i,j);
	    line_st = PHY_GetTxLs(phy_num);
	    printf("the line state read from CTSR is %d\n",line_st);
	    return (0);
                
	case 12:     /* exit from phy test */
	    PHY_ClearPhys(2,FDDICON_MASTER);
	    return 0;
    }       /* switch */
    return 0;
}

/**************************************************************************
 *  Function    test_phy_regs
 *
 *  Description
 *      This function reads all the registers in the specified phy.
 *
 *  Parameters:
 *      int i - phy #
 *
 *  Return:     ERROR - 1; OK - 0
 **************************************************************************/  
static int test_phy_regs (i)
int i;
{
    PHY_REG_TYPE *r_ptr;

    r_ptr = phys[i].base_addr;
    printf("mr %x, cr %x, icr %x, icmr %x, ctsr %x\n",r_ptr->mr,r_ptr->cr,r_ptr->icr,r_ptr->icmr,r_ptr->ctsr);
    printf("ijtr %x, isra %x, isrb %x, crsr %x, rcra %x\n",r_ptr->ijtr,r_ptr->isra,r_ptr->isrb,r_ptr->crsr,r_ptr->rcra);
    printf("rcrb %x, rcmra %x, rcmrb %x, ntr %x, nptr %x\n",r_ptr->rcrb,r_ptr->rcmra,r_ptr->rcmrb,r_ptr->ntr,r_ptr->nptr);
    printf("cncr %x, cnpcr %x, str %x, sptr %x, cscr %x\n",r_ptr->cncr,r_ptr->cnpcr,r_ptr->str,r_ptr->sptr,r_ptr->cscr);
    printf("cspcr %x, letr %x, clecr %x, udr %x, idr %x\n",r_ptr->cspcr,r_ptr->letr,r_ptr->clecr,r_ptr->udr,r_ptr->idr);

    return (0);
}


#ifdef __FEBRIDGE

Br_Switch_A_Phys( int state ) {
	A_Port_Config_State = state;
	Restart_Phys(0);
}

Br_Switch_B_Phys( int state ) {
	B_Port_Config_State = state;
	Restart_Phys(1);
}


Restart_Phys(int phynum)
{

	if( phynum == 1 )		/* if B port */
		PHY_Isolate(phynum);
	else {
		PHY_Isolate(phynum);
		PHY_Stop(phynum); 
		PHY_FS_Kill(phynum);  
		PHY_Insert_NULL(phynum);	
	}
	PHY_ClearInt (phynum);
    GetPhyDefault(phynum,FDDI_BRIDGE);
	ResetPlayerRegs (phynum);
	PHY_Run(phynum);
	PHY_Isolate(phynum); 
}
void PHY_Insert_NULL(phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *ptr;
    PHY_CTRL_TYPE *i_ptr;

    i_ptr = phys[phy_num].image_ptr;
    ptr = phys[phy_num].base_addr;
    ptr->cr =  PHY_CR_BIND_BREQ | PHY_CR_BIE;
}

void PHY_FS_Kill(phy_num)
uint32 phy_num;
{
    volatile PHY_REG_TYPE *ptr;


	/*
    if (phys[phy_num].status == DRV_MODE_STOP)
        return;
	*/
    ptr = phys[phy_num].base_addr;
    ptr->mr = 0;

    if (phys[phy_num].id == PHY_ID_PLAYERPLUS) 
      PHY_SetRR38(ptr);
    phys[phy_num].status = DRV_MODE_STOP;
}

/***************************************************************************
 *  Function    ResetOfflineBPlayerRegs
 *
 *  Return: void
 **************************************************************************/
static void ResetOfflineBPlayerRegs ()
{
    volatile PHY_REG_TYPE *r_ptr;
    byte i;
    PHY_DEFAULT_TABLE *tbl_ptr;


	if( A_Port_Config_State == FRONT &&
		B_Port_Config_State == FRONT )  {
    	r_ptr = (&(phy_br_default[3]))->base_addr;
	}
	else
	if( A_Port_Config_State == BACK &&
		B_Port_Config_State == BACK )  {
    	r_ptr = (&(phy_br_default_AB2BP[3]))->base_addr;
	}
	else
	if( A_Port_Config_State == FRONT &&
		B_Port_Config_State == BACK )  {
    	r_ptr = (&(phy_br_default_B2BP[3]))->base_addr;
	}
	else
	if( A_Port_Config_State == BACK &&
		B_Port_Config_State == FRONT )  {
    	r_ptr = (&(phy_br_default_A2BP[3]))->base_addr;
	}


    tbl_ptr = &phybr_deftbl;
        
    /* h/w bug in PLAYER+ chip */
    PHY_SetRR38(r_ptr);

    /***
        setup mode register - TQL is always high so that when unconnected,
        the LED is off.
    ***/
    r_ptr->mr = tbl_ptr->mr;
    
    /***
        set configuration register - initial with isolated
    ***/
    /* PHY_Isolate(phy_num); */
	r_ptr->cr = 0xdb;

    /***
        clear interrupt condition register
    ***/
    PHY_ClearIcr(r_ptr);

}
#endif
