/****************************************************************************
 *  File    bsi_reg.c
 *
 *  Description
 *      This file contains all the functions to access the BSI registers
 *
 *  Copyright (c) 1992, 1993 Hughes Lan Systems
 *
 *  Author: J. Jang
 **************************************************************************/


#include <fddi.h>
#include <drv.h>
#include <bsi.h>



extern BSI_TYPE bsis[];
extern BSI_ADDR_TYPE bsi_dev[];


/*
 *-----------------------------------------------------------------
 *      local functions
 *-----------------------------------------------------------------
 */
/* static byte SetCondReg(byte,byte); */



/****************************************************************************
 *  Function    BSIREG_READREV
 *
 *  Description
 *      This function set the mailbox register (MBAR) and read the revision
 *      number.
 *
 *  Parameter:
 *      uint16 bsi_num
 *
 *  Return: void
 ***************************************************************************/
void BSIREG_ReadRev (bsi_num)
uint32 bsi_num;
{
    volatile BSI_REG_TYPE *r_ptr;
    uint32  mbox, bsi_mbox;

    r_ptr = (BSI_REG_TYPE *)(bsi_dev+bsi_num)->base_addr;
    r_ptr->pcar = 0;

    /* read revision number */
    (bsis+bsi_num)->rev_num = r_ptr->mbar << 24;
    (bsis+bsi_num)->rev_num |= r_ptr->mbar << 16;
    (bsis+bsi_num)->rev_num |= r_ptr->mbar << 8;
    (bsis+bsi_num)->rev_num |= r_ptr->mbar;
        
#ifdef __FEBRIDGE
    mbox = (uint32) shmalloc(sizeof(uint32));	/* mbox in shared RAM */
#else
    mbox = (uint32) lmalloc(sizeof(uint32));
#endif
    if (mbox == 0L)
      return;
    (bsis+bsi_num)->mbox_addr = mbox;
    bsi_mbox = CPU2BSI(mbox);
    /* set mailbox address */
    r_ptr->mbar = bsi_mbox >> 24 & 0xff;      
    r_ptr->mbar = bsi_mbox >> 16 & 0xff;      
    r_ptr->mbar = bsi_mbox >> 8 & 0xff;      
    r_ptr->mbar = bsi_mbox & 0xff;      
}

/**********************************************************************
 *  Function BSIREG_ResetBsi
 *
 *  Description
 *    This function resets the BSI by set the MRST bit in MR0.
 *
 *  Parameter:
 *    uint32 bsi_num
 *
 *  Return:  void
 *********************************************************************/
void BSIREG_ResetBsi (bsi_num)
uint32 bsi_num;
{
  volatile BSI_REG_TYPE *r_ptr;

  r_ptr = (BSI_REG_TYPE *)(bsi_dev+bsi_num)->base_addr;
  r_ptr->mr0 = BSI_MR0_MRST;
}

/**********************************************************************
 *  Fruction    BSIREG_SetVST
 *
 *  Description
 *      This function sets the VST bit in R0CR0/R1CR0. The void
 *      stripping is used by FDDI bridge.
 *
 *  Parameter:
 *      uint16 bsi_num
 *
 *  Return: void
 **********************************************************************/
void BSIREG_SetVST(r_ptr)
volatile BSI_REG_TYPE *r_ptr;
{
    r_ptr->r0cr0 |= BSI_RxCR0_VST;
    r_ptr->r1cr0 |= BSI_RxCR0_VST;
}

/**********************************************************************
 *  Fruction    BSIREG_SetSAT
 *
 *  Description
 *      This function sets the SAT bit in R0CR0/R1CR0. The source
 *      address transparency is mainly used by FDDI bridge.
 *
 *  Parameter:
 *      uint16 bsi_num
 *
 *  Return: void
 **********************************************************************/
void BSIREG_SetSAT(ptr)
volatile BSI_REG_TYPE *ptr;
{
    ptr->r0cr0 |= BSI_RxCR0_SAT;
    ptr->r1cr0 |= BSI_RxCR0_SAT;
}

 
/****************************************************************************
 *  Function    BSIREG_SetITR
 *
 *  Description
 *      This function sets the BSI indicate threshold register
 *
 *  Parameters:
 *      uint16 bsi_num
 *      byte new_val - threshold value
 *
 *  Return: void
 ***************************************************************************/
void BSIREG_SetITR(bsi_num,new_val)
uint32 bsi_num;
uint32 new_val;
{
    byte val, i=0;
    volatile BSI_REG_TYPE *r_ptr;
    BSI_CTRL_TYPE *i_ptr;
    
    r_ptr = (bsi_dev+bsi_num)->base_addr;
    i_ptr = (bsis+bsi_num)->image_ptr;
    
    do {
        val = r_ptr->iar;
        r_ptr->iar = val | BSI_IAR_EXCI1 | BSI_IAR_EXCI2;
    } while ((r_ptr->star & BSI_STAR_CWI != 0) && (i++ < CWI_REPEAT_MAX));

    r_ptr->itr = (byte )new_val;
    i_ptr->itr = (byte )new_val;
    
    i = 0;
    do {
        val = r_ptr->iar;
        r_ptr->iar = val & ~BSI_IAR_EXCI1 & ~BSI_IAR_EXCI2;
    } while ((r_ptr->star & BSI_STAR_CWI != 0) && (i++ < CWI_REPEAT_MAX));
}

 
/****************************************************************************
 *  Function    BSIREG_SetIHLR
 *
 *  Description
 *      This function sets the BSI indicate header length register
 *
 *  Parameters:
 *      uint16 bsi_num
 *      byte new_val - threshold value
 *
 *  Return: void
 ***************************************************************************/
void BSIREG_SetIHLR(bsi_num,new_val)
uint32 bsi_num;
uint32 new_val;
{
    byte val, i=0;
    volatile BSI_REG_TYPE *r_ptr;
    
    r_ptr = (bsi_dev+bsi_num)->base_addr;
    
    do {
        val = r_ptr->ihlr;
        r_ptr->ihlr = (byte )new_val;
    } while ((r_ptr->star & BSI_STAR_CWI != 0) && (i++ < CWI_REPEAT_MAX));
}


/****************************************************************************
 *  Function    BSIREG_SetNsar
 *
 *  Description
 *      This function sets the BSI NoSpace Attention Register. It is used
 *      to clear a bit after writting PSP to the Limit RAM reg. NOTE that
 *      the LDIx bits need to be treated carefully. If the host clears
 *      LDI bits, the PSP descriptors will be messed up.
 *
 *  Parameters:
 *      uint16 bsi_num
 *      byte mask
 *
 *  Return: void
 ***************************************************************************/
void BSIREG_SetNsar(bsi_num,mask)
uint32 bsi_num;
byte mask;
{
#define LOW_SPACE_MASK  0x2a
    byte val, i=0;
    volatile BSI_REG_TYPE *r_ptr;

    r_ptr = bsi_dev[bsi_num].base_addr;
    mask |= LOW_SPACE_MASK;
    do {
        val = r_ptr->nsar;
        r_ptr->nsar = val & mask;
    } while ((r_ptr->star & BSI_STAR_CWI != 0) && (i++ < CWI_REPEAT_MAX));
}


/****************************************************************************
 *  Function    BSIREG_ClearIAR
 *
 *  Description
 *      This function clears the BSI Indicate Attention Register. It is used
 *      to clear a bit after writting PSP to the Limit RAM reg.
 *
 *  Parameters:
 *      uint16 bsi_num
 *      byte mask
 *
 *  Return: void
 ***************************************************************************/
void BSIREG_ClearIAR (r_ptr)
volatile BSI_REG_TYPE *r_ptr;
{
    byte val, i=0;

    do {
        val = r_ptr->iar;
        r_ptr->iar = 0;
    } while ((r_ptr->star & BSI_STAR_CWI != 0) && (i++ < CWI_REPEAT_MAX));
}


/****************************************************************************
 *  Function    BSIREG_ClearRAR
 *
 *  Description
 *      This function clears the BSI Request Attention Register. It is used
 *      to clear a bit after writting PSP to the Limit RAM reg.
 *
 *  Parameters:
 *      uint16 bsi_num
 *
 *  Return: void
 ***************************************************************************/
void BSIREG_ClearRAR (bsi_num)
uint32 bsi_num;
{
    byte val, i=0;
    volatile BSI_REG_TYPE *r_ptr;

    r_ptr = bsi_dev[bsi_num].base_addr;
#if 0
    while (i++ < CWI_REPEAT_MAX) {
      val = r_ptr->rar;
      r_ptr->rar = 0;
      if ((r_ptr->star & BSI_STAR_CWI) == 0)
	return;
    }
    ReSchedule();
#endif
    do {
      val = r_ptr->rar;
      r_ptr->rar = 0;
    } while ((r_ptr->star & BSI_STAR_CWI != 0) && (i++ < CWI_REPEAT_MAX));
}


/****************************************************************************
 *  Function    BSIREG_ClearSTAR
 *
 *  Description
 *      This function clears the BSI State Attention Register. 
 *
 *  Parameters:
 *      uint16 bsi_num
 *      byte mask
 *
 *  Return: void
 ***************************************************************************/
void BSIREG_ClearSTAR (bsi_num)
uint32 bsi_num;
{
    byte val;
    volatile BSI_REG_TYPE *r_ptr;    
    int i=0;

    r_ptr = bsi_dev[bsi_num].base_addr;
    do {
        val = r_ptr->star;
        r_ptr->star = 0;
    } while ((r_ptr->star & BSI_STAR_CWI != 0) && (i++ < CWI_REPEAT_MAX));
}

/****************************************************************************
 *  Function    BSIREG_SetSTAR
 *
 *  Description
 *      This function sets the BSI State Attention Register - one of the
 *      3 state machines.
 *
 *  Parameters:
 *      uint16 bsi_num
 *      byte mask - [BSI_STAR_INSTOP | BSI_STAR_RQSTOP | BSI_STAR_SPSTOP]
 *
 *  Return: void
 ***************************************************************************/
void BSIREG_SetSTAR (bsi_num,mask)
uint32 bsi_num;
byte mask;
{
    byte val;
    volatile BSI_REG_TYPE *r_ptr;    
    int i = 0;

    r_ptr = bsi_dev[bsi_num].base_addr;
    do {
        val = r_ptr->star;
        r_ptr->star = mask;
    } while ((r_ptr->star & BSI_STAR_CWI != 0) && (i++ < CWI_REPEAT_MAX));
}

/**************************************************************************
 *  Funciton    BSIREG_BsiRun
 *
 *  Description
 *      This function sets BSI to RUN mode
 *
 *  Parameter:
 *      uint16 bsi_num
 *
 *  Return: void
 *************************************************************************/
void BSIREG_BsiRun (bsi_num)
uint32 bsi_num;
{
    volatile BSI_REG_TYPE *r_ptr;
    BSI_CTRL_TYPE *i_ptr = bsis[bsi_num].image_ptr;

    r_ptr = (BSI_REG_TYPE *)(bsi_dev+bsi_num)->base_addr;

    /* put 3 state machines to run mode */
    BSIREG_ClearSTAR(bsi_num);
    bsis[bsi_num].bsi_status = 0L;
    /* restore interrupts */
    r_ptr->mnr = i_ptr->mnr;
}
 

/**************************************************************************
 *  Function    BSIREG_BsiStop
 *
 *  Description
 *      This function sets the BSI to STOP mode
 *
 *  Parameter:
 *      uint16 bsi_num
 *
 *  Return: void
 **************************************************************************/
void BSIREG_BsiStop (bsi_num)
uint32 bsi_num;
{
    volatile BSI_REG_TYPE *r_ptr;

    r_ptr = (BSI_REG_TYPE *)(bsi_dev+bsi_num)->base_addr;
    /* put 3 state machines in STOP mode */
    BSIREG_SetSTAR(bsi_num,BSI_STAR_INSTOP|BSI_STAR_RQSTOP|BSI_STAR_SPSTOP);
    bsis[bsi_num].bsi_status = BSI_IND_STOP | BSI_REQ_STOP | BSI_SP_STOP;
    r_ptr->mnr = 0;     /* disable all the BSI interrupts */
}

/**************************************************************************
 *  Function    BSIREG_EnableParity
 *
 *  Description
 *      This function sets the BSI to check the parity error.
 *
 *  Parameter:
 *      uint16 bsi_num
 *
 *  Return: void
 **************************************************************************/
void BSIREG_EnableParity (bsi_num)
uint32 bsi_num;
{
    volatile BSI_REG_TYPE *r_ptr;
    BSI_CTRL_TYPE *i_ptr;

    r_ptr = (BSI_REG_TYPE *)(bsi_dev+bsi_num)->base_addr;
    i_ptr = (bsis+bsi_num)->image_ptr;
    r_ptr->mr0 = i_ptr->mr0 = (i_ptr->mr0 | BSI_MR0_FLOW);
}

/**************************************************************************
 *  Function    BSIREG_DisableParity
 *
 *  Description
 *      This function disable the BSI to check the parity error.
 *
 *  Parameter:
 *      uint16 bsi_num
 *
 *  Return: void
 **************************************************************************/
void BSIREG_DisableParity (bsi_num)
uint32 bsi_num;
{
    volatile BSI_REG_TYPE *r_ptr;
    BSI_CTRL_TYPE *i_ptr;

    r_ptr = (BSI_REG_TYPE *)(bsi_dev+bsi_num)->base_addr;
    i_ptr = (bsis+bsi_num)->image_ptr;
    r_ptr->mr0 = i_ptr->mr0 = (i_ptr->mr0 & ~BSI_MR0_FLOW);
}




