/***********************************************************************
 *  File    bmac_isr.c
 *
 *  Description
 *          This module processes the BMAC interrups.
 *
 *  Copyright (c) 1992-93 Hughes Lan Systems
 *
 *  Author: Jia-wei Jang
 *********************************************************************/

#include <fddi.h>
#include <drv.h>
#include <bsi.h>
#include <bmac.h>
#include <monp.h>


#define COUNT_DEBUG     0   /* to debug mac related counts */
#define RINGOP_DEBUG    0
#define CLAIM_DEBUG     0


#define RINGOP_MAX_COUNT    2   /* max ringop flag interrupt count */

#if COUNT_DEBUG
int ringop_cnt[2];
int rx_cnt[2];
int cp_cnt[2];
int err_cnt[2];
int nocp_cnt[2];  /* not copy */
int over_cnt[2];  /* overflow */
#endif


/*
 *-----------------------------------------------------------------------
 *             Imported variables
 *-----------------------------------------------------------------------
*/
extern BMAC_TYPE bmacs[];
extern DRV_MIBS_TYPE drv_mibs[];


/*
 *-----------------------------------------------------------------------
 *             Local function prototypes
 *-----------------------------------------------------------------------
*/
#ifdef DO_LINT
static void RELR1_isr (byte);
static void TTE_isr (uint32);

#else
static void RELR1_isr ();
static void TTE_isr ();
static void COLR_isr();
#endif


/**************************************************************************
 *  Function    BMACISR_InitDebug
 *
 *  Description
 *    This function initializes the debug variables
 *
 *  Parameters:  none
 *
 *  Return: void
 *************************************************************************/
void BMACISR_InitDebug ()
{
#if COUNT_DEBUG
  int i;

  for (i=0; i < 2; i++) {
    ringop_cnt[i] = rx_cnt[i] = err_cnt[i] = nocp_cnt[i] = over_cnt[i] = cp_cnt[i]=0;
  }
#endif
}

/**************************************************************************
 *  Function BMAC_PollRingOp
 *
 *  Description
 *    This function polls the RELR0 for RingOp flag
 *
 *  Parameter: bmac_num
 *
 *  Return: void
 *************************************************************************/
void BMAC_PollRingOp (mac_num)
uint32 mac_num;
{
  register volatile BMAC_REG_TYPE *r_ptr;
  byte reg;
  uint32 i = 0, sig_val = 0;     /* value with signal to SMT */
#ifdef __FEBRIDGE
  static call_cnt = 0;

  if(! (call_cnt++ % 20) )
		return;
#endif

  r_ptr = bmacs[mac_num].reg_base;

  /* polling TVX Expired. */
  if (r_ptr->telr0 & BMAC_TELR0_TVXEXP) {
    (*(init_mibs.d_mib[mac_num].macTVXExpiredCount))++;
    r_ptr->telr0 &= ~BMAC_TELR0_TVXEXP;
  }
  if (r_ptr->telr0 & BMAC_TELR0_TRTEXP) {
    /* increment late count if TRT expired */
    (*(init_mibs.d_mib[mac_num].macLateCount))++;
    r_ptr->telr0 &= ~BMAC_TELR0_TRTEXP;
  }

  if ((reg = r_ptr->relr0)) {
    if (reg & (BMAC_RELR0_ROP | BMAC_RELR0_RNOP)) {
      /* clear the latch register */
      reg &= ~BMAC_RELR0_DUPADD;  /* don't touch dupAddr */
      do {
	r_ptr->relr0 ^= reg;  
      } while (((r_ptr->esr & BMAC_ESR_CWI) != 0) && (++i < CWI_REPEAT_MAX));

      if ((sig_val = r_ptr->cts0 & BMAC_CTS0_ROP))   /* ringOp is set */ {
	bmacs[mac_num].ring_op = TRUE;
	BSI_ChkUnserReq(mac_num);
	/* only counted when RingOp from reset to set */
	(*(init_mibs.d_mib[mac_num].macRingOpCount))++;  
#if RINGOP_DEBUG
	{
	  int ticks;
	  ticks = RealTimeTicks();
	  printf("ring_op in mac %d up, %d\n",mac_num,ticks);
	}
#endif  /* RINGOP_CEBUG */
      }
      else {
	bmacs[mac_num].ring_op = FALSE;
#if RINGOP_DEBUG
	{
	  int ticks;
	  ticks = RealTimeTicks();
	  printf("ring_op in mac %d is down, %d   ",mac_num,ticks);
	  printf("RELR0: %x,  RELR1 %x\n",reg,r_ptr->relr1);
	}
#endif  /* RINGOP_CEBUG */
      }
      CSP_SMTSendSignal(SIG_RING_OP,mac_num,sig_val);
    }
  }
}

/**************************************************************************
 *  Function    BMAC_Isr
 *
 *  Description
 *      These functions process the BMAC interrupt.
 *
 *  Parameter:
 *      uint32 mac_num
 *
 *  Return: void
 *************************************************************************/
void BMAC_PriIsr ()
{
  BMAC_Isr(BMAC_B);
  clear_ipnd(1 << 5);
}
void BMAC_SecIsr ()
{
  BMAC_Isr(BMAC_A);
  clear_ipnd(1 << 4);
}

void BMAC_Isr (mac_num)
uint32 mac_num;
{
    volatile BMAC_REG_TYPE *r_ptr;
    byte icr, reg;
    uint32 sig_val = 0;     /* value with signal to SMT */

    r_ptr = bmacs[mac_num].reg_base;
	if (r_ptr->or & BMAC_OR_IRPT) {
      printf("Fatal Error: BMAC %d IRPT bit is set\n",mac_num);
	  r_ptr->or &= ~BMAC_OR_IRPT;
	}
    icr = r_ptr->icr;

    if ((icr & r_ptr->imr) == 0)     /* interrupt was masked */
      return;

    if (icr & BMAC_ICR_RNG) {   /* process Ring Event Latch Reg */
      /* only check for Ring_OP flag. The other bits are masked off.
           The SMT core code needs driver to pass the value of
           BMAC_CTS0 with 0-6 bits mask off */
        /* check relr1 */
	reg = r_ptr->relr1;
	if (reg != 0) 
	  RELR1_isr(reg,r_ptr,mac_num,&sig_val);
    }

    if (icr & BMAC_ICR_TTE) {
        reg = r_ptr->telr0;
        if (reg != 0) {
            TTE_isr(reg,r_ptr,mac_num,&sig_val);
        }
    }
    if (icr & BMAC_ICR_COE) {
        reg = r_ptr->colr;
        if (reg != 0) {
            COLR_isr(reg,r_ptr,mac_num);
        }
    }
    if (icr & BMAC_ICR_IERR) {  /* fatal internal error */
        reg = r_ptr->ielr;
        if (reg & BMAC_IELR_RSMERR)
	  printf("Fatal Error: BMAC %d internal unknown Rx state\n",mac_num);
        if (reg & BMAC_IELR_TSMERR)
	  printf("Fatal Error: BMAC %d internal unknown Tx state\n",mac_num);
#if 0
	/* don't clear it for debugging */
        r_ptr->ielr ^= reg;        /* clear it */
#endif
	reset();                   /* reset the board */
    }

    if (sig_val != 0)
      CSP_SMTSendSignal(SIG_MAC_INTERRUPT,mac_num,sig_val);
}

/************************************************************************
 *  Function    RELR1_isr
 *
 *  Description
 *      This function processes the Ring Event Latch Register 1. It won't
 *      clear the relr0. This function also clears the corresponding mask
 *      bit in REMR1
 *
 *  Parameter:
 *      byte val - register content
 *
 *  Return: void
 ***********************************************************************/
static void RELR1_isr (val,r_ptr,mac_num,signal)
byte val;
volatile BMAC_REG_TYPE *r_ptr;
uint32 mac_num;
uint32 *signal;
{
    byte remr1=0;
    uint32 i=0;

    if (val & BMAC_RELR1_OTRBCN & BMAC_REMR1_OTRBCN) {
#if CLAIM_DEBUG
	  printf("rx other beacon on mac %d\n",mac_num);
#endif
      remr1 |= BMAC_REMR1_OTRBCN;
      *signal |= MAC_OTHER_BEACON;
	  r_ptr->fr |= BMAC_FR_CLM;
      PHY_PstatsMACPkts(mac_num);
      MONP_MLstats(mac_num,MLSTATS_BCN_EVNT);
      if (bmacs[mac_num].BeaconType == EXTERNAL_BEACON)
	BSI_AbortReq(mac_num);
    }
    if (val & BMAC_RELR1_MYBCN & BMAC_REMR1_MYBCN) {
	  r_ptr->fr |= BMAC_FR_CLM;
      remr1 |= BMAC_REMR1_MYBCN;
      *signal |= MAC_MY_BEACON;
#if CLAIM_DEBUG
	  printf("rx my beacon on mac %d, fr %x\n",mac_num,r_ptr->fr);
#endif
      PHY_PstatsMACPkts(mac_num);
      MONP_MLstats(mac_num,MLSTATS_BCN_EVNT);
      if (bmacs[mac_num].BeaconType == EXTERNAL_BEACON)
	BSI_AbortReq(mac_num);
    }
    if (val & BMAC_RELR1_MYCLM & BMAC_REMR1_MYCLM) {
      remr1 |= BMAC_REMR1_MYCLM;
      *signal |= MAC_MY_CLAIM;
 /*     r_ptr->or |= BMAC_OR_IRR;   */
#if CLAIM_DEBUG
	  printf("rx my claim on mac %d\n",mac_num);
#endif
	  PHY_PstatsMACPkts(mac_num);
      MONP_MLstats(mac_num,MLSTATS_CLM_EVNT);
    }
    if (val & (BMAC_RELR1_HICLM | BMAC_RELR1_LOCLM) & (BMAC_REMR1_HICLM | BMAC_REMR1_LOCLM)) {
#if CLAIM_DEBUG
	  printf("rx other claim on mac %d\n",mac_num);
#endif
 /*     r_ptr->or |= BMAC_OR_IRR;    */
      PHY_PstatsMACPkts(mac_num);
      MONP_MLstats(mac_num,MLSTATS_CLM_EVNT);
    }

    r_ptr->remr1 &= ~remr1;    /* mask off interrupts */

    do {
        r_ptr->relr1 ^= val;
    } while (((r_ptr->esr & BMAC_EMR_CWI) != 0) && (++i < CWI_REPEAT_MAX));
}

/************************************************************************
 *  Function    TTE_Isr
 *
 *  Description
 *      This function processes Token and Timer event interrupt
 *
 *  Parameter:
 *      uint32 mac_num
 *
 *  Return: void
 ***********************************************************************/
static void TTE_isr (telr,r_ptr,mac_num,signal)
byte telr;
volatile BMAC_REG_TYPE *r_ptr;
uint32 mac_num;
uint32 *signal;
{
    byte val = 0;
    uint32 i=0;
    byte temr = r_ptr->temr0;

    if (telr & BMAC_TELR0_ENTRMD & temr) { /* entering restricted mode */
      val |= BMAC_TEMR0_ENTRMD;
      *signal |= MAC_RMODE;
    }
    if (telr & BMAC_TELR0_CBERR & temr) {
      val |= BMAC_TEMR0_CBERR;
      *signal |= MAC_TRT_IN_T4T5;
      r_ptr->temr0 &= ~BMAC_TEMR0_CBERR;  /* disable interrupt */
      (*(init_mibs.d_mib[mac_num].macLateCount))++;
    }

#if 0
    if (telr & BMAC_TELR0_DUPTKR) {
        drv_mibs[mac_num].macDupTokenCount++;
    }

    if (telr & BMAC_TELR0_TKCAPT) {
        drv_mibs[mac_num].macTokenCaptureCount++;
    }
    if (telr & BMAC_TELR0_TKPASS) {
        drv_mibs[mac_num].macTokenPassCount++;
        /* calculate the average token pass period */
        /* or we can use TKCT token received counter to do the same thing */
        /* if we use the later method, reset this mask bit to reduce the
            interrupt overhead */
    }

    if (telr & BMAC_TELR0_RLVD & BMAC_TEMR0_RLVD) {
        latency_cnt = r_ptr->rlct1 << 16;
        latency_cnt = r_ptr->rlct2 << 8;
        latency_cnt = r_ptr->rlct3 & BMAC_COUNTER_MASK;
        (*(init_mibs.d_mib[mac_num].macRingLatencyCount)) = latency_cnt;
    }
#endif

    do {
	r_ptr->telr0 ^= telr;
    } while (((r_ptr->esr & BMAC_EMR_CWI) != 0) && (++i < CWI_REPEAT_MAX));
}

/************************************************************************
 *  Function    COLR_isr
 *
 *  Description
 *      This function process Counter Overflow Interrupt
 *
 *  Parameters
 *      byte mask - content of COLR
 *      BMAC_REG_TYPE *r_ptr
 *      uint32 mac_num
 *
 *  Return: void
 ****************************************************************************/
static void COLR_isr (mask,r_ptr,mac_num)
byte mask;
volatile BMAC_REG_TYPE *r_ptr;
uint32 mac_num;
{
#define COUNTER_OVERFLOW_MASK       0x00100000  /* 20 bits overflow reg */
    DRV_FDDI_MIBS *mib_ptr;
    register i=0;

    mib_ptr = &(init_mibs.d_mib[mac_num]);
    if (mask & BMAC_COLR_FRRCV) {
#if COUNT_DEBUG
      rx_cnt[mac_num] += 1;
      printf("mac %d, frcv %d\n",mac_num,rx_cnt[mac_num]);
#endif
        *(mib_ptr->macFrameRxCount) += COUNTER_OVERFLOW_MASK;
    }
    if (mask & BMAC_COLR_FREI) {
#if COUNT_DEBUG
      err_cnt[mac_num] += 1;
      printf("mac %d, err %d\n",mac_num,err_cnt[mac_num]);
#endif
        *(mib_ptr->macErrIsolateCount) += COUNTER_OVERFLOW_MASK;
    }
    if (mask & BMAC_COLR_FRLST) {
#if COUNT_DEBUG
      over_cnt[mac_num] += 1;
      printf("mac %d, lost %d\n",mac_num,over_cnt[mac_num]);
#endif
        *(mib_ptr->macLostFrameCount) += COUNTER_OVERFLOW_MASK;
    }
    if (mask & BMAC_COLR_FRCOP) {
#if COUNT_DEBUG
      cp_cnt[mac_num] += 1;
      printf("mac %d, copy %d, mib %x\n",mac_num,cp_cnt[mac_num],mib_ptr->macFrameCopiedCount);
#endif
        *(mib_ptr->macFrameCopiedCount) += COUNTER_OVERFLOW_MASK;
      }
    if (mask & BMAC_COLR_FRNCOP) {
#if COUNT_DEBUG
      nocp_cnt[mac_num] += 1;
      printf("mac %d, not copied %d\n",mac_num,nocp_cnt[mac_num]);
#endif
        *(mib_ptr->macFrameNotCopiedCount) += COUNTER_OVERFLOW_MASK;
    }
    if (mask & BMAC_COLR_FRTRX)
        *(mib_ptr->macFrameTxCount) += COUNTER_OVERFLOW_MASK;
    if (mask & BMAC_COLR_TKRCVD)
        *(mib_ptr->macTokenRxCount) += COUNTER_OVERFLOW_MASK;

    do {
        r_ptr->colr ^= mask;
    } while (((r_ptr->esr & BMAC_EMR_CWI) != 0) && (++i < CWI_REPEAT_MAX));
}

/***************************************************************************
 *  Function  PrintIntCounts
 *
 *  Description
 *    This function prints out the # of interrupts of each overflow count
 *    register. It is used mainly for debug
 *
 *  Parameter:
 *    int mac_num
 *
 *  Return: void
 *************************************************************************/
void PrintIntCounts (mac_num)
uint32 mac_num;
{
#if COUNT_DEBUG
  int *ptr, i,j;

   printf("rx int cnt %d, mib %d\n",rx_cnt[mac_num],*(init_mibs.d_mib[mac_num].macFrameRxCount));
   printf("err int cnt %d\n",err_cnt[mac_num]);
   printf("copy int cnt %d, mib %d\n",cp_cnt[mac_num],*(init_mibs.d_mib[mac_num].macFrameCopiedCount));
   printf("not copy int cnt %d\n",nocp_cnt[mac_num]);
   printf("overflow int cnt %d\n",over_cnt[mac_num]);
   printf("ringop count %d, mib %d\n",ringop_cnt[mac_num],*(init_mibs.d_mib[mac_num].macRingOpCount));

  if (mac_num == 0)
    ptr = (int *)0xa0000ba0;
  else
    ptr = (int *)0xa0000aa0;

  for (i=j=0; i < 20; j++,i++) {
    printf("%x  ",*ptr++);
    if (j == 3) {
	  j = 0;
      printf("\n");
	  }
  }
#endif
}
