 
/*********************************************************************
 *  File    bsi.h
 *
 *  Description
 *      This file containa all the definition for the BSI chip. It is
 *      local to this directory. No module outside this directory should
 *      include this head file. All the data types and definitions
 *      exported to the external modules are defined in fddi_drv.h
 *
 *  Copyright (c) 1992, 1993 Hughes Lan Systems
 *
 *  Author:   Jia-wei Jang
 *
 *  $Log:   /usr/shasta/pvcs/fddicon/drv/include/bsi.h_v  
 * 
 *    Rev 1.60.1.0   30 Jul 1993 09:42:38   jang
 * changed some function prototype due to DBD/PKT change
 * 
 *    Rev 1.60   08 Jul 1993 17:40:04   jang
 * changed BSI_TXSLEEP_TIME to 6000
 * 
 *    Rev 1.59   25 Jun 1993 13:55:52   gregs
 * new NEXT_QSLOT_ADDR, BSI2CPU_ADDRMASK, IND_LMOP_THRES for Fddi-eth bridge.
 * 
 *    Rev 1.58   23 Jun 1993 16:53:26   jang
 * increased the smt channel psp number
 * 
 *    Rev 1.57   07 Jun 1993 17:30:12   jang
 * deleted icr from bsibr_default_table and bsicon_default_table
 * 
 *    Rev 1.56   07 Jun 1993 17:05:36   jang
 * changed the data structures to move variables into internal data ram
 * 
 *    Rev 1.55   02 Jun 1993 08:58:30   jang
 * changed BSI_BOOT_MON_PSPS from 90 to 80
 * 
 *    Rev 1.54   20 May 1993 11:41:32   jang
 * added 2 new defines for beacon/claim frame FC
 * 
 *    Rev 1.53   18 May 1993 16:28:40   jang
 * deleted base_ptr fro bsi_reqq_type, changed
 * IND_LMOP_THRES, and BSI_SendReq()
 * 
 *    Rev 1.52   07 May 1993 17:16:04   jang
 * changed function prototype MEM_GetDataBuf()
 * 
 *    Rev 1.51   06 May 1993 14:28:52   jang
 * direct beacon
 * 
 *    Rev 1.50   05 May 1993 14:56:42   jang
 * added DRV_TXBUF_CNT
 * 
 *    Rev 1.49   29 Apr 1993 09:51:48   pedror
 * descrese the DBD number
 * 
 *    Rev 1.48   29 Apr 1993 09:48:38   gregs
 * Changed BSI2CPU_ADDR_MASK for the fddi-eth bridge.
 * 
 *    Rev 1.47   28 Apr 1993 14:56:32   jang
 * changed the return code type of BSI_SendReq()
 * 
 *    Rev 1.46   21 Apr 1993 15:26:22   jang
 * changed the function prototype
 * 
 *    Rev 1.45   20 Apr 1993 16:00:46   jang
 * changed the total dbd number to be allocated during application
 * 
 *    Rev 1.44   10 Apr 1993 16:16:28   jang
 * added icr config for boot
 * 
 *    Rev 1.43   06 Apr 1993 06:27:56   gregs
 * added buffer counts for fddi-eth bridge
 * 
 *    Rev 1.42   05 Apr 1993 07:37:16   jang
 * added more defines for request machine stop mode
 * 
 *    Rev 1.41   31 Mar 1993 18:33:42   jang
 * added one field in BSI_TYPE and changed CNF_LMOP_THRES
 * 
 *    Rev 1.40   25 Mar 1993 09:50:58   jang
 * added chip_test flag to indicate whether or not chip is functional
 * 
 *    Rev 1.39   23 Mar 1993 12:03:48   jang
 * chagned function prototype of BSI_WritePSP() and MEM_SetChPsp()
 * 
 *    Rev 1.38   18 Mar 1993 18:58:18   jang
 * 
 *    Rev 1.37   05 Mar 1993 10:01:46   jang
 * changed function ptototype of BSI_Init() and added default psps for the boot code
 * 
 *    Rev 1.36   02 Mar 1993 15:14:32   jang
 * 
 *    Rev 1.35   01 Mar 1993 19:38:18   jang
 * chnaged BSI_LLC_PSP_NUM from 20 to 120
 * 
 *    Rev 1.34   01 Mar 1993 15:13:00   jang
 * 
 *    Rev 1.33   23 Feb 1993 09:48:34   jang
 * added lmop threshold to ch 2
 * 
 *    Rev 1.32   22 Feb 1993 14:18:26   jang
 * moved some constants to fddimap.h to be shared with bridge code
 * 
 *    Rev 1.31   19 Feb 1993 18:04:48   jang
 * 
 *    Rev 1.30   16 Feb 1993 10:46:02   jang
 * changed function prototype BSI_ProcessCNF() and added BSI_DBD_ADDR_MASK
 * 
 *    Rev 1.29   12 Feb 1993 10:25:32   jlin
 * changed BSI_BUF_CTRL from 2 words to 8 words to align with
 * bsi burst boundary
 * 
 *    Rev 1.28   08 Feb 1993 14:19:40   jang
 * added function prototype
 * 
 *    Rev 1.27   06 Feb 1993 12:03:16   jang
 * fixed typerr BSI_PSP_SIZE
 * 
 *    Rev 1.26   06 Feb 1993 09:32:26   jang
 * added BSI_rcv_task() function prototype
 * 
 *    Rev 1.25   04 Feb 1993 19:22:04   jang
 * added BSI_BUF_CTRL
 * 
 *    Rev 1.24   03 Feb 1993 13:33:20   jang
 * this version works with lanhawk dumping data
 * 
 *    Rev 1.23   30 Jan 1993 16:28:04   jang
 * deleted some function prototype
 * 
 *    Rev 1.22   26 Jan 1993 11:13:40   jang
 * added one more define for ICR and more comment for indq
 * 
 *    Rev 1.21   19 Jan 1993 15:04:48   jang
 ********************************************************************/

#ifndef _BSI_H_
#define _BSI_H_

#include <fddi.h>
#include <drv.h>
#include <bmac.h>
#include <phy.h>
#include <pkt.h>
#include <dbd.h>
#include <fddimap.h>

/*---------------------------------------------------------------
 *      define MACROs
 *---------------------------------------------------------------
*/

#ifdef __FEBRIDGE	/* SHRAM is at 0x50000000 ************************* */

#define NEXT_QSLOT_ADDR(addr) (addr=((((addr)+8) & 0x500003ff) | ((addr) & 0x5ffffc00)))
#define BSI2CPU_ADDRMASK    0x50000000 

#else 				/* DRAM is at 0x10000000 ************************** */

/** advance to the next available descriptor queue slot **/
#define NEXT_QSLOT_ADDR(addr) (addr=((((addr)+8) & 0x100003ff) | ((addr) & 0x1ffffc00)))
/****
  convert a CPU address to BSI recognize address - word alignment.
  The DRAM starts from 1000.0000, but BSI always starts from 0000.0000, 
  therefore we have to mask off 29th-33rd bits.
****/
#define BSI2CPU_ADDRMASK    0x10000000 

#endif				/*                       ************************** */

#define CPU2BSI(addr)   ((uint32)addr & 0x0fffffff)
#define BSI2CPU(addr)   ((uint32)addr | BSI2CPU_ADDRMASK)
#define BSI2SRAM(addr)  (addr)

/* it convers a physical address in s to 28 bits BSI address & copy it to t */ 
#define SET_BSIADDR28(t, s)\
        ((t) = ((t) & ~BSI_ADDR28) | ((uint32)s & BSI_ADDR28))


#define SET_BSI_FIRST_DUD(t)\
        ((t) = ((t) & ~BSI_ADDRFL) | BSI_FIRST)
#define SET_BSI_LAST_DUD(t)\
        ((t) = ((t) & ~BSI_ADDRFL) | BSI_LAST)
#define SET_BSI_MIDDLE_DUD(t)\
        ((t) &= ~BSI_ADDRFL)
#define SET_BSI_ONLY_DUD(t)\
        ((t) = ((t) & ~BSI_ADDRFL) | BSI_ONLY)


/** check for whether or not the descriptor is NULL **/
#define NULL_DESC(ptr)  ((((GENERIC_DESC_TYPE*)ptr)->w0==0\
                            && ((GENERIC_DESC_TYPE*)ptr)->w1==0) ? TRUE : FALSE)

/** to nullify the descriptor **/
#define NULLIFY_DESC(ptr)    (((GENERIC_DESC_TYPE *)ptr)->w0 =\
                                       ((GENERIC_DESC_TYPE*)ptr)->w1 = 0L)
#define NULLIFY_DESC1ST(ptr)    (((GENERIC_DESC_TYPE *)ptr)->w0 = 0L)
#define NULLIFY_DESC2ND(ptr)    (((GENERIC_DESC_TYPE *)ptr)->w1 = 0L)




/***
FDDI frame type: mask 0-3 bits, only compare 4-7 bits
SMT frame 0x41 to 0x4f
MAC frame 0xc1 to 0xcf
LLC asy frame 0x50 to 0x57
LLC syn frame 0xd0 to 0xd7
***/
#define FDDI_FC_MASK            0X30
#define FC_LLC_FRAME            0X10
#define FC_LLC_OUR_FRAME        0X20
#define FC_SMT_FRAME            0X01
#define FC_MAC_FRAME            0xc0  
#define FC_BEACON_FRAME         0xc2
#define FC_CLAIM_FRAME          0xc3
#define BSI_SMT_FRAME           1
#define BSI_LLC_FRAME           2



/***
descriptor size and rx buffer size
***/
#define BSI_DESC_SLOT_SIZE  8       /* 1 descriptor slot has 8 bytes */
#define BSI_QUEUE_SIZE  1024        /* 1 descriptor is 1024 bytes */
#define BSI_QUEUE_MAX_SLOT    (BSI_QUEUE_SIZE / BSI_DESC_SLOT_SIZE)
#define BSI_QUEUE_NUM   12
#define BSI_QUEUE_BLK_SIZE  (BSI_QUEUE_SIZE * BSI_QUEUE_NUM)
#define BSI_BUF_CTRL        32  /* reserved 32 bytes for ctrl */
#define BSI_DBD_BUF_MASK    0xffffffe0  /* mask of db_buffer */
#define BSI_DBD_ADDR_MASK   0xfffff000  /* to the address of dbdp */
#define BSI_PSP_SIZE        (DRV_BUF_SIZE - BSI_BUF_CTRL)
#define BSI_DBD_BUF_SIZE    BSI_PSP_SIZE
#define BSI_MBOX_SIZE       4       /* bsi MBAR size is 4 bytes */

#define INC_PKT_CNT(addr) tmp= (int *)((int)addr & BSI_DBD_ADDR_MASK); \
                            tmp_cnt = *tmp; \
			    *tmp = tmp_cnt + 1

/***
bsi request and indicate channels and descriptors
***/
#define BSI_REQCH_NUM           2   /* 1 bsi has 2 request channels */
#define BSI_REQCH0              0
#define BSI_REQCH1              1

#define BSI_INDCH_NUM           3   /* 1 bsi has 3 indicate channels */
#define BSI_INDCH0              0
#define BSI_INDCH1              1
#define BSI_INDCH2              2

#define BSI_REQ_DESC_NUM        3   /* request descriptors number */
#define BSI_REQQ                0
#define BSI_CNFQ                1
#define BSI_ODUDQ               2
#define CNFQ_INIT_NUM           100  /* init cnfq number */

#define BSI_IND_DESC_NUM        2   /* indicate descriptors number */
#define BSI_IDUDQ               0
#define BSI_PSPQ                1
#define IDUDQ_INIT_NUM          CNFQ_INIT_NUM 


/** # of bufs put in each PSP descriptor initially **/
/* For the bridge */
#define SMT_PSP_NUM_BR         5     /* for smt/mac frames */
#define LLC_PSP_NUM_BR         5     /* for internal llc frames */
#define MON_PSP_NUM_BR         25
#define MON_PSP_BUFS_BR        (BSI_MON_PSP_NUM )
#define DBD_NUM_BR             64

/* For the concentrator */
/* About 1.3 Mbytes memory is needed */
#define BSI_SMT_PSP_NUM         35     /* for smt/mac frames */
#define BSI_LLC_PSP_NUM         15     /* for internal llc frames */
#define BSI_MON_PSP_NUM         (BSI_QUEUE_SIZE/BSI_DESC_SLOT_SIZE-24)
#define BSI_MON_PSP_BUFS        (BSI_MON_PSP_NUM * 2)
/* have 12 hidden DBDs in 6 psp descriptors */
#define BSI_DBD_NUM             ((BSI_SMT_PSP_NUM+BSI_LLC_PSP_NUM) * 2+12+DRV_TXBUF_CNT)
#define BSI_PSP_PIPELINE_NUM    2
#define BSI_BOOT_SMT_PSPS       10
#define BSI_BOOT_LLC_PSPS       23
#define BSI_BOOT_MON_PSPS       80


/***
2 BSI chips in the master card
***/
#define BSI_NUM_UNKNOWN     255
#define BSI_MAX_RESET_THRESHOLD     2

/***
    default threshold value for indicate channel 2
***/
#define CNF_LMOP_THRES          20
#define REQ_LMOP_THRES          5       /* update lmop counts for req ch */
#ifdef __FEBRIDGE
#define IND_LMOP_THRES          (SMT_PSP_NUM_BR / 3)
#else
#define IND_LMOP_THRES          (BSI_SMT_PSP_NUM / 3)
#endif
#define LMOP_THRES_MIN          1       /* minimum lmop threshold */
#define LMOP_THRES_PSP          10      /* lmop threshold for ch 2 */
#define LMOP_ADDR_MASK          0xfffff000  /* mask off last 12 bits */
#define IDUD_ADDR_MASK          0x3ffffffc  /* mask off fl bits and last 2 bits */

/* BSIs control bus base address */

#define BSI_MAX             MAX_MAC_COUNT      /* max number of BSI chip in the card */
#define BSI_0               PRIMARY_MAC
#define BSI_1               SECONDARY_MAC
#define BSI_A               BSI_1  /* secondary ring by default */
#define BSI_B               BSI_0  /* primary ring by default */

#define TOTAL_DESC_SIZE     (BSI_QUEUE_SIZE * (BSI_MAX * (BSI_REQCH_NUM * BSI_REQ_DESC_NUM + BSI_INDCH_NUM * BSI_IND_DESC_NUM)+1))

/*------------------------------------------------------------------------
 *   BSI descriptors base address, total 1 M bytes. The 1st 1K bytes are
 *   reserved. The 1st 4 bytes are reserved for BSI 1 mailbox, the next 4
 *   bytes are reserved for BSI 2 mailbox. the reset bytes are used for
 *   program control flow trace.
 *------------------------------------------------------------------------
 */
#define BSI_MEMORY_BASE     0x10000000
#define BSI_BLOCK_RESERVED  1024
#define BSI_BLOCK_BEGIN     (BSI_MEMORY_BASE + BSI_BLOCK_RESERVED)
#define BSI_BLOCK_END       0x100fffff
#define BSI_MAILBOX_BASE    BSI_MEMORY_BASE
#define BSI_1_MAILBOX       BSI_MAILBOX_BASE  
#define BSI_2_MAILBOX       (BSI_1_MAILBOX + BSI_MBOX_SIZE)
#define BSI_MEM_FREE        (BSI_2_MAILBOX + BSI_MBOX_SIZE)
#define BSI_REQ_UID_MAX     64
#define BSI_REQ_UID_MASK    0x3f
#define BSI_MAX_TXQ         (BSI_REQ_UID_MAX-24)  /* max tx pkts allowed in req */

/***
bsi revision number
***/
#define BSI_1_REV_A             0
#define BSI_1_REV_B             0x00000004
#define BSI_2_REV_A             0x00000040

/***
bsi channels and their status
***/
#define LMOP_WRITE              1
#define LMOP_READ               0
#define PTOP_WRITE              LMOP_WRITE
#define PTOP_READ               LMOP_READ


/***
    bsi channel type
***/
#define BSI_REQ_CH          0
#define BSI_IND_CH          1    

#define BSI_SET             1
#define BSI_RESET           0

#define BSI_OFFSET_MASK     0x00000fff  /* only last 12 bits are used */
#define BSI_ADDR28	    0x0fffffff

#define BSI_ADDRFL_MASK	    0xc0000000
#define BSI_FIRST	    0x80000000
#define BSI_MIDDLE          0L
#define BSI_LAST	    0x40000000
#define BSI_ONLY	    0xc0000000
#define BSI_IDUD_CNT_MASK  0x00001fff




#define POLL_REG_ONE_SEC    100

/****
    mode register 0
****/    
#define BSI_MR0		0x00	/* MR0 offset */
#define BSI_MR0_SMLB	0x80	/* Small Bursts */
#define BSI_MR0_SMLQ	0x40	/* Small Queue */
#define BSI_MR0_VIRT	0x20	/* Virtual Address Mode */
#define BSI_MR0_BIGEND	0x10	/* Big Endian Data Format */
#define BSI_MR0_FLOW	0x08	/* Flow Parity */
#define BSI_MR0_MRST	0x04	/* Master Reset */
#define BSI_MR0_FABCLK	0x02	/* Fast Abus Clock */
#define BSI_MR0_TEST	0x01	/* Test Mode */

/***
    MR1     only for BSI-2
***/
#define BSI_MR1         0x01
#define BSI_MR1_DEFAULT 0       /* always set this reg to 0 */
#define BSI_MR1_EAM     0X01
#define BSI_MR1_ASM     0X04
#define BSI_MR1_ATM     0X08

/****
    PCAR
****/
#define BSI_PCAR	0x02	/* PCAR offset */
#define BSI_PCAR_BP	0xc0	/* Byte Pointer */
#define BSI_PCAR_PTRW	0x20	/* PTOP Read/Write */
#define BSI_PCAR_A	0x1f	/* Pointer RAM Address */


#define BSI_MBAR	0x03	/* MBAR offset */
    
#define BSI_MAR		0x04	/* MAR offset */
#define BSI_MAR_STA	0x80	/* State Attention Register */
#define BSI_MAR_NSA	0x40	/* No Space Attention Register */
#define BSI_MAR_SVA	0x20	/* Service Attention Register */
#define BSI_MAR_RQA	0x10	/* Request Attention Register */
#define BSI_MAR_INA	0x08	/* Indicate Attention Register */
#define BSI_MAR_RES	0x07	/* Reserved */

#define BSI_MNR		0x05	/* MNR offset */
#define BSI_MNR_STAN	0x80	/* State Attention Register Notify */
#define BSI_MNR_NSAN	0x40	/* No Space Attention Register Notify */
#define BSI_MNR_SVAN	0x20	/* Service Attention Register Notify */
#define BSI_MNR_RQAN	0x10	/* Request Attention Register Notify */
#define BSI_MNR_INAN	0x08	/* Indicate Attention Register Notify */
#define BSI_MNR_RES	0x07	/* Reserved */

#define BSI_STAR	0x06	/* STAR offset */
#define BSI_STAR_ERR	0x80	/* Error */
#define BSI_STAR_BPE	0x40	/* BMAC Parity Error */
#define BSI_STAR_CPE	0x20	/* Control Bus Parity Error */
#define BSI_STAR_CWI	0x10	/* Conditional Write Inhibit */
#define BSI_STAR_CMDE	0x08	/* Command Error */
#define BSI_STAR_SPSTOP	0x04	/* Status/Space Stop */
#define BSI_STAR_RQSTOP	0x02	/* Request Stop */
#define BSI_STAR_INSTOP	0x01	/* Indicate Stop */

#define BSI_STNR	0x07	/* STNR offset */
#define BSI_STNR_ERRN	0x80	/* Error */
#define BSI_STNR_BPEN	0x40	/* BMAC Parity Error */
#define BSI_STNR_CPEN	0x20	/* Control Bus Parity Error */
#define BSI_STNR_CWIN	0x10	/* Conditional Write Inhibit */
#define BSI_STNR_CMDEN	0x08	/* Command Error */
#define BSI_STNR_SPSTOPN	0x04	/* Status/Space Stop */
#define BSI_STNR_RQSTOPN	0x02	/* Request Stop */
#define BSI_STNR_INSTOPN	0x01	/* Indicate Stop */

#define BSI_SAR		0x08	/* SAR offset */
#define BSI_SAR_RES	0xf0	/* Reserved */
#define BSI_SAR_ABR0	0x08	/* RCHN0 Abort Request */
#define BSI_SAR_ABR1	0x04	/* RCHN1 Abort Request */
#define BSI_SAR_LMOP	0x02	/* Limit RAM Operation */
#define BSI_SAR_PTOP	0x01	/* Pointer RAM Operation */

#define BSI_SNR		0x09	/* SNR offset */
#define BSI_SNR_RES	0xf0	/* Reserved */
#define BSI_SNR_ABR0N	0x08	/* RCHN0 Abort Request Notify */
#define BSI_SNR_ABR1N	0x04	/* RCHN1 Abort Request Notify */
#define BSI_SNR_LMOPN	0x02	/* Limit RAM Operation Notify */
#define BSI_SNR_PTOPN	0x01	/* Pointer RAM Operation Notify */

#define BSI_NSAR	0x0a	/* NSAR offset */
#define BSI_NSAR_NSR0	0x80	/* No Status Space on RCHN0 */
#define BSI_NSAR_NSR1	0x40	/* No Status Space on RCHN1 */
#define BSI_NSAR_LDI0	0x20	/* Low Data Space on ICHN0 */
#define BSI_NSAR_NSI0	0x10	/* No Status Space on ICHN0 */
#define BSI_NSAR_LDI1	0x08	/* Low Data Space on ICHN1 */
#define BSI_NSAR_NSI1	0x04	/* No Status Space on ICHN1 */
#define BSI_NSAR_LDI2	0x02	/* Low Data Space on ICHN2 */
#define BSI_NSAR_NSI2	0x01	/* No Status Space on ICHN2 */

#define BSI_NSNR	0x0b	/* NSNR offset */
#define BSI_NSNR_NSR0N	0x80	/* No Status Space on RCHN0 Notify */
#define BSI_NSNR_NSR1N	0x40	/* No Status Space on RCHN1 Notify */
#define BSI_NSNR_LDI0N	0x20	/* Low Data Space on ICHN0 Notify */
#define BSI_NSNR_NSI0N	0x10	/* No Status Space on ICHN0 Notify */
#define BSI_NSNR_LDI1N	0x08	/* Low Data Space on ICHN1 Notify */
#define BSI_NSNR_NSI1N	0x04	/* No Status Space on ICHN1 Notify */
#define BSI_NSNR_LDI2N	0x02	/* Low Data Space on ICHN2 Notify */
#define BSI_NSNR_NSI2N	0x01	/* No Status Space on ICHN2 Notify */

#define BSI_LAR		0x0c	/* LAR offset */
#define BSI_LAR_LRA	0xf0	/* Limit RAM Register Address */
#define BSI_LAR_LMRW	0x08	/* LMOP Read/Write */
#define BSI_LAR_RES	0x06	/* Reserved */
#define BSI_LAR_LRD8	0x01	/* Limit RAM Data Bit 8 */

#define BSI_LDR		0x0d	/* LDR offset */

#define BSI_RAR		0x0e	/* RAR offset */
#define BSI_RAR_USRR0	0x80	/* Unserviceable Request on RCHN0 */
#define BSI_RAR_RCMR0	0x40	/* Request Complete on RCHN0 */
#define BSI_RAR_EXCR0	0x20	/* Exception on RCHN0 */
#define BSI_RAR_BRKR0	0x10	/* Breakpoint on RCHN0 */
#define BSI_RAR_USRR1	0x08	/* Unserviceable Request on RCHN1 */
#define BSI_RAR_RCMR1	0x04	/* Request Complete on RCHN1 */
#define BSI_RAR_EXCR1	0x02	/* Exception on RCHN1 */
#define BSI_RAR_BRKR1	0x01	/* Breakpoint on RCHN1 */

#define BSI_RNR		0x0f	/* RNR offset */
#define BSI_RNR_USRR0N	0x80	/* Unserviceable Request on RCHN0 Notify */
#define BSI_RNR_RCMR0N	0x40	/* Request Complete on RCHN0 Notify */
#define BSI_RNR_EXCR0N	0x20	/* Exception on RCHN0 Notify */
#define BSI_RNR_BRKR0N	0x10	/* Breakpoint on RCHN0 Notify */
#define BSI_RNR_USRR1N	0x08	/* Unserviceable Request on RCHN1 Notify */
#define BSI_RNR_RCMR1N	0x04	/* Request Complete on RCHN1 Notify */
#define BSI_RNR_EXCR1N	0x02	/* Exception on RCHN1 Notify */
#define BSI_RNR_BRKR1N	0x01	/* Breakpoint on RCHN1 Notify */

#define BSI_R0CR0	0x10	/* R0CR0 offset */

#define BSI_R1CR0	0x11	/* R1CR0 offset */
#define BSI_RxCR0_TT	0xc0	/* Transmit Threshhold */
#define BSI_RxCR0_PRE	0x20	/* Preempt/Prestage */
#define BSI_RxCR0_HLD	0x10	/* Hold */
#define BSI_RxCR0_FCT	0x08	/* Frame Control Transparency */
#define BSI_RxCR0_SAT	0x04	/* Source Address Transparency */
#define BSI_RxCR0_VST	0x02	/* Void Stripping */
#define BSI_RxCR0_FCS	0x01	/* Frame Check Sequence Disable */

#define BSI_R0EFSR	0x12	/* R0EFSR offset */

#define BSI_R1EFSR	0x13	/* R1EFSR offset */
#define BSI_RxEFSR_VDL	0x80	/* Valid Data Length */
#define BSI_RxEFSR_VFCS	0x40	/* Valid FCS */
 #define BSI_RxEFSR_EE	0x30	/* Expected E Indicator (Error Detected) */
#define BSI_RxEFSR_EA	0x0c	/* Expected A Indicator (Address Recognized) */
#define BSI_RxEFSR_EC	0x03	/* Expected C Indicator (Frame Copied) */

#define BSI_IAR		0x14	/* IAR offset */
#define BSI_IAR_RES	0xc0	/* Reserved */
#define BSI_IAR_EXCI0	0x20	/* Exception on ICHN0 */
#define BSI_IAR_BRKI0	0x10	/* Breakpoint on ICHN0 */
#define BSI_IAR_EXCI1	0x08	/* Exception on ICHN1 */
#define BSI_IAR_BRKI1	0x04	/* Breakpoint on ICHN1 */
#define BSI_IAR_EXCI2	0x02	/* Exception on ICHN2 */
#define BSI_IAR_BRKI2	0x01	/* Breakpoint on ICHN0 */

#define BSI_INR		0x15	/* INR offset */
#define BSI_INR_RES	0xc0	/* Reserved */
#define BSI_INR_EXC0N	0x20	/* Exception on ICHN0 Notify */
#define BSI_INR_BRK0N	0x10	/* Breakpoint on ICHN0 Notify */
#define BSI_INR_EXC1N	0x08	/* Exception on ICHN1 Notify */
#define BSI_INR_BRK1N	0x04	/* Breakpoint on ICHN1 Notify */
#define BSI_INR_EXC2N	0x02	/* Exception on ICHN2 Notify */
#define BSI_INR_BRK2N	0x01	/* Breakpoint on ICHN0 Notify */

#define BSI_ITR		0x16	/* ITR offset */
#define ITR_THRESHOLD_DEFAULT           1

#define BSI_IMR		0x17	/* IMR offset */
#define BSI_IMR_SM	0x40	/* Sort Mode - EXTERNAL-INTERNAL */
#define BSI_IMR_SKIP	0x20	/* Skip Enabled */
#define BSI_IMR_FPP	0x10	/* Frame Per Page */
#define BSI_IMR_BOT1	0x04	/* Breakpoint on Threshold for ICHN1 */
#define BSI_IMR_BOT2	0x08	/* Breakpoint on Threshold for ICHN2 */
#define BSI_IMR_BOB	0x02	/* Breakpoint on Burst */
#define BSI_IMR_BOS	0x01	/* Breakpoint on Service Opportunity */

#define BSI_ICR            0x18     /* ICR offset */
#define BSI_ICR_CC_PRO     0xdb    /* copy promiscuously */
#define BSI_ICR_CC0_DA     0x9b    /* copy only recoginze DA */
#define BSI_ICR_CC0_NO     0x1b    /* do not copy */
#define BSI_ICR_CC0_ONLY   0x5b    /* copy if DA & ~SA */
#define BSI_ICR_CC1_DA     0xd3    /* copy only recoginze DA */
#define BSI_ICR_CC1_NO     0xc3    /* do not copy */
#define BSI_ICR_CC1_ONLY   0xcb    /* copy if DA & ~SA */
#define BSI_ICR_CC2_DA     0xda    /* copy only recoginze DA */
#define BSI_ICR_CC2_NO     0xd8    /* do not copy */
#define BSI_ICR_CC2_ONLY   0xd9    /* copy if DA & ~SA */
#define BSI_ICR_CC0	0xc0	/* Copy Control ICHN0 */
#define BSI_ICR_RES0	0x20	/* Reserved */
#define BSI_ICR_CC1	0x18	/* Copy Control ICHN1 */
#define BSI_ICR_RES1	0x04	/* Reserved */
#define BSI_ICR_CC2	0x03	/* Copy Control ICHN2 */
#define BSI_CP_PROMIS   DRV_COPY_PROMISC
#define BSI_CP_RESTORE  DRV_COPY_NON_PROMISC

#define BSI_IHLR	0x19	/* IHLR offset */
#define BSI_IHLR_CON_DEFAULT    0

#define BSI_ACR         0x1A
#define BSI_R0CR1       0X1B
#define BSI_R1CR1       0x1c
#define BSI_CMP		0x1f	/* CMP offset */

/* IDUD Frame Status definitions: C, A, and E indicators */
#define BSI_IDUD_FRS_MASK               0x00ff0000
#define IDUD_FRS_SET                    0x02
#define IDUD_FRS_RESET                  0x01
#define IDUD_FRS_TERMINATE              0x03
#define IDUD_FRS_NONE                   0x00

/* IDUD Indicate Status definitions */
#define BSI_IDUD_IS_LAST_PAGEX           0
#define BSI_IDUD_IS_PAGEX                1
#define BSI_IDUD_IS_HEADER               2
#define BSI_IDUD_IS_HEADER_PAGEX         3
#define BSI_IDUD_IS_INTERMEDIATE         4
#define BSI_IDUD_IS_BURST                5
#define BSI_IDUD_IS_THRESHOLD            6
#define BSI_IDUD_IS_SERVOP               7
#define BSI_IDUD_IS_NO_DATA_SPACE        8
#define BSI_IDUD_IS_NO_HEADER_SPACE      9
#define BSI_IDUD_IS_NO_INFO_SPACE       10
#define BSI_IDUD_IS_INCOMPLETE_INFO     11
#define BSI_IDUD_IS_FIFO_OVERRUN        12
#define BSI_IDUD_IS_BAD_FRAME           13
#define BSI_IDUD_IS_PARITY_ERR          14
#define BSI_IDUD_IS_INTERNAL_ERR        15

 /* Request Class Values */
#define BSI_REQ_RQCLS_NONE	0	/* None */
#define BSI_REQ_RQCLS_APR1	1	/* Async Priority 1 */
#define BSI_REQ_RQCLS_APR2	2	/* Async Priority 2 */
#define BSI_REQ_RQCLS_APR3	3	/* Async Priority 3 */
#define BSI_REQ_RQCLS_SYN	4	/* Sync */
#define BSI_REQ_RQCLS_IMM	5	/* Immediate, issue no token */
#define BSI_REQ_RQCLS_IMMN	6	/* Immediate, issue non-restrict tok */
#define BSI_REQ_RQCLS_IMMR	7	/* Immediate, issue restricted token */
#define BSI_REQ_RQCLS_ASYN	8	/* Async */
#define BSI_REQ_RQCLS_RBEG	9	/* Restricted Begin */
#define BSI_REQ_RQCLS_REND	10	/* Restricted End */
#define BSI_REQ_RQCLS_RCNT	11	/* Restricted Continue */
#define BSI_REQ_RQCLS_ASYND	12	/* Async without THT */
#define BSI_REQ_RQCLS_RBEGD	13	/* Restricted Begin without THT */
#define BSI_REQ_RQCLS_RENDD	14	/* Restricted End without THT */
#define BSI_REQ_RQCLS_RCNTD	15	/* Restricted Continue with THT */
 /* THT = Token Hold Timer */

 /* Confirmation Class Values */
 /* CNF only on exception */
#define BSI_REQ_CNFCLS_NONE	0
 /* Transmitter confirm, CNF on exception or completion */
#define BSI_REQ_CNFCLS_TEND	1
 /* Transmitter confirm, CNF on exception, completion or intermediate */
#define BSI_REQ_CNFCLS_TINT	3
 /* Transmitter confirm, CNF on exception or completion */
#define BSI_REQ_CNFCLS_FEND	5
 /* Full confirm, CNF on exception, completion or intermediate */
#define BSI_REQ_CNFCLS_FINT	7
 /* Transmitter confirm, CNF on exception or completion, repeat frame */
#define BSI_REQ_CNFCLS_TENDR	9
 /* Transmitter confirm, CNF on exc, compl or intermed, repeat frame */
#define BSI_REQ_CNFCLS_TINTR	11
 /* CNF only on exception, repeat frame */
#define BSI_REQ_CNFCLS_NONER	12
 /* Full confirm, CNF on exception or completion, repeat frame */
#define BSI_REQ_CNFCLS_FENDR	13
 /* Full confirm, CNF on exc, compl or intermediate, repeat frame */
#define BSI_REQ_CNFCLS_FINTR	15

 /* CNF Request Status Values */
#define BSI_CNF_RS_NONE		0
#define BSI_CNF_RS_PREEMPTED	1
#define BSI_CNF_RS_PART_DONE	2
#define BSI_CNF_RS_SERV_LOSS	3
#define BSI_CNF_RS_COMPLETED_BC	5
#define BSI_CNF_RS_COMPLETED_OK	6
#define BSI_CNF_RS_BAD_CONF	7
#define BSI_CNF_RS_UNDERRUN	8
#define BSI_CNF_RS_HOST_ABORT	9
#define BSI_CNF_RS_BAD_RINGOP	10
#define BSI_CNF_RS_MAC_ABORT	11
#define BSI_CNF_RS_TIMEOUT	12
#define BSI_CNF_RS_MAC_RESET	13
#define BSI_CNF_RS_CON_FAILURE	14
#define BSI_CNF_RS_FATAL_ERROR	15

#define BSI_CNF_FT_ANY_OTHER	0
#define BSI_CNF_FT_TOKEN	1
#define BSI_CNF_FT_OTHER_VOID	2
#define BSI_CNF_FT_MY_VOID	3

 /* E/A/C Indicator Values */
#define BSI_INDICATOR_NONE	0
#define BSI_INDICATOR_R		1
#define BSI_INDICATOR_S		2
#define BSI_INDICATOR_T		3

 /* Terminating Condition Values */
#define BSI_TC_OTHER		0	/* other condition */
#define BSI_TC_ED		1	/* ED Symbol */
#define BSI_TC_FORMAT_ERR	2	/* Frame Format Error */
#define BSI_TC_FRAME_STRIPPED	3	/* Frame Stripped */

/* Number of Descriptors Pipelined for output */
#define BSI_CNF_PIPELINE_SIZE	2
#define BSI_IDUD_PIPELINE_SIZE	2
    

/* BSI control bus registers */
typedef struct bsi_cbus {
   byte mr0;		/* Mode Register 0 */
   byte mr1;	        /* mord register 1. BSI-2 */
   byte pcar;		/* Pointer RAM Control and Address Register */
   byte mbar;		/* Mailbox Address Register */
   byte mar;		/* Master Attention Register */
   byte mnr;		/* Master Notify Register */
   byte star;		/* State Attention register */
   byte stnr;		/* State Notify register */
   byte sar;		/* Service Attention register */
   byte snr;		/* Service Notify register */
   byte nsar;		/* No Space Attention register */
   byte nsnr;		/* No Space Notify register */
   byte lar;		/* Limit Address register */
   byte ldr;		/* Limit Data register */
   byte rar;		/* Request attention */
   byte rnr;		/* Request notify */
   byte r0cr0;		/* RCHN0 Request configuration register */
   byte r1cr0;		/* RCHN1 Request configuration register */
   byte r0efsr; 	    /* RCHN0 Request Expected Frame Status */
   byte r1efsr;	        /* RCHN1 Request Expected Frame Status */
   byte iar;	        /* Indicate Attention register */
   byte inr;            /* Indicate Notify register */
   byte itr;	        /* Indicate Threshold */
   byte imr;	        /* Indicate Mode register */
   byte icr;	        /* Indicate Configuration register */
   byte ihlr;	        /* Indicate Header Length */
   byte acr;	        /* Address Configuration Reg */
   byte r0cr1;	        /* Request Channel 0 configuration register 1 */
   byte r1cr1;	        /* Request Channel 1 configuration register 1 */
   byte res_1d;	        /* Reserved */
   byte res_1e;	        /* Reserved */
   byte cmp;	        /* Conditional Write Comparison register */
} BSI_REG_TYPE;


/***
some of the BSI registers image
***/
typedef struct bsi_ctrl_struct {
   byte mr0;		/* Mode Register 0 */
   byte itr;	        /* Indicate Threshold */
   byte inr;
   byte mnr;            /* master notify register, use to restore mnr */
} BSI_CTRL_TYPE;



extern FDDI_CON_TYPE fddi_board;



/***
    a data structure describes IDUD and PSP queue 
***/    
typedef struct inram_ind_queue {
   uint32       base;  /* start address of q. assign when initialize */
   uint32       next_slot;  /* point to the next slot to be processed */
   uint32       ql; /* point to the last slot BSI can write new IDUD */
   uint32       qp;  /* point to the slot to read rx IDUD */
                            /* pspq used it to point to the next pspq
			       to be read by BSI to get the dbd ptr
			       for the address to rx pkt */
   uint32       lmop_threshold;  /* # of desc to write before updating QLR */
   uint32       lmop_count;  /* # of desc has written after last QLR updated */
   uint32       psp_count;   /* # of psps in this queue */
   uint32       last_res;    /* last reserve variable */
}   BSI_INDQ_TYPE;

#define REQ_MARK        0x4d514552
typedef struct req_msg_ { /* store the PKTs in the req desc */
  struct req_msg_ *next;
  int req_mark;
  struct req_msg_ *link;
  char *bufp;   /* pointer to the PKT/DBD */
} BSI_REQ_MSG;

#define CNF_NORMAL          0   /* for cnf status */
#define CNF_NO_SPACE        0x01
typedef struct nf_req_queue {
   uint32       base;  /* start address of q. assign when initialize */
   uint32       next_slot;  /* point to the next slot to be processed */
   uint32       qp;   /* point to the last free slot */
   uint32       ql;  /* point to the new slot to write */
   uint16       lmop_threshold;  /* # of desc to write before updating QLR */
   uint16       lmop_count;  /* # of desc has written after last QLR updated */
   byte         uid;             /* user id put in queue */
   byte         status;     
   uint16       filler_s;
   int          last_filler;
}   BSI_REQQ_TYPE;


/***
    double link list to hold the buffer list for each indicate channel
***/
#if 0
typedef struct _buf_list_ {
  struct _buf_list_ *next;
  struct _buf_list_ *prev;
  byte *mem_ptr;         /* pointer to the structure contain data buf */
  void (*free_func)();    /* the function to free buf_ptr */
  uint32 buf_addr;       /* start address of the data buffer */
} IND_BUF_LIST;
#define MAX_INDBUFS             130  
#endif

/***
    channel information and control
***/    
typedef struct bsi_ind_info {
    uint32 status;    /* BSI_CH_RUN | BSI_CH_STOP  */
    byte ipi_addr;        /* offset of IPU Pointer */
    byte npi_addr;        /* offset of next psp */
    uint32 no_space_cnt;  /* # of no space interrupt */
    uint32 low_space_cnt;
} BSI_INDCH_INFO;

/****
  this data structure is put in internal data ram
****/
typedef struct inram_ind_info {
  BSI_INDQ_TYPE *idudq;
  BSI_INDQ_TYPE *pspq;
  BSI_INDCH_INFO    *info;
  byte iqli_addr;     /* offset of IDUD queue limit reg */
  byte pqli_addr;     /* offset of PSP queue limit reg */
  byte iqpi_addr;       /* offset of IDUD Queue Pointer */
  byte pqpi_addr;       /* offset of PSP Queue Pointer */
  /* END. NO MORE VARIABLE */
} BSI_INCH_TYPE;


typedef struct _ind_ram {
    byte iqli_addr;     /* offset of IDUD queue limit reg */
    byte pqli_addr;     /* offset of PSP queue limit reg */
    byte ipi_addr;        /* offset of IPU Pointer */
    byte iqpi_addr;       /* offset of IDUD Queue Pointer */
    byte pqpi_addr;       /* offset of PSP Queue Pointer */
    byte npi_addr;        /* offset of next psp */
} IND_RAM_REGS;

typedef struct sram_req_info {
    byte    rqlr_addr;              /* offset of REQ queue limit reg */
    byte    cqlr_addr;              /* offset of CNF queue limit reg */
    byte    opr_addr;               /* pointer ram ODU Pointer address */
    byte    olpr_addr;              /* ODUD List Pointer */
    byte    cqpr_addr;              /* CNF Queue Pointer */
    byte    rqpr_addr;              /* REQ Queue Pointer */
    short   queued_cnt;             /* # of oduds in process */
    uint32  status;   /* BSI_CH_RUN | BSI_CH_STOP  */
    uint32  err_count;              /* error counts in this channel */
    uint32  no_space_cnt;           /* cnf descriptor has no space */
    bool    excep_flag;             /* TRUE when exception occur */
   BSI_REQ_MSG  *req_head;
   BSI_REQ_MSG  *req_tail; 
} BSI_REQCH_INFO;

typedef struct inram_sap_req_info {
    BSI_REQQ_TYPE *reqq;            /* queue # == BSI_REQ_DESC_NUM */
    BSI_REQQ_TYPE *cnfq;
    BSI_REQQ_TYPE *odudq;
    BSI_REQCH_INFO *info_ptr;
    /* NO MORE VARIABLE */
} BSI_REQCH_TYPE;

typedef struct _req_ram {
    byte  rqlr_addr;              /* offset of REQ queue limit reg */
    byte  cqlr_addr;              /* offset of CNF queue limit reg */
    byte  opr_addr;               /* pointer ram ODU Pointer address */
    byte  olpr_addr;              /* ODUD List Pointer */
    byte  cqpr_addr;              /* CNF Queue Pointer */
    byte  rqpr_addr;              /* REQ Queue Pointer */
}   REQ_RAM_REGS;

/***
this data structure holds the default value for BSI chip during
initialization
***/
typedef struct _bsi_default_table {
    byte    mr0;
    byte    mr1;
    byte    mnr;
    byte    stnr;
    byte    snr;
    byte    nsnr;
    byte    rnr;
    byte    r0cr0;
    byte    r1cr0;
    byte    r0efsr;
    byte    r1efsr;
    byte    inr;
    byte    itr;
    byte    imr;
    byte    ihlr;
    byte    acr;
    byte    r0cr1;
    byte    r1cr1;
} BSI_DEFAULT_TABLE;

/****
  WARNING: it will be put in internal data ram. only 8 words are allowed.
****/
typedef struct bsi_addr {
    BSI_REG_TYPE    *base_addr;       /* ctrl bus base registers */
    BSI_REQCH_TYPE   *rch0;             /* request channel 0 */
    BSI_REQCH_TYPE   *rch1;             /* request channel 1 */
    BSI_INCH_TYPE   *ich0;             /* indicate channel 0 */
    BSI_INCH_TYPE   *ich1;             /* indicate channel 1 */
    BSI_INCH_TYPE   *ich2;             /* indicate channel 2 */
    int          timer_count;       /* timer to check tx sleep */
#define BSI_TXSLEEP_TIME      6000  /* 6 second, 1 ms unit */
    int             last_res;
} BSI_ADDR_TYPE;

typedef struct bsi_db {
    uint32    mbox_addr;               /* mailbox address */
    uint32  rev_num;                   /* revision number */
    uint32  reset_count;               /* # of reset due to internal error */
    uint32  isr;                        /* idud shadow register offset */
    uint32  osr;                        /* odud shadow register offset */
    uint32  icr_val;                    /* incicate configuration value */
    uint32  icr_val_boot;               /* incicate config value for boot */
    uint32  ich0_psp_num;               /* # of PSP to put in ind channel 0 */
    uint32  ich1_psp_num;
    uint32  ich2_psp_num;
    uint32  ich0_psp_boot;               /* # of PSP to put in ind channel 0 */
    uint32  ich1_psp_boot;
    uint32  ich2_psp_boot;
    bool    self_test;                  /* self test flag */
    bool    do_scrub;                   /* doing scrubbing */
    bool    chip_test;                  /* 1 if chip is OK */
    bool    direct_beacon;              /* 1 if sending direct beacon */
    BSI_CTRL_TYPE   *image_ptr;
    uint32  bsi_status;               /* bsi state machine status */
#define BSI_IND_STOP    1  /* indicate machine stop */
#define BSI_REQ_STOP    0x00000002  /* request machine stop */
#define BSI_SP_STOP     0x00000004
#define REQ_STOP_RINGOP 0x00000008  /* req machine stop due to ringop reset */
#define REQ_STOP_NOCNF  0x00000010
} BSI_TYPE;


#define REQ_STOP    (BSI_REQ_STOP | REQ_STOP_RINGOP | REQ_STOP_NOCNF)

/***********************************
This structure stores all the unchanged data for BSI.
***********************************/
typedef struct bsi_default_db {
    BSI_REG_TYPE    *base_addr;       /* ctrl bus base registers */
    uint32  isr;                        /* idud shadow register offset */
    uint32  osr;                        /* odud shadow register offset */
    uint32  ich0_psp_num;               /* # of PSP to put in ind channel 0 */
    uint32  ich1_psp_num;               /* # of PSP to put in ind channel 1 */
    uint32  ich2_psp_num;               /* # of PSP to put in ind channel 2 */
    uint32  ich0_psp_boot;               /* # of PSP to put in ind channel 0 */
    uint32  ich1_psp_boot;               /* # of PSP to put in ind channel 1 */
    uint32  ich2_psp_boot;               /* # of PSP to put in ind channel 2 */
    uint32  icr_val;                    /* incicate configuration value */
    uint32  icr_val_boot;               /* ind config for booter */
} BSI_INIT_TYPE;


 /* general BSI descriptor */
typedef struct descr {
   uint32 w0;
   uint32 w1;
}  GENERIC_DESC_TYPE;


typedef struct idud_descr {
   unsigned    cnt:13;		/* Byte Count of IDU */
   unsigned    res:2;		/* Reserved */
   unsigned    vc:1;		/* VCOPY */
   unsigned    frs_c:2;		/* C Indicator */
   unsigned    frs_a:2;		/* A Indicator */
   unsigned    frs_e:2;		/* E Indicator */
   unsigned    frs_vfcs:1;	/* Valid FCS */
   unsigned    frs_vdl:1;	/* Valid Data Length */
   unsigned    fra_tc:2;	/* Terminating Condition */
   unsigned    fra_aflag:1;	/* AFLAG */
   unsigned    fra_mflag:1;	/* MFLAG */
   unsigned    is:4;		/* Indicate Status */
   uint32      loc;		/* location of IDU */
} IDUD_DESCR_TYPE;

typedef struct psp_descr {
   unsigned    cnt:13;		/* Byte Count */
   unsigned    res1b:3;		/* reserved */
   unsigned    res1a:16;	/* reserved */
   uint32      loc;		/* Location */
} PSP_DESCR_TYPE;

typedef struct odud_descr {
   unsigned    cnt:13;		/* Byte Count of ODU */
   unsigned    res1b:3;		/* Reserved */
   unsigned    res1a:16;	/* Reserved */
   uint32      loc;		/* Location of ODU */
} ODUD_DESCR_TYPE;

typedef struct req_descr {
   unsigned    fc:8;		/* Frame Control */
   unsigned    rqcls:4;		/* Request/Release Class */
   unsigned    cnfcls:4;	/* Confirmation Class */
   unsigned    size:8;		/* Size of REQ in Frames */
   unsigned    uid:6;		/* User Identification */
   unsigned    res:2;		/* Reserved */
   uint32      loc;		/* Location of ODUD array */
} REQ_DESCR_TYPE;

typedef struct cnf_descr {
   /* first word */
   unsigned    cfc:8;		/* Confirmed Frame Count */
   unsigned    tfc:8;		/* Transmitted Frame Count */
   unsigned    frs_c:2;		/* C Indicator */
   unsigned    frs_a:2;		/* A Indicator */
   unsigned    frs_e:2;		/* E Indicator */
   unsigned    frs_vfcs:1;	/* Valid FCS */
   unsigned    frs_vdl:1;	/* Valid Data Length */
   unsigned    fra_tc:2;	/* Terminating Condition */
   unsigned    fra_aflag:1;	/* AFLAG */
   unsigned    fra_mflag:1;	/* MFLAG */
   unsigned    rs:4;		/* Request Status */
   /* second word */
   unsigned    res:8;		/* Reserved */
   unsigned    cs_ft:2;		/* Frame Type */
   unsigned    cs_f:1;		/* Full Confirm */
   unsigned    cs_u:1;		/* Unexpected Frame Status */
   unsigned    cs_p:1;		/* Parity */
   unsigned    cs_e:1;		/* Exception */
   unsigned    cs_r:1;		/* Ring-Op */
   unsigned    cs_t:1;		/* Transmit Class */
   unsigned    fc:8;		/* Frame Control */
   unsigned    uid:6;		/* User Identification */
   unsigned    fl:2;		/* last bit */
}  CNF_DESCR_TYPE;

#if 0
/****
 contains the buffer write to PSP for receive buffer. The data buffer is
 alignment with PSP_BUF_ALIGN bytes.
****/
typedef struct psp_list {
    struct psp_list *next;
    struct psp_list *prev;
    byte *buf_ptr;  /* used to store the rx pkt. alignment to 4k */
} PSP_LIST;
#endif
/***
function prototypes
***/

extern status_type BSI_ResetBsi(uint32);


/* drv_mem.c */
extern uint32 MEM_SetChPsp(uint32,uint32,BSI_INCH_TYPE *,uint32,bool);
extern int MEM_InitMem(void);
extern void MEM_ResetMem(void);
extern int MEM_InitPSPPool(char *,int);
extern char *MEM_GetFreePsp(void);
extern byte *MEM_GetDataBuf(void);
extern void MEM_FreeTxDBD(uint32);
extern void MEM_DbdWaiting(uint32, DBD *);
extern void MEM_PspWaiting(uint32,uint32,uint32);
extern void MEM_FreeDataBuf(uint32);
extern void MEM_SetBsiPsp(uint32);
extern status_type MEM_InitSharedPspPool(void);
extern DBD *MEM_GetIndBufs(BSI_INCH_TYPE *,uint32);
extern void MEM_FreePSPBuf(uint32);
extern char *MEM_GetFreePsp(void);
extern int MEM_InitPSPPool(char *,int);
extern PKT *MEM_GetPkt(void);


/* bsi.c */
extern status_type BSI_Init(uint32,int);
extern void BSI_CMTTestInit(void);
extern status_type BSI_ResetBsi(uint32);
extern status_type BSI_WritePsp(uint32, BSI_INCH_TYPE *,uint32,bool);
extern void BSI_SetInr(uint32,uint32,volatile byte *);
extern bool BSI_ChkCh2NoSpace(uint32,bool);
extern status_type BSI_WriteCNF(uint32,BSI_REQCH_TYPE *,bool);
extern bool BSI_QueueFull(uint32,uint32);
extern int BSI_SendReq(uint32,uint32,BSI_REQCH_TYPE *,PKT *,bool);
extern status_type BSI_ChkIdudStatus(IDUD_DESCR_TYPE *,uint32);
extern void BSI_ProcessCNF(uint32);
extern bool BSI_DoDiag(uint32,uint32,uint32);
extern void BSI_ShutdownIndChnl(uint32,uint32);
extern status_type BSI_WriteIDUD (uint32,BSI_INCH_TYPE *,bool);
extern bool BSI_CNF_IDUD_Qfull(uint32,uint32);
/* extern void BSI_rcv_task(void); */
extern byte *BSI_GetDbufCtrl(uint32);
extern status_type BSI_DoMACScrub(uint32,uint32);
extern void BSI_FreeQueuedBufs();
extern void BSI_ChkDBDs();
extern void BSI_StopDirectBeacon(uint32);



/* bsi_reg.c */
extern void BSIREG_ReadRev(uint32);
extern void BSIREG_ResetBsi(uint32);
extern void BSIREG_SetVST(volatile BSI_REG_TYPE *);
extern void BSIREG_SetSAT(volatile BSI_REG_TYPE *);
extern void BSIREG_SetITR(uint32,uint32);
extern void BSIREG_SetIHLR(uint32,uint32);
extern void BSI_AbortReq(uint32);
extern int BSI_SendDirectBeacon(uint32,byte *);



/* bsi_isr.c */
extern PKT *RxLLCFrameIsr(uint32); 


#endif


