/***************************************************************************
 *  Module  lockport.c
 *
 *  Description:
 *      This module contains all the functions to do the port lock features.
 *  
 *  Copyright (c) Hughes LAN Systems 1992, 1993
 *
 *  Author: Jia-Wei jang
 *
 *  $Log:   /user/pvcs/fddicon/collector/lockport.c_v  
 * 
 *    Rev 1.6   26 Feb 1993 15:57:02   nayan
 * removed debug printf statements.
 * 
 *    Rev 1.5   17 Feb 1993 18:18:32   jang
 * 
 *    Rev 1.4   11 Feb 1993 14:31:24   jang
 * deleted my_ports[]
 * 
 *    Rev 1.3   09 Feb 1993 16:20:44   jang
 * added a function to test whether or not the address list is
 * full
 * 
 *    Rev 1.2   08 Feb 1993 16:32:14   nayan
 * chagned src_addr_num from byte to word
 * 
 *    Rev 1.1   07 Jan 1993 09:55:14   jang
 * moved define far to 10bt.h so it can be shared by other files
 * 
 *    Rev 1.0   28 Dec 1992 15:30:44   jang
 * Initial revision.
 ***************************************************************************/


#include <fddi.h>
#include <uierror.h>
#include <monp.h>
#include <10bt.h>
#include <nvrfddi.h>
#include <dips.h>



/*
 *=======================================================================
 *  Exported variables
 *=======================================================================
 */
PORT_INFO  *my_ports;

/*
 *=======================================================================
 *  IMPORT VARIABLES
 *=======================================================================
 */



/*
 *=======================================================================
 *  Local variables
 *=======================================================================
 */
static NVR_PORT_LOCK nvr_plock;


/*
 *=======================================================================
 *  Functions Prototype
 *=======================================================================
 */
static int cmp_addr();
static int PutPortLockRec();



/*
 *=======================================================================
 *  Export Functions
 *=======================================================================
 */


/*************************************************************************
 *  Function    GetPortLockRec
 *
 *  Description
 *    This function gets port locking record from NVRAM
 *
 *  Parameter: 
 *    PORT_INFO * - pointer to port locking record
 *
 *  Return: [NVRAM_OK | 
 ************************************************************************/
int GetPortLockRec ()
{
  int err;

  if ((err = Nvram_Load(NVR_PORT_LOCK_ADDR,&nvr_plock,NVR_PORT_LOCK_SIZE)) !=
      NVRAM_AOK) {
    /* create a new one */
    memset(&nvr_plock,0,sizeof(NVR_PORT_LOCK));
    if ((err=PutPortLockRec()) != NVRAM_AOK) {
      if (get_debug() == DEBUG_ON)
	printf("Error: can't create a new port locking record\n");
      return (err);
    }
  }

  if (nvr_plock.nvr_port_lock_marker != NVR_PORT_LOCK_MARKER) {
    if (get_debug() == DEBUG_ON)
      printf("Error: port lock marker unmatch\n");
  }

  my_ports = (PORT_INFO *) &(nvr_plock.port_lock);
  return (err);
}

/*************************************************************************
 *  Function    CHK_LOCK_ADDR
 *
 *  Description:
 *      This function compares the source address field in the RX packet
 *      with the address link list. It returns TRUE if the address is in
 *      the list; return FALSE otherwise. Note also that the source is
 *      not validated by this function to save some time.
 *
 *  Parameter:
 *      port_num : port receives the packet
 *      source - [LOCK_SRC_ADDR | LOCK_DEST_ADDR]
 *      *addr_ptr - point to the 1st byte of the address field in
 *                   the rx packet
 *
 *  Return: [TRUE | FALSE]
 ************************************************************************/
bool chk_lock_addr(port_num,source,addr_ptr)
uint32 port_num;
uint32 source;
byte far *addr_ptr;
{
    byte *ptr;
    register int i;
    byte result;
    int  len;
    
    if (source == LOCK_SRC_ADDR) {
        ptr = nvr_plock.port_lock[port_num].src_addr[0];
        len = nvr_plock.port_lock[port_num].src_addr_num;
    }
    else {
        ptr = nvr_plock.port_lock[port_num].dest_addr[0];
        len = nvr_plock.port_lock[port_num].dest_addr_num;
    }
    for (i=0; i < len; i++, ptr += LOCK_ADDR_LEN) {
        result = cmp_addr(addr_ptr,ptr);
        if (result == 0)    /* address match */
            return (TRUE);
    }
    
    return (FALSE);
}

/*************************************************************************
 *  Function    IsPortLockAddrFull
 *
 *  Description
 *    This function returns TRUE if the port lock address list is full;
 *    FALSE otherwise.
 *
 *  Parameters:
 *    uint32 port_Num
 *    uint32 source - [LOCK_SRC_ADDR | LOCK_DEST_ADDR]
 *
 *  Return: [TRUE | FALSE]
 ************************************************************************/
bool IsPortLockAddrFull (port_num, source)
uint32 port_num;
uint32 source;
{
  uint32 len;

  if (source == LOCK_SRC_ADDR) {
    len = nvr_plock.port_lock[port_num].src_addr_num;
  }
  else 
    len = nvr_plock.port_lock[port_num].dest_addr_num;

  if (len >= LOCK_ADDR_MAX)
    return (TRUE);

  return (FALSE);
}


/*************************************************************************
 *  Function    ADD_ADDR
 *
 *  Description:
 *      This function adds the address list in the nvr_plock.port_lock. It returns
 *      ERROR if the address list is already full or if the new address
 *      is already in the list.
 *
 *  Parameter:
 *      port_num - port number
 *      source - [LOCK_SRC_ADDR | LOCK_DEST_ADDR]
 *      addr - new address to add
 *
 *  Return: [OK | error codes]
 *************************************************************************/
status_type add_addr(port_num,source,addr)
uint32 port_num, source;
byte addr[];
{
    register int i;
    int j;
    bool find;  /* TRUE when find an insertion point */
    byte tmp_addr[LOCK_ADDR_LEN], *ptr, *tmp;
    status_type (*enqueue)();
    bool non_zero = FALSE;
    int *len; 

    for (i=0; i < LOCK_ADDR_LEN; i++) {
        if (addr[i] != 0) {
            non_zero = TRUE;
            break;
        }
    }
    if (!non_zero)
        return (INV_PHY_ADDR);
    for (i=0; i < LOCK_ADDR_LEN; i++) {
        if (addr[i] != 0xff) {
            non_zero = TRUE;
            break;
        }
    }
    if (!non_zero)
        return (INV_PHY_ADDR);
    if (source == LOCK_SRC_ADDR) {
        len = (int *)&nvr_plock.port_lock[port_num].src_addr_num;
        ptr = nvr_plock.port_lock[port_num].src_addr[0];
    }
    else {
        len = (int *)&nvr_plock.port_lock[port_num].dest_addr_num;
        ptr = nvr_plock.port_lock[port_num].dest_addr[0];
    }

#if 0 
    printf("len = %d src_addr_num = %d \n", *len, 
                                nvr_plock.port_lock[port_num].src_addr_num);
#endif 
 
    if (*len >= LOCK_ADDR_MAX) /* list is full */
        return (LOCK_ADDR_FULL);
    if (*len == 0) {  /* list is empty */
        *len = 1;
        for (i = 0; i < LOCK_ADDR_LEN; i++) {
            ptr[i] = addr[i];
        }
        return (OK);
    }
    /* check to see if the address is already in the list */
    tmp = ptr;
    for (i = 0; i < *len; i++, tmp += LOCK_ADDR_LEN) {
        if (!cmp_addr(addr,tmp))
            return (LOCK_ADDR_DUPLICATE);
    }

    /* now can insert the new address */
    find = FALSE;    
    for (j=0; j < *len ; ptr+= LOCK_ADDR_LEN,j++) {
        for (i=0; i < LOCK_ADDR_LEN; i++) {
            if (addr[i] < ptr[i]) {
                find = TRUE;
                break;
            }
            if (addr[i] > ptr[i])
                break;
        }
        if (find)
            break;
    }
    if (find) {
        for (; j < *len; j++, ptr += LOCK_ADDR_LEN) {
            for (i=0; i < LOCK_ADDR_LEN; i++) {
                tmp_addr[i] = ptr[i];
                ptr[i] = addr[i];
                addr[i] = tmp_addr[i];
            }
        }   /* for j */
        /* append the last one */

        for (i=0; i < LOCK_ADDR_LEN; i++)   
            ptr[i] = addr[i];   
    }
    else {   /* must the biggest one */
        for (i=0; i < LOCK_ADDR_LEN; i++)   
            ptr[i] = addr[i];   
    }

    *len += 1;

    PutPortLockRec();
    return (OK);
}

/*************************************************************************
 *  Function    CLEAR_ALL_ADDR
 *
 *  Description:
 *      This function clears the entire address list for a given port.
 *
 *  Parameter:
 *      port_num - port number
 *      source - [LOCK_SRC_ADDR | LOCK_DEST_ADDR]
 *
 *  Return: void
 *************************************************************************/
void clear_all_addr(port_num,source)
uint32 port_num, source;
{
    if (source == LOCK_SRC_ADDR) {
        nvr_plock.port_lock[port_num].src_addr_num = 0;
    }
    else if (source == LOCK_DEST_ADDR) {
        nvr_plock.port_lock[port_num].dest_addr_num = 0;
    }
}

/*************************************************************************
 *  Function    GET_NEXT_ADDR
 *
 *  Description:
 *      This function gets the next address in the list to the given
 *      address. If the given address is not in the list, the next address
 *      will be the one is immediately bigger than the given address.
 *
 *  Parameter:
 *      port_num - port number
 *      source - [LOCK_SRC_ADDR | LOCK_DEST_ADDR]
 *      addr - the address to compare
 *      
 *  Return: a pointer to the desired address; NULL if no address found
 *************************************************************************/
byte *get_next_addr(port_num,source,addr)
uint32 port_num,source;
byte addr[];
{
    register int i;
    byte *ptr, retn_code, *head_ptr;
    int len;

    if (source == LOCK_SRC_ADDR) {
        len = nvr_plock.port_lock[port_num].src_addr_num;
        ptr = nvr_plock.port_lock[port_num].src_addr[0];
    }
    else if (source == LOCK_DEST_ADDR) {
        len = nvr_plock.port_lock[port_num].dest_addr_num;
        ptr = nvr_plock.port_lock[port_num].dest_addr[0];
    }
    head_ptr = ptr;
    for (i=0; i < len; i++,ptr += LOCK_ADDR_LEN) {
        if ((retn_code=cmp_addr(addr,ptr)) == 0) {  /* 2 address are equal */
            if ( i == len - 1)      /* if it is the last one */
                return (NULL);
            else
                return (ptr+LOCK_ADDR_LEN);
        }
        if (retn_code == 1) {
            return(ptr);
        }
    }

    return (NULL);
}

/*************************************************************************
 *  Function    DELETE_ADDR
 *
 *  Description
 *      This functin purges an address from the given port
 *
 *  Parameter
 *      port_num
 *      source - [LOCK_SRC_ADDR | LOCK_DEST_ADDR]
 *      addr - the address to purge
 *
 *  Return: [OK | FDDI_ERROR]
 ************************************************************************/
status_type delete_addr(port_num,source,addr)
uint32 port_num, source;
byte addr[];
{
    byte *ptr;
    int  *len;
    register uint32 i, j;
    uint32 result;
    bool find = FALSE;
    
    if (source == LOCK_SRC_ADDR) {
        ptr = nvr_plock.port_lock[port_num].src_addr[0];
        len = (int *)&nvr_plock.port_lock[port_num].src_addr_num;
    }
    else if (source == LOCK_DEST_ADDR) {
        ptr = nvr_plock.port_lock[port_num].dest_addr[0];
        len = (int *)&nvr_plock.port_lock[port_num].dest_addr_num;
    }
        
    for (i = 0; i < *len; i++)  {
        if ((result = cmp_addr(addr,ptr)) != 0)
            ptr += LOCK_ADDR_LEN;
        else  {  /* find the match address */
            find = TRUE;
            break;
        }
    }
    if (find) {
        for (; i < *len; i++) {
            for (j=0; j < LOCK_ADDR_LEN; j++) {
                *ptr = *(ptr+LOCK_ADDR_LEN);
                ptr++;
            }
        }
        *len -= 1;
	PutPortLockRec();
        return (OK);
    }

    return (FDDI_ERROR);
}

#if 0
/*****************************************************************************
 *  Function    PRINT_LOCK_ADDR
 *
 *  Description:
 *      This function prints the address list to the user interface
 *
 *  Parameter:
 *      outport -
 *      port_num: RIC port
 *      source - [LOCK_SRC_ADDR | LOCK_DEST_ADDR]
 *
 *  Return: void
 ****************************************************************************/
void print_lock_addr(outport,port_num,source)
uint32 outport,port_num,source;
{
    int len;
    byte *ptr;
    int i;
    register j;
    
    if (source == LOCK_SRC_ADDR) {
        len = nvr_plock.port_lock[port_num].src_addr_num;
        ptr = nvr_plock.port_lock[port_num].src_addr[0];
    }
    else if (source == LOCK_DEST_ADDR) {
        len = nvr_plock.port_lock[port_num].dest_addr_num;
        ptr = nvr_plock.port_lock[port_num].dest_addr[0];
    }
    if (len == 0) {
        oprintf(outport,"*n/a*\n");
        return;
    }
    if (len > LOCK_ADDR_MAX) {
/*
        oprintf(outport,"fatal error. total number of addresses are %d\n",len);
*/
        return;
    }
    for (i=0; i < len; i++,ptr += LOCK_ADDR_LEN) {
        for (j=0; j < LOCK_ADDR_LEN; j++) {
            if (ptr[j] < 0x10) {
                if (j == 5)
                    oprintf(outport,"0%1x ",ptr[j]);
                else
                    oprintf(outport,"0%1x.",ptr[j]);
            }
            else {
                if (j == 5)
                    oprintf(outport,"%2x ",ptr[j]);
                else
                    oprintf(outport,"%2x.",ptr[j]);
            }
        }
        if ( i & 0x01)      /* 2 addresses per line */
            oprintf(outport,"\n%19s",pblank);
        else
            oprintf(outport,"%13s",pblank);
    }

    oprintf(outport,"\n");    
}
#endif
 
/*
 *------------------------------------------------------------------------
 *  Local Functions
 *------------------------------------------------------------------------
 */

/*************************************************************************
 *  Function    CMP_ADDR
 *
 *  Description:
 *      This function compares the two address.
 *
 *  Parameter:
 *      src - source address
 *      dest - dest address
 *
 *  Return:
 *      0 - if two address are equal
 *      1 - if src < dest
 *      2 - if src > dest
 *************************************************************************/
static int cmp_addr(src,dest)
byte src[];
byte dest[];
{
    register int i;

    for (i = 0; i  < LOCK_ADDR_LEN; i++) {
        if (src[i] < dest[i])
            return (1);
        if (src[i] > dest[i])
            return (2);
    }
    return (0);
}




/*************************************************************************
 *  Function PutPortLockRec
 *
 *  Description
 *    This function put the entire port lock address table back to the
 *    NVRAM
 *    PORT_INFO * - pointer to port locking record
 *
 *  Return: [NVRAM_OK | 
 ************************************************************************/
static int PutPortLockRec ()
{
  int err;


  nvr_plock.nvr_port_lock_marker = NVR_PORT_LOCK_MARKER;

  if ((err = Nvram_Updt(NVR_PORT_LOCK_ADDR,&nvr_plock,NVR_PORT_LOCK_SIZE)) != NVRAM_AOK) {
    if (get_debug() == DEBUG_ON)
      printf("Error: can't write new port locking record to NVRAM\n");
  }

  return (err);
}


