/**			       
*
*	Program Name:	Stanley - multi-port bridge
*
*	Filename:	stpc.c
*
*	$Log:   /b/gregs/bridge/stp/stpc.c_v  $
 * 
 *    Rev 1.0   30 Jul 1993 13:51:42   vinay
 * Initial revision.
 * 
 *    Rev 1.0   30 Mar 1992 17:40:38   pvcs
 * Initial revision.
*
*	Creation Date:	6.90
*
*	Date:		9.11.91
*
*	Version:	1.0
*
*	Programmers:	K Kong
*
*	Modifications:	K Kong	9.11.91
*			Adding functions SetAgeTimeToForwardDelay() and
*			RestoreAgeTime() to modify and restore the 
*			ago time when a change in topoplgy has been
*			detected.
*
*	Comments:	Port to the intel 960 platform from 8086.
*			This is based on BLL's implementation for
*			the 8080/8050 system.
*			Reference is made to the IEEE standard 802.1D-1990:-
*			MAC bridges.
*
*  			This is STP routines to handle the 
*			transmission and the DBD buffer management.  
*			They are as follows :        
*              		1) stp_stack_init - SNMP stack initialization 
*              		5) SendFromSTP    - transmit a packet to one channel
*              		6) SendToSTP      - transmit a packet to STP protocol
*
*	Copyright (c) 1991 by Hughes LAN Systems
*
**/

#include <types.h>
#include <target.h>
#include <krnl.h>
#include <netbuf.h>
#include <prcctl.h>
#include <bridges.h>
#include <nvrecs.h>
#include <stp.h>




extern	PACKET	getfree();
extern PRCCTL prc;
extern	NVR_BSTATUS	*BridgeStatus;

typedef	struct
	{
	Boolean	TopologyChange;
	uint	OriginalAgeCounter;
	}
	TopologyChangeAge;

static	TopologyChangeAge	AgeCounter;

void SetProcState(int port_no, int state);
void SetHelloTime(int HelloTime);
void SetMaxAge(int age);
void SetForwardDelay(int delay);



static TIMER	stp_timer;		/* stp all-purpose timer      */

static byte stphdr[20] =
	{
	0x01, 0x80, 0xc2, 0x00, 0x00, 0x00,	/* destination group address */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	/* source address */
	0x00, 0x00,        	/* length field */
	0x42, 0x42, 0x03,	/* LLC */
	0x00, 0x00, 0x00	/* STP Protocol Identifier & Version Identifier */
	};			/* outgoing STP LLC and MAC header */

static PrintPacket(unsigned char *p, int length)

	{
	while (length--)
		{
		printf("%X ", *p++);
		}
	}

stp_stack_init()			/* buffer init */
	{
	int	port;

	/* finish init the STP outgoing header */
	/*	Each port has it's own MAC address.
	 *	
	ncopy(&stphdr[6], MyNid(0));
	***********************/
	/* STP algorithm and protocol initialisation */
	initialisation();

	/* init stp timer */
	CreatTimer(&stp_timer);	/* init SNMPs timer */
	/* 
	 *	Disable STP if it has been disabled.
	 *	Otherwise, disable those ports which have been
	 *	disabled.
	 */
	if (prc.Prc_StpMode != STP_MODE_DISABLE)
		{
		port = NumberOfStpPorts;
		for (port = 0; port < NumberOfStpPorts; port++)
			{
			if (((prc.Prc_PortState >> port) & 0x01) == 0)
				disable_port(port + 1);
			}
		StartTimerCall(&stp_timer, 3000, tick, (int)&stp_timer);
		}
	else
		disable_stp();
	}	

/* Invoke spanning tree processing routines on received frame */
SendToSTP(dp)
PACKET	dp;

	{
	/*
	 *	Send to stp if we are in stp mode only
	 */
	if (prc.Prc_StpMode != STP_MODE_DISABLE)
		{
		/* process frames contents */
		if(((Config_bpdu *)dp->db_buffer)->type == 0x00)
			{
			received_config_bpdu(dp->db_rcvportno + 1, 
				(Config_bpdu *)dp->db_buffer);
			}
		else
			{
			received_tcn_bpdu(dp->db_rcvportno + 1, 
				(Tcn_bpdu *)dp->db_buffer);
			}
		}

	putfree(dp);		/* free the buffer */
	}

/* transmit a packet to one or both channels */
SendFromSTP(port_no, pp, ln)
int	port_no;
byte	*pp;
int	ln;

	{
	register PACKET dp;
	byte	*bp;

	if (prc.Prc_StpMode != STP_MODE_DISABLE)
		{
		/* get a free buffer */
		if ((dp = getfree()) == NULL)
			return 1;
		bp = dp->db_buffer;
		dp->db_indent = 0;

		/* build the MAC and LLC header */
		/*
		 *	llc length is total length - the MAC header,
		 *	i.e. excluding the src address, dst address and length
		 */
		*(ushort *)&stphdr[12] = htons(ln - 14);
		memcpy(pp, stphdr, sizeof(stphdr));
		memcpy(bp, pp, ln);
		ncopy(bp + 6, MyNid(port_no - 1));

		/* send the frame */
		dp->db_xmtportlist = (1 << (port_no - 1));/* to this port */
		dp->db_actcnt = 60;  /* minimum frame size is 60 bytes */
		dp->nb_flags = PKT_IEEEHDR;	/* indicate an llc frame */
		/* 
		 *	Free the dbd after transmitting
		 */

		nim_xmt(dp, 60);
		putfree(dp);
		}
	return 0;
	}

initialisation()

	{
	Int	port_no;

	bridge_info.designated_root.priority = BridgeStatus->BridgePriority;
	bridge_info.bridge_id.priority = BridgeStatus->BridgePriority;
	/*
	 *	The bridge address is the MAC address of Port 1.
	 *	item 11c
	 */
	ncopy(bridge_info.bridge_id.stp_address, MyNid(0));
	ncopy(bridge_info.designated_root.stp_address, MyNid(0));
	/*
	 *	Assume I am the root on start up
	 */
	bridge_info.root_path_cost = Zero; 
	bridge_info.root_port = No_port; 

	bridge_info.max_age = BridgeStatus->MaxAge;
	bridge_info.bridge_max_age = BridgeStatus->MaxAge;
	bridge_info.hello_time = BridgeStatus->HelloTime;
	bridge_info.bridge_hello_time = BridgeStatus->HelloTime;
	bridge_info.forward_delay = BridgeStatus->ForwardDelay;
	bridge_info.bridge_forward_delay = BridgeStatus->ForwardDelay;
	bridge_info.topology_change_time = bridge_info.bridge_max_age 
					+ bridge_info.bridge_forward_delay;
	bridge_info.hold_time = BRIDGE_HOLD_TIME;

	bridge_info.topology_change_detected = False; 
	bridge_info.topology_change = False;
	AgeCounter.TopologyChange = False;
	stop_tcn_timer();
	stop_topology_change_timer();

	for(port_no = One; port_no <= NumberOfStpPorts; port_no++)
		{
		port_info[port_no].port_id = (BridgeStatus->priority[port_no - 1 ] << 8) | (port_no);
		port_info[port_no].path_cost = (Int)BridgeStatus->PathCost[port_no - 1];
		initialize_port(port_no);
		}

	port_state_selection();
	config_bpdu_generation();
	start_hello_timer();
	}

/*
 * name		config_bpdu_generation
 *
 * synopsis	config_bpdu_generation(void)
 *
 * description	Send bridge configuration pdus on those ports which
 *		are designated ports, have not been disabled and
 *		are configured to run spanning tree protocol.
 *
 * returns	nothing
 */

config_bpdu_generation()

	{
	Int	port_no;

	for (port_no = One; port_no <= NumberOfStpPorts; port_no++)
		{
		if (designated_port(port_no) &&
			(port_info[port_no].state != Disabled) &&
			(prc.Prc_StpMode != STP_MODE_DISABLE))
			{
			transmit_config(port_no);
			}
		}
	}


set_bridge_priority(new_bridge_id)
unsigned int new_bridge_id;

	{
	Boolean root;
	Int	port_no;

	root = root_bridge();

	for(port_no = One; port_no <= NumberOfStpPorts; port_no++)
		{
		if(designated_port(port_no))
			{
			port_info[port_no].designated_bridge.priority = new_bridge_id;
			ncopy(port_info[port_no].designated_bridge.stp_address,
				MyNid(0));
			/*
			port_info[port_no].designated_bridge = new_bridge_id;
			*/
			}
		}

	bridge_info.bridge_id.priority = new_bridge_id;
	ncopy(bridge_info.bridge_id.stp_address, MyNid(0));

	configuration_update();

	port_state_selection();

	if((root_bridge()) && (!root))
		{
		bridge_info.max_age = BridgeStatus->MaxAge;
		bridge_info.bridge_max_age = BridgeStatus->MaxAge;
		bridge_info.hello_time = BridgeStatus->HelloTime;
		bridge_info.bridge_hello_time = BridgeStatus->HelloTime;
		bridge_info.forward_delay = BridgeStatus->ForwardDelay;
		bridge_info.bridge_forward_delay = BridgeStatus->ForwardDelay;

		topology_change_detection();
		stop_tcn_timer();
		config_bpdu_generation();
		start_hello_timer();
		}
	}

void SetHelloTime(int HelloTime)

	{
	bridge_info.bridge_hello_time = HelloTime;
	if (bridge_info.root_port == 0)
		/*
		 *	If I am the root, then the update the 
		 *	system wide hello_time.
		 */
		bridge_info.hello_time = HelloTime;
	}

void SetMaxAge(int age)
	{
	bridge_info.bridge_max_age = age;
	/*
	 *	Update the topology_change_time as well.
	 */
	bridge_info.topology_change_time = bridge_info.bridge_max_age 
					+ bridge_info.bridge_forward_delay;
	if (bridge_info.root_port == 0)
		/*
		 *	update the system wide max_age if I am the root. 
		 */
		bridge_info.max_age = age;
	}

void SetForwardDelay(int delay)
	{
	bridge_info.bridge_forward_delay = delay;
	/*
	 *	Update the topology_change_time as well.
	 */
	bridge_info.topology_change_time = bridge_info.bridge_max_age 
					+ bridge_info.bridge_forward_delay;
	if(bridge_info.root_port == 0)
		/*
		 *	update the system wide forware delay if I am the root. 
		 */
		bridge_info.forward_delay = delay;
	}
/*
 *	For debugging
 */
static char *State2String[] =
	{
	"Disabled",
	"Listening",
	"Learning",
	"Forwarding",
	"Blocking",
	""
	};
void SetProcState(int port_no, int state)

	{
	/**
	printf("Changing port %d state from %s to %s    ", 
		port_no,
		State2String[prc.Prc_PortStpState[port_no - 1]],
		State2String[state]);
	***/
	port_no--;
	prc.Prc_PortStpState[port_no] = state;
	if (prc.Prc_StpMode != STP_MODE_DISABLE)
		{
		/*
		 *	spanning tree in on
		 */
		if (state == Blocking || state == Disabled)
			{
			LedsOff(1 << port_no);
			}
		else
			{
			LedsOn(1 << port_no);
			}

		}
	}
/*
 *	Check if all ports has been disabled
 */
IsAllPortDisabled()
	{
	return (prc.Prc_PortState & 0x0f) == 0;
	}

disable_stp()

	{
	Int	port_no;
	int	port_state;

	port_state = prc.Prc_PortState;
	for(port_no = One; port_no <= NumberOfStpPorts; port_no++)
		{
		disable_port(port_no);
		/*
		 *	Turn on the leds if it is on
		 */
		if ((port_state >> (port_no - 1)) & 0x01)
			LedsOn(1 << (port_no - 1));
		}
	StopTimer(&stp_timer);
	} 


enable_stp()

	{
	Int	port_no;

	for(port_no = One; port_no <= NumberOfStpPorts; port_no++)
		{
		/*
		 *	enable those ports which have not been
		 *	disabled.
		 */
		if (((prc.Prc_PortState >> (port_no - 1)) & 0x01) != PORT_MODE_DISABLE)
			enable_port(port_no);
		}
	StartTimerCall(&stp_timer, 100, tick, (int)&stp_timer);
	}
/*
 * name		SetAgeTimeToForwardDelay
 *
 * synopsis	SetAgeTimeToForwardDelay(void)
 *
 * description	Set the age time to be the forward delay time if we have
 *		not done it already.
 *		We ago out "prc.Prc_AgeCnt" number of records per
 *		100 ms. To get an effective age time of "forward delay"
 *		seconds by age out total number of records in the system
 *		divided by forward delay(in 100 ms).
 *
 * returns	nothing
 */

SetAgeTimeToForwardDelay()
	{
	if (AgeCounter.TopologyChange != True)
		{
		/*
		 *	We save the original counter and set the new
		 *	ago out counter.
		 */
		AgeCounter.OriginalAgeCounter = prc.Prc_AgeCnt;
		if (bridge_info.forward_delay == 0)
			bridge_info.forward_delay = DEFAULT_FORWARD_DELAY;
		prc.Prc_AgeCnt = ADR_REC_CNT / (bridge_info.forward_delay * 10);
		AgeCounter.TopologyChange = True;
		}
	}
/*
 * name		RestoreAgeTime
 *
 * synopsis	RestoreAgeTime(void)
 *
 * description	Restore the original age time(i.e. age counter).
 *
 * returns	nothing
 */	

RestoreAgeTime()
	
	{
	if (AgeCounter.TopologyChange == True)
		{
		prc.Prc_AgeCnt = AgeCounter.OriginalAgeCounter; 
		AgeCounter.TopologyChange = False;
		}
	}
/*
 *	If the user modify the age time after we have modified the
 *	Prc_AgeCnt and before we restore it, we want to update our
 *	OriginalAgeCounter such that we can restore the latest counter
 *	later on RestorAgeTime().
 */

NewPrcAgeCnt(uint counter)

	{
	if (AgeCounter.TopologyChange == True)
		{
		AgeCounter.OriginalAgeCounter = counter;
		/*
		 *	We don't want to use new AgeCnt yet!
		 */
		prc.Prc_AgeCnt = ADR_REC_CNT / (bridge_info.forward_delay * 10);
		}
	}

StpClearCounters()

	{
	int	port;

	for (port = 1; port <= NumberOfStpPorts; port++)
		{
		StpConfigTxBPDU[port] = 0;	
		StpConfigRxBPDU[port] = 0;
		StpTopChgTxBPDU[port] = 0;	
		StpTopChgRxBPDU[port] = 0;
		StpForwardTransitions[port] = 0;
		}
	StpTopChgDet = 0;
	/*StpTopChgUpTime = 0;*/
	}
