/**************************************************************************
 *  Copyright (c) 1992 Hughes LAN Systems                                 *
 *  All rights Reserved                                                   *
 *                                                                        *
 *  This file contains unpublished proprietary source code of Hughes LAN  *
 *  Systems.                                                              *
 **************************************************************************
                                                                           
   This file contains the service routines for MULTIPORTBRIDGE-MIB
   It is directly derived from the MIB specification using the AWK program 
   'MIBSVC.AWK'
   Should significant changes be needed in this file that are NOT          
   reflected back into the MIB specification, then modify this header to   
   indicate that this is the case.                                         
                                                                           
 *************************************************************************/

#define MULTIPORTBRIDGE-MIB_SVC
#define ENABLE 2
#define DISABLE 1

#include <types.h> 
#include <target.h>
#include <krnl.h>
#include <dbd.h>
#include <snmp.h>
#include <asn1.h>
#include <syteksnm.h>
#include <bitmask.h>
#include <prcctl.h>

/* Additional include files */
#include <bridges.h>
#include <prcadr.h>
#include <nvrecs.h>
#include <sncvar.h>
#include <filter.h>
#include "834CmdSnmp.h"


/* External Definitions */
extern NVR_BSTATUS  *BridgeStatus;
extern PRCCTL       prc;
extern SV           sv_var[];
extern UDPDIR       *udpdir;
extern uint         *PFilterList;
extern PROTCNTLREC  Protocols[];
extern PRCCTL *prcctl;

/*************************************************************************
*  Procedure    :   svc_chPortState
*  Path         :   1.3.6.1.4.1.26.2.25.2.1.2
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*              "The status of each port on the bridge. If a port is
*           enabled then it can receive and transmit packets. If
*           it is disabled it will only receive packets, and not
*           forward them."
*
**************************************************************************/

int svc_chPortState(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    int rtn_code = 0;
    uint PortMask;


    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */
    case SNMP_GET:
        rtn_code = chk_portidx(*compc, *compl);
        if (rtn_code == 0)
        {
            /* PortState is in the form of a bitmask */
            PortMask = Indx2Mask( *compl );

            if( BridgeStatus->PortState & PortMask )
                *(uint *)value = ENABLE;
            else
                *(uint *)value = DISABLE;
            break;
        }
        else
			return(NO_SUCH_NAME);
        break;

    case SNMP_SET:
        {
            if( *compc != 1 || *compl > 4 )
                return(rtn_code = NO_SUCH_NAME);

            PortMask = Indx2Mask( *compl );
            switch( *value )
            {
                 case ENABLE:
                     BridgeStatus->PortState |= PortMask;
                     prcctl->Prc_PortState   |= (byte)PortMask;
		             if( prcctl->Prc_StpMode == (byte) STP_MODE_ENABLE) 
			             enable_port( *compl );
		             LedsOn(PortMask);
                     SaveBStatus();
                     break;

                 case DISABLE:      
                     BridgeStatus->PortState &= ~PortMask;
                     prcctl->Prc_PortState   &= (byte)~PortMask;
		             if( prcctl->Prc_StpMode == (byte) STP_MODE_ENABLE) 
			               disable_port( *compl );
		             LedsOff(PortMask);
                     SaveBStatus();
                     break;
  
                default:
                    return(rtn_code = BAD_VALUE);
                    break;
            } /* end of *value switch   */
            break;
        }

    case SNMP_TEST:
        rtn_code = chk_portidx(*compc, *compl);
        if(rtn_code == 0)
        {
            /* FILL IN ACTUAL TEST OPERATIONS HERE */
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_chRestrictInboundUnicast
*  Path         :   1.3.6.1.4.1.26.2.25.2.1.3
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*              "The status of the restrict inbound uni-cast mode for
*           each bridge port. A port's inbound uni-cast restrict
*           mode must be enabled for the channel to respond to
*           restrict inbound uni-cast flags set on any address 
*           which lives on that bridge port."
*
**************************************************************************/

int svc_chRestrictInboundUnicast(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    int rtn_code = 0;
    uint PortMask;


    switch(service)
    {

    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */

    case SNMP_GET:
        rtn_code = chk_portidx(*compc, *compl);
        if (rtn_code == 0)
        {
            /* Parameter is in the form of a bitmask */
            PortMask = Indx2Mask( *compl );

            if( BridgeStatus->RestrictInUCast & PortMask )
                *(uint *)value = ENABLE;
            else
                *(uint *)value = DISABLE;
            break;

        }
        else
            return NO_SUCH_NAME;
        break;

    case SNMP_SET:
        {
            PortMask = Indx2Mask(*compl);
            switch(*value)
            {
                 case ENABLE:
                     BridgeStatus->RestrictInUCast |= PortMask;
                     prcctl->Prc_RestInUcast       |= (byte)PortMask;
                     SaveBStatus();
                     break;

                 case DISABLE:      
                     BridgeStatus->RestrictInUCast &= ~PortMask;
                     prcctl->Prc_RestInUcast       &= (byte)~PortMask;
                     SaveBStatus();
                     break;
  
                default:
                    return(BAD_VALUE);
            }
            break;
        }

    case SNMP_TEST:
        rtn_code = chk_portidx(*compc, *compl);
        if(rtn_code == 0)
        {
            /* FILL IN ACTUAL TEST OPERATIONS HERE */
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_chRestrictInboundMulticast
*  Path         :   1.3.6.1.4.1.26.2.25.2.1.4
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*              "The status of the restrict inbound Multi-cast mode for
*           each bridge port. A port's inbound Multi-cast restrict
*           mode must be enabled for the channel to respond to
*           restrict inbound Multi-cast flags set on any address 
*           which lives on that bridge port."
*
**************************************************************************/

int svc_chRestrictInboundMulticast(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    int rtn_code = 0;
    uint PortMask;


    switch(service)
    {

    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */

    case SNMP_GET:
        rtn_code = chk_portidx(*compc, *compl);
        if (rtn_code == 0)
        {
            /* Parameter is in the form of a bitmask */
            PortMask = Indx2Mask( *compl );

            if( BridgeStatus->RestrictInMCast & PortMask )
                *(uint *)value = ENABLE;
            else
                *(uint *)value = DISABLE;
            break;

        }
        else
            return NO_SUCH_NAME;
        break;

    case SNMP_SET:
        {
            PortMask = Indx2Mask(*compl);
            switch(*value)
            {
                 case ENABLE:
                     BridgeStatus->RestrictInMCast |= PortMask;
                     prcctl->Prc_RestInMcast       |= (byte)PortMask;
                     SaveBStatus();
                     break;

                 case DISABLE:      
                     BridgeStatus->RestrictInMCast &= ~PortMask;
                     prcctl->Prc_RestInMcast       &= (byte)~PortMask;
                     SaveBStatus();
                     break;
  
                default:
                    return(BAD_VALUE);
            }
            break;
        }

    case SNMP_TEST:
        rtn_code = chk_portidx(*compc, *compl);
        if(rtn_code == 0)
        {
            /* FILL IN ACTUAL TEST OPERATIONS HERE */
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_chRestrictOutbound
*  Path         :   1.3.6.1.4.1.26.2.25.2.1.5
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*              "The status of the restrict outbound mode for
*           each bridge port. A port's outbound Multi-cast restrict
*           mode must be enabled for the channel to respond to
*           restrict outbound flags set on any address 
*           which lives on that bridge port."
*
**************************************************************************/

int svc_chRestrictOutbound(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    int rtn_code = 0;
    uint PortMask;


    switch(service)
    {

    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */

    case SNMP_GET:
        rtn_code = chk_portidx(*compc, *compl);
        if (rtn_code == 0)
        {
            /* Parameter is in the form of a bitmask */
            PortMask = Indx2Mask( *compl );

            if( BridgeStatus->RestrictOutbound & PortMask )
                *(uint *)value = ENABLE;
            else
                *(uint *)value = DISABLE;
            break;

        }
        else
            return NO_SUCH_NAME;
        break;

    case SNMP_SET:
        {
            PortMask = Indx2Mask(*compl);
            switch(*value)
            {
                 case ENABLE:
                     BridgeStatus->RestrictOutbound |= PortMask;
                     prcctl->Prc_RestOut            |= (byte)PortMask;
                     SaveBStatus();
                     break;

                 case DISABLE:      
                     BridgeStatus->RestrictOutbound &= ~PortMask;
                     prcctl->Prc_RestOut            &= (byte)~PortMask;
                     SaveBStatus();
                     break;
  
                default:
                    return(BAD_VALUE);
            }
            break;
        }

    case SNMP_TEST:
        rtn_code = chk_portidx(*compc, *compl);
        if(rtn_code == 0)
        {
            /* FILL IN ACTUAL TEST OPERATIONS HERE */
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_chPacketSize
*  Path         :   1.3.6.1.4.1.26.2.25.2.1.6
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The range of frame sizes to be forwarded onto an
*                          Ethernet/802.3 network by this bridge.  The value
*                          standard(1) represents the standard range of 60
*                          through 1536 bytes. The value extended(2)
*                          represents the (non-standard) range of 16 through
*                          2100 bytes."
*
**************************************************************************/

int svc_chPacketSize(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    int rtn_code = 0;
    uint PortMask;


    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */
    case SNMP_GET:
        rtn_code = chk_portidx(*compc, *compl);
        if (rtn_code == 0)
        {
            /* PortState is in the form of a bitmask */
            PortMask = Indx2Mask( *compl );

            if( BridgeStatus->PacketSize & PortMask )
                *(uint *)value = EXT_TYPE;
            else
                *(uint *)value = STD_TYPE;
            break;
         }
         else
			 return( NO_SUCH_NAME);
         break;

    case SNMP_SET:
        {
            if( *compc != 1 || *compl > 4 )
                return(rtn_code = NO_SUCH_NAME);

            PortMask = Indx2Mask( *compl );
            switch( *value )
            {
                 case EXT_TYPE:
                     BridgeStatus->PacketSize |= PortMask;
                     prcctl->Prc_PacketSize   |= (byte)PortMask;
                     SaveBStatus();
                     break;

                 case STD_TYPE:      
                     BridgeStatus->PacketSize &= ~PortMask;
                     prcctl->Prc_PacketSize   &= (byte)~PortMask;
                     SaveBStatus();
                     break;
  
                default:
                    return(rtn_code = BAD_VALUE);
                    break;
            } /* end of *value switch   */
            break;
        }

    case SNMP_TEST:
        rtn_code = chk_portidx(*compc, *compl);
        if(rtn_code == 0)
        {
            /* FILL IN ACTUAL TEST OPERATIONS HERE */
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_chReceiveDeadmanTimer
*  Path         :   1.3.6.1.4.1.26.2.25.2.1.7
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                          "The deadman timer for frame reception on an
*                          interface.  "
*
**************************************************************************/

int svc_chReceiveDeadmanTimer(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    int rtn_code = 0;
    uint PortMask;


    switch(service)
    {

    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */

    case SNMP_GET:
        rtn_code = chk_portidx(*compc, *compl);
        if (rtn_code == 0)
        {
            /* Parameter is in the form of a bitmask */
            PortMask = Indx2Mask( *compl );

            if( BridgeStatus->RxThreshhold & PortMask )
                *(uint *)value = ENABLE;
            else
                *(uint *)value = DISABLE;
            break;

        }
        else
            return NO_SUCH_NAME;
        break;

    case SNMP_SET:
        {
            PortMask = Indx2Mask(*compl);
            switch(*value)
            {
                 case ENABLE:
                     BridgeStatus->RxThreshhold |= PortMask;
                     SaveBStatus();
                     break;

                 case DISABLE:      
                     BridgeStatus->RxThreshhold &= ~PortMask;
                     SaveBStatus();
                     break;
  
                default:
                    return(BAD_VALUE);
            }
            break;
        }

    case SNMP_TEST:
        rtn_code = chk_portidx(*compc, *compl);
        if(rtn_code == 0)
        {
            /* FILL IN ACTUAL TEST OPERATIONS HERE */
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_multiPortProtocolFilterPort
*  Path         :   1.3.6.1.4.1.26.2.25.3.1.1
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*         "The port number of the port for which this entry contains
*          protocol filtering information."
*
**************************************************************************/

int svc_multiPortProtocolFilterPort(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    int rtn_code = 0;


    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */
    case SNMP_GET:
        rtn_code = chk_portidx(*compc, *compl);
        if (rtn_code == 0)
        {
            *value = (UINT_32_T) *compl;
        }
        else
            return NO_SUCH_NAME;
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_multiPortProtocolFilterMode
*  Path         :   1.3.6.1.4.1.26.2.25.3.1.2
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*         "Identifies whether the protocol types selected for each  
*          port entry are being blocked (filtered) or permited 
*          (forwarded). Setting this value to invalid(3), will cause 
*          all protocols types for this entry to be marked as 
*          (not selected). See protocolFilterType object below. "
*
**************************************************************************/

int svc_multiPortProtocolFilterMode(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    int  rtn_code = 0;
    uint PortMask;

    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */

    case SNMP_GET:
        rtn_code = chk_portidx(*compc, *compl);
        if (rtn_code == 0)
        {
            PortMask = Indx2Mask(*compl);
            /* If protocol filtering is enabled */
            if(BridgeStatus->ProtFilter & PortMask) {

               /* Get the filtering mode */
               if( BridgeStatus->FilterMode & PortMask )
                   *value = (UINT_32_T) 2; /* Permit Mode */
               else
                   *value = (UINT_32_T) 1; /* Block Mode */
            }
            else
                   *value = (UINT_32_T) 3; /* Invalid */
        }
        else
            return NO_SUCH_NAME;
        break;

    case SNMP_SET:
        {
            if(*value < 1 || *value > 3)
				return(BAD_VALUE);

            PortMask = Indx2Mask(*compl);
            switch(*value) {

            case 1: /* Block Mode */
                BridgeStatus->FilterMode &= ~PortMask;
                prcctl->Prc_ProtFltMode  &= (byte) ~PortMask;

                /* Enable protocol filtering only if there */
                /* are protocols types selected            */
                ProtFilEnaDis( PortMask );
                break;

            case 2: /* PermitMode */
                BridgeStatus->FilterMode |= PortMask;
                prcctl->Prc_ProtFltMode  |= (byte) PortMask;

                /* Enable protocol filtering only if there */
                /* are protocols types selected            */
                ProtFilEnaDis( PortMask );
                break;

            case 3: /* Invalid */

                /* Clear All Protocols from the protocol */
                /* filter table                          */
                ClearAllUProts( PortMask );  /* User specifed */
                ClearAllPProts( PortMask );  /* Pre-defined */

                /* Disable Protcol Filtering Mode */
                ProtFilEnaDis( PortMask );
				break;
            }
            SaveBStatus();
        }
        break;

    case SNMP_TEST:
        rtn_code = chk_portidx(*compc, *compl);
        if(rtn_code == 0)
        {
            if(*value < 1 || *value > 3)
             return(BAD_VALUE);
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_multiPortProtocolFilterType
*  Path         :   1.3.6.1.4.1.26.2.25.3.1.3
*  Access       :   RW
*  Syntax       :   OCTET VT_STRING
*  Description  :   
*                   "An instance of this object-type exists for each entry in
*                    the Filter Table of a HLS multi port ethernet bridge. The
*                    format of the OCTET STRING is: 
*         1st Octet:   1 =  IP protocol selected
*                      2 =  IP protocol not selected
*         2nd Octet:   1 =  ARP protocol selected
*                      2 =  ARP protocol not selected
*         3rd Octet:   1 =  REVARP protocol selected
*                      2 =  REVARP protocol not selected
*         4th Octet:   1 =  TOP protocol selected
*                      2 =  TOP protocol not selected
*         5th Octet:   1 =  V2 protocol selected
*                      2 =  V2 protocol not selected
*         6th Octet:   1 =  NOVELL protocol selected
*                      2 =  NOVELL protocol not selected
*         7th Octet:   1 =  LAT protocol selected
*                      2 =  LAT protocol not selected
*         8th Octet:   1 =  BRIDGE protocol selected
*                      2 =  BRIDGE protocol not selected
*         9th Octet:   1 =  XNS protocol selected
*                      2 =  XNS protocol not selected
*         10th Octet:  1 =  DEC protocol selected
*                      2 =  DEC protocol not selected
*         11th Octet:  1 =  APPLE protocol selected
*                      2 =  APPLE protocol not selected
*         12th Octet:  1 =  USP1 protocol selected
*                      2 =  USP1 protocol not selected
*         13th Octet:  1 =  USP2 protocol selected
*                      2 =  USP2 protocol not selected
*         14th Octet:  1 =  USP3 protocol selected
*                      2 =  USP3 protocol not selected
*         15th Octet:  1 =  USP4 protocol selected
*                      2 =  USP4 protocol not selected
*         16th Octet:  1 =  USP5 protocol selected
*                      2 =  USP5 protocol not selected
*         17th Octet:  1 =  USP6 protocol selected
*                      2 =  USP6 protocol not selected
*         18th Octet:  1 =  USP7 protocol selected
*                      2 =  USP7 protocol not selected
*         19th Octet:  1 =  USP8 protocol selected
*                      2 =  USP8 protocol not selected
*         20th Octet:  1 =  USP9 protocol selected
*                      2 =  USP9 protocol not selected
*         21th Octet:  1 =  USP10 protocol selected
*                      2 =  USP10 protocol not selected
*         Identifies whether the protocol referenced by this entry
*         has been selected to be permited or blocked, based on the 
*         current protocol filtering mode set on the port. For 
*         octets 10 - 21, there must be a corresponding entry in 
*         the User-Specified Protocol table, before the USP can
*         be selected in the protocol filter table."
*
**************************************************************************/

int svc_multiPortProtocolFilterType(service, compc, compl, lenp, value, index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
OCTET_P      value;
int          index;
{
    int  rtn_code = 0;
    uint PortMask;
    uint UserVal;
    uint cnt;
    UDPDIR    *udp      = udpdir;
    OCTET_P tmpval        = value;
    extern PROTCNTLREC Protocols[];
    PROTCNTLREC *plist  = &Protocols[0];


    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */
    case SNMP_GET:
        rtn_code = chk_portidx(*compc, *compl);
        if (rtn_code == 0)
        {
            PortMask = Indx2Mask(*compl);

            /* First 1-11 Octets are for the Pre-defined protocol types */
            for(cnt=0; cnt < PROTNUM; cnt++) {
                if( cnt != TCP_P ) {
                    if( BridgeStatus->PFilterList[cnt] & PortMask )
                        *tmpval =  1;  /* Portocol type is set */
                    else        
                        *tmpval =  2;  /* Portocol type is not set */
                tmpval++;
                }
           
            }

            /* 12-21 Octets are for the User-specified protocol types */
            for(cnt=0; cnt < MAXUSRTYPE; cnt++) {
            
                if( (udp+cnt)->PrtSet & PortMask )  
                    *tmpval =  1;  /* Portocol type is set */
                else
                    *tmpval =  2;  /* Portocol type is not set */
                tmpval++;
            }
            *lenp = 21;
        }
        else
            return NO_SUCH_NAME;
        break;

    case SNMP_SET:
        {

            PortMask = Indx2Mask(*compl);

            if(CheckSetValues(tmpval, *lenp))
                return( BAD_VALUE );

            /* Set/Clear the pre-defined protocol types first */
            for( cnt=0; cnt < PROTNUM; cnt++ ) {

                if( cnt != TCP_P ) {
                    if( *tmpval == 1 ) {

                        /* Set values in the actual protocol filter table */
                        SetPortProt( &((plist+cnt)->prots), PortMask, PTYPE);

                        /* Set bits in administrative structure */
                        SetPortBit( (PFilterList+cnt), PortMask);
                    }
                    else
                    if( *tmpval == 2 ) { 

                        /* Clear values in the actual protocol filter table */
                        ClearPortProt( &((plist+cnt)->prots), PortMask, PTYPE);

                        /* Clear bits in administrative structure */
                        ClearPortBit( (PFilterList+cnt), PortMask);
                    }
                tmpval++;
                }
            }
            /* Set/Clear the user-specified protocol types second */
            for( cnt=0; cnt < MAXUSRTYPE; cnt++ ) {
            
                if( *tmpval == 1) {

                    /* Check for a valid entry in the USP directory */
                    if( udp[cnt].UProtTyp > 1500 ) {
                        /* If valid, Add port info to directory */
                        udp[cnt].PrtSet |= PortMask;
                        UserVal = udp[cnt].UProtTyp;
                        SetPortProt(&UserVal, PortMask, UTYPE);
                    }
                    /* Not a valid entry in the USP directory, ignore set */
                    else
                        ;
                }
                else
                if( *tmpval == 2) {

                    /* Check for a valid entry in the USP directory */
                    if( udp[cnt].UProtTyp > 1500 ) {
                        /* If valid, Add port info to directory */
                        udp[cnt].PrtSet &= ~PortMask;
                        UserVal = udp[cnt].UProtTyp;
                        ClearPortProt(&UserVal, PortMask, UTYPE);
                    }
                    /* Not a valid entry in the USP directory, ignore set */
                    else
                        ;
                }
                tmpval++;
            }

        ProtFilEnaDis( PortMask );
        SaveBStatus();
        SaveProtocolTbl();
        }
        break;

    case SNMP_TEST:
        rtn_code = chk_portidx(*compc, *compl);
        if(rtn_code == 0)
        {
            if(*lenp != 21)     
                return(BAD_VALUE);
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_userDefinedProtocolDirectoryIndex
*  Path         :   1.3.6.1.4.1.26.2.25.4.1.1
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*         "The port number of the port for which this entry contains
*          protocol filtering information."
*
**************************************************************************/

int svc_userDefinedProtocolDirectoryIndex(service,compc,compl,lenp,value,index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    int rtn_code = 0;

    if(*compl >= 10) 
        rtn_code  = NO_SUCH_NAME;

    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */
    case SNMP_GET:
        if (rtn_code == 0)
        {
            *value = (UINT_32_T) *compl;
        }
        else
            return NO_SUCH_NAME;
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_userDefinedProtocolDirectoryType
*  Path         :   1.3.6.1.4.1.26.2.25.4.1.2
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*
**************************************************************************/

int svc_userDefinedProtocolDirectoryType(service,compc,compl,lenp,value,index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    
	int    rtn_code = 0; 
	uint   PortMask;
	uint   type;
    UDPDIR     *udp = udpdir;

    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */

    case SNMP_GET:
        if(*compl > 10) 
            rtn_code = NO_SUCH_NAME;

        if (rtn_code == 0)
        {
			type = (udp+(*compl-1))->UProtTyp;
			bswap(&type);
            *value = (UINT_32_T) type;
        }
        else
            return NO_SUCH_NAME;
        break;

    case SNMP_SET:
        {
			/* Make sure that this protocol directory entry is not in use */
			if( (udp+(*compl-1))->PrtSet == 0 ) {
                type = *value;  
			    bswap(&type);
                (udp+(*compl-1))->UProtTyp = type;
			    SaveProtocolTbl();
			}
			else
				return(ERR_INSTANCE);
        }
        break;

    case SNMP_TEST:
        if(*compl > 10) 
            rtn_code = NO_SUCH_NAME;
        if(rtn_code == 0)
        {
			/* type value must be greater than 1518(base 10) */
            if(*value < 0x05EF || *value > 0xffff)
             return(BAD_VALUE);
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_userDefinedProtocolDirectoryName
*  Path         :   1.3.6.1.4.1.26.2.25.4.1.3
*  Access       :   RW
*  Syntax       :   OCTET VT_STRING
*  Description  :   
*                   "An instance of this object-type exists for each entry in
*                    the Filter Table of a HLS multi port ethernet bridge. The
*                    format of the OCTET STRING is: 
*
**************************************************************************/

int svc_userDefinedProtocolDirectoryName(service,compc,compl,lenp,value,index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
OCTET_P      value;
int          index;
{
    int  rtn_code = 0;
    uint UserVal;
    uint cnt;
    UDPDIR    *udp      = udpdir;
    char *tmpval        = (char *)value;
    PROTCNTLREC *plist  = Protocols;


    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */
    case SNMP_GET:

        if(*compl > 10) 
            rtn_code = NO_SUCH_NAME;

        if (rtn_code == 0)
        {
            /* 1-8 Octets are for the Pre-defined protocol type name */
            for(cnt=0; cnt < 8; cnt++) {
            
                if( (strlen((udp+(*compl-1))->UProtStr) > 0 ) )
                    /*Portocol type is set*/ 
                    *tmpval = (udp+(*compl-1))->UProtStr[cnt];  
                else
                    /* Portocol type is not set */
                    *tmpval = (char) NULL;  
                tmpval++;
            }
            *lenp = 8;
        }
        else
            return NO_SUCH_NAME;
        break;

    case SNMP_SET:
        {
            /*Portocol type is not in use*/ 
            if( (udp+(*compl-1))->PrtSet == 0 ){
                for(cnt=0; cnt < 8; cnt++) {
                    (udp+(*compl-1))->UProtStr[cnt] = *tmpval;  
                    tmpval++;
	            }
			}
			else
				return(BAD_VALUE);
        }

        SaveProtocolTbl();
        break;

    case SNMP_TEST:
        if(*compl > 10) 
            rtn_code = NO_SUCH_NAME;
        if(rtn_code == 0)
        {
            if(*lenp > 8)     
                return(BAD_VALUE);
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/*    Bit Mask Filter Define Routines */


/**************************************************************************
*  Procedure    :   svc_bmaskDefIndex
*  Path         :   1.3.6.1.4.1.26.2.25.5.1.1
*  Access       :   RO
*  Syntax       :   DisplayString
*  Description  :   
*         "The index of the bit mask table for which this entry contains bit
*          mask filtering information."
*
**************************************************************************/

int svc_bmaskDefIndex(service,compc,compl,lenp,value,index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
char         *value;
int          index;
{
    int rtn_code = 0;

    if(*compl >= 10) 
        rtn_code  = NO_SUCH_NAME;

    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */
    case SNMP_GET:
        if (rtn_code == 0)
        {
            *value = (char)(((*compl)-1) + 'a');
			*lenp = 1;
        }
        else
            return NO_SUCH_NAME;
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_bmaskDefValue
*  Path         :   1.3.6.1.4.1.26.2.25.5.1.2
*  Access       :   RW
*  Syntax       :   OCTET VT_STRING
*  Description  :   
*                   "Identifies the value of the bit mask for this entry
*                    This entry must be 8 octets in length. Each octet
*                    should contain either a valid hex value such as 
*                    '3' or and 'X'. A value of 'X' will mean, 
*                    don't care.
*
**************************************************************************/

int svc_bmaskDefValue(service,compc,compl,lenp,value,index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
OCTET_P      value;
int          index;
{
    uint       cnt;
	int        Index;
    int        rtn_code  = 0;
    char       *tmpval   = (char *)value;
	unsigned char       *bmvalue;
    char       Buffer[9];
	RECORDMASK ValueMask;

    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */
    case SNMP_GET:

        if(*compl > 10) 
            rtn_code = NO_SUCH_NAME;

        if (rtn_code == 0)
        {
			bmvalue = BridgeStatus->BMElement[(*compl)-1].ValueString;
            /* 1-8 Octets are for bit mask filter value */
            for(cnt=0; cnt < 8; cnt++) {
            
                if( (strlen(bmvalue)) > 0 ) 
                    /* bit mask filter value is set*/ 
                    *tmpval = bmvalue[cnt];  
                else
                    /* bit mask filter value is not set */
                    *tmpval = (char) NULL;  
                tmpval++;
            }
            *lenp = 8;
        }
        else
            return NO_SUCH_NAME;
        break;

    case SNMP_SET:
        {
			/* Setup */
			memset(Buffer, NULL, sizeof(Buffer));
			memset(&ValueMask, NULL, sizeof(RECORDMASK));
            Index = (*compl)-1;
			bmvalue = BridgeStatus->BMElement[Index].ValueString;

            /* bitmask can only be defined if it's corresponding 'status' */ 
			/* entry has been set to valid, and the definition is not in  */
			/* use by a boolean bitmask expression.                       */
            if( ((strlen(bmvalue)) != 0) && 
				(BridgeStatus->BMElement[Index].BitMask.bm_count == 0))  {

				memcpy(Buffer, tmpval, 8);

				if( HexToMask(Buffer, &ValueMask, 4) ) 
                    return(BAD_VALUE); 
	           
			   /* Define bit mask value and name */
		       memcpy(&BridgeStatus->BMElement[Index].BitMask.bm_mask,
							 &ValueMask.AddressMask, 4);

				memcpy(&BridgeStatus->BMElement[Index].BitMask.bm_mtch, 
							 &ValueMask.AddressValue, 4);
				
				memcpy(BridgeStatus->BMElement[Index].ValueString, Buffer,
							 sizeof(Buffer));
			}
			else
				return(BAD_VALUE);
        }

		SaveBStatus();
        break;

    case SNMP_TEST:

        if(*compl > 10) 
            rtn_code = NO_SUCH_NAME;

        if(rtn_code == 0)
        {
            if(*lenp > 8)     
                return(BAD_VALUE);
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_bmaskDefOffset
*  Path         :   1.3.6.1.4.1.26.2.25.5.1.3
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "Identifies the offset into a packet, that a comparison
*                    between the bytes in the packet, and the bytes defined
*                    by bmaskDefValue, for this entry, will begin."
*
**************************************************************************/

int svc_bmaskDefOffset(service,compc,compl,lenp,value,index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    
	int    rtn_code = 0; 
	uint   offset;
	unsigned char       *bmvalue;

    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */

    case SNMP_GET:
        if(*compl > 10) 
            rtn_code = NO_SUCH_NAME;

        if (rtn_code == 0)
        {
			offset = BridgeStatus->BMElement[(*compl)-1].BitMask.bm_ofst;
            *value = (UINT_32_T) offset;
        }
        else
            return NO_SUCH_NAME;
        break;

    case SNMP_SET:
        {
			bmvalue = BridgeStatus->BMElement[(*compl)-1].ValueString;
			/* Make sure that this bit mask entry is not in use, and */
			/* a valid entry exist.                                  */
			if( (BridgeStatus->BMElement[(*compl)-1].BitMask.bm_count == 0 )
				&& (*bmvalue != NULL) ) {

                offset = (uint) *value;  
                BridgeStatus->BMElement[(*compl)-1].BitMask.bm_ofst = offset;
			    SaveBStatus();
			}
			else
				return(BAD_VALUE);
        }
        break;

    case SNMP_TEST:
        if(*compl > 10) 
            rtn_code = NO_SUCH_NAME;
        if(rtn_code == 0)
        {
			/* offset value must be greater than 0 and less than 256 */
            if(*value > 256 || *value < 0)
             return(BAD_VALUE);
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}


/**************************************************************************
*  Procedure    :   svc_bmaskDefCurrentUses
*  Path         :   1.3.6.1.4.1.26.2.25.5.1.4
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*                   "Identifies the number of times that this bit mask 
*                    definition has been used as part of a boolean bit
*                    mask expression. A bit mask definition which is in
*                    use cannot be deleted or modified.
*
**************************************************************************/

int svc_bmaskDefCurrentUses(service,compc,compl,lenp,value,index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    int rtn_code = 0;

    if(*compl >= 10) 
        rtn_code  = NO_SUCH_NAME;

    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */
    case SNMP_GET:
        if (rtn_code == 0)
        {
         *value=(UINT_32_T)BridgeStatus->BMElement[(*compl)-1].BitMask.bm_count;
			
        }
        else
            return NO_SUCH_NAME;
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_bmaskDefStatus
*  Path         :   1.3.6.1.4.1.26.2.25.5.1.5
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*     "Identifies whether the value and offset fields associated with this
*     entry are valid or invalid. If this object is set to invalid, then
*     the associated value and offset fields will be set to null. If
*     this object is set to valid, then the associated value and offset
*     fields will be set to default values.   VALUE:  FFFFFFFF
*                                             OFFSET: 0"
*
**************************************************************************/

int svc_bmaskDefStatus(service,compc,compl,lenp,value,index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    
	int           rtn_code = 0; 
	uint          Index;
	uint          offset;
    char          Buffer[9];
	RECORDMASK    ValueMask;
    unsigned char *bmvalue;

    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */

    case SNMP_GET:
        if(*compl > 10) 
            rtn_code = NO_SUCH_NAME;

        if (rtn_code == 0)
        {
			/* Validity of entry is determined implicitly, by checking */
			/* for a non-null entry in the ValueString element of the  */
			/* BMElement structure which is contained in NVRAM         */
			bmvalue = BridgeStatus->BMElement[(*compl)-1].ValueString;
            if( *bmvalue == '\0' )
                *value = (UINT_32_T) 2;  /* Entry is invalid */
			else
                *value = (UINT_32_T) 1;  /* Entry is valid */
        }
        else
            return NO_SUCH_NAME;
        break;

    case SNMP_SET:
        {
			/* Make sure that this bit mask entry is not in use.         */
			/* Changing or deleting a bit mask definition while it is in */
			/* use could produce disastrous effects on bridge operation. */
			if( (BridgeStatus->BMElement[(*compl)-1].BitMask.bm_count == 0 )) {

				switch(*value) {

					/* invalid */
					case 2:
                         memset(&BridgeStatus->BMElement[(*compl)-1], 
								NULL, sizeof(BMDEFINE));
                         SaveBStatus();
                         break;

					/* valid */
					case 1:

						/* Set Offset Value to Default */
                        BridgeStatus->BMElement[(*compl)-1].BitMask.bm_ofst = 0;

						/* Set Bit Mask Value to Default */
			            memset(Buffer, NULL, sizeof(Buffer));
             			memset(&ValueMask, NULL, sizeof(RECORDMASK));
                        Index = (*compl)-1;
             			bmvalue = BridgeStatus->BMElement[Index].ValueString;

			            strcpy(Buffer, "f0000000");

		             	if( HexToMask(Buffer, &ValueMask, 4) ) 
                                return(BAD_VALUE); 
	           
		                /* Define bit mask value and name */
	                    memcpy(&BridgeStatus->BMElement[Index].BitMask.bm_mask,
                               &ValueMask.AddressMask, 4);
            
		             	memcpy(&BridgeStatus->BMElement[Index].BitMask.bm_mtch, 
                               &ValueMask.AddressValue, 4);
				
			            memcpy(BridgeStatus->BMElement[Index].ValueString,
                               Buffer, sizeof(Buffer));
			            SaveBStatus();
                        break;
				}
			}
			else
				return(BAD_VALUE);
        }
        break;

    case SNMP_TEST:
        if(*compl > 10) 
            rtn_code = NO_SUCH_NAME;

        if(rtn_code == 0)
        {
			/* Status value must be valid(1) or invalid(2) */
            if(*value != 1 && *value != 2)
                return(BAD_VALUE);

			if( (BridgeStatus->BMElement[(*compl)-1].BitMask.bm_count != 0 )) 
				return(BAD_VALUE);
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_bMaskExpPort
*  Path         :   1.3.6.1.4.1.26.2.25.6.1.1
*  Access       :   RO
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*       "The port number of the port for which this entry contains bit mask
*        filtering information."
*
**************************************************************************/

int svc_bMaskExpPort(service,compc,compl,lenp,value,index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
char         *value;
int          index;
{
    int rtn_code = 0;

    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */
    case SNMP_GET:
	
        rtn_code = chk_portidx(*compc, *compl);
        if (rtn_code == 0)
        {
            *value = (char)(*compl);
        }
        else
            return NO_SUCH_NAME;
        break;
    }
    return 0;
}

/**************************************************************************
*  Procedure    :   svc_bMaskExpMode
*  Path         :   1.3.6.1.4.1.26.2.25.6.1.2
*  Access       :   RW
*  Syntax       :   INTEGER VT_NUMBER
*  Description  :   
*	  "Describes how a packet, which matches the bit mask expression
*	   string for this port, will be handled. Before an instance of this
*	   object type can be set, a valid expression must exist in the
*	   corresponding instance of the bMaskExpString object. When an
*	   instance of this object type is set to none, the corresponding
*	   instance of the bMaskExpString is cleared by setting it to null." 
*
**************************************************************************/

int svc_bMaskExpMode(service,compc,compl,lenp,value,index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
UINT_32_T    *value;
int          index;
{
    
	int    rtn_code = 0; 
	uint   PortMask;
	uint   offset;
    uint   idx;

    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */

    case SNMP_GET:
        rtn_code = chk_portidx(*compc, *compl);
        if (rtn_code == 0)
        {
		    switch(BridgeStatus->BMExpression[(*compl)-1].BMMode) {

		    	case BLOCK:  *value = (UINT_32_T) 1; /* block */
                             break;

		    	case PERMIT: *value = (UINT_32_T) 2; /* permit */
		     	             break;

				case NULL:   *value = (UINT_32_T) 3; /* invalid */
                             break;
			}
        }
        else
            return NO_SUCH_NAME;
        break;

    case SNMP_SET:
        {
			/* There must be a valid expression, prior to allowing a set. */
			idx = (*compl) - 1;
			if( BridgeStatus->BMExpression[idx].BMExpString == NULL ) 
				return(BAD_VALUE);

            PortMask = Indx2Mask( *compl );
            switch( *value ) {

				case 1:   /* block mode */
						  BridgeStatus->BMExpression[idx].BMMode = BLOCK;
						  prcctl->Prc_BitMskFltMode &= (byte)~PortMask;
						  prcctl->Prc_BitMskFlt     |= (byte)PortMask;
						  break;

				case 2:   /* permit mode */
						  BridgeStatus->BMExpression[idx].BMMode = PERMIT;
						  prcctl->Prc_BitMskFltMode |= (byte)PortMask;
						  prcctl->Prc_BitMskFlt     |= (byte)PortMask;
						  break;

				case 3:   /* invalid */
						  /* Turn off the bit mask */
						  prcctl->Prc_BitMskFlt &= (byte)~PortMask;

						  /* Delete the bit mask expression for this port */
                          DecrementDefinition(
							  &BridgeStatus->BMExpression[idx].BMExpRevPolish);
						  memset(&BridgeStatus->BMExpression[idx], NULL,
                              sizeof(BMEXPRESSION));
					      break;
			}
			SaveBStatus();
        }
        break;

    case SNMP_TEST:

        rtn_code = chk_portidx(*compc, *compl);

        if(rtn_code == 0)
        {
			/* value must be greater less than 3 and greater than 0 */
            if(*value < 1 || *value > 3)
             return(BAD_VALUE);
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}


/**************************************************************************
*  Procedure    :   svc_bMaskExpString
*  Path         :   1.3.6.1.4.1.26.2.25.6.1.3
*  Access       :   RW
*  Syntax       :   OCTET VT_STRING
*  Description  :   
*                  "Boolean expression made up of variables a - j, defined in
*                   bmaskDefTab, and the NOT ,!, AND ,&, OR ,|, parenthesis
*                   ,( ), operators. The variables a - j, must be defined via
*                   the bmaskDefTab objects or via the bridge console prior
*                   to use. A usage example: (a&b)|c."
*
**************************************************************************/

int svc_bMaskExpString(service,compc,compl,lenp,value,index)
unsigned int service;
byte         *compc;
unsigned int *compl;
int          *lenp;
OCTET_P      value;
int          index;
{
	uint       idx;
	uint       PortMask;
    int        rtn_code  = 0;
	char       valbuf[BMFOLEN];
	char       expbuf[BMFOLEN];
	char       revpolishbuf[BMFOLEN];
	char       buf[16];
	RECORDMASK ValueMask;

    switch(service)
    {
    case SNMP_GETNXT:
        rtn_code = get_nxtportidx((int *)compc,
                    (unsigned int *)compl, rtn_code);
        /* FALL TRHROUGH */
    case SNMP_GET:

        rtn_code = chk_portidx(*compc, *compl);
		idx = (*compl) - 1;
        if (rtn_code == 0)
        {
            PortMask = Indx2Mask( *compl );
			memset(buf, NULL, 16);
			memcpy(buf, BridgeStatus->BMExpression[idx].BMExpString, 16); 
			memcpy(value, buf, 16); 
            *lenp = 16;
        }
        else
            return NO_SUCH_NAME;
        break;

    case SNMP_SET:
        {
			/* SetUp */
		    idx = (*compl) - 1;
            PortMask = Indx2Mask( *compl );
            memset(valbuf, NULL, BMFOLEN);
            memset(expbuf, NULL, BMFOLEN);
            memset(revpolishbuf, NULL, BMFOLEN);
			memcpy(valbuf, value, *lenp);
            strlwr(valbuf);

			/* Check Syntax and Parse Expression */
			RemoveBlank(&expbuf, &valbuf);
			if( ParseExpression(&revpolishbuf, &expbuf) )
				return(BAD_VALUE);
		
			/* Delete Old Expression If Any */
            DecrementDefinition(
						   &BridgeStatus->BMExpression[idx].BMExpRevPolish);
			memset(&BridgeStatus->BMExpression[idx], NULL, 
													 sizeof(BMEXPRESSION));
		   /* Add New Expression */
		   strupr(expbuf);
		   memcpy(&BridgeStatus->BMExpression[idx].BMExpRevPolish,
				   revpolishbuf, sizeof(revpolishbuf));
		   memcpy(&BridgeStatus->BMExpression[idx].BMExpString, expbuf,
				   sizeof(expbuf));
           IncrementDefinition(&BridgeStatus->BMExpression[idx].BMExpRevPolish);
		   memcpy(&prcctl->Prc_BitMskFltExp[idx], &revpolishbuf,
														 sizeof(revpolishbuf));
		   BridgeStatus->BMExpression[idx].BMMode = BLOCK;
		   prcctl->Prc_BitMskFltMode &= (byte)~PortMask;
           prcctl->Prc_BitMskFlt     |= (byte)PortMask;
		   SaveBStatus();
        }

        break;

    case SNMP_TEST:

        rtn_code = chk_portidx(*compc, *compl);
        if(rtn_code == 0)
        {
            if( *lenp >= 16  || *lenp < 1)     
                return(BAD_VALUE);
        }
        else
            return(rtn_code);
        break;
    }
    return 0;
}


/****************************************************************************
**
**   CHK_PORTIDX ROUTINE
**
**   Check the instance of a tabular object which is indexed by port number.
**
**   Return: 0:            If instance count 1 and instance value is between
**                         1 and 4.
**           NO_SUCH_NAME: All other conditions.
**
****************************************************************************/

chk_portidx(uint inst_len, uint index) {

   if( (inst_len != 1) || (index < 1) || (index > 4) )
       return( NO_SUCH_NAME );
   else
       return(0);
}


/****************************************************************************
**
**   GET_NXTPORTIDX ROUTINE
**
**   Increment the instance of a tabular object which is indexed
**   by port number.
**
**   Return: 0:            If instance count 1 and instance value is between
**                         1 and 4.
**           NO_SUCH_NAME: All other conditions.
**
****************************************************************************/

get_nxtportidx( uint *inst_len, uint *index, int retval ) {

    if( *inst_len == 0 ) {

        *inst_len = 1;
        *index    = 1;
    }
    else
    if( retval != 0 )
        return(ERR_INSTANCE);
    else
        *index += 1;

    return(0);
}


/*****************************************************************************
**                                                                           *
**  Start Indx2Mask()                                                        *
**  Convert an snmp port index value into a port bitmask.                    *
**  PORT1, etc... are defined in bridges.h as bitmasks.                      *
**                                                                           *
******************************************************************************/
Indx2Mask( uint indx )  {

    switch( indx )  {
        case 1: 
            return( PORT1 );
        case 2:
            return( PORT2 );
        case 3:
            return( PORT3 );
        case 4:
            return( PORT4 );
        case 5:
            return( PORT5 );
    }
}

CheckSetValues( char *tmpval, uint len )  {

    uint cnt;

    for(cnt=0; cnt < len; cnt++) {
      if( (*(tmpval+cnt) != 1) && (*(tmpval+cnt) != 2) )
       return(BAD_VALUE);
    }
    return(0);
}
