static char rcsid[] = "$Header: ufs_alloc.c,v 86.1 86/07/21 14:57:31 bog Exp $";

/************************************************************************\
**									**
**				Copyright 1986				**
**			VALID LOGIC SYSTEMS INCORPORATED		**
**									**
**	This listing contains confidential proprietary information	**
**	which is not to be disclosed to unauthorized persons without	**
**	written consent of an officer of Valid Logic Systems 		**
**	Incoroporated.							**
**									**
**	The copyright notice appearing above is included to provide	**
**	statutory protection in the event of unauthorized or 		**
**	unintentional public disclosure.				**
**									**
\************************************************************************/

/*
 * ufs_alloc.c
 *
 * Routines related to block and inode allocation in a Unix File System.
 * Taken from sys/ufs_alloc.c 0786 by Bill O. Gallmeister.
 */
#include "../h/param.h"
#include "../h/systm.h"
#include "../h/mount.h"
#include "../h/fs.h"
#include "../h/conf.h"
#include "../h/buf.h"
#include "../h/inode.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/quota.h"
#include "../h/kernel.h"

#include "saio.h"	/* Nifty standalone stuff. */

/*
 * Externs for communicating to lower layers.
 */
extern u_long		hashalloc();
extern ino_t		ialloccg();
extern daddr_t		alloccg();
extern daddr_t		alloccgblk();
extern daddr_t		fragextend();
extern daddr_t		blkpref();
extern daddr_t		mapsearch();
extern int		inside[], around[];
extern unsigned char	*fragtbl[];

/*
 * Externs for communicating to higher layers.
 */
extern int sysdebug;		/* Print system-level debugging statements. */
extern int errno;		/* Global error summary (like UNIX). */

/*
 * Allocate a block in the file system.
 * 
 * The size of the requested block is given, which must be some
 * multiple of fs_fsize and <= fs_bsize.
 *
 * A preference may be optionally specified. If a preference is given
 * the following hierarchy is used to allocate a block:
 *   1) allocate the requested block.
 *   2) allocate a rotationally optimal block in the same cylinder.
 *   3) allocate a block in the same cylinder group.
 *   4) quadradically rehash into other cylinder groups, until an
 *      available block is located.
 * If no block preference is given the following heirarchy is used
 * to allocate a block:
 *   1) allocate a block in the cylinder group that contains the
 *      inode for the file.
 *   2) quadradically rehash into other cylinder groups, until an
 *      available block is located.
 *
 * Returns a disk block number (NOT file system block number!).
 * Returns -1 on failure.
 *
 * Implicitly: Changes CG and FS structures; changes incore inode.
 */
daddr_t
sa_alloc(io, bpref, size)
	register struct iob *io;
	daddr_t bpref;
	int size;
{
	daddr_t bno;
	register struct fs *fs = io->i_fs;
	int cg;
	register struct inode *ip = io->i_ino;
	
	if ((unsigned)size > fs->fs_bsize || fragoff(fs, size) != 0) {
		if (sysdebug)
		{
			printf("alloc: bad size: dev = 0x%x, bsize = %d, ",
		    		ip->i_dev, fs->fs_bsize);
			printf("size = %d, fs = %s\n",
		    		size, fs->fs_fsmnt);
		}
		return(-1);
	}
	if (size == fs->fs_bsize && fs->fs_cstotal.cs_nbfree == 0)
		goto nospace;
	if (freespace(fs, fs->fs_minfree) <= 0)
		goto nospace;
	if (bpref >= fs->fs_size)
		bpref = 0;
	if (bpref == 0)
		cg = itog(fs, ip->i_number);
	else
		cg = dtog(fs, bpref);
	bno = (daddr_t)hashalloc(io, cg, (long)bpref, size,
		(u_long (*)())alloccg);
	if (bno <= 0)
		goto nospace;
	ip->i_blocks += btodb(size);
	ip->i_flag |= IUPD|ICHG;
	return (bno);
nospace:
	errno = ENOSPC;
	printf("write fails; file system full\n");
	exit (2);
}

/*
 * Reallocate a fragment to a bigger size
 *
 * The number and size of the old block is given, and a preference
 * and new size is also specified. The allocator attempts to extend
 * the original block. Failing that, the regular block allocator is
 * invoked to get an appropriate block.
 */
daddr_t
realloccg(io, bprev, bpref, osize, nsize)
	register struct iob *io;
	daddr_t bprev, bpref;
	int osize, nsize;
{
	daddr_t bno;
	register struct fs *fs;
	int cg;
	register struct inode *ip = io->i_ino;
	
	fs = io->i_fs;
	if ((unsigned)osize > fs->fs_bsize || fragoff(fs, osize) != 0 ||
	    (unsigned)nsize > fs->fs_bsize || fragoff(fs, nsize) != 0)
	{
		if (sysdebug)
		{
			printf("realloccg: bad size: dev = 0x%x, bsize = %d, ",
		    		ip->i_dev, fs->fs_bsize);
			printf("osize = %d, nsize = %d, fs = %s\n",
				osize, nsize, fs->fs_fsmnt);
		}
		return(-1);
		
	}
	if (freespace(fs, fs->fs_minfree) > 0)
	{
		if (bprev == 0) {
			if (sysdebug)
			{
				printf("realloccg: bad bprev: dev = 0x%x, ",
					ip->i_dev);
				printf("bsize = %d, bprev = %d, fs = %s\n",
					fs->fs_bsize, bprev, fs->fs_fsmnt);
			}
			return(-1);
		}
		cg = dtog(fs, bprev);
		bno = fragextend(io, cg, (long)bprev, osize, nsize);
		if (bno != 0) {
			ip->i_blocks += btodb(nsize - osize);
			ip->i_flag |= IUPD|ICHG;
			return (bno);
		}
		if (bpref >= fs->fs_size)
			bpref = 0;
		bno = (daddr_t)hashalloc(io, cg, (long)bpref, nsize,
			(u_long (*)())alloccg);
		if (bno > 0) {
			sa_free(io, bprev, (off_t)osize);
			ip->i_blocks += btodb(nsize - osize);
			ip->i_flag |= IUPD|ICHG;
			return (bno);
		} else
			return (-1);
	}
	else
	{
		/*
		 * No space available.
		 */
		errno = ENOSPC;
		printf("write fails; file system full\n");
		exit(2);
	}
}

/*
 * Allocate an inode in the file system.
 * 
 * A preference may be optionally specified. If a preference is given
 * the following hierarchy is used to allocate an inode:
 *   1) allocate the requested inode.
 *   2) allocate an inode in the same cylinder group.
 *   3) quadradically rehash into other cylinder groups, until an
 *      available inode is located.
 * If no inode preference is given the following heirarchy is used
 * to allocate an inode:
 *   1) allocate an inode in cylinder group 0.
 *   2) quadradically rehash into other cylinder groups, until an
 *      available inode is located.
 *
 * Notes to myself:
 *	The inode allocated is opened up via openi() -- this means it ends up
 *	in the global pool of inodes.  Watch out for leaving it lying around.
 *	(shouldn't happen -- next time we do an openi using this iob, we will
 *	close this inode (see sys.c for the good word on openi/closei.
 */
struct inode *
ialloc(io, ipref, mode)
	register struct iob *io;
	ino_t ipref;
	int mode;
{
	ino_t ino;
	register struct fs *fs;
	struct inode ibuf;
	register struct inode *ip = &ibuf;
	int cg;
	
	fs = io->i_fs;
	if (fs->fs_cstotal.cs_nifree == 0)	/* No free inodes left. */
		goto noinodes;
	if (ipref >= fs->fs_ncg * fs->fs_ipg)	/* Ridiculous preference. */
		ipref = 0;
	cg = itog(fs, ipref);	/* Preferred cylinder group. */
	/*
	 * Hashalloc() hammers bits in the CG structures written on disk,
	 * but does not return a new inode structure.
	 */
	/*
	 * Notes to myself:
	 * When ialloc is called, does io->i_ino contain anything useful?
	 */
	ino = (ino_t)hashalloc(io, cg, (long)ipref, mode, ialloccg);
	if (ino == 0)
		goto noinodes;
	/* Snag new inode from disk. */
	if (openi(ino, io) < 0) {
		/* Retrieval barf. */
		if (sysdebug)
			printf("ialloc: openi(0x%x, 0x%x) fails.\n",
				ino, io);
		return (NULL);
	}
	ip = io->i_ino;
	if (ip->i_mode) {	/* Ach, nein! This compartment in use! */
		if (sysdebug) {
			printf("ialloc: dup alloc: mode = 0%o, inum = %d, ",
				ip->i_mode, ip->i_number);
			printf("fs = %s\n", fs->fs_fsmnt);
		}
		return(NULL);
	}
	bclear (ip->i_db, (NDADDR + NIADDR) * sizeof ip->i_db[0]);
	if (ip->i_blocks) {
		if (sysdebug)
			printf("free inode %s/%d had %d blocks\n",
			    fs->fs_fsmnt, ino, ip->i_blocks);
		ip->i_blocks = 0;
	}
	return (ip);
noinodes:
	printf("ialloc: out of inodes\n");
	errno = ENOSPC;
	return (NULL);
}

/*
 * Find a cylinder to place a directory.
 *
 * The policy implemented by this algorithm is to select from
 * among those cylinder groups with above the average number of
 * free inodes, the one with the smallest number of directories.
 *
 * This is part of the so-called "Fast File System for Unix (TM)".
 */
ino_t
dirpref(fs)
	register struct fs *fs;
{
	int cg, minndir, mincg, avgifree;

	avgifree = fs->fs_cstotal.cs_nifree / fs->fs_ncg;
	minndir = fs->fs_ipg;
	mincg = 0;
	for (cg = 0; cg < fs->fs_ncg; cg++)
		if (fs->fs_cs(fs, cg).cs_ndir < minndir &&
		    fs->fs_cs(fs, cg).cs_nifree >= avgifree) {
			mincg = cg;
			minndir = fs->fs_cs(fs, cg).cs_ndir;
		}
	return ((ino_t)(fs->fs_ipg * mincg));
}

/*
 * Select the desired position for the next block in a file.  The file is
 * logically divided into sections. The first section is composed of the
 * direct blocks. Each additional section contains fs_maxbpg blocks.
 * 
 * If no blocks have been allocated in the first section, the policy is to
 * request a block in the same cylinder group as the inode that describes
 * the file. If no blocks have been allocated in any other section, the
 * policy is to place the section in a cylinder group with a greater than
 * average number of free blocks.  An appropriate cylinder group is found
 * by maintaining a rotor that sweeps the cylinder groups. When a new
 * group of blocks is needed, the rotor is advanced until a cylinder group
 * with greater than the average number of free blocks is found.
 * 
 * If a section is already partially allocated, the policy is to
 * contiguously allocate fs_maxcontig blocks.  The end of one of these
 * contiguous blocks and the beginning of the next is physically separated
 * so that the disk head will be in transit between them for at least
 * fs_rotdelay milliseconds.  This is to allow time for the processor to
 * schedule another I/O transfer.
 */
daddr_t
blkpref(ip, lbn, indx, bap)
	struct inode *ip;
	daddr_t lbn;
	int indx;
	daddr_t *bap;
{
	register struct fs *fs;
	int cg, avgbfree;
	daddr_t nextblk;

	fs = ip->i_fs;
	/*
	 * If index is a multiple of # blocks per group
	 * (first block in a group?),
	 * or the block for the index isn't allocated...
	 */
	if (indx % fs->fs_maxbpg == 0 || bap[indx - 1] == 0) {
		if (lbn < NDADDR) {
			/* Direct block. Pick cyl grp same as inode's. */
			cg = itog(fs, ip->i_number);
			/*
			 * Return first available frag in the cyl grp.
			 * (Assume first fs_frag frags is the superblock)
			 */
			return (fs->fs_fpg * cg + fs->fs_frag);
		}
		/*
		 * Indirect block.
		 *
		 * Find a cylinder with greater than average number of
		 * unused data blocks.
		 */
		avgbfree = fs->fs_cstotal.cs_nbfree / fs->fs_ncg;
		/*
		 * The next two loops walk through cyl groups. If we
		 * find one with a greater than average # free blocks,
		 * we use it. In this case, return block near beginning
		 * of that cyl group again.
		 */
		for (cg = fs->fs_cgrotor + 1; cg < fs->fs_ncg; cg++)
			if (fs->fs_cs(fs, cg).cs_nbfree >= avgbfree) {
				fs->fs_cgrotor = cg;
				return (fs->fs_fpg * cg + fs->fs_frag);
			}
		for (cg = 0; cg <= fs->fs_cgrotor; cg++)
			if (fs->fs_cs(fs, cg).cs_nbfree >= avgbfree) {
				fs->fs_cgrotor = cg;
				return (fs->fs_fpg * cg + fs->fs_frag);
			}
		return (NULL);
	}
	/*
	 * One or more previous blocks have been laid out. If less
	 * than fs_maxcontig previous blocks are contiguous, the
	 * next block is requested contiguously, otherwise it is
	 * requested rotationally delayed by fs_rotdelay milliseconds.
	 */
	nextblk = bap[indx - 1] + fs->fs_frag;
	if (indx > fs->fs_maxcontig &&
	    bap[indx - fs->fs_maxcontig] + blkstofrags(fs, fs->fs_maxcontig)
	    != nextblk)
		return (nextblk);
	if (fs->fs_rotdelay != 0)
		/*
		 * Here we convert ms of delay to frags as:
		 * (frags) = (ms) * (rev/sec) * (sect/rev) /
		 *	((sect/frag) * (ms/sec))
		 * then round up to the next block.
		 */
		nextblk += roundup(fs->fs_rotdelay * fs->fs_rps * fs->fs_nsect /
		    (NSPF(fs) * 1000), fs->fs_frag);
	return (nextblk);
}

/*
 * Implement the cylinder overflow algorithm.
 *
 * The policy implemented by this algorithm is:
 *   1) allocate the block in its requested cylinder group.
 *   2) quadradically rehash on the cylinder group number.
 *   3) brute force search for a free block.
 *
 * The allocator routine takes the following arguments:
 *   1) a struct iob * (for disk access).
 *   2) an integer cg #.
 *   3) an integer block preference.
 *   4) an integer size to allocate.
 */
u_long
hashalloc(io, cg, pref, size, allocator)
	register struct iob *io;
	register int cg;
	long pref;
	register int size;	/* size for data blocks, mode for inodes */
	u_long (*allocator)();
{
	register struct fs *fs = io->i_fs;
	long result;
	register int i, icg = cg;

	/*
	 * 1: preferred cylinder group
	 */
	result = (*allocator)(io, cg, pref, size);
	if (result)
		return (result);
	/*
	 * 2: quadratic rehash
	 */
	for (i = 1; i < fs->fs_ncg; i *= 2) {
		cg += i;
		if (cg >= fs->fs_ncg)
			cg -= fs->fs_ncg;
		result = (*allocator)(io, cg, 0, size);
		if (result)
			return (result);
	}
	/*
	 * 3: brute force search
	 * Note that we start at i == 2, since 0 was checked initially,
	 * and 1 is always checked in the quadratic rehash.
	 */
	cg = (icg + 2) % fs->fs_ncg;
	for (i = 2; i < fs->fs_ncg; i++) {
		result = (*allocator)(io, cg, 0, size);
		if (result)
			return (result);
		cg++;
		if (cg == fs->fs_ncg)
			cg = 0;
	}
	return (NULL);
}

/*
 * Determine whether a fragment can be extended.
 *
 * Check to see if the necessary fragments are available, and 
 * if they are, allocate them.
 */
daddr_t
fragextend(io, cg, bprev, osize, nsize)
	register struct iob *io;
	register int cg;
	long bprev;
	register int osize, nsize;
{
	register struct fs *fs;
	register struct cg *cgp;
	long bno;
	register int frags, bbase;
	int i, cc;
	register struct inode *ip = io->i_ino;

	fs = io->i_fs;
	if (fs->fs_cs(fs, cg).cs_nffree < nsize - osize)
		return (NULL);
	frags = numfrags(fs, nsize);
	bbase = fragoff(fs, bprev);
	if (bbase > (bprev + frags - 1) % fs->fs_frag) {
		/* cannot extend across a block boundry */
		return (NULL);
	}
	/*
	 * Read in the Cylinder Group Summary Structure from disk.
	 */
	io->i_bn = fsbtodb(fs, cgtod(fs, cg)) + io->i_boff;
	io->i_cc = fs->fs_cgsize;	/* Hope it's big enough! */
	io->i_ma = io->i_buf;
	cc = devread(io);

	cgp = (struct cg *)io->i_buf;
	if ((cc <= 0) || (cgp->cg_magic != CG_MAGIC) ||
		(cgp->cg_cs.cs_nifree == 0))
			return (NULL);
	cgp->cg_time = time.tv_sec;
	bno = dtogd(fs, bprev);
	for (i = numfrags(fs, osize); i < frags; i++)
		if (isclr(cgp->cg_free, bno + i))
		{
			if (sysdebug)
				printf("fragextend: isclr(0x%x, 0x%x)\n",
					cgp->cg_free, bno + i);
			return (NULL);
		}
	/*
	 * The current fragment can be extended.
	 * Deduct the count on fragment being extended into;
	 * increase the count on the remaining fragment (if any);
	 * allocate the extended piece.
	 */
	for (i = frags; i < fs->fs_frag - bbase; i++)
		if (isclr(cgp->cg_free, bno + i))
			break;
	cgp->cg_frsum[i - numfrags(fs, osize)]--;
	if (i != frags)
		cgp->cg_frsum[i - frags]++;
	for (i = numfrags(fs, osize); i < frags; i++) {
		clrbit(cgp->cg_free, bno + i);
		cgp->cg_cs.cs_nffree--;
		fs->fs_cstotal.cs_nffree--;
		fs->fs_cs(fs, cg).cs_nffree--;
	}
	fs->fs_fmod++;
	devwrite(io);	/* Update CG structure on disk. */
	writefs(io);	/* Update FS structure on disk. */
	/*
	 * Used to be:
	 *	return (bno);
	 * Upon review of sys/ufs_alloc.c saw that the return from 
	 * fragextend was:
	 *	return (bprev);
	 * so I changed this one, we'll see what happens...
	 */
	return (bprev);
}

/*
 * Determine whether a block can be allocated.
 *
 * Check to see if a block of the apprpriate size is available,
 * and if it is, allocate it.
 */
daddr_t
alloccg(io, cg, bpref, size)
	register struct iob *io;
	int cg;
	daddr_t bpref;
	int size;
{
	register struct fs *fs = io->i_fs;
	register struct cg *cgp;
	int bno, frags;
	int allocsiz;
	register int i;

	if (fs->fs_cs(fs, cg).cs_nbfree == 0 && size == fs->fs_bsize)
		return (0);
	io->i_bn = fsbtodb(fs, cgtod(fs,cg)) + io->i_boff;
	io->i_ma = io->i_buf;
	io->i_cc = fs->fs_cgsize;
	if (devread(io) < 0)
	{
		if (sysdebug)
			printf("alloccg: devread() failed.\n");
		errno = io->i_error;
		return(0);
	}
	cgp = (struct cg *) io->i_buf;
	if (cgp->cg_cs.cs_nbfree == 0 && size == fs->fs_bsize)
		return (0);
	cgp->cg_time = time.tv_sec;
	if (size == fs->fs_bsize) {
		bno = alloccgblk(fs, cgp, bpref);
		if (devwrite(io) < 0)
		{
			if (sysdebug) printf("alloccg: devwrite() 1 failed.\n");
			errno = io->i_error;
			return(0);
		}
		return (bno);
	}
	/*
	 * check to see if any fragments are already available
	 * allocsiz is the size which will be allocated, hacking
	 * it down to a smaller size if necessary
	 */
	frags = numfrags(fs, size);
	for (allocsiz = frags; allocsiz < fs->fs_frag; allocsiz++)
		if (cgp->cg_frsum[allocsiz] != 0)
			break;
	if (allocsiz == fs->fs_frag) {
		/*
		 * no fragments were available, so a block will be 
		 * allocated, and hacked up.
		 */
		if (cgp->cg_cs.cs_nbfree == 0) {
			return (0);
		}
		bno = alloccgblk(fs, cgp, bpref);
		bpref = dtogd(fs, bno);
		for (i = frags; i < fs->fs_frag; i++)
			setbit(cgp->cg_free, bpref + i);
		i = fs->fs_frag - frags;
		cgp->cg_cs.cs_nffree += i;
		fs->fs_cstotal.cs_nffree += i;
		fs->fs_cs(fs, cg).cs_nffree += i;
		fs->fs_fmod++;
		cgp->cg_frsum[i]++;
		if (devwrite(io) < 0)
		{
			if (sysdebug) printf("alloccg: devwrite() 1 failed.\n");
			errno = io->i_error;
			return(0);
		}
		return (bno);
	}
	bno = mapsearch(fs, cgp, bpref, allocsiz);
	if (bno < 0)
		return (0);
	for (i = 0; i < frags; i++)
		clrbit(cgp->cg_free, bno + i);
	cgp->cg_cs.cs_nffree -= frags;
	fs->fs_cstotal.cs_nffree -= frags;
	fs->fs_cs(fs, cg).cs_nffree -= frags;
	fs->fs_fmod++;
	cgp->cg_frsum[allocsiz]--;
	if (frags != allocsiz)
		cgp->cg_frsum[allocsiz - frags]++;
	if (devwrite(io) < 0)
	{
		if (sysdebug) printf("alloccg: devwrite() 1 failed.\n");
		errno = io->i_error;
		return(0);
	}
	return (cg * fs->fs_fpg + bno);
}

/*
 * Allocate a block in a cylinder group.
 *
 * This algorithm implements the following policy:
 *   1) allocate the requested block.
 *   2) allocate a rotationally optimal block in the same cylinder.
 *   3) allocate the next available block on the block rotor for the
 *      specified cylinder group.
 * Note that this routine only allocates fs_bsize blocks; these
 * blocks may be fragmented by the routine that allocates them.
 */
daddr_t
alloccgblk(fs, cgp, bpref)
	register struct fs *fs;
	register struct cg *cgp;
	daddr_t bpref;
{
	daddr_t bno;
	int cylno, pos, delta;
	short *cylbp;
	register int i;

	if (bpref == 0) {
		bpref = cgp->cg_rotor;
		goto norot;
	}
	bpref &= ~(fs->fs_frag - 1);
	bpref = dtogd(fs, bpref);
	/*
	 * if the requested block is available, use it
	 */
	if (sa_isblock(fs, cgp->cg_free, fragstoblks(fs, bpref))) {
		bno = bpref;
		goto gotit;
	}
	/*
	 * check for a block available on the same cylinder
	 */
	cylno = cbtocylno(fs, bpref);
	if (cgp->cg_btot[cylno] == 0)
		goto norot;
	if (fs->fs_cpc == 0) {
		/*
		 * block layout info is not available, so just have
		 * to take any block in this cylinder.
		 */
		bpref = howmany(fs->fs_spc * cylno, NSPF(fs));
		goto norot;
	}
	/*
	 * check the summary information to see if a block is 
	 * available in the requested cylinder starting at the
	 * requested rotational position and proceeding around.
	 */
	cylbp = cgp->cg_b[cylno];
	pos = cbtorpos(fs, bpref);
	for (i = pos; i < NRPOS; i++)
		if (cylbp[i] > 0)
			break;
	if (i == NRPOS)
		for (i = 0; i < pos; i++)
			if (cylbp[i] > 0)
				break;
	if (cylbp[i] > 0) {
		/*
		 * found a rotational position, now find the actual
		 * block. A screwup if none is actually there.
		 */
		pos = cylno % fs->fs_cpc;
		bno = (cylno - pos) * fs->fs_spc / NSPB(fs);
		if (fs->fs_postbl[pos][i] == -1) {
			printf("alloccgblk: cyl groups corrupted: ");
			printf("pos = %d, i = %d, fs = %s\n",
			    pos, i, fs->fs_fsmnt);
			return(NULL);
		}
		for (i = fs->fs_postbl[pos][i];; ) {
			if (sa_isblock(fs, cgp->cg_free, bno + i)) {
				bno = blkstofrags(fs, (bno + i));
				goto gotit;
			}
			delta = fs->fs_rotbl[i];
			if (delta <= 0 || delta > MAXBPC - i)
				break;
			i += delta;
		}
		printf("alloccgblk: can't find blk in cyl: ");
		printf("pos = %d, i = %d, fs = %s\n", pos, i, fs->fs_fsmnt);
		return(NULL);
	}
norot:
	/*
	 * no blocks in the requested cylinder, so take next
	 * available one in this cylinder group.
	 */
	bno = mapsearch(fs, cgp, bpref, (int)fs->fs_frag);
	if (bno < 0)
		return (NULL);
	cgp->cg_rotor = bno;
gotit:
	sa_clrblock(fs, cgp->cg_free, (long)fragstoblks(fs, bno));
	cgp->cg_cs.cs_nbfree--;
	fs->fs_cstotal.cs_nbfree--;
	fs->fs_cs(fs, cgp->cg_cgx).cs_nbfree--;
	cylno = cbtocylno(fs, bno);
	cgp->cg_b[cylno][cbtorpos(fs, bno)]--;
	cgp->cg_btot[cylno]--;
	fs->fs_fmod++;
	return (cgp->cg_cgx * fs->fs_fpg + bno);
}
	
/*
 * Determine whether an inode can be allocated, and if it can,
 * allocate it by hammering bits in the appropriate CG structure.
 *
 * Check to see if an inode is available, and if it is,
 * allocate it.
 *
 * We use the following policy:
 *   1) allocate the requested inode.
 *   2) allocate the next available inode after the requested
 *      inode in the specified cylinder group.
 *
 * Returns an inode number; returns 0 on failure to allocate.
 */
ino_t
ialloccg(io, cg, ipref, mode)
	register struct iob *io;
	int cg;
	daddr_t ipref;
	int mode;
{
	register struct fs *fs;
	register struct cg *cgp;
	int i,cc;

	fs = io->i_fs;
	if (fs->fs_cs(fs, cg).cs_nifree == 0)
		return (0);

	/*
	 * Read in CG structure from disk.
	 */
	io->i_bn = fsbtodb(fs, cgtod(fs, cg)) + io->i_boff;
	io->i_cc = fs->fs_cgsize;	/* Hope it's big enough! */
	io->i_ma = io->i_buf;
	cc = devread(io);

	cgp = (struct cg *)io->i_buf;
	if ((cc <= 0) || (cgp->cg_magic != CG_MAGIC) ||
		(cgp->cg_cs.cs_nifree == 0))
			return (0);
	cgp->cg_time = time.tv_sec;
	if (ipref) {
		ipref %= fs->fs_ipg;
		if (isclr(cgp->cg_iused, ipref))
			goto gotit;
	} else
		ipref = cgp->cg_irotor;
	for (i = 0; i < fs->fs_ipg; i++) {
		ipref++;
		if (ipref >= fs->fs_ipg)
			ipref = 0;
		if (isclr(cgp->cg_iused, ipref)) {
			cgp->cg_irotor = ipref;
			goto gotit;
		}
	}
	return (0);
gotit:
	setbit(cgp->cg_iused, ipref);
	cgp->cg_cs.cs_nifree--;
	fs->fs_cstotal.cs_nifree--;
	fs->fs_cs(fs, cg).cs_nifree--;
	fs->fs_fmod++;
	if ((mode & IFMT) == IFDIR) {
		cgp->cg_cs.cs_ndir++;
		fs->fs_cstotal.cs_ndir++;
		fs->fs_cs(fs, cg).cs_ndir++;
	}
	cc = devwrite(io);	/* Update CG record on disk. */
	writefs(io);		/* Update superblock on disk. */
	return (cg * fs->fs_ipg + ipref);
}

/*
 * Free a block or fragment.
 *
 * The specified block or fragment is placed back in the
 * free map. If a fragment is deallocated, a possible 
 * block reassembly is checked.
 */
sa_free(io, bno, size)
	register struct iob *io;
	daddr_t bno;
	off_t size;
{
	register struct inode *ip = io->i_ino;
	register struct fs *fs;
	register struct cg *cgp;
	register struct buf *bp;
	int cg, blk, frags, bbase;
	register int i;

	fs = io->i_fs;
	if ((unsigned)size > fs->fs_bsize || fragoff(fs, size) != 0) {
		printf("sa_free: bad size: ");
		printf("dev = 0x%x, bsize = %d, size = %d, fs = %s\n",
		    ip->i_dev, fs->fs_bsize, size, fs->fs_fsmnt);
		_stop ("sa_free");
	}
	cg = dtog(fs, bno);
	if (badblock(fs, bno)) {
		printf("bad block %d, ino %d\n", bno, ip->i_number);
		return;
	}
	io->i_ma = io->i_buf;
	io->i_cc = fs->fs_cgsize;
	io->i_bn = fsbtodb(fs, cgtod(fs, cg)) + io->i_boff;
	if (devread (io) < 0) 
		return -1;
	cgp = (struct cg *) io->i_buf;
	if (cgp->cg_magic != CG_MAGIC)
		return -1;
	cgp->cg_time = time.tv_sec;
	bno = dtogd(fs, bno);
	if (size == fs->fs_bsize) {
		if (sa_isblock(fs, cgp->cg_free, fragstoblks(fs, bno))) {
			printf("free: freeing free block: ");
			printf("dev = 0x%x, block = %d, fs = %s\n",
			    ip->i_dev, bno, fs->fs_fsmnt);
			return -1;
		}
		sa_setblock(fs, cgp->cg_free, fragstoblks(fs, bno));
		cgp->cg_cs.cs_nbfree++;
		fs->fs_cstotal.cs_nbfree++;
		fs->fs_cs(fs, cg).cs_nbfree++;
		i = cbtocylno(fs, bno);
		cgp->cg_b[i][cbtorpos(fs, bno)]++;
		cgp->cg_btot[i]++;
	} else {
		bbase = bno - (bno % fs->fs_frag);
		/*
		 * decrement the counts associated with the old frags
		 */
		blk = blkmap(fs, cgp->cg_free, bbase);
		fragacct(fs, blk, cgp->cg_frsum, -1);
		/*
		 * deallocate the fragment
		 */
		frags = numfrags(fs, size);
		for (i = 0; i < frags; i++) {
			if (isset(cgp->cg_free, bno + i)) {
				printf("free: freeing free frag: ");
				printf("dev = 0x%x, block = %d, fs = %s\n",
				    ip->i_dev, bno + i, fs->fs_fsmnt);
			}
			setbit(cgp->cg_free, bno + i);
		}
		cgp->cg_cs.cs_nffree += i;
		fs->fs_cstotal.cs_nffree += i;
		fs->fs_cs(fs, cg).cs_nffree += i;
		/*
		 * add back in counts associated with the new frags
		 */
		blk = blkmap(fs, cgp->cg_free, bbase);
		fragacct(fs, blk, cgp->cg_frsum, 1);
		/*
		 * if a complete block has been reassembled, account for it
		 */
		if (sa_isblock(fs, cgp->cg_free, fragstoblks(fs, bbase))) {
			cgp->cg_cs.cs_nffree -= fs->fs_frag;
			fs->fs_cstotal.cs_nffree -= fs->fs_frag;
			fs->fs_cs(fs, cg).cs_nffree -= fs->fs_frag;
			cgp->cg_cs.cs_nbfree++;
			fs->fs_cstotal.cs_nbfree++;
			fs->fs_cs(fs, cg).cs_nbfree++;
			i = cbtocylno(fs, bbase);
			cgp->cg_b[i][cbtorpos(fs, bbase)]++;
			cgp->cg_btot[i]++;
		}
	}
	fs->fs_fmod++;
	devwrite (io);
	return 0;
}

/*
 * Free an inode.
 *
 * The specified inode is placed back in the free map.
 */
ifree(io, mode)
	register struct iob *io;
	int mode;
{
	register struct fs *fs = io->i_fs;
	register struct cg *cgp;
	register struct inode *ip = io->i_ino;
	int cg, cc, ino = ip->i_number;

	if ((unsigned)ino >= fs->fs_ipg*fs->fs_ncg) {
		printf("ifree: range: ");
		printf("dev = 0x%x, ino = %d, fs = %s\n",
		    ip->i_dev, ino, fs->fs_fsmnt);
		return;
	}
	cg = itog(fs, ino);
	/*
	 * Read in CG structure from disk.
	 */
	io->i_bn = fsbtodb(fs, cgtod(fs, cg)) + io->i_boff;
	io->i_cc = fs->fs_cgsize;	/* Hope it's big enough! */
	io->i_ma = io->i_buf;
	cc = devread(io);
	cgp = (struct cg *)io->i_buf;
	if ((cc <= 0) || (cgp->cg_magic != CG_MAGIC) ||
		(cgp->cg_cs.cs_nifree == 0))
		{
			if (sysdebug)
				printf("ifree: fail: %s\n",
					(cc <= 0) ? "cc < 0" :
					(cgp->cg_magic != CG_MAGIC) ?
					"cg_magic mismatch" :
					(cgp->cg_cs.cs_nifree == 0) ?
					"cs_nifree == 0" : "UNKNOWN!");
			return (NULL);
		}

	/*
	 * Mess up CG structure.
	 */
	cgp->cg_time = time.tv_sec;
	ino %= fs->fs_ipg;
	if (isclr(cgp->cg_iused, ino)) {
		printf("ifree: freeing free inode: ");
		printf("dev = 0x%x, ino = %d, fs = %s\n",
		    ip->i_dev, ino, fs->fs_fsmnt);
		return (NULL);
	}
	clrbit(cgp->cg_iused, ino);
	if (sysdebug) if (isclr(cgp->cg_iused, ino))
		printf("ifree: inode bit cleared.\n");
		else printf("ifree: inode bit not cleared!\n");
	cgp->cg_cs.cs_nifree++;
	fs->fs_cstotal.cs_nifree++;
	fs->fs_cs(fs, cg).cs_nifree++;
	if ((mode & IFMT) == IFDIR) {
		cgp->cg_cs.cs_ndir--;
		fs->fs_cstotal.cs_ndir--;
		fs->fs_cs(fs, cg).cs_ndir--;
	}
	fs->fs_fmod++;
	cc=devwrite(io);	/* Put CG structure back on disk. */
	writefs(io);		/* Update superblock on disk. */
}

/*
 * Find a block of the specified size in the specified cylinder group.
 *
 * It is a screwup if a request is made to find a block and none are
 * available.
 */
daddr_t
mapsearch(fs, cgp, bpref, allocsiz)
	register struct fs *fs;
	register struct cg *cgp;
	daddr_t bpref;
	int allocsiz;
{
	daddr_t bno;
	int start, len, loc, i;
	int blk, field, subfield, pos;

	/*
	 * find the fragment by searching through the free block
	 * map for an appropriate bit pattern
	 */
	if (bpref)
		start = dtogd(fs, bpref) / NBBY;
	else
		start = cgp->cg_frotor / NBBY;
	len = howmany(fs->fs_fpg, NBBY) - start;
	loc = scanc((unsigned)len, (caddr_t)&cgp->cg_free[start],
		(caddr_t)fragtbl[fs->fs_frag],
		(int)(1 << (allocsiz - 1 + (fs->fs_frag % NBBY))));
	if (loc == 0) {
		len = start + 1;
		start = 0;
		loc = scanc((unsigned)len, (caddr_t)&cgp->cg_free[start],
			(caddr_t)fragtbl[fs->fs_frag],
			(int)(1 << (allocsiz - 1 + (fs->fs_frag % NBBY))));
		if (loc == 0)
			return (-1);
	}
	bno = (start + len - loc) * NBBY;
	cgp->cg_frotor = bno;
	/*
	 * found the byte in the map
	 * sift through the bits to find the selected frag
	 */
	for (i = bno + NBBY; bno < i; bno += fs->fs_frag) {
		blk = blkmap(fs, cgp->cg_free, bno);
		blk <<= 1;
		field = around[allocsiz];
		subfield = inside[allocsiz];
		for (pos = 0; pos <= fs->fs_frag - allocsiz; pos++) {
			if ((blk & field) == subfield)
				return (bno + pos);
			field <<= 1;
			subfield <<= 1;
		}
	}
	printf("mapsearch: block not in map: ");
	printf("bno = %d, fs = %s\n", bno, fs->fs_fsmnt);
	return (-1);
}

/*
 * Scanning subroutine from UNIX: ~sys/tty.c.
 */
scanc(len, cp, table, mask)
	register int len;
	register u_char *cp, table[];
	register int mask;
{
	if (len == 0)
		return(0);
	while ((table[*cp++] & mask) == 0 && --len)
		;
	return (len);
}

/*
 * Block operations Clrblock() and Isblock() emanate from sys/ufs_subr.c.
 */

/*
 * ISBLOCK:
 *
 * check if a block is available
 */
sa_isblock(fs, cp, h)
	struct fs *fs;
	unsigned char *cp;
	daddr_t h;
{
	unsigned char mask;

	switch (fs->fs_frag) {
	case 8:
		return (cp[h] == 0xff);
	case 4:
		mask = 0x0f << ((h & 0x1) << 2);
		return ((cp[h >> 1] & mask) == mask);
	case 2:
		mask = 0x03 << ((h & 0x3) << 1);
		return ((cp[h >> 2] & mask) == mask);
	case 1:
		mask = 0x01 << (h & 0x7);
		return ((cp[h >> 3] & mask) == mask);
	default:
		printf("sa_isblock: bad fs->fs_frag 0x%x.\n", fs->fs_frag);
		return (NULL);
	}
}

/*
 * CLRBLOCK:
 *
 * Take a block out of the map
 */
sa_clrblock(fs, cp, h)
	struct fs *fs;
	u_char *cp;
	daddr_t h;
{

	switch ((fs)->fs_frag) {
	case 8:
		cp[h] = 0;
		return;
	case 4:
		cp[h >> 1] &= ~(0x0f << ((h & 0x1) << 2));
		return;
	case 2:
		cp[h >> 2] &= ~(0x03 << ((h & 0x3) << 1));
		return;
	case 1:
		cp[h >> 3] &= ~(0x01 << (h & 0x7));
		return;
	default:
		printf("sa_isblock: bad fs->fs_frag 0x%x.\n", fs->fs_frag);
		return;
	}
}

/*
 * put a block into the map
 */
sa_setblock(fs, cp, h)
	struct fs *fs;
	unsigned char *cp;
	daddr_t h;
{

	switch (fs->fs_frag) {

	case 8:
		cp[h] = 0xff;
		return;
	case 4:
		cp[h >> 1] |= (0x0f << ((h & 0x1) << 2));
		return;
	case 2:
		cp[h >> 2] |= (0x03 << ((h & 0x3) << 1));
		return;
	case 1:
		cp[h >> 3] |= (0x01 << (h & 0x7));
		return;
	default:
		_stop("setblock");
	}
}

extern	int around[9];
extern	int inside[9];
extern	u_char *fragtbl[];

/*
 * Update the frsum fields to reflect addition or deletion 
 * of some frags.
 */
fragacct(fs, fragmap, fraglist, cnt)
	struct fs *fs;
	int fragmap;
	long fraglist[];
	int cnt;
{
	int inblk;
	register int field, subfield;
	register int siz, pos;

	inblk = (int)(fragtbl[fs->fs_frag][fragmap]) << 1;
	fragmap <<= 1;
	for (siz = 1; siz < fs->fs_frag; siz++) {
		if ((inblk & (1 << (siz + (fs->fs_frag % NBBY)))) == 0)
			continue;
		field = around[siz];
		subfield = inside[siz];
		for (pos = siz; pos <= fs->fs_frag; pos++) {
			if ((fragmap & field) == subfield) {
				fraglist[siz] += cnt;
				pos += siz;
				field <<= siz;
				subfield <<= siz;
			}
			field <<= 1;
			subfield <<= 1;
		}
	}
}

/*
 * Check that a specified block number is in range.
 */
badblock(fs, bn)
	register struct fs *fs;
	daddr_t bn;
{

	if ((unsigned)bn >= fs->fs_size) {
		printf("bad block %d, ", bn);
		_stop("bad block");
		return (1);
	}
	return (0);
}
