/*
 * terminal
 * communicate with other computers or devices using a tty port
 * Terminal was developed at the Wisconsin State Hygiene Lab.
 * It is based on the program UTALK from the University of New South Wales,
 * but it has been extensively modified.
 * Modified by David E. Miran 11/27/79 ,1/21/80, and 4/3/80   
 * further modified for access checking subroutine 7/31/80
	version of 7/27/82
 */
/* ********** must be compiled with a v7 c compiler ******** */
/*
	terminal  [-l] [-o] [-n] [-px] [-sn] [-f filename] [-h]
 *
 *	Talk to another machine via a tty port.
 *	  Except during file transfer, terminal is completely 
 *	  transparent to normal UNIX tty codes.
 *	  You may receive files to local pathnames,
 *	  and optionally keep a log of all that is received.
 *
 *
 *			  by
 *		     piers lauder
 *	      DEPT. OF COMPUTER  SCIENCE
 *		 UNIVERSITY OF SYDNEY
 *		      july  1977
 *
 *	parameters:-
 *		 -l:  start with log on ( default: you are asked )
 *		 -o:  start with log off
 *		 -h:  half duplex mode (local echoing of characters typed)
 *		-px:  tty port id is 'x' ( default: 'g' )
 *		-sn:  tty port speed is 'n' ( default: '2400' )
 *		 -f:  next parameter is filename to be transmitted on receipt of CTRL-f
 *		 -n:  echo nl to cr on local port
 *
 *	commands:-
 *	  -- type CTRL-y to quit --
 *		  CTRL-r to receive file ( local file name will be requested ) ( CTRL-r to stop )
 *		  CTRL-f to send file ( if no file name has been specified - one will be requested ) ( DEL to stop )
 *		  CTRL-t to toggle copy status ( read to file or read to log : on/off )
 *		  CTRL-b to turn on block raw mode for receive (more efficient)
 *		  CTRL-c to turn off block raw mode.
 *
 *	files:-
 *		/dev/ttyx	- tty ports
 *		log.mmmddhhmm	- log file ( with date )
 */


#include	<stdio.h>
#define QUIT		031		/* CTRL-y to terminate */
#define RECV		022		/* CTRL-r to receive file */
#define SEND		006		/* CTRL-f to send file */
#define COPY		024		/* CTRL-t to toggle copy status */
#define BLKON		02		/* CTRL-b to turn on block raw recieve*/
#define BLKOFF		03		/* CTRL-c to turn off block raw */

#define	NOTNICE		-40		/* run at high priority */


char	ttyl[]		"/dev/ttyx";
char	ttyr[]		"/dev/ttyg";
char	logf[]		"log.mmmddhhmm";

#define SIGDEL		2
#define SIGQUIT		3
#define SIGCOPY		6
#define	SIGSTAT		6
#define	SIGRECV		7
#define	SIGKILL		9

struct	sgtty
{
	int	sgspeeds;
	char	sgerase, sgkill;
	int	sgflags;
}
	ttylstat, ttyrstat;

struct	spd
{
	int	baud, speed;
}
	speeds[]
{
  	110, (3<<8)|3,
	150, (5<<8)|5,
  	300, (7<<8)|7,
	600, (8<<8)|8,
  	1200, (9<<8)|9,
  	2400, (11<<8)|11,
  	4800, (12<<8)|12,
  	9600, (13<<8)|13,
	0
};

int	sspeed	{(11<<8)|11};

int	saveflags;
#define XTABS	2
#define LCASE	4
#define ECHO	010
#define CRMOD	020
#define RAW	040
#define ANYP	0300
#define	CR3	030000
#define BLOCK	0100000

#define RFLAGS	(ANYP|RAW|XTABS)
#define	CFLAGS	(RFLAGS|CR3)
#define BFLAGS	(BLOCK|RAW|ANYP|XTABS)

#define LOGON		1
#define LOGONINIT	2
#define LOGOFFINIT	4
int	readstatus	0;
#define READTOFILE	010
#define LOGNONEMPTY	020
#define	COPYON		(LOGON|READTOFILE)
int	loging	0;
int	recving 0;

struct	inode
{
	int	i_fill1[2];
	int	i_flags;
	int	i_fill2[15];
}
	ibuf;

FILE	*recvbuf, *logbuf;

int	stopflag;
long	tcount;
int	ls;

#define NAMEZ		132

char	recvf[NAMEZ];

int	c;
int	nlflag;
char	nl		'\n';
char	at		'@';
int	remoterd, remotewr;
char	*name;
char	*file;
char	openfail[]	"%s: cannot open %s\n";
char	requestname[]	"%s: %s filename ? ";
int	parent;
int	hdup	0;	/* set if half duplex mode */
char cl[500];
int clpos, clread;
unsigned	uid, gid, setid;

extern char *ctime();



main( argc , argv )  char **argv;
{
	register struct spd  *sp;
	register  s, x;
	char  *p1, *p2;
	int  child;
	extern  togcopy(), quit();
	extern  recvfile(), stop();

  nice( NOTNICE );


  uid = getuid() & 0377;
  gid = getgid() & 0377;
  setid = (uid&0377)|((gid&0377)<<8);

  name = *argv;

  /*
   *	read parameters
   */
	while (--argc) {
		argv++;

		if (strcmp(*argv,"-f") == 0) {
			if (--argc) file = *++argv;
			continue;
		}
		if (strcmp(*argv,"-h") == 0) {
			hdup++;
			continue;
		}

		if (strcmp(*argv,"-l") == 0) {
			readstatus =| LOGONINIT;
			continue;
		}

		if (strcmp(*argv,"-n") == 0) {
			nlflag++;
			continue;
		}

		if (strcmp(*argv,"-o") == 0) {
			readstatus =| LOGOFFINIT;
			continue;
		}

		if (strncmp(*argv,"-p",2) == 0) {
			ttyr[8] = argv[0][2];
			printf("%s: Using communication line %s\n",name,ttyr);
			continue;
		}

		if (strncmp(*argv, "-s",2) == 0) {
			s = atoi(&argv[0][2]);
			for ( sp = speeds ; x = sp->baud ; sp++ )
				if ( x == s )  {
					sspeed = sp->speed;
					printf("%s:  Communication line speed %d\n",name,x);
					break;
				}
			if ( x )  continue;
			ls = s;
			printf("%s: invalid speed %d\nValid speeds are 110, 150, 300, 600, 1200, 2400, 4800, 9600\n",name,ls);
			exit( 1 );
		}

		printf("%s: unknown option: %c\nValid options are: -o, -l, -n, -h, -pn, -sn, -f filename\n",name,**argv);
		exit( 1 );
	}

/* fprintf(stderr,"past option checking \n");
  /*
   *	open log file
   */
  if ( openlog() == 0 )  exit( 2 );
  if ( (readstatus & (LOGONINIT|LOGOFFINIT)) == 0 )  {
	printf( "%s: log on ? " , name );
	read( 0 , &c , 1 );
	if ( (c & 0177) == 'n' )
		readstatus =& ~LOGON;
  }else
	if ( (readstatus & (LOGONINIT|LOGOFFINIT)) == LOGOFFINIT )
		readstatus =& ~LOGON;

  /*
   *	set up remote tty
   */
 if ( ! (saccess(ttyr, 4))) {
	printf(openfail , name , ttyr );
	exit( 3 );
 }
  if ( (remoterd = open( ttyr , 0 )) < 0 || (remotewr = open( ttyr , 1 )) < 0 )  {
	printf( openfail , name , ttyr );
	exit( 3 );
  }
  chmod( ttyr , 0 );
  gtty( remotewr , &ttyrstat );
  ttyrstat.sgspeeds = sspeed;
  ttyrstat.sgerase = '\0';
  ttyrstat.sgkill = '\0';
  ttyrstat.sgflags = RFLAGS;
  stty( remotewr , &ttyrstat );

  /*
   *	set up local tty
   */
  ttyl[8] = ttyn( 1 );
  fstat( 1 , &ibuf );
  chmod( ttyl , 0600 );
  gtty( 1 , &ttylstat);
  if ((ttylstat.sgspeeds&0377) < (ttyrstat.sgspeeds&0377)) {
		printf("terminal:  ERROR - you must use a terminal at least as fast as the communications line\n");
		chmod(ttyr, 0666);
		exit(1);
	}
  saveflags = ttylstat.sgflags;
  ttylstat.sgflags = (RAW|(saveflags & ANYP));
  stty( 1 , &ttylstat );

  parent = getpid();		/* remember ancestor */
  signal( SIGDEL , stop );	/* for DEL during sendfile()/recvfile() */

  /*
   *	fork off child to read remote to local
   */

  if ( (child = fork()) == 0 )  {
	signal( SIGCOPY , togcopy );	/* to toggle log status */
	signal( SIGQUIT , quit );	/* for terminate from parent */
	signal( SIGRECV , recvfile );	/* to receive a file */

	for(;;)
		if ((clread = read(remoterd, &cl[0], 500)) > 0 ) {
			write(1,&cl[0], clread);
			for (clpos = 0; clpos < clread; clpos++) {
				c = cl[clpos];
			if ( readstatus & COPYON )
				if ( (c =& 0177) != '\r' )  
					if ( readstatus & READTOFILE )  {
						tcount++;
						putc( c , recvbuf );
					}else  {
						readstatus =| LOGNONEMPTY;
						putc( c , logbuf );
					}
			}
		}
  }

  /*
   *	parent sends local to remote
   */

	if (child < 0) {
		fprintf(stderr,"%s: Cannot fork\n",name);
		goto out2;
	}

  for(;;)
	if ( read( 0 , &c , 1 ) == 1 )  {

		switch ( c & 0177 )  {
		  case QUIT:	goto out;

		  case RECV:	kill( child , SIGRECV );
				stopflag = 0;
				while ( !stopflag )  sleep( 2 );
				continue;

		  case SEND:	sendfile();
				continue;

		  case COPY:	kill( child , SIGCOPY );
				continue;

		  case BLKON:	gtty(remotewr, &ttyrstat);
				ttyrstat.sgflags = BFLAGS;
				ttyrstat.sgerase = 80;
				ttyrstat.sgkill = '\n';
				stty(remotewr, &ttyrstat);
				printf("BLOCK input mode on\r\n");
				continue;

		  case BLKOFF:	gtty(remotewr, &ttyrstat);
				ttyrstat.sgflags = RFLAGS;
				ttyrstat.sgerase = 0;
				ttyrstat.sgkill = 0;
				stty(remotewr, &ttyrstat);
				printf("BLOCK input mode off\r\n");
				continue;


		  case '\r':	if ( nlflag )  write( 1 , &nl , 1 );
		}

		write( remotewr , &c ,1 );
		if (hdup) write(1, &c, 1); /* half duplex mode local echo */
	}

out:

  /*
   *	cleanup
   */
  kill( child , SIGQUIT );
  wait( &readstatus );
  readstatus =>> 8;

out2:

  resetty();
  chmod( ttyl , ibuf.i_flags );
  chmod( ttyr , 0666 );
  ttyrstat.sgflags = ANYP;
  stty( remotewr , &ttyrstat );

  printf( "\nQUIT\n" );

  if ( readstatus & LOGNONEMPTY )  {
	printf( "%s: keep log file ? " , name );
	read( 0 , recvf , 132 );
	if (((recvf[0] & 0177) == 'n') || ((recvf[0] & 0177) == 'N')) {
removelog:
		unlink( logf );
		loging = 0;
	}
  }else
	goto removelog;

  if (loging)
	printf( "%s: %s\n" , name , logf );

  exit( 0 );
}




/*
 *	open log file
 */
openlog()
{
	register  x;
	register char  *p1, *p2;
	char  *t;
	long tvec;

  time(&tvec);
  t = ctime(&tvec);
  logf[4] = t[4];
  for ( x = 0 , p1 = &logf[5] , p2 = t+5 ; x < 4 ; x++ )  {
	if ( (*p1++ = *p2++) == ' ' )  *(p1-1) = '0';
	*p1++ = *p2++;
	p2++;
  }
  if ((logbuf = fopen(logf,"w")) == NULL) {
	printf( "%s: cannot create %s\n" , name , logf );
	return( 0 );
  }
  chown( logf , setid );
  chmod( logf , 0644 );
  readstatus =| LOGON;
  loging = 1;
  return( 1 );
}




/*
 *	catch signal to receive a file
 */
recvfile()
{
	register  s, fid;

  signal( SIGRECV , recvfile );
  resetty();

  if ( recving == 0 )  {
	stopflag = 0;
	printf( requestname , name , "receive" );
	s = read( 0 , recvf , NAMEZ );
	if ( stopflag || s < 2 )  goto out;
	recvf[--s] = '\0';

	if (!(saccess(recvf, 2))) {
		printf( "%s: cannot create %s\n" , name , recvf );
		goto out;
	}
	if ((recvbuf = fopen(recvf, "w")) == NULL) {
		printf( "%s: cannot create %s\n" , name , recvf );
		goto out;
	}

	chown( recvf , setid );
	chmod( recvf , 0644 );
	recving = 1;

	tcount = 0;
	readstatus =| READTOFILE;

  }else  {

	status();
	fclose(recvbuf);
	recving = 0;
	printf( "%s: %s %s\n" , name , recvf , "done" );
	readstatus =& ~READTOFILE;
  }

out:

  sleep( 3 );
  setty();
  kill( parent , SIGDEL );
}



/*
 *	give status report on file transfer
 */
status()
{
  printf( "%s: %D bytes transferred\n" , name , tcount );
}




/*
 *	send local file to remote
 */
sendfile()
{
	register char  *p;
	register  s, x;
	char  f[512];
	char  n[NAMEZ];
	int  fid;
	struct { char *cp; };
	extern  stop();

  resetty();
  stopflag = 0;


  if ( file == 0 )  {
	printf( requestname , name , "send" );
	x = read( 0 , n , NAMEZ );
	if ( stopflag || x < 2 )  goto out;
	n[--x] = '\0';
	file = n;
  }

 if (! (saccess(file, 4))) {
	printf( openfail , name , file );
	goto out;
  }
  if ( (fid = open( file , 0 )) < 0 )  {
	printf( openfail , name , file );
	goto out;
  }

  tcount = 0;

  while ( (s = read( fid , f , 512 )) > 0 )  {
  	if ( stopflag )  break;
  	tcount =+ s;
  	for ( p = f , x.cp = &f[s] ; p < x.cp ; )
  		if ( *p++ == '\n' )  *(p-1) = '\r';
  	write( remotewr , f , s );
  	if ( stopflag )  break;
  }

  status();
  close( fid );

  printf( "%s: %s %s\n" , name , file , (stopflag ? "stopped" : "done" ) );

out:

  setty();
  file = 0;
}



/*
 *	catch interrupt signals during sendfile()
 */
stop()
{
  signal( SIGDEL , stop );
  stopflag++;
}



int	s2;


/*
 *	reset tty to original status
 */
resetty()
{
	s2 = ttylstat.sgflags;
	ttylstat.sgflags = saveflags;
	stty( 1 , &ttylstat );
}



/*
 *	set tty to raw status
 */
setty()
{
	ttylstat.sgflags = s2;
	stty( 1 , &ttylstat );
}



/*
 *	catch signal to toggle copy status
 */
togcopy()
{
	register  before, after;

  signal( SIGCOPY , togcopy );

  before = readstatus;
  after = (readstatus =^ (recving ? READTOFILE : LOGON));

  printf( "%s: %s\r\n" , name ,
		after  & READTOFILE ? "recv on"  :
		after  & LOGON      ? "log on"   :
		before & READTOFILE ? "recv off" : "log off"
	);
}



/*
 *	catch signal to terminate child
 */
quit()
{
  if (loging) fclose(logbuf);
  if (recving) fclose(recvbuf);
  exit( readstatus );
}

