/**************************************************************************/
/*    Dispwin is a routine which gets data from the pipe, formatting them */
/*    and then display them in a tabular form on a window.                */
/*                                                                        */
/*    Algorithm:                                                          */
/*      step1: Allocate memory space to the screen and each input line    */
/*             dynamically.                                               */
/*      step2: Check if the input filename is NULL, if not get terminal   */
/*             type. Otherwise, exit.                                     */
/*      step3: Initialize the screen. Change any tty charcteristic if     */
/*             required.                                                  */
/*      step4: Get input lines through the pipe ,counting input lines and */
/*             # of fields in a line. Calculate the window size from the  */
/*             sum of each maximum field length in a line and the desired */
/*             number of lines on a page.                                 */
/*      step5: Display the first page. If the # of records is more than   */
/*             one page, type in the required key for the desired page    */
/*             the user wants. If the # of fields is greater than the     */
/*             width of the screen, the same thing is performed. Many     */
/*             other key functions are available and this can be referred */
/*             by keying '?' to obtain the MENU FILE. Key in 'q' to exit. */
/*      step6: Delete the window and restore any tty characteristic if    */
/*             changed.                                                   */
/*                                                                        */
/*      Date implemented: May 6, 1982                                     */
/**************************************************************************/

#include "../includes/dispwin.h"      /* this includes all the window functional routines */


typedef struct {
	char *tr_fptr[MAXFLDNUM];        /* each field of a record        */
} trrec_t;

typedef struct SCRREC {
       trrec_t  tr_scrrec[MAXRECNUM];    /* screen records (each line)    */
       int      max_flen[MAXFLDNUM];     /* MAX field length for each dictionary fields */
       int      tr_nfields;              /* Number of dictionary fields   */
       int      tr_nlinrec;              /* Number of screen line records */
       int      s_fld;                   /* starting field */
       int      e_fld;                   /* ending   field */
       int      s_line;                  /* starting record line */
       int      e_line;                  /* ending   record line */
} scr_t;
scr_t	*scr;

int     ln_cnt;                           /* no. of input lines            */

WINDOW *win;

main(argc,argv)
  int argc;
  char **argv;
{
    if(argc == 2)
	dispwin(argv[1]);
    else
	fprintf(stderr, "usage: %s {-|<file>}\n", argv[0]);
}

dispwin (filename)
  char    *filename;             /* input filename */
{
    FILE       *fildes;
    FILE       *terminal;        /* terminal file   */
    char       *termtyp;         /* terminal type   */
    char       *line;            /* input line from pipe */
    int         c;               /* input key       */
    int         sum_len=0;       /* window width    */
    int         begy=0;          /* initial win row */
    int         begx;            /* initial win col */
    int         lineno=0;        /* screen line #   */
    int         i;
    int         sbx, sby, smx, smy; /* temp (x,y)co-ordinates for desired window    */

    scr = (scr_t *)malloc(sizeof(scr_t));          /* allocate space for the screen */
    line = (char *)malloc(MAXRECLEN);              /* allocae space for each record */
    if (strcmp("-", filename) == 0)
	fildes = stdin;
    else if ((fildes = fopen (filename, "r")) == NULL) {
	fprintf(stderr, "dispwin: can't open %s\n", filename);
	exit();
    }
    if ((termtyp = getenv ("TERM")) != NULL) {
     LINES = 0; COLS = 0;
      win = initscr();                           /* initialize the screen */
      wrefresh(win);
      wmove(win,0,0);
      mvcur(win->_cury,win->_curx,0,0);
      fprintf(stderr,"Dispwin is starting ");
      if (strcmp("-",filename) != 0) {
	 mvcur(win->_cury,win->_curx,0,21);
	 fprintf(stderr,"for %s", filename);
      }
      wclear(win);
      while((getrec(fildes,line)) != EOF){        /* get all input lines through the pipe */
	ln_cnt++;
	parse(line,&scr->tr_scrrec[lineno++]);   /* parse each line calculating the # of fields from the */
      }                                          /* first line in term of field separator and the max.   */

      scr->tr_nlinrec = ln_cnt-1;                /* field length to calculate the actual record length   */

      if (scr->tr_nlinrec >= 1){
	 scr->s_fld = 0;                         /* initial start field  */
	 for (i=scr->s_fld; i < scr->tr_nfields; i++) {
	     if ((sum_len +=  scr->max_flen[i] + 1 ) >80) {
		sum_len = 80;
		break;
	      }
	 }
	 scr->e_fld = (sum_len == 80)?i:i-1;      /* initial end field    */

	 begx = (sum_len == 80)?0:(80 - (sum_len+1))/2;
	 scr->s_line = 1;
	 scr->e_line = (scr->tr_nlinrec >= REC_PG)? REC_PG:scr->tr_nlinrec;

	 win->_begx = begx;      win->_begy = begy;
	 win->_maxx = (sum_len == 80)?80:sum_len+2;
	 win->_maxy = 2*scr->e_line+2;

	 draw_window(filename);                    /* display the first page, if more than one page or the # of fields exceeding */
						   /* the scrren width, performed the desired key function                       */
      if (!(terminal = fopen("/dev/tty", "rw")))       {
	      fprintf(stderr,"open error\n");
	      exit();
      }
      crmode();                                     /* set tty characteristics */
      noecho();
      while ((c = getc(terminal)) != 'q') {         /* loop to get the key     */
	 switch(c)   {
		  /* move forward one page  */
	 case 'r' :  if (scr->e_line == scr->tr_nlinrec) {
			mvcur(win->_cury,win->_curx,22,0);
			fprintf(stderr,"status: bottom of file          ");
			mvcur(win->_cury,win->_curx,0,0);
			break;
		     }

		     scr->e_line = ((scr->e_line + REC_PG)> scr->tr_nlinrec)
			      ?scr->tr_nlinrec: scr->e_line + REC_PG;
		     scr->s_line = scr->e_line - REC_PG + 1;
		     wclear(win);
		     draw_window(filename);
		     mvcur(win->_cury,win->_curx,0,0);
		     break;

		  /* move backward one page */
	 case 'e' :  if (scr->s_line == 1) {
			mvcur(win->_cury,win->_curx,22,0);
			fprintf(stderr,"status: top of file             ");
			mvcur(win->_cury,win->_curx,0,0);
			break;
		     }

		     scr->s_line = (scr->s_line  - REC_PG < 0)? 1 : scr->s_line - REC_PG;
		     scr->e_line = scr->s_line + REC_PG - 1;
		     wclear(win);
		     draw_window(filename);
		     mvcur(win->_cury,win->_curx,0,0);
		     break;

		  /* move forward one third of a page */
	 case 'f' :  if (scr->e_line == scr->tr_nlinrec) break;
		     scr->e_line =((scr->e_line  + ONE_THIRD_PG)> scr->tr_nlinrec)
			      ?scr->tr_nlinrec: scr->e_line + ONE_THIRD_PG;
		     scr->s_line = scr->e_line - REC_PG + 1;
		     wclear(win);
		     draw_window(filename);
		     mvcur(win->_cury,win->_curx,0,0);
		     break;

		  /* move backward one third of a page */
	 case 'd' :  if (scr->s_line == 1) {
			mvcur(win->_cury,win->_curx,22,0);
			fprintf(stderr,"status: top of file             ");
			mvcur(win->_cury,win->_curx,0,0);
			break;
		     }

		     scr->s_line = (scr->s_line  - ONE_THIRD_PG < 0)? 1 :scr->s_line - ONE_THIRD_PG;
		     scr->e_line = scr->s_line + REC_PG - 1;
		     wclear(win);
		     draw_window(filename);
		     mvcur(win->_cury,win->_curx,0,0);
		     break;

		  /* move forward one record  */
	 case 'j' :  if (scr->e_line == scr->tr_nlinrec){
			mvcur(win->_cury,win->_curx,22,0);
			fprintf(stderr,"status: bottom of file          ");
			mvcur(win->_cury,win->_curx,0,0);
			break;
		     }

		     scr->e_line =((scr->e_line  + ONE_REC)> scr->tr_nlinrec)
				  ?scr->tr_nlinrec: scr->e_line + ONE_REC;
		     scr->s_line = scr->e_line - REC_PG + 1;
		     wclear(win);
		     draw_window(filename);
		     mvcur(win->_cury,win->_curx,0,0);
		     break;

		  /* move backward one record */
	 case 'k' :  if (scr->s_line == 1) {
			mvcur(win->_cury,win->_curx,22,0);
			fprintf(stderr,"status: top of file             ");
			mvcur(win->_cury,win->_curx,0,0);
			break;
		     }

		     scr->s_line = (scr->s_line  - ONE_REC < 0)? 1 : scr->s_line - ONE_REC;
		     scr->e_line = scr->s_line + REC_PG - 1;
		     wclear(win);
		     draw_window(filename);
		     mvcur(win->_cury,win->_curx,0,0);
		     break;

		  /* move left one column */
	  case 'l' :
		     scr->s_fld = (scr->s_fld == scr->tr_nfields - 1) ? 0 :scr->s_fld + 1;
		     scr->e_fld = ((scr->s_fld - 1) >= 0) ? (scr->s_fld - 1):(scr->tr_nfields - 1);

		    sum_len = 0;
		    for (i=scr->s_fld; ((scr->s_fld > scr->e_fld)?(i - scr->tr_nfields) : i) <= scr->e_fld; i++){
			 if ((sum_len +=  scr->max_flen[(i - scr->tr_nfields) >= 0 ? (i - scr->tr_nfields) : i] + 1) > 80){
			    sum_len = 80;
			    break;
			 }
		    }
		   if (sum_len == 80)
		       scr->e_fld = (i <= scr->tr_nfields) ? i : i - scr->tr_nfields;
		   else
		       scr->e_fld = (i <= scr->tr_nfields) ? i - 1 : i - scr->tr_nfields - 1;

		     wclear(win);
		     draw_window(filename);
		     mvcur(win->_cury,win->_curx,0,0);
		     break;


		  /* move right one column */
	 case 'h' :
		    scr->s_fld = (scr->s_fld == 0) ? (scr->tr_nfields - 1): (scr->s_fld - 1);
		    scr->e_fld = ((scr->s_fld - 1)>= 0) ? (scr->s_fld - 1):(scr->tr_nfields - 1);

		     sum_len = 0;
		    for (i=scr->s_fld; ((scr->s_fld > scr->e_fld)?(i - scr->tr_nfields) : i) <= scr->e_fld; i++) {
			 if ((sum_len +=  scr->max_flen[(i - scr->tr_nfields) >= 0 ? (i - scr->tr_nfields) : i] + 1 ) > 80){
			    sum_len = 80;
			    break;
			 }
		    }
		   if (sum_len == 80)
		      scr->e_fld = (i <= scr->tr_nfields) ? i : i - scr->tr_nfields;
		   else
		      scr->e_fld = (i <= scr->tr_nfields ) ? i - 1 : i - scr->tr_nfields - 1;

		     wclear(win);
		     draw_window(filename);
		     mvcur(win->_cury,win->_curx,0,0);
		     break;

		  /* move to top of file */
	 case 't' :  if (scr->s_line == 1) {
			mvcur(win->_cury,win->_curx,22,0);
			fprintf(stderr,"status: top of file             ");
			mvcur(win->_cury,win->_curx,0,0);
			break;
		     }
		    scr->s_line = 1;
		    scr->e_line = ( scr->tr_nlinrec >= REC_PG)? REC_PG:scr->tr_nlinrec;
		    wclear(win);
		    draw_window(filename);
		    mvcur(win->_cury,win->_curx,0,0);
		    break;

		  /* move to bottom of file */
	 case 'y' :  if (scr->e_line == scr->tr_nlinrec) {
			mvcur(win->_cury,win->_curx,22,0);
			fprintf(stderr,"status: bottom of file          ");
			mvcur(win->_cury,win->_curx,0,0);
			break;
		     }
		    scr->e_line = scr->tr_nlinrec;
		    scr->s_line = scr->e_line - REC_PG + 1;
		    wclear(win);
		    draw_window(filename);
		    mvcur(win->_cury,win->_curx,0,0);
		    break;

	 case '?' : sbx = win->_begx; sby= win->_begy;
		    smx = win->_maxx; smy= win->_maxy;
		    wclear(win);
		    help(terminal);
		    wclear(win);
		    win->_begx = sbx; win->_begy = sby;
		    win->_maxx = smx; win->_maxy = smy;
		    draw_window(filename);
		    mvcur(win->_cury,win->_curx,0,0);
		    break;

	 default  :
		     mvcur(win->_cury,win->_curx,23,0);
		     fprintf(stderr,"status: unrecognized command key");
		     mvcur(win->_cury,win->_curx,0,0);
		     break;

	 }
	}
	mvcur(win->_cury, win->_curx, 24,0);
	delwin(win);
	endwin();
	nocrmode();    /* restore tty characteristics */
	echo();
	}
  }
}

/************************/
getrec(file,line)             /* get a line through a pipe or std. input file */
/************************/
  FILE     *file;
  char     *line;
{
    char   *ptr;
    int    c;
    int     i;
    for(i=0; i<MAXRECLEN - 1 && (c=getc(file)) != EOF && c!= '\n'; ++i){
       line[i] = c;
    }
    if (c == '\n')
      line[i] = c;
   return((c == EOF)?c:0);

}

/**********************/
parse(line,rec)                /* parse an input line for # of fields and max. length for each field */
/**********************/
char *line;
trrec_t *rec; 
{
  int      nfields, fld_cnt;

  nfields=0; fld_cnt=0;
  while( *line ) {

     switch(*line ) {
      case COLON:
      case NEWLINE:
	    scr->max_flen[nfields] = MAX(fld_cnt, scr->max_flen[nfields]);
	    rec->tr_fptr[nfields] = (char *) malloc(fld_cnt+1);
	    cpystr(line-fld_cnt, rec->tr_fptr[nfields], fld_cnt);
	    nfields++;
	    if (ln_cnt == 1)
		scr->tr_nfields = nfields;

	    fld_cnt = 0;
            break;

	  default:
	    fld_cnt++;
    
     }
     if (*line == '\n') line = '\0';
     else line ++;
     
  }
  return;
}

/****************************/
cpystr(line,s,fcnt)             /* copy each field to the space allocated */
/****************************/
char *line;
char *s;
int  fcnt;
{
   int  i=0;

   while ( i++ < fcnt)
      *s++ = *line++;
   *s = '\0';
   return;
}


/*****************************/
format_rec( rec, string)             /* format each input line  */
/*****************************/
trrec_t *rec;
char    *string;

{
    int i;
    char fmt[8];
    for (i=scr->s_fld; ((scr->s_fld > scr->e_fld)?(i - scr->tr_nfields):i) < scr->e_fld; i++){
       sprintf(fmt, "%s%-ds", "|%-",scr->max_flen[(i - scr->tr_nfields) >= 0 ? (i - scr->tr_nfields) : i]);
       sprintf(string, fmt, (rec->tr_fptr)[(i - scr->tr_nfields) >= 0 ? (i - scr->tr_nfields) : i]);
       string = string + strlen(string);
    }
    sprintf(fmt, "%s%-ds|", "|%-",scr->max_flen[(i - scr->tr_nfields) >= 0 ? (i - scr->tr_nfields) : i]); 
    sprintf(string, fmt, (rec->tr_fptr)[(i - scr->tr_nfields) >= 0 ? (i - scr->tr_nfields) : i]);
    string = string + strlen(string);

  return;
}

/*****************************/
format_boundary(win_char)            /* format each window boundary for each corresponding input line */
/*****************************/
char *win_char;
{
      int i,j;
      int cnt=0;

      for (i=scr->s_fld; ((scr->s_fld > scr->e_fld)?(i - scr->tr_nfields):i) < scr->e_fld; i++)  {
	  *(win_char + cnt++) = '|';
	  for (j=0; j < scr->max_flen[(i - scr->tr_nfields) >= 0 ? (i - scr->tr_nfields) : i]; j++)
	      *(win_char + cnt++) = '_';
      }
      *(win_char + cnt++) = '|';
      for (j=0; j < scr->max_flen[(i - scr->tr_nfields) >= 0 ? (i - scr->tr_nfields) : i]; j++)
	  *(win_char + cnt++) = '_';
      *(win_char + cnt) = '|';

   return;
}


/****************************/
show_rec(record, dcnt)               /* add each formatted line to window */
/****************************/
char *record;
int  dcnt;
{
	int      x, y,i;

	   y = dcnt; x = 0;
	   wmove(win,y,x);
	   waddstr(win,record);
   return;
}


/*****************************/
draw_window(filename)                /* display the formatted lines on a table in the window */
/*****************************/
char *filename;
{
   int    i,j;
   int    cnt=0;
   char   *string;
   char   *win_char;

   string = (char *)malloc(MAXRECLEN);
   win_char = (char *)malloc(MAXRECLEN);

   format_rec(&scr->tr_scrrec[cnt], string);
   show_rec(string, cnt);

   format_boundary(win_char);
   show_rec(win_char, cnt+1);

     for (i=scr->s_line; i<= scr->e_line; i++) {
       format_rec(&scr->tr_scrrec[i], string);
       show_rec(string, 2*(i - scr->s_line + 1));
       format_boundary(win_char);
       show_rec(win_char, 2*(i- scr->s_line + 1) + 1);
     }

       touchwin(win);
       wrefresh(win);

       if (strcmp (filename, "-") != 0) {
	  mvcur(win->_cury,win->_curx,22,0);
	  fprintf(stderr,"FILE : %s", filename);
       }
       mvcur(win->_cury,win->_curx,22,44);
       fprintf(stderr,"Records %4d thru %4d of %4d",scr->s_line,scr->e_line,scr->tr_nlinrec);

       mvcur(win->_cury,win->_curx,0,0);

       return;
}


  char *helpstr[]={
		  "                 HELP MENU              ",
		  " ______________________________________ ",
                  "|   ?    |    display this menu        |",
		  "|________|_____________________________|",
                  "|   r    |   move forward one page     |",
		  "|________|_____________________________|",
                  "|   e    |   move backward one page    |",
		  "|________|_____________________________|",
                  "|   f    |   move forward 1/3 page     |",
		  "|________|_____________________________|",
                  "|   d    |   move backward 1/3 page    |",
		  "|________|_____________________________|",
                  "|   t    |   move to top of file       |",
		  "|________|_____________________________|",
                  "|   y    |   move to bottom of file    |",
		  "|________|_____________________________|",
                  "|   h    |   shift right one column    |",
		  "|________|_____________________________|",
                  "|   l    |   shift left one column     |",
		  "|________|_____________________________|",
		  "|   q    |   quit, exit dispwin        |",
		  "|________|_____________________________|",

   };


/***********************/
help(terminal)              /* display the menu file       */
/***********************/
{
  int c;
  int y;
  int x=0;
  int begx=19;
  int begy=0;
        
  win->_begx = begx;      win->_begy = begy;
  win->_maxx = 42;        win->_maxy = 22;

  for (y=0; y <= 21; y++) {
      wmove(win,y,x);
      waddstr(win,helpstr[y]);
  }
  touchwin(win);
  wrefresh(win);
  
  mvcur(win->_cury,win->_curx,23,0);
  fprintf(stderr,"Hit any key to continue");
  mvcur(win->_cury,win->_curx,0,0);
  c = getc(terminal);
  return; 

}
