#
/*
Name:
	AUTHUSERS Calendar System Command

Function:
	To add, delete, or list the authorized users of a user's
	calendar system.

Algorithm:
	Add:	Search through the old list of authorized users and if the new user
	to be added is not already on that list then add him.

	Delete:	Search through the old list of authorized users and if the user to
	be deleted is on that list then remove his name form the list.

	List:	List the contents of home-dir/exec/auth.inputters file

Parameters:
	None

Returns:
	ENOERR	Normal exit
	ENOENT	Bad home directory or no auth.inputters file
	ESPIPE	Illegal lseek in auth.inputters
	EACCES	Unable to empty auth.inputters file

Files and Programs:
	home-dir/exec/auth.inputters		place where list of authorized users
						is stored.

Installation Instructions:
	Compile as follows: cc -o $EXECDIR/authusers authusers.c library
	Change owner to root
	Deny write permissions except for owner

History:
	Oct 79,	Release 1 Version 1,G. Maltagliati, coded in shell.
	Nov 80,	Release 5 Version 1,G. Maltagliati, coded in C for
		efficiency.
	Aug 82, Release 7 Version 2, J. Neyer...re-write exec support


*/

#include <stdio.h>
#include <pwd.h>	/* data from /etc/passwd */
#include "../includes/error.h"
#include "../includes/getdef.h"
#define AUTHFILE "exec/auth.inputters"
#define ACTION "Add, delete, or list authorized users (a, d, l): "
#define OACTION "Enter a, d, l only: "
#define OPENERR "authusers: cannot open exec/auth.inputters\n"
#define SEEKERR "authusers: lseek error--exec/authusers\n"
#define CLOSEERR "authusers: cannot close exec/authusers\n"
#define AUSER "Enter new authorized user's userid: "
#define BADUSER "Invalid user.  Re-enter valid userid: "
#define DUSER "Enter the userid of the person you wish to delete: "
#define DELERR "You cannot delete your own userid nor root from the list\n"
#define FILERR "authusers: no authorized users in list\n"
#define NEWLINE '\n'
#define EOS '\0'

char id[] = "%W% %H%";


/*

Name:
	main

Function:
	Finds whether the user wants to add, delete, or list the authorized
	users of his calendar system and calls the appropriate subroutine
	to accomplish the user's intended action.

Algorithm:
	Move to the user's home directory and prepare the auth.inputters file.
	Prompt the user for one of the actions add, delete, or list and then
	call the appropriate subroutine to accomplish that action.

Parameters:
	None

Returns:
	ENOERR	Normal exit
	ENOENT	Bad home directory or no auth.inputters file
	ESPIPE	Illegal lseek to beginning or end of auth.inputters file

Files and Programs:
	home-dir/exec/auth.inputters		place where the list of authorized users
						is stored.


*/
main( argc, argv, envp)

int argc;
char *argv[], **envp;

{
    FILE *fp;
    char    act[5];
    char   *action = act;                               /* action to be performed */


    if (chdir (findenv("HOME", envp)))
    {
	printf("authusers: bad HOME directory\n");
	exit (ENOENT);
    }
    printf(ACTION);
    while (!(gets(action)) || (*action != 'a' && *action != 'd' && *action != 'l'))
							/* Invalid answer */
	    fprintf(stderr, OACTION);
    if ((fp = fopen (AUTHFILE, "rw")) == NULL)                /* Open auth.inputters for reading and writing */
    {
	fprintf(stderr, OPENERR);
	exit (ENOENT);
    }
    switch (*action)
    {
	case 'a': 					/* Add a new authorized user */
	    adduser (fp);
	    break;
	case 'd': 					/* Delete an authorized user from the current list */
	    deleteuser(fp);             /* delete a user */
	    break;
	case 'l': 					/* List all current authorized users */
	    listuser (fp);
	    break;
    }
    fclose (fp);
    exit (ENOERR);
}
/*

Name:
	adduser

Function:
	Add an authorized user to a user's authorized user list.

Algorithm:
	Prompt the user for a userid to add to the user's authorized user's list.
	Search through the old list and if the user to be added is not on that
	list already, then add the name to the bottom of the list.

Parameters:
	None

Returns:
	ESPIPE	Illegal lseek in home-dir/exec/auth.inputters file

Files and Programs:
	home-dir/exec/auth.inputters		place where list of authorized users
						is stored.


*/
adduser (fp)
FILE *fp;
{
    char    user_buf[10];
    char   *user = user_buf,                            /* Userid input */
            buffer[513],				/* Holds list of current authorized users */
           *buf_ptr = &buffer,				/* Pointer to buffer */
            compare[9],					/* Holds one userid retrieved from buffer */
           *comp_ptr,					/* Pointer to compare */
            newuser[9],					/* Holds userid (with a newline added to the end) to be */
							/* put on the authorized users list */
           *end_ptr;					/* Pointer to null character in newuser */
    register int    flag = 1,				/* Flags the end of comparisons */
                    writeflag = 1,			/* Flags whether to write the userid out or not */
                    num_chars;				/* Number of characters in buffer */


    printf(AUSER);
    while (!(gets(user)) && getpwnam(user))
	fprintf(stderr, BADUSER);
    if (num_chars = fread (buffer, sizeof(*buffer), 512, fp));  /* Read in all authorized users into buffer */

/* Search through the buffer until EOS is reached or until the user being */
/* entered is found in the buffer and therefore does not need to be added */
	while (flag)
	{
	    comp_ptr = &compare;
	    while (*buf_ptr != NEWLINE)			/* Extract one userid from the buffer */
	    {
		*comp_ptr++ = *buf_ptr++;
		num_chars--;
	    }
	    *comp_ptr = EOS;
	    if (!strcmp (compare, user))			/* The user to be entered is already on the list */
		writeflag = flag = 0;			/* Turn flags off */
	    if (--num_chars == 0)			/* EOS has been reached */
		flag = 0;
	    buf_ptr++;
	}
    if (writeflag == 1)
    {
	fclose(fp);
	fp = fopen(AUTHFILE, "a");
	fprintf(fp, "%s\n", user);                    /* Put the newuser on the list */
	printf ("User *** %s *** has been added to the authorized user's list\n", user);
    }
    else						/* User is already on the authorized user's list */
	printf ("User *** %s *** is already on the authorized user's list\n", user);
}
/*

Name:
	deleteuser

Function:
	Delete a user from a calendar system user's authorized users list

Algorithm:
	Prompt the user for a userid to be deleted. Copy the old list into a buffer.
	Search through the buffer. If the name was found in the old list and the name
	is not the userid of the person logged in, then copy all  names left after
	the userid to be deleted to the position in the buffer where the userid was
	located. Now output the buffer to the auth.inputters file.
	If the name was not found on the list or it was the first or second
	name print a diagnostic.

Parameters:

Returns:
	ESPIPE	Illegal lseek in auth.inputters
	EACCES	Cannot empty the auth.inputters file

Files and Programs:
	None

Assumptions:
	The first two names in the file must be the user himself and root.


*/
deleteuser (fp)
FILE *fp;
{
    char    user_buf[10];
    char   *user = user_buf,                            /* Userid input */
            buffer[513],				/* Holds list of current authorized users */
           *buf_ptr = buffer,				/* Pointer to buffer */
            compare[9],					/* Holds one userid retrieved from buffer */
           *comp_ptr,					/* Pointer to compare */
           *hold_ptr;					/* Pointer into buffer */
    register int    flag = 1,				/* Flags the end of comparisons */
		   count,		/* counts userids scanned */
                    delflag = 1,			/* Flags whether to delete the userid  or not */
                    num_chars;				/* Number of characters in buffer */


    printf(DUSER);
    while (!(gets(user)) && getpwnam (user))
	printf(BADUSER);
    if (num_chars = fread (buffer, sizeof(*buffer), 512, fp));  /* Read in all authorized users into buffer */
    {
	buffer[num_chars] = EOS;

/* Search through the buffer until EOS is reached or until the user being */
/* deleted is found in the buffer and therefore will be deleted */
	for( count = 0; flag; count++) {
	    comp_ptr = &compare;
	    while (*buf_ptr != NEWLINE)			/* Extract one userid from the buffer */
	    {
		*comp_ptr++ = *buf_ptr++;
		num_chars--;
	    }
	    *comp_ptr = EOS;
	    if (!strcmp (compare, user))			/* The user to be deleted is on the list */
		delflag = flag = 0;			/* Turns delete flag on, and loop flag off */
	    if (--num_chars == 0)			/* EOS has been reached */
		flag = 0;
	    buf_ptr++;
	}
    }
    if (delflag == 0 && count > 2)
    {
							/* User is to be delete fromto the authorized user's list */
	fclose(fp);
	if ((fp = fopen (AUTHFILE, "w")) == NULL)      /* Empty the auth.inputters file */
	{
	    fprintf(stderr, OPENERR);
	    exit (EACCES);
	}
/* Set one pointer to the last character being pointed at and another to the */
/* first character after the newline of the user being deleted */
	hold_ptr = buf_ptr;
	buf_ptr = buf_ptr - strlen (compare) - 1;
	copy (hold_ptr, buf_ptr);			/* Removes the user to be deleted */
	fprintf(fp, "%s", buffer);            /* Puts out the new authorized users list */
	printf ("User *** %s *** has been deleted from the authorized user's list\n", user);
    }
    else
	if (delflag)		/* userid was not in the file */
	    printf ("User *** %s *** was not on the authorized user's list\n", user);
	else		/* root or userid of logged-in user */
	    write (2, DELERR, sizeof (DELERR));
}
/*

Name:
	listuser

Function:
	List all authorized users of a users calendar system

Algorithm:
	If there are any names in the authorized users list then output them.
	Otherwise, print a diagnostic and exit.

Parameters:
	None

Returns:
	ENOENT	auth.inputters file is empty

Files and Programs:
	None


*/
listuser (fp)
FILE *fp;
{
    register int    num_chars;				/* The number of characters read in a buffer */
    char    buffer[512];				/* The input/output buffer */


    if ((num_chars = fread (buffer, sizeof(*buffer), 512, fp)))  /* Read in all authorized users into buffer */
	fwrite (buffer, sizeof(*buffer), num_chars, stdout);   /* Print on the user's terminal the list of authorized users */
    else						/* Auth.inputters file is empty */
    {
	fprintf(stderr, "num_chars = %d\n", num_chars);
	fprintf (stderr, FILERR);
	exit (ENOENT);
    }
}
