#include "x25.h"
#include "com.h"
#include "stdio.h"

#define	HL_SIZE	600		/* maximum HDLC data size */
#define INC8(A) ( ( A == 8 ) ? 0 : A+1 )  /* Increment modulo 8 */

/**********************************************************************
File : x25_tl.c
Date : Dec28th 1989
Author : Gilles Lejeune
Desc : list of X25 <-> asynchronous test functions
List :
	init_test    : initialize test
	end_test     : end a test
	x25_error    : handle X25 error
        asy_error    : handle asynchronous error
        send_x25     : send an X25 packet
        send_asy     : send an asynchronous string
        expect_x25   : expect an X25 packet
        expect_asy   : expect an asynchronous string
        put_x25_mess : put X25 packet type in a string

Comment : Those functions are used as library functions in the
          scenarios of the X25/Async tests. They are very basic
          as they are now and are shown as examples. They can be
          modified or taken as example for more sofisticated functions.

**********************************************************************/

/*----- Global data -----*/

extern	long	_stdvt ;

int		ps,pr ;		/* X25 P(S) and P(R) */
int		lcn = 0 ;	/* received LCN */
int		lcgn = 0 ;	/* received LCGN */

/****** Tests initialization *************************
parameters :
	pstr : test description string pointer
******************************************************/

init_test(pstr) 
char	*pstr ;
{
com_crctlmb(2) ;          /* initialize 2 pipes of communications */
com_flush(PORT_A) ;       /* flush port A */
com_flush(PORT_B) ;       /* flush port B */
disablecur(_stdvt) ;      /* disable cursor */
clears() ;                /* clear screen */
disp_init(pstr) ;         /* display init screen */
ps = pr = 0 ;             /* initialize variables */
lcgn = 0 ;
lcn = 1 ;
}

/****** Successfull end of test function ****************/

end_test()
{
dispend() ;
com_flush(PORT_A) ;
com_flush(PORT_B) ;
getchar() ;
}

/****** X25 error function *****************************
parameter :
	ptr : error message pointer
********************************************************/

x25_error(ptr)
char	*ptr ;
{
disp_x25_error(ptr) ;
com_flush(PORT_A) ;
getchar() ;
}

/****** asynchronous error function ********************
parameter :
	ptr : pointer on error message
********************************************************/

asy_error(ptr)
char	*ptr ;
{
disp_asy_error(ptr) ;
com_flush(PORT_B) ;
getchar() ;
}

/****** send an X25 packet ***************************************
parameter :
	type : type of packet
	pdata : pointer on data buffer ( if necessary )

Comment : This function will display the message to be sent,build
          a standard packet and send it using the communication
          pipe with the HDLC task.
******************************************************************/

send_x25(type,pdata)
int	type ;
char	*pdata ;
{
struct	x25_st	x25_enc ;
struct	x25_st	*px25  ;
MB_MESS		mess_st ;
MB_MESS		*pmess  ;
char		*pframe ;
char		dispbuf[100] ;
char		facbuf[10] ;

strcpy(dispbuf,"X25 <- ") ;
put_x25_mess(type,dispbuf) ;
disp_x25(dispbuf) ;

px25 = &x25_enc ;
pmess = &mess_st ;

px25->x25_type = type ;	
px25->x25_q = 0 ;
px25->x25_d = 0 ;
px25->x25_mod = 1 ;
px25->x25_lcgn = lcgn ;
px25->x25_lcn = lcn ;
px25->x25_ps = ps ;
px25->x25_pr = pr ;
px25->x25_m = 0 ;
px25->x25_cga = (unsigned char*)"" ;
px25->x25_cgal = 0 ;
px25->x25_cda = (unsigned char*)"0017" ;
px25->x25_cdal = 4 ;
if ( ( px25->x25_type == PT_CR ) || ( px25->x25_type == PT_CA ) )
	{
	facbuf[0] = 0 ;
	px25->x25_data = facbuf ;
	px25->x25_dtl = 1 ;
	}
if ( px25->x25_type == PT_DT )
	{
	px25->x25_data = (unsigned char *)pdata ;
	px25->x25_dtl = strlen(pdata) ;
	ps = INC8(ps) ;
	}
if ( px25->x25_type == PT_CLR )
	{
	facbuf[0] = 1;
	facbuf[1] = 2 ;
	px25->x25_data = facbuf ;
	px25->x25_dtl = 2 ;
	}
if ( ( px25->x25_type == PT_RST ) || ( px25->x25_type == PT_RSTC ) )
	px25->x25_lcn = 0 ;

if ( ( pframe = com_gptr(HL_SIZE) ) == NULL )
	puts("HDLC tx alloc problem") ;

pmess->len = x25_b(pframe,px25) ;
pmess->pdata = pframe ;
pmess->port = PORT_A ;
pmess->type = CT_DATA ;

com_snd(pmess) ;
}

/****** send an asynchronous string ******************************
parameter :
	ptr : pointer on data string

Comment : This function will display the message to be sent and
          send it using the communication pipe with the asynchronous
          task.
******************************************************************/

send_asy(ptr)
char	*ptr ;
{
char		*pdata ;
MB_MESS		mess_st ;
MB_MESS		*pmess = &mess_st ;
int		len ;
char		dispbuf[100] ;

strcpy(dispbuf,"ASYNC -> ") ;
strcat(dispbuf,ptr) ;
disp_async(dispbuf) ;

if ( ( pdata = com_gptr(HL_SIZE)) == NULL )
	puts("TX ASY alloc problem") ;

strcpy(pdata,ptr) ;

pmess->pdata = pdata ;
pmess->len = strlen(pmess->pdata) ;
pmess->port = PORT_B ;
pmess->type = CT_DATA ;

com_snd(pmess) ;
}

/****** expect an X25 packet **********************************
parameters :
        type : type of packet expected
        delay ( in ms ) : maximum delay to expect the packet
        ptr : pointer on a user buffer to put the received data
return :
        type of packet received
        -1 : nothing received after delay

Comment : this function will check every 50 ms, the communication
          pipe with the HDLC task for a reception of message and
          this until the maximum delay expires. If a message is
          received, it will be decoded to extract the mesage type
          and the data ( if necessary ). 
*****************************************************************/

expect_x25(type,delay,ptr)
int	type,delay ;
char	*ptr ;
{
MB_MESS		mess_st ;
MB_MESS		*pmess = &mess_st ;
int		rec,time ;
struct	x25_st	x25_dec ;
struct	x25_st	*px25 = &x25_dec ;
unsigned	char	cdabuf[10],cgabuf[10] ;
char		dispbuf[100] ;

pmess->port = PORT_A ;

px25->x25_cda = cdabuf ;
px25->x25_cga = cgabuf ;
px25->x25_data = (unsigned char*) ptr ;

for(time = 0 ; time <= delay/50 ; time++ )
	{
	rec = com_chkmb(pmess) ;
	if ( rec == CT_DATA )
		{
		x25_ub(pmess->pdata,pmess->len,px25) ;
		com_rel	(pmess->pdata) ;
		strcpy(dispbuf,"X25 -> ") ;
		put_x25_mess(px25->x25_type,dispbuf) ;
		disp_x25(dispbuf) ;
		return(px25->x25_type) ;
		}
	pause(50+MS) ;
	}
return(-1) ;
} 

/****** expect an asynchronous string ***************************
parameters :
        delay ( in ms ) : maximum delay to expect thestring 
        ptr : pointer on a user buffer to put the received data
return :
	0  : string received
        -1 : nothing received after delay

Comment : this function will check every 50 ms, the communication
          pipe with the asynchronous task for a reception of message and
          this until the maximum delay expires. If a message is
          received, the data will be copied into the user buffer. 
*****************************************************************/

expect_asy(delay,ptr)
int	delay ;
char	*ptr ;
{
MB_MESS		mess_st ;
MB_MESS		*pmess = &mess_st ;
int		rec,time ;
char		dispbuf[100] ;

pmess->port = PORT_B ;

for(time = 0 ; time <= delay/50 ; time++ )
	{
	rec = com_chkmb(pmess) ;
	if ( rec == CT_DATA )
		{
		strcpy(ptr,pmess->pdata) ;
		com_rel	(pmess->pdata) ;
		strcpy(dispbuf,"ASYNC <- ") ;
		strcat(dispbuf,ptr) ;
		disp_async(dispbuf) ;
		return(0) ;
		}
	pause(50+MS) ;
	}
return(-1) ;
} 

/****** put the X25 packet type received into a string *****
parameter :
	type : type of packet
        dispbuf : pointer on a user buffer to copy the string
***********************************************************/

put_x25_mess(type,dispbuf)
int	type ;
char	*dispbuf ;
{
switch(type)
	{
	case PT_CR :
		strcat(dispbuf,"Call Req") ;
		break ;
	case PT_CA :
		strcat(dispbuf,"Call Conf") ;
		break ;
	case PT_CLR :
		strcat(dispbuf,"Clear Req") ;
		break ;
	case PT_CLC :
		strcat(dispbuf,"Clear Conf") ;
		break ;
	case PT_IT :
		strcat(dispbuf,"IT packet") ;
		break ;
	case PT_ITC :
		strcat(dispbuf,"IT Conf") ;
		break ;
	case PT_RR :
		strcat(dispbuf,"Rec Ready") ;
		break ;
	case PT_RNR :
		strcat(dispbuf,"Rec Not Ready") ;
		break ;
	case PT_REJ :
		strcat(dispbuf,"Reject") ;
		break ;
	case PT_RES :
		strcat(dispbuf,"Reset") ;
		break ;
	case PT_REC :
		strcat(dispbuf,"Reset Conf") ;
		break ;
	case PT_RST :
		strcat(dispbuf,"Restart") ;
		break ;
	case PT_RSTC :
		strcat(dispbuf,"Restart Conf") ;
		break ;
	case PT_DIAG :
		strcat(dispbuf,"Diagnostic") ;
		break ;
	case PT_DT :
		strcat(dispbuf,"Data") ;
		break ;
	default :
		break ;
	}
}
		

