/* udchain.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */
/*
 *	Use-Definition Chain Calculation
 */

/*
 *	See: Aho and Ullman, Principles of Compiler Design, Section 12.5
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: udchain.c,v 5.5 89/05/12 12:53:23 pcc Rel-3_0 $";
/* static char ID[] = "@(#)udchain.c	15.3	of 86/10/10"; */
#endif

/*	Import
 */

# include <activity.h>
# include <bool.h>
# include <udchain.h>
# include <assert.h>
# include <blocks.h>
# include <dag.h>
# include <flow.h>
# include <identifier.h>
# include <dagsymbol.h>
# include <longset.h>
# include <storage.h>
# include <erroro.h>

# define UD_SET_INIT	1024		/* initial UD set size */
# define UD_SET_INCR	256		/* amount to increase UD set size */

extern int ddebug;

/*	Export
 */
DAG_Node *UDDagPtrs;		/* array of pointers to DAG nodes that are
				 * definitions.  All UD information is stored
				 * as subsets of this set
				 */
int UDSetSize;			/* current size of UD Chain Sets */
int UDInitSetSize = UD_SET_INIT;/* initial size of UD Chain Sets */

/*	Import
 */

extern int ddebug;

/*	Private
 */

	/* Cases for which we keep track of UD information in symbol table */
#define InterestingCases	case REG: \
				case NAME: \
				case LNAME: \
				case PNAME: \
				case STATNAME

static void Gen2Def();
static void UDSetup();
static Boolean Eval1Node();
static Boolean Eval2Node();
static void Done1();
static void Done2();
static void Gen1Dly();
static void Gen2Dly();
static DAG_Node UndefinedDAG;	/* a DAG node used to provide an initial value
				 * for every identifier
				 */
static LongSet NonamePtrs;	/* for keeping track of pointer assigns about
				 * which we know nothing
				 */
static LongSet StartDefs;	/* The set of initial definitions of each
				 * identifier */

static int NextDefFree;		/* index in UDDagPtrs of next free def */

static LongSet GTmpset;		/* Tunnel from UDSecondWalk to Gen2Dly */
static BasicBlock GBlock;	/* ditto */

static Boolean EN2_CalcIn;	/* Implicit parameter to Eval2Node */
static BasicBlock EN2_b;	/* Implicit parameter to Eval2Node */


/*
 * Solve data flow equations for UD chains
 */

/*
 * Initialization for UD Chains
 */
void
InitUDInfo()
{
	NextDefFree = 0;

	if( UDDagPtrs != NULL ) {
		free(UDDagPtrs);
		DecreaseSpace(s_UDptrs, UDSetSize * sizeof(DAG_Node));
	}
	
	UDSetSize = UDInitSetSize;
	UDDagPtrs = GetArray(s_UDptrs, UDSetSize, DAG_Node);
	CheckStorage(UDDagPtrs, "storage for for UD chains (set size = %d).", UDSetSize);

	if( NonamePtrs != NULL )
		DestroySet(NonamePtrs);
	NonamePtrs = CreateSet(UDSetSize);

	if( StartDefs != NULL )
		DestroySet(StartDefs);
	StartDefs = CreateSet(UDSetSize);
}

/*
 * The actual work of setting up the UD Chains
 */
void
UDChains()
{
	int i;
	BasicBlock b;

	NullSet(NonamePtrs);

	UDSetup();

	for( i = 0; i < NumReachableNodes; i++)
	{
		UDFirstWalk(FlowGraph[DFN[i]].block);
	}

	for( i = 0; i < NumReachableNodes; i++)
	{
		b = FlowGraph[DFN[i]].block;
		if( i == 0 )
				/* provide initial defs of everything */
			CopySet(b->ud.In, StartDefs);
		else
			NullSet(b->ud.In);

		UDSecondWalk(b,False);
	}

	UDDataFlow();
}

void
UDDataFlow()
{
	int i;
	LongSet NewIn;
	Boolean changed;
	FlowIndex n;
	BasicBlock b, bb;
	PredNode p;

	/* Provide in, out for initial block */

	b = FlowGraph[0].block;
	CopySet(b->ud.In, StartDefs);
	Difference(b->ud.Out, b->ud.In, b->ud.Kill);
	Union( b->ud.Out, b->ud.Out, b->ud.Gen );

	/* Provide initial Out for other blocks */

	for( i = 1; i < NumReachableNodes; i++)
	{
		b = FlowGraph[DFN[i]].block;
		CopySet(b->ud.Out, b->ud.Gen);
	}

	NewIn = CreateSet(UDSetSize);
	do
	{
		changed = False;
		for( i = 1; i < NumReachableNodes; i++ )
		{
			n = DFN[i];
			b = FlowGraph[n].block;
			NullSet(NewIn);
			for( p = FlowGraph[n].preds; p != NULL; p = p->next)
			{
				bb = FlowGraph[p->p].block;
				if( bb->reachable )
					Union(NewIn, NewIn, bb->ud.Out);
			}
			if( Not(SetEq(NewIn,b->ud.In)))
			{
				changed = True;
				CopySet(b->ud.In,NewIn);
			}
			Difference(NewIn, NewIn, b->ud.Kill);
			Union( b->ud.Out, NewIn, b->ud.Gen );
		}
	} while( changed );

	if( ddebug > 1 )
		PUDInfo();
	DestroySet(NewIn);
}

static DefsIndex
NewDef()
{
	int n;

	n = NextDefFree++;
	if( n >= UDSetSize )
	{
		UDDagPtrs = (DAG_Node *) realloc((char *)UDDagPtrs,
		     (unsigned)((UDSetSize + UD_SET_INCR) * sizeof(DAG_Node)));
		CheckStorage(UDDagPtrs, "storage for UD chains (set size = %d).", UDSetSize);
		UDSetSize += UD_SET_INCR;
		assert(n < UDSetSize);
		IncreaseSpace(s_UDptrs, UD_SET_INCR * sizeof(DAG_Node));

		/* Might be a good idea to expand set of noname pointers */
		/* ExpandSet( NonamePtrs, UDSetSize); */

	}
	return(n);
}

/*
 * UD chain initialization once the DAG has been built.  Main idea here
 * is to provide a set of definitions and an initial value for each
 * name in the symbol table
 */
static void
UDSetup()
{
	Identifier id;
	LongSet sdefs;
	DefsIndex newdef;

	NullSet(StartDefs);
	if( UndefinedDAG == NULL )
	{
		UndefinedDAG = (DAG_Node) calloc((unsigned)1,sizeof(*UndefinedDAG));
		CheckStorage(UndefinedDAG, "storage for Undefined DAG", 0);
		UndefinedDAG->op = ERROR;
		UndefinedDAG->FGindex = NullFlow;
	}

	for( id=FirstId; id<=MaxIdentifier; ++id )
	switch(IdOp(id))
	{
	InterestingCases:
		sdefs = GetDefs(id);
		if( sdefs == NULL )
			sdefs = InitDefs(id, UDSetSize);
		newdef = NewDef();
		UDDagPtrs[newdef] = UndefinedDAG;
		Insert((int)newdef,sdefs);
		Insert((int)newdef,StartDefs);
		break;
	}
}

/*
 * First walk over DAG to collect UD information
 */
void
UDFirstWalk( b )
	BasicBlock b;
{

	if( b->ud.Gen == NULL )
	{
		b->ud.Gen = CreateSet(UDSetSize);
		b->ud.In = CreateSet(UDSetSize);
		b->ud.Out = CreateSet(UDSetSize);
		b->ud.Kill = CreateSet(UDSetSize);
	}
	NullSet(b->ud.In);

	(void) ActivityDag(b->Dag, Eval1Node);
}

static Boolean
Eval1Node(d)
	DAG_Node d;
{
	AttachedID a;
	Identifier id;
	DefsIndex newdef;
	LongSet sdefs;

	if( (d->op == UNARY MUL  &&  !d->is_fetch)  ||  callop(d->op) )
	{
		d->UDIndex = newdef = NewDef();
		UDDagPtrs[newdef] = d;
		if( d->indirect == NoId )
		{
			/* We have no idea what may be killed by this
			 * assign/call.
			 * Create a single def, and make it a possible
			 * def for all globals and any locals or
			 * parameters whose addresses were taken.
			 * (Note that if the address of any parameter
			 * is taken, readero marks ALL parameters as
			 * having their addresses taken.)
			 * Remember the def so that it will
			 * never be killed
			 */
			Insert((int)newdef,NonamePtrs);
			for( id=FirstId; id<=MaxIdentifier; id++ )
			if( IdOp(id) == NAME || WasAddressed(id))
			{
				sdefs = GetDefs(id);
				assert(sdefs != NULL);
				Insert((int)newdef,sdefs);
			}
		}
		else
		{
			sdefs = GetDefs(d->indirect);
			assert(sdefs != NULL);
			Insert((int)newdef,sdefs);
		}
	}

	for( a = d->attached ; a != NULL; a = a->next)
	switch(IdOp(a->id))
	{
	InterestingCases:
		/*
		 * Note: for this code, we don't need to worry about
		 * delays.
		 */

		a->UDIndex = Gen1Def(a->id,d);
		break;
	}
	if (d->delayed != NULL && d->special_delay) {
		/* Do special delayed store.  This mimics the
		 * algorithm used in dagtree.c */

		Gen1Dly(d);
	}

	UpdateActivity(d, Done1);
	return False;
}

/*
 * The activity count on node "d" has just gone to 0.  Do delayed stores
 * that are waiting on "d".
 */

static void
Done1(d)
	DAG_Node d;
{
	assert(d->activity == 0);
	if( d->delayed && !(d->special_delay) )
		Gen1Dly(d);
}

/*
 * "d" has a delayed store waiting on it.  Do it and (recursively) any
 * delayed stores waiting on this delayed store
 */

static void
Gen1Dly(d)
	DAG_Node d;
{
	assert(optype(d->op) == LTYPE);
	switch(IdOp(d->leaf_id))
	{
	InterestingCases:
		d->UDIndex = Gen1Def(d->leaf_id,d->delayed);
		break;
	}
	assert(d->delayed->activity > 0);
	--(d->delayed->activity);
	if( d->delayed->activity == 0 )
		Done1(d->delayed);
}

/*
 * This routine also called by loop invariant motion code
 */
int
Gen1Def(id,d)
	Identifier id;
	DAG_Node d;
{
	LongSet sdefs;
	DefsIndex newdef;

	sdefs = GetDefs(id);
	assert(sdefs != NULL);
	newdef = NewDef();
	UDDagPtrs[newdef] = d;
	Insert((int)newdef,sdefs);
	return newdef;
}

/*
 * This routine called by loop invariant code to fix up the UD information
 * for a pre-header
 */
void
UDNewBlock(preheader)
	BasicBlock preheader;
{
	preheader->ud.Gen  = CreateSet(UDSetSize);
	preheader->ud.In   = CreateSet(UDSetSize);
	preheader->ud.Out  = CreateSet(UDSetSize);
	preheader->ud.Kill = CreateSet(UDSetSize);
	NullSet(preheader->ud.Kill);
	NullSet(preheader->ud.Gen);
}

/*
 * Second walk over DAG to collect UD information
 */

void
UDSecondWalk(b,CalculateIn)
	register BasicBlock b;
	Boolean CalculateIn;
{
	GBlock = b;
	GTmpset = CreateSet(UDSetSize);
	NullSet(GTmpset);

	NullSet(b->ud.Gen);
	NullSet(b->ud.Kill);
	CopySet(b->ud.Out, b->ud.In);

	EN2_CalcIn = CalculateIn;		/* Set implicit parameter */
	EN2_b = b;				/* Ditto */
	(void) ActivityDag(b->Dag, Eval2Node);
	DestroySet(GTmpset);
}

static Boolean
Eval2Node(d)
	register DAG_Node d;
{
	register AttachedID a;
	register int newdef;

	/* Only attach In sets in places where they are used
	 */

	if( EN2_CalcIn &&
	    ( d->attached != NULL || d->delay_count != 0 ||
	      d->op == NAME || d->op == LNAME || d->op == PNAME ||
	      d->op == REG || d->op == STATNAME || callop(d->op) ||
	      (d->op == UNARY MUL && d->is_fetch)))
	{
		if( d->In == NULL )
			d->In = CreateSet(UDSetSize);
		CopySet(d->In, EN2_b->ud.Out);
	}

	if( (d->op == UNARY MUL  &&  !d->is_fetch) ||  callop(d->op) )
	{
		newdef = d->UDIndex;
		Insert(newdef,EN2_b->ud.Gen);
		Insert(newdef,EN2_b->ud.Out);

#ifdef ImplementAnotherTime
		if( d->indirect != NoId )
		{
			/* We know what the assign or call may kill,
			 * and have created a def for this symbol.
			 * The def has been included in Gen and Out.
			 * However, do not kill other defs unless
			 * we know that the symbol is atomic.
			 */

			if(the thing is atomic)
				Gen2Def(d->indirect, newdef, b, GTmpset);
		}
#endif
	}

	/* For each attached identifier, remove all previous
	 * definitions from Out and Gen, and add this one.
	 * Also update Kill to show that all other defs are
	 * killed here.
	 * This is slightly complicated by the fact that
	 * we must not kill a definition that is a Noname Pointer
	 */

	for( a = d->attached ; a != NULL; a = a->next)
	switch(IdOp(a->id))
	{
	InterestingCases:
		Gen2Def(a->id, a->UDIndex, EN2_b, GTmpset);
		break;
	}
	if (d->delayed != NULL && d->special_delay) {
		/* Do special delayed store.  This mimics the
		 * algorithm used in dagtree.c */
		Gen2Dly(d);
	}

	UpdateActivity(d, Done2);
	return False;
}

/*
 * The activity count on node "d" has just gone to 0.  Do delayed stores
 * that are waiting on "d".
 */

static void
Done2(d)
	DAG_Node d;
{
	assert(d->activity == 0);
	if( d->delayed && !(d->special_delay) )
		Gen2Dly(d);
}

/*
 * "d" has a delayed store waiting on it.  Do it and (recursively) any
 * delayed stores waiting on this delayed store
 */

static void
Gen2Dly(d)
	DAG_Node d;
{
	assert(optype(d->op) == LTYPE);
	switch(IdOp(d->leaf_id))
	{
	InterestingCases:
		Gen2Def(d->leaf_id, d->UDIndex, GBlock, GTmpset);
		break;
	}
	assert(d->delayed->activity > 0);
	--(d->delayed->activity);
	if( d->delayed->activity == 0 )
		Done2(d->delayed);
}

/*
 * Routine called by the second DAG walk to generate UD information.
 */

static void
Gen2Def(id, newdef, b, tmpset)
	Identifier id;
	register DefsIndex newdef;
	register BasicBlock b;
	register LongSet tmpset;
{
	register LongSet sdefs;
	register DAG_Node d;

	d = UDDagPtrs[newdef];
	if( d->in_cond == NotConditional )
	{
		sdefs = GetDefs(id);
		assert(sdefs != NULL);
		Difference(tmpset, sdefs, NonamePtrs);
		Difference(b->ud.Out, b->ud.Out, tmpset);
		Difference(b->ud.Gen, b->ud.Gen, tmpset);
		Union(b->ud.Kill, b->ud.Kill, tmpset); /* include this def */
	}
	Insert((int)newdef,b->ud.Gen);
	Insert((int)newdef,b->ud.Out);
}

static void
PUDSet(s)
	LongSet s;
{
	int i;
	DAG_Node n;
	int count = 0;

	i = FirstElement(s);
	if( i != NoElement )
	{
		printf("\t\t");
		do
		{
			if( count >= 10 )
			{
				printf("\n\t\t");
				count = 0;
			}

			n = UDDagPtrs[i];
			printf("%x (%d), ", n, n->order);
			count++;

			i = NextElement(i,s);
		} while( i != NoElement );
	}
	if( count != 0 )
		printf("\n");
}

static void
PUDBlock(b)
	BasicBlock b;
{
	printf("UD Block: %o (#%d)\n", b, b->blocknum);
	printf("\tGen:\n"); PUDSet(b->ud.Gen);
	if( ddebug > 2 )
	{
		printf("\tKill:\n"); PUDSet(b->ud.Kill);
		printf("\tIn:\n"); PUDSet(b->ud.In);
		printf("\tOut:\n"); PUDSet(b->ud.Out);
	}
}

void
PUDInfo()
{
	int n;
	for( n = 0; n < NumBlocks; n++ )
	{
		PUDBlock(FlowGraph[n].block);
	}
}
