/* instruct.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */
/*
 *	This module implements operations for the "Instruction"
 *	type.
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: instruct.c,v 5.5 89/05/12 12:51:10 pcc Rel-3_0 $";
/* static char ID[] = "@(#)instruct.c	15.2	of 86/10/25"; */
#endif

/*	Import
 */

# include <stdio.h>
# include <string.h>
# include <assert.h>
# include <instruct.h>
# include <storage.h>
# include <tree.h>
# include <pcc.h>
# include <erroro.h>

/*
 *	Private
 */

static Instruction Free	   = NULL;	/* Available instructions */

/*
 *	File name management
 */

typedef struct FNP {
	struct FNP *next;		/* a linked list */
	char *name;			/* the file name */
} FNPType, *FilenameList;

static FilenameList Files	= NULL;	/* a list of file names */

/*
 *	Storage Management
 */

void
InstFree(i)
	Instruction i;
{
	PccFreeInst(i);		/* handle machine dependent stuff */
	switch (InstType(i)) {
		case FcnStart:
			if (i->u.fb.fname != NULL) {
				DecreaseSpace(s_Text, strlen(i->u.fb.fname)+1);
				free(i->u.fb.fname);
			}
			break;
		case Passed:
			DecreaseSpace(s_Text, strlen(i->u.tx.line)+1);
			free(i->u.tx.line);
			break;
		case Expr:
			if (i->u.ex.root != NULL)
				walkf(i->u.ex.root, TreeFree);
			break;
		case Switch:
			free(i->u.sw.swtable);
			break;

		case FreeInst:
			break;

#ifdef FORT
		case EntryPoint:
			if (i->u.ent.ename != NULL)
				free(i->u.ent.ename);
			break;
#endif

		default:
			break;
	}
	i->next = Free;
	Free = i;
}

void
InitList(list)
	InstrList *list;
{
	list->first = NULL;
	list->last = NULL;
}

void
FreeIList(list)
	InstrList *list;
{
	Instruction p, next;

	p = list->first;
	while (p != NULL) {
		next = p->next;
		InstFree(p);
		p = next;
	}
	InitList(list);
}

static Instruction
AllocInstruction()
{
	Instruction ni, c;
	int i;

	if (Free == NULL) {
		c = GetArray(s_Instruct, InstrChunk, Instr);
		CheckStorage(c, "storage for instructions", 0);

		/* Turn array into linked list (backwards!) */

		ni = NULL;
		for (i = 0; i < InstrChunk; ++i) {
			c->next = ni;
			ni = c++;
		}

		/*
		 * Now ni points to the head of the free list (the
		 * last entry in the chunk).
		 */
		
	} else
		ni = Free;

	Free = ni->next;
	return ni;
}

char *
StoreText(p)
	char *p;
{
	char *saved = NULL;
	int len;

	if (p != NULL) {
		len = strlen(p);
		saved = malloc((unsigned) (len + 1));
		CheckStorage(saved, "storage for instruction text (%d)", len+1);
		IncreaseSpace(s_Text, len + 1); 
		(void) strcpy(saved, p);
	}
	return saved;
}

/*
 *	File name handling
 */

char *
EnterFilename(s)			/* save a file name */
	char * s;
{
	FilenameList fp;

	if (Files == NULL || strcmp(s, Files->name) != 0) {  /* New Name */
		fp = GetStorage(s_NoStats, FNPType);
		CheckStorage(fp, "storage for file names", 0);
		fp->next = Files;
		fp->name = StoreText(s);
		Files = fp;
	}
	return Files->name;
}

Instruction
CreateInstruction(t)
	InstTag t;
{
	Instruction p;

	p = AllocInstruction();
	assert(p != NULL);
	p->next = NULL;
	p->tag = t;
	return p;
}

void
ConcatInstructions(list, extra)		/* append extra to list */
	InstrList *list;
	InstrList *extra;
{
	assert(list->first==NULL? list->last==NULL : list->last!=NULL && list->last->next==NULL);
	assert(extra->first==NULL? extra->last==NULL : extra->last!=NULL && extra->last->next==NULL);
	if (list->last == NULL)
		list->first = extra->first;
	else
		list->last->next = extra->first;

	if (extra->last == NULL)
		; /* list->last = list->last; */
	else
		list->last = extra->last;
	InitList(extra);
}

void
AppendInstruction(i, list)
	Instruction i;
	InstrList *list;
{
	assert(list->first==NULL? list->last==NULL : list->last!=NULL && list->last->next==NULL);
	if (list->last == NULL)
		list->first = i;
	else
		list->last->next = i;
	list->last = i;
	assert(i->next == NULL);	/* i->next = NULL; */
}

Instruction
AddInstruction(t, list)
	InstTag t;
	InstrList *list;
{
	Instruction i;

	i = CreateInstruction(t);
	AppendInstruction(i, list);
	return i;
}

void
DelInstruction(i,list)
	Instruction i;
	InstrList *list;
{
	Instruction p, pp;

	assert(i!=NULL && list->first!=NULL && list->last!=NULL && list->last->next==NULL);
	for (pp=NULL, p = list->first; p!=i; pp=p, p=p->next)
		assert(p != NULL);
	if (pp == NULL)
		list->first = p->next;
	else
		pp->next = p->next;
	if (i == list->last)
		list->last = pp;
	InstFree(i);
}

LabelType
BranchTarget(i)
	Instruction i;
{
	NODE *t;

	if (IsUBranch(i))
		return i->u.ub.target;
	else
	if (IsCBranch(i)) {
		t = i->u.ex.root;
		t = t->in.right;	/* target tree */
		assert(t->in.op == STLABEL);
		return t->tn.lval;
	} else {
		InternalFault("Bad instr for branch target, tag = %d\n",
					(int) i->tag);
	}
	/*NOTREACHED*/
}

/*	Debug aids
 */

void
PrintInstruction(p)
	Instruction p;
{
	int i;

	printf("@%x: next: %x ", (unsigned int) p, (unsigned int) (p->next));
	switch (p->tag) {
	case FreeInst:
		printf("Free");
		break;

	case FcnStart:
		printf("Function Start: %s", p->u.fb.fname);
		PccPrintInst(p);
		break;
	
	case FcnEnd:
		printf("Function End");
		break;

	case Passed:
		printf("Passed\n");
		printf("\t%s", p->u.tx.line);
		break;

	case Block_Start:
		printf("Block Start: ftnno %d autooff %d",
			p->u.bs.ftnno, p->u.bs.autooff);
#ifdef FORT
		printf(" argloc %d", p->u.bs.argloc);
#endif
		PccPrintInst(p);
		break;

	case Block_End:
		printf("Block End: " );
		PccPrintInst(p);
		break;

	case Label:
		printf("Label: %d", p->u.lb.lab_num);
		break;

	case UBranch:
		printf("U-Branch: %d", p->u.ub.target);
		break;

	case Return:
		printf("Return %d %d", p->u.rt.r_lab, p->u.rt.r_count);
		break;
	
#ifdef FORT
	case OnlyFileName:
		printf("File name: line %d file %s", p->u.ofn.lineno,
				p->u.ofn.filename);
		break;

	case EntryPoint:
		printf("Entry Point: name: %s pro_label %d\n", p->u.ent.ename,
				p->u.ent.pro_label);
		break;

	case CopyOut:
		printf("Copy Out Here");
		break;
#endif


	case Expr:
		printf("Expr: line %d file %s\n", p->u.ex.lineno, p->u.ex.filename);
		fwalk(p->u.ex.root, eprint, 2);
		break;

	case Switch:
		printf("Switch: %d cases, reg %d, default %d",
			p->u.sw.ncases, p->u.sw.swregister,
			p->u.sw.defaultlabel);
		for( i = 0; i < p->u.sw.ncases; i++ )
			printf("\n%d %d", p->u.sw.swtable[i].sval,
				p->u.sw.swtable[i].slab );
		break;

	case ParamFetch:
		printf("Parameter Fetch: reg %d offset %d type ",
			p->u.pf.reg, p->u.pf.offset);
		tprint(p->u.pf.type);
		break;

	case ParamHere:
		printf("Parameter Fetches Here");
		PccPrintInst(p);
		break;

	case ParamSave:
		printf("Parameter Save: %d bytes", p->u.ps.npassed);
		break;

#ifdef FORT
	case ArithIf:
		printf("Arithmetic If: lab1 %d lab2 %d lab3 %d",
			p->u.aif.labs[0], p->u.aif.labs[1], p->u.aif.labs[2]);
		break;
#endif

#ifdef FORT
	case f77_Special:
		printf("f77 Special: ");
		PccPrintInst(p);
		break;
#endif
	}
	printf("\n");
}

void
PrintIList(l)
	InstrList l;
{
	Instruction i;

	for (i=l.first; i != NULL; i=i->next)
		PrintInstruction(i);
}
