/* idiom.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */

/*
 *	Use side-effect operators where feasible.
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: idiom.c,v 5.5 89/05/12 12:50:52 pcc Rel-3_0 $";
/* static char ID[] = "@(#)idiom.c	15.3 of 86/10/23"; */
#endif

#include <activity.h>
#include <idiom.h>
#include <dag.h>
#include <bool.h>
#include <assert.h>
#include <cmanifest.h>
#include <blocks.h>
#include <temp.h>
#include <identifier.h>
#include <ops.h>
#include <prune.h>

/*
 * Export
 */

int idebug = 0;
int NoIdioms = 0;

/*
 * Private
 */

static Boolean IdiomNode();
static Boolean TrySideEffectOp();
static Boolean TrySimpleASGOP();
static void ChangeDangling();
static void FinishSimple();
static void ChangeRight();
static Boolean PostWorthwhile();
static void FinishPost();
static void StartPost();

void
FindIdioms()
{
	BasicBlock b;

	if( NoIdioms )
		return;

	for( b = FirstBlock; b != NULL; b = b->next )
	    if( b->reachable )
	    {
		/* ActivityDag returns True if it needs a restart */

		while(ActivityDag(b->Dag, IdiomNode))/*NOTHING*/;
	    }

}

/*
 *	Called by ActivityDag to do idiom processing for each node.
 */

static Boolean
IdiomNode(d)
	DAG_Node d;
{
	UpdateActivity(d,(void(*)())NULL);

	if( d->delayed != NULL ) {
		assert(d->delayed->activity > 0);
		--d->delayed->activity;
	}

	/*
	 * Look for possible ASGOPS
	 */

	if( d->op == ASSIGN )
	{
		if( OpAssignable(d->u.in.right->op) )
			return TrySideEffectOp(d);

	}
	else
	if( OpAssignable(d->op) || 
	    (d->op == OCONVTREE && OpAssignable(d->u.in.left->op)) )
		return TrySimpleASGOP(d);

	return False;
}

/*
 * "d" is an assignment with a right subtree whose root is a potential
 * assignment operator.  See if it really can be made into an assignment
 * operator, and if it can, do it.
 */
static Boolean
TrySideEffectOp(d)
	DAG_Node d;
{
	DAG_Node aop;		/* op on rhs of ASSIGN */
	DAG_Node lhs;		/* lhs of ASSIGN */
	DAG_Node opleft;	/* left subtree of op on rhs */
	DAG_Node opright;	/* right subtree of op on rhs */

	Boolean dIsRef;		/* is the assign op referenced anywhere? */
	Boolean opIsRef;	/* was the op on the rhs referenced? */
	Boolean opleftIsRef;	/* was the left subtree of the rhs referenced?*/

	AttachedID aid;
	
	aop = d->u.in.right;
	lhs = d->u.in.left;
	opleft = aop->u.in.left;
	opright = aop->u.in.right;

	if( lhs->op != UNARY MUL || opleft->op != UNARY MUL ||
			lhs->u.in.left != opleft->u.in.left ||
			lhs->in_degree > 1 || WillHaveCarrier(lhs) )
/**/		return False;

	dIsRef = d->in_degree > 0 || WillHaveCarrier(d);
	opIsRef = aop->in_degree > 1 || WillHaveCarrier(aop);
	opleftIsRef = opleft->in_degree > 1 || WillHaveCarrier(opleft);

	if( idebug )
	{
		printf("TryASGOP d %x(%s) aop %x(%s) lhs %x(%s)\n",
			d, opst[d->op], aop, opst[aop->op],
			lhs, opst[lhs->op]);
		printf("	opleft %x(%s) opright %x(%s)\n",
			 opleft, opst[opleft->op],
			 opright, opst[opright->op]);
		if( idebug > 1 )
			printf("\td ref %d op ref %d opleft ref %d\n",
				dIsRef, opIsRef, opleftIsRef);
	}

	if( !dIsRef && !opIsRef && PostWorthwhile(d,aop))
	{
		/* Left subtree of plus or minus is referenced later on in the
		 * DAG, and nothing else is referenced.  Do this as post
		 * increment/decrement.
		 */

		if( idebug )
			printf("\tDo as INCR/DECR\n");

		d->op = (aop->op == PLUS ) ? INCR : DECR;

		StartPost(d, opleft);
		ChangeDangling(d, opleft);
		ChangeRight(d);
		FinishPost(d, d);
		return True;

	}
	else
	if( opIsRef )
	{
		/* If the op on the rhs is referenced elsewhere or stored away,
		 * let generated code use this value
		 * NOTE: The decision not to use ASGOP if in_degree>1 and
		 * attached == NULL should be based on cost
		 * NOTE: It is possible to re-write the DAG to use an ASGOP
		 * here.  Doing so would add more complexity to this code,
		 * so, for the moment, we don't.
		 */

		if( idebug )
			printf("op referenced, so abandon\n");
		return False;
	}
	else
	if( opleftIsRef )
	{
		/* If fetch of value is referenced elsewhere, give up.
		 * The decision to give up if no attached and
		 * in_degree > 1 should be based on cost
		 */

		if( idebug )
			printf("not INCR/DECR, opleft ref, so abandon\n");
		return False;
	}
	else
	{
		if( idebug )
			printf("\tDo as ASGOP\n");

		d->op = ASG aop->op;
		ChangeRight(d);

		/* In the best of all possible worlds, this tree would
		 * never appear more than once in the output.
		 * 
		 *	assert( d->in_degree <= 1 || d->attached != NULL );
		 *
		 * In practice, there is a bug which means that it might.
		 * If we find one here, ensure that it doesn't.
		 */

		if( d->in_degree > 1 && d->attached == NULL )
		{
			aid = MakeTemp(d, 'e', d->type);
			d->carrier = aid->id;
		}

		/* Finally, check that the lhs of the op is no longer
		 * active.  The "sticky" algorithm should have seen
		 * to it that enough temps were attached that this
		 * could never happen.
		 */

		assert(opleft->op == LEAFNOP || opleft->activity == 0);

		return True;
	}

	/*NOTREACHED*/
}

/*
 * change d to have the same right subtree as its right child.
 */
static void
ChangeRight(d)
	DAG_Node d;
{
	DAG_Node aop;
	DAG_Node opright;

	aop = d->u.in.right;
	opright = aop->u.in.right;

	d->u.in.right = opright;
	aop->in_degree--;
	aop->activity--;	/* as a defensive measure, keep the activity
				 * count on aop right */
	opright->in_degree++;

	PruneTree(aop, NotConditional, RecursivePrune);		/* Try to make aop vanish */
}

/*
 * Find all dangling references to opleft and change them to
 * references to d.  If this kills all references to opleft, make
 * it go away.
 */
static void
ChangeDangling(d, opleft)
	DAG_Node d;
	DAG_Node opleft;
{
	DAG_Node tmp;
	int ty;

	for( tmp = d->next; tmp != NULL; tmp = tmp->next )
	if( tmp->in_cond == NotConditional )
	{
		ty = optype(tmp->op);
		if( ty != LTYPE )
		{
			if(tmp->u.in.left == opleft && !tmp->evaluated)
			{
				tmp->u.in.left = d;
				opleft->in_degree--;
				opleft->activity--;
				d->in_degree++;
				if( idebug )
				{
					printf("Change left subtree of %x from %x to %x\n",
						tmp, opleft, d);
				}
			}
			if(ty == BITYPE &&
			   tmp->u.in.right == opleft && !tmp->evaluated)
			{
				tmp->u.in.right = d;
				opleft->in_degree--;
				opleft->activity--;
				d->in_degree++;
				if( idebug )
				{
					printf("Change right subtree of %x from %x to %x\n",
						tmp, opleft, d);
				}
			}
		}
	}

}

static Boolean
TrySimpleASGOP(d)
	DAG_Node d;
{
	DAG_Node aop;
	DAG_Node opleft;
	DAG_Node killed;
	AttachedID aid;

	aop = d;
	opleft = d->u.in.left;
	if( d->op == OCONVTREE )
	{
		aop = opleft;
		if( aop->in_degree != 1 || WillHaveCarrier(aop) )
/**/			return False;

		opleft = aop->u.in.left;
	}

	if( optype(opleft->op) != LTYPE )
/**/		return False;

	aid = FindAttachedMember(d->attached, opleft->leaf_id);

	/* the check of prev_attach here excludes the following abomination:
	 *	k = i;
	 *	i = a[j] + 2;
	 *	t = i * 7;
	 *	foo(t);
	 *	i = k + 1;
	 * Here, the left subtree of the + is a leaf named "i"; however
	 * the "i" attached to the plus was last attached to nothing, as
	 * it was killed by the foo(t) call.
	 */
	if( aid == NULL || aid->prev_attach != opleft )
/**/		return False;

	/* The changes that we make in the DAG for INCR/DECR may cause the
	 * code for this node to be attached to a tree generated much
	 * later in the DAG.  We don't want to do this if there is any
	 * chance that the identifier will be stored into later on; hence
	 * we check that killed_by is NULL.
	 * We also check neither the assignment nor the attached identifier
	 * have been marked sticky.  If they have, we dare not use
	 * INCR/DECR, as these might wind up embedded in the wrong place.
	 */
	killed = aid->killed_by;
	if ( d->in_degree == 0 && killed == NULL && !aid->sticky &&
			!opleft->sticky && !d->sticky && d->delay_count == 0 &&
			d->attached->next == NULL && PostWorthwhile(d,aop) )
	{
		if( idebug )
		{
			printf("INCR/DECR Idiom: root %x(%s) aop %x(%s) opleft %x(%s)",
				d, opst[d->op],
				aop, opst[aop->op],
				opleft, opst[opleft->op]);
		}

		aop->op = (aop->op == PLUS) ? INCR : DECR;

		if( idebug )
		{
			printf(" new op %s\n", opst[aop->op]);
		}

		/* Re-arrange the DAG to reflect the new op.  The order of
		 * these operations must be preserved.
		 * 1) delete the attached identifier from d.  The third
		 *    argument should be irrelevant, since we checked for
		 *    stickiness above.
		 * 2) move the attached identifiers from the left subtree to
		 *    d, and attach a temp to the left subtree.
		 * 3) change any outstanding references to the left subtree to
		 *    point to d
		 */
		DeleteID(opleft->leaf_id, d, True);
		d->carrier = NoId;
		assert(d->attached == NULL);
		StartPost(aop, opleft);
		ChangeDangling(d, opleft);

		FinishPost(d, aop);
		FinishSimple(d, aop, killed);

		return True;
	}
	else
	if( opleft->activity == 0 || opleft->carrier != opleft->leaf_id )
	{
		if( idebug )
		{
			printf("ASGOP Idiom: root %x(%s) aop %x(%s) opleft %x(%s)",
				d, opst[d->op],
				aop, opst[aop->op],
				opleft, opst[opleft->op]);
			printf(" new op %s\n", opst[ASG aop->op]);
		}

		/* Delete the original ID from the new ASGOP.  At the moment
		 * we can do this without regard for stickiness, since the
		 * ASGOP will never be embedded.  If FinishSimple is ever
		 * allowed to embed, it must be told about sticky attached
		 * ID's
		 */
		DeleteID(opleft->leaf_id, d, True);

		aop->op = ASG aop->op;
		FinishSimple(d, aop, killed);

		return True;
	}
	return False;
}

/* Is it worthwhile to change aop into a post increment?
 * Main idea here is to do INCR/DECR stuff only if ALL attached identifiers
 * are still valid carriers and if nothing else uses the
 */
static Boolean
PostWorthwhile(d, aop)
	DAG_Node d;
	DAG_Node aop;
{
	DAG_Node opleft;
	DAG_Node opright;
	AttachedID aid;

	if(aop->op != PLUS && aop->op != MINUS )
/**/		return False;

	if( d->in_cond != NotConditional  ||  aop->in_cond != NotConditional )
/**/		return False;	/* FOR NOW, refuse to touch
				 * conditionals */

	opright = aop->u.in.right;
	opleft = aop->u.in.left;

	if( opright->op == ICON )
	{
		if( opleft->type == FLOAT || opleft->type == DOUBLE )
/**/			return False;
	}
	else
	if( opright->op == FCON )
	{
		 if( opleft->type != FLOAT && opleft->type != DOUBLE)
/**/			return False;
	}
	else
/**/		return False;

	/* If there are any delayed stores here, forget it
	 */

	if( opleft->delay_count != 0 )
	{
		if( idebug > 1 )
			printf("No INCR/DECR because of delayed stores (left op)\n");
/**/		return False;
	}

	aid = opleft->attached;
	for( ; aid != NULL; aid=aid->next)
	{
		/*
		 * Does this ID get killed before the proposed ASGOP?
		 */
		if( aid->killed_by != NULL &&
				aid->killed_by->order < d->order )
		{
			if( idebug > 1 )
				printf("No INCR/DECR because id %d killed at %d(%x)\n",
					aid->id, aid->killed_by->order, aid->killed_by);
/**/			return False;
		}
	}
	/* At this point, we can safely say that there are no delayed stores
	 * and that we will not create any.
	 * However, it is possible that opleft is used  between
	 * opleft and d.  In this case, doing INCR/DECR will not do anything
	 * for these references, so use ASGOP instead.
	 * Note that since there were no delayed stores, and everything is
	 * valid as a carrier, we can pick the entire attached identifier
	 * list up and move it.
	 */

	return opleft->activity == opleft->in_degree - 1   &&
		(opleft->activity > 0  ||  opleft->attached != NULL );
}

/*
 * Start of handling for INCR/DECR.  Main idea here is to transfer the
 * attached ID list from the LHS to the INCR/DECR
 * NOTE: THIS DESTROYS UD CHAINS!!!!!
 */

static void
StartPost(aop, opleft)
	DAG_Node aop;		/* Node that will eventually be the INCR/DECR */
	DAG_Node opleft;	/* Node that will be the LHS of same */
{
	AttachedID aid;

	if( opleft->attached != NULL )
	{
		aop->attached = opleft->attached;
		aop->carrier = opleft->carrier;
		aop->carrier_required = opleft->carrier_required;
		opleft->attached = NULL;
		if( optype(opleft->op) == LTYPE )
			opleft->carrier = opleft->leaf_id;
		else {
			opleft->carrier = NoId;
			opleft->carrier_required = False;
		}
	}
}

/*
 * Clean up post increment DAG.  The main idea here is add and delete temps
 * as necessary
 */
static void
FinishPost(d, aop)
	DAG_Node d;		/* The root node */
	DAG_Node aop;		/* The INCR/DECR (may not be == d for
				 * simple assigns */
{
	DAG_Node opleft;	/* The lhs of the INCR/DECR */
	AttachedID aid;

	opleft = aop->u.in.left;
	if( opleft->activity != 0 )
	{
		/* We have not moved all outstanding left subtree
		 * references.  Thus we need to attach a temp to the left
		 * subtree.  Note that this should only happen in the
		 * simple case.
		 */
		assert(optype(opleft->op) == LTYPE);
		(void) MakeTemp(opleft, 'h', opleft->type);
	}

	/* If the in_degree of this node is greater than
	 * one, we have to keep the temp attached to ensure
	 * that the INCR/DECR is not done twice.
	 * If the in_degree of this node is exactly one or zero,
	 * we can dispense with the temp.  (PostWorthwhile()
	 * has seen to it that if there is a temp attached,
	 * it is alone - but we check anyway).
	 * NOTE: if aop != d, the attached id will be on aop, but the
	 * indegree test should still be done on d, since aop must have
	 * indegree of exactly 1 (this is checked in TrySimpleASGOP())
	 */
	aid = aop->attached;
	if( aid != NULL &&
	    IsTemp(aid->id) &&
	    (d->in_degree == 0 ||
		(d->in_degree == 1 && !aop->sticky && aid->next == NULL )) )
	{
		if( aid->id == aop->carrier ) {
			aop->carrier = NoId;
			aop->carrier_required = False;
		}
		DeleteID(aid->id, aop, True);
		if( d->in_degree == 0 && aop->attached == NULL )
		{
			/* If we get here, we were unable to move
			 * any outstanding refs to the lhs (or there
			 * were none).  In this case we may as well
			 * convert the INCR/DECR to ASG PLUS/MINUS
			 */
			aop->op = (aop->op == INCR) ? ASG PLUS
						    : ASG MINUS;
			if( idebug )
			{
				printf(" new op changed to %s\n",
					opst[aop->op]);
			}
		}
	}
}

/* Finish off the work of converting a simple assignment and arithmetic op
 * to an ASGOP or an INCR/DECR.  If d and aop are not equal, an implied
 * conversion has been added.  If this conversion has in_degree 0 and
 * nothing attached, get rid of it.  Also, need to apply the correct type
 * to the ASGOP
 */
static void
FinishSimple(d, aop, killed)
	DAG_Node d;
	DAG_Node aop;
	DAG_Node killed;
{
	if( d != aop )
	{
		d->op = UNARYNOP;	/* conversion no longer needed */
		d->carrier_required = False;
		aop->type = d->type;
		PruneTree(d, NotConditional, RecursivePrune);	/* try to make d go away */
	}

	/* At this point, it would be nice if we could allow ASGOPs with
	 * indegree 1 to be embedded.  Hovever, this doesn't work. Consider
	 * the following (extracted from V7 ed):
	 *	*lp++ = c;
	 *	if( c == '\n' || lp > (linp + 72 ) )
	 * The code above decides to use a pre-increment, and then the code
	 * below cleverly embeds the pre-increment in the right hand side of
	 * the conditional.
	 *
	 * FOR NOW, then, we disable this check:

	 * See if we can allow this to be embedded

	 * if( d->in_degree == 1 && d->attached == NULL && d->delay_count == 0
	 * 	&& killed == NULL )
	 * {
	 * 	d->carrier = NoId;
	 *	!! Be sure to clear carrier_required if that is
	 *	!! appropriate here.
	 * 	d->activity--;
	 * 	IncrementActivity(d);
	 * 	if( idebug )
	 * 		printf("Embedding %x\n", d);
	 * }
	 * LATER NOTE:  Before this code is re-activated, will also need to
	 *	know that the original attached ID was not sticky
	 */
}
