/* dagtree.c */

/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */

/*
 *	dagtree - convert DAGs back into trees
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: dagtree.c,v 5.5 89/05/12 12:50:13 pcc Rel-3_0 $";
/* static char ID[] = "@(#)dagtree.c	15.3	of 86/10/20"; */
#endif

/*
 *	This module is responsible for generating trees that perform
 *	the same computation as a DAG.  These trees are ultimately
 *	written on the intermediate file as the new program.
 */

/*
 *	Import
 */

#include <activity.h>
#include <assert.h>
#include <bool.h>
#include <dagtree.h>
#include <dag.h>
#include <tree.h>
#include <blocks.h>
#include <instruct.h>
#include <ops.h>
#include <identifier.h>
#include <dagsymbol.h>
#include <cost.h>
#include <idiom.h>
#include <storage.h>
#include <target.h>
#include <erroro.h>
#include <config.h>
#ifdef FILLSPILL
#include <option.h>
#endif /* FILLSPILL */

/*
 *	Globals
 */

int rdebug = 0;		/* Rewrite trees */

static InstrList delay_list;		/* tunnel for UpdateActivity */
static InstrList *GenTreeList;		/* tunnel for ActivityDag */

/*
 *	Forward
 */

static void GenCode();
static void BuildTree();
static TreeNode GenStore();
static void GenTrees();
static void GenDelay();
static void DoDelay();
static TreeNode GenSideEffects();
static TreeNode GenComma();
static void CleanUnreachableDAG();

/*
 * Local
 */

/* For nodes not in conditionals, we must generate code (and not embed) if
 *	a) there are attached identifiers
 *   or b) there is a carrier and this is not a leaf (idiom.c may have removed
 *	   the attached ID and left the carrier).  Values carried by loop
 *	   temps also show up as nodes with carriers and no attached ID's.
 *	   For these, we do NOT want to generate code.
 *
 *	NB: THIS SHOULD BE DONE USING A FLAG.  FIX ASAP.
 *
 * This macro is defined here to ensure that GenTrees() and TreeRef()
 * are consistent.
 */

#define MayNotEmbed(n)	((n)->attached != NULL || \
			((n)->carrier != NoId && optype((n)->op) != LTYPE && \
			 !(n)->prev_valid_carrier) )

void
RewriteDAGS()
{
	BasicBlock b;
	InstrList new;
	Instruction i, new_i, ni;

	for (b=FirstBlock; b != NULL; b = b->next)
	    if( b->reachable ) {

		/* Move all unrecognised instructions (except the last)
		   to the front
		 */

		InitList(&new);
		i = b->code.first;
		while (i != NULL) {
			ni = i->next;
			if (InstType(i) != Expr && 
			    InstType(i) != UBranch &&
 			    InstType(i) != Return){
				new_i = AddInstruction(InstType(i), &new);
				*(new_i) = *i;	/* Preserve the instruction */
				new_i->next = NULL; /* but NOT its successor */
				i->tag = FreeInst;
			}
			i = ni;
		}

		/* Build the code for the block */

		if (rdebug)
			printf("Rebuilding Block # %d\n", b->blocknum);

		GenTrees(b->Dag, &new);

		i = b->code.last;
		if( i != NULL && (IsUBranch(i) || IsReturn(i)))
		{			/* it isn't in the DAG yet */
			new_i = AddInstruction(InstType(b->code.last), &new);
			*(new_i) = *i;
			/* new_i->next == NULL since i->next == NULL */
			i->tag = FreeInst;
		}

		FreeIList(&(b->code));
		b->code = new;
	    }
	    else
	    {
		CleanUnreachableDAG(b->Dag);
	    }
}

/*	Temporary fix until we do this right.  In unreachable blocks, we must
 *	clean up the tree nodes that we keep to describe STASGs and the like.
 *	Eventually this will be fixed right by storing the nodes in e_tree.
 */

static void
CleanUnreachableDAG(n)
	DAG_Node n;
{
	for(; n != NULL; n = n->next )
	{
		if( n->new_tree != NULL )
		{
			TreeFree(n->new_tree);
			n->new_tree = NULL;
		}
	}
}

static Boolean
GenNode(n)
	DAG_Node n;
{
	Operator op;
	Instruction i;
# ifdef FILLSPILL
	TreeNode store;
	TreeNode load;
# endif /* FILLSPILL */

	op = n->op;
	BuildTree(n);

	if (rdebug) {
		printf("Rebuilt tree for # %d\n", n->order);
		fwalk(n->new_tree, eprint, 0);
	}

	if ( MayNotEmbed(n) || (n->in_degree == 0)) {

		/* Generate code */

		if (op != LEAFNOP) {	/* Really generate code */
			if (n->in_cond == NotConditional) {
# ifdef FILLSPILL
				store = NULL;
				load = NULL;
				if( !NoFillSpill && callop(n->new_tree->in.op) )
					NewCall( n, &store, &load );
				if( store != NULL ) {
					if( fdebug ) {
						printf( "Store Regs\n" );
						DBprint( store );
						printf( "Function CALL\n" );
						DBprint( n->new_tree );
					}
					i = AddInstruction( Expr, GenTreeList );
					i->u.ex.lineno = n->line_number;
					i->u.ex.filename = n->file_name;
					i->u.ex.root = store;
				}
# endif /* FILLSPILL */
				i = AddInstruction(Expr, GenTreeList);
				GenCode(i, n->new_tree, n);
# ifdef FILLSPILL
				if( load != NULL ) {
					if( fdebug ) {
						printf( "Load Regs\n" );
						DBprint( load );
					}
					i = AddInstruction( Expr, GenTreeList );
					i->u.ex.lineno = n->line_number;
					i->u.ex.filename = n->file_name;
					i->u.ex.root = load;
				}
# endif /* FILLSPILL */
			}
			/* Side effects in conditionals are being
			 * collected by ActivityDag and will be
			 * attached to their controlling node.
			 */
		}

		/* Output accumulated delayed stores, if any */

		if (delay_list.first != NULL) {
			if (rdebug) {
				printf("Output Delay List at # %d\n",
					n->order);
				PrintIList(delay_list);
			}
			ConcatInstructions(GenTreeList, &delay_list);
		}

		/* If this node has special delay, do it now. */

		if (n->special_delay) {
			if (rdebug)
				printf("Special delayed store @ # %d\n",
					n->order);
			assert(n->delayed != NULL);
			GenDelay(GenTreeList, n);
		}
	}

	return False;
}

static void
GenTrees(g, list)	/* Build a new list for g in list */
	DAG_Node g;
	InstrList *list;
{

	InitList(&delay_list);

	GenTreeList = list;		/* Set Implicit parameter */

	(void) ActivityDag(g, GenNode);

	assert(delay_list.first == NULL);	/* lost a delayed store */

}

static void
GenCode(i, t, n)		/* Generate code for t at n into i */
	Instruction i;
	TreeNode t;
	DAG_Node n;
{
	i->u.ex.lineno = n->line_number;
	i->u.ex.filename = n->file_name;
	i->u.ex.root = CopyTree(t);
}

static void
TreeSkeleton(t, op, type)
	TreeNode t;
	Operator op;
	TWORD type;
{
	t->in.op = op;
	t->in.type = type;
	t->in.identifier = NULL;
}

static void
BuildTree(n)		/* Build tree for node n */
	DAG_Node n;
{
	Operator op;
	TreeNode t;
	int kind;
	Identifier excluded;		/* id in list not to be assigned to */
	AttachedID aid;

	op = n->op;
	kind = optype(op);

	/* Build Basic Tree */

	t = SpecialTree(n);
	if (t == NULL) { /* Not Special */
		t = TreeAllocate();
		if (op == STASG || op == STARG || op == STCALL ||
		    op == UNARY STCALL )
		{
			*t = *(n->new_tree);	/* Special Case */
		} else
		if (kind == LTYPE  && op != FCON && op != LEAFNOP &&
		    op != STLABEL)
		{
			assert(HasAllocation(n->leaf_id));
			*t = *(IdTree(n->leaf_id));
			t->tn.type = n->type;	/* type from DAG not symtab */
		} else {

			/* Build it from scratch */

			TreeSkeleton(t, op, n->type);
			if (op == FCON)
				t->fpn.dval = n->u.fpn.dval;
			else
			if( op == STLABEL )
			{
				t->tn.lval = n->u.tn.lval;
				t->tn.rval = n->u.tn.rval;
			}
		}

		if (kind != LTYPE)
		{
			t->in.left = TreeRef(n->u.in.left);
			if (kind == BITYPE)
				t->in.right = TreeRef(n->u.in.right);
			else
				t->tn.rval = n->u.tn.rval;
		}
	}

	/*
	 *	t contains a tree that will evaluate the node.
	 *	Usually, this is also the source of the value to
	 *	be assigned.  But, if the value is already available
	 *	in a previously valid carrier, we'll use that.
	 */

	n->e_tree = t;		/* the evaluation tree */

	if (n->prev_valid_carrier) {
		/*
		 * Get the value from the (already set) carrier for
		 * the assignment.
		 */
		assert(n->carrier != NoId);
		t = IdTree(n->carrier);
	}

	excluded = NoId;		/* Nothing excluded, yet */

	/* If this op is an asgop of something simple, make sure that we
	 * do not generate a redundant assignment.  Also need to be sure
	 * that the left subtree is the real variable and not some carrier.
	 */

	if( asgop(op) && optype(n->u.in.left->op) == LTYPE )
	{
		assert(!n->prev_valid_carrier); /* we've a mess now! */
		excluded = n->u.in.left->leaf_id;
		t->in.left = IdTree(excluded);
	}

	UpdateActivity(n, DoDelay);

	/*
 	 *	If this node is a conditional operator, obtain the side
	 *	effects from the list generated by ActivityDag. We do
	 *	it here rather than before we generate the ref to the RHS
	 *	because it makes the implementation slightly easier.
	 *	These side effects will occur in the correct place.
	 *
	 *	Note: We need to think about the fact that this is
	 *	is going into new_tree but not into e_tree.  This
	 *	may change when we embed side effects.
	 */

	if (n->op == COLON) {		/* two sets of side effects */
		TreeNode lst, rst;
		DagNodeList lln, rln;

		lln = n->chain.first_cond;
		assert(lln != NULL && lln->n == NULL);	/* left chain */

		rln = lln->next;
		assert(rln != NULL);			/* right chain */
		while (rln->n != NULL) rln = rln->next; /* find start */
		assert(rln->n == NULL);

		lst = GenSideEffects(lln->next, n->u.in.left, t->in.left);
		rst = GenSideEffects(rln->next, n->u.in.right, t->in.right);

		if (lst != NULL)
			t->in.left = lst;
		if (rst != NULL)
			t->in.right = rst;
	} else
	if (n->op == ANDAND || n->op == OROR) {
		TreeNode st;		/* side effect tree */
		DagNodeList ln;

		ln = n->chain.first_cond;
		assert(ln != NULL && ln->n == NULL);	/* starting marker */
		st = GenSideEffects(ln->next, n->u.in.right, t->in.right);

		if (st != NULL)			/* there are side effects */
			t->in.right = st;
	}

	/* Assign the value to the attached identifiers */

	for (aid = n->attached; aid != NULL; aid = aid->next) {

		if (! SameId(aid->id, excluded)) {
			t = GenStore(aid->id, t);
			++(n->count);
		}
	} /* for ... */

	if (n->new_tree != NULL)
		TreeFree(n->new_tree);
		
	n->new_tree = t;
}

static TreeNode
GenStore(id, t)
	Identifier id;			/* identifier */
	TreeNode t;		/* to be assigned */
{

	TreeNode right, left;

	right = t;
	left = IdTree(id);
	t = TreeAllocate();
	t->tn.op = ASSIGN;
	/* Fill in type info */
	t->in.identifier = NULL;	/* This IS required */
	t->in.type = left->in.type;
	t->in.left = left;
	t->in.right = right;
	return t;
}

/*
 *	This generates the list of side effects for a conditional
 *	operator.  It stops at end of list, or when a marker is
 *	encountered.
 */

static TreeNode
GenSideEffects(ln, child, child_tree)
	DagNodeList ln;
	DAG_Node child;		/* appropriate child node */
	TreeNode child_tree;	/* associated tree */
{
	TreeNode tl;
	DagNodeList pln;	/* previous ln */

	tl = NULL; pln = NULL;
	while (ln != NULL && ln->n != NULL) {
		if (tl == NULL)
			tl = ln->n->new_tree;		/* no , needed */
		else
			tl = GenComma(tl, ln->n->new_tree); /* tl, ln */
		pln = ln;
		ln = ln->next;
	}

	/*
 	 *	Now see if the last side effect also generates the value
	 *	of the child node.  If it does, we don't need to add
	 *	a reference to the child.  If not, we do.
 	 */

	if (tl != NULL) {		/* we've found side effects */
		assert(pln != NULL && pln->n != NULL);
		if (pln->n != child) {
			tl = GenComma(tl, child_tree);
		}
	}
	return tl;
}

static TreeNode
GenComma(tl, tr)
	TreeNode tl, tr;
{
	TreeNode comma;

	comma = TreeAllocate();
	TreeSkeleton(comma, COMOP, tr->in.type);   /* use type of r.h.s */
	comma->in.left = tl;
	comma->in.right = tr;
	return comma;
}

TreeNode
TreeRef(n)				/* Reference node n for a tree */

	DAG_Node n;
{
	TreeNode t;

	assert(n->new_tree != NULL);

	/* The following decision is machine dependent */

	if (Literal(n->op))	/* Use the literal value */
		if( n->op == STLABEL ) {
			t = n->new_tree;
			assert(t->tn.op == n->op);
		} else {
			if (n->carrier != NoId &&
			    IdRefCost(n->carrier) < n->cost)
			{	/* Use the carrier */

				t = IdTree(n->carrier);
			} else {
				/* Search the assignment tree for it */

				t = n->e_tree;
				assert(t->tn.op == n->op);
			}
		}
	else
	if ( n->carrier != NoId ) /* if there is a carrier, use it! */
		t = IdTree(n->carrier);
	else
	if ( !MayNotEmbed(n) ) /* anonymous (embedded) subexpression */
		t = n->new_tree;
	else
	{

		/* The node has no carrier. */

		t = n->e_tree;
	}

	return t;
}


/*
 *	Handle Delayed Store Operations
 */

static void
DoDelay(n)				/* When activity == 0 */
	DAG_Node n;
{
	assert(n->activity == 0);
	if (n->delayed != NULL && !(n->special_delay)) {/* ordinary delay */
		if (rdebug) {
			printf("Ordinary delayed store generated @ # %d\n",
				n->order);
		}
		assert(optype(n->op) == LTYPE);
		GenDelay(&delay_list, n);
	}
		
}

static void
GenDelay(list, n)			/* Generate a delay */
	InstrList *list;
	DAG_Node n;
{
	TreeNode t;
	Instruction i;

	t = GenStore(n->leaf_id, TreeRef(n->delayed));

	i = AddInstruction(Expr, list);
	GenCode(i, t, n);
	TreeFree(t);

	if (rdebug) {
		printf("Delayed store generated:\n");
		PrintInstruction(i);
	}

	/* Decrement activity of node with value */

	assert(n->delayed->activity > 0);
	--(n->delayed->activity);
	if (n->delayed->activity == 0)
		DoDelay(n->delayed);
}
