/* dagsymbol.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */
/*
 *	Directed Acyclic Graph Symbol Table Manager
 *	See: Aho and Ullman, Principles of Compiler Design, Section 12.3
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: dagsymbol.c,v 5.5 89/05/12 12:50:07 pcc Rel-3_0 $";
/* static char ID[] = "@(#)dagsymbol.c	15.5	of 86/11/27"; */
#endif

/*
 *	Internally, the symbol table is a two-dimensional array of
 *	records.  Each row represents a symbol table space.  Currently
 *	there is a space for globals and one for everything else.  This
 *	means that re-hashing isn't necessary at the end of a function.
 *
 *	The outside world sees the symbol table as a linear array
 *	of identifiers.  An "Identifier" is a unique integer that
 *	names the NAME, PNAME, etc.  It is dense, so is suitable
 *	for use as an array index or set element.
 *
 *	The mapping from Identifier values to symbol table entries
 *	is done via an expandable array.
 */


/*	Import
 */

# include <string.h>
# include <assert.h>
# include <option.h>
# include <identifier.h>
# include <dag.h>
# include <dagsymbol.h>
# include <tree.h>
# include <bool.h>
# include <longset.h>
# include <storage.h>
# include <pcc.h>
# include <flow.h>
# include <erroro.h>

/*
 *	Forward
 */

static void EnterRegisterDefinitions();
static TempDefList CreateTempDef();
static void FreeTempDef();

/*
 *	Internal
 */

/*
 *	Symbol Table Spaces
 *	Note: you can't just change NumTables, the code in InitTable
 *	and ResetTable would have to be checked.
 */

#define NumTables	2		/* Number of separate symbol tables */
#define LocalSpace	0		/* cleared at fcn start */
#define GlobalSpace	1		/* survives function */

#define space_map(o)	((o) == NAME ? 1 : 0)

typedef int TableIndex;		/* Index within table */

/*
 *	The structure of the Symbol Table
 */

/*
 *	Keys
 *
 *	For each kind of leaf (expect ICON), pass1 provides a
 *	unique id "cookie."  For REGs, the cookie is the register
 *	number and is stored in lval.  For LNAME, PNAME, NAME, etc.,
 *	the cookie as a small integer and is stored in rval.
 *
 *	For ICONs, the cookie is the ICON's value, and is stored
 *	in lval.  Since the entire value is needed, the key value
 *	field must be a CONSZ.
 */

typedef struct KY {
	Operator op;		/* What operator is used to describe it? */
	CONSZ cookie;		/* It's unique (within space) id cookie */
	TWORD type;		/* type of the associated id */
} KeyType;

#define SameKey(k1, k2)		((k1).op == (k2).op \
				&& (k1).cookie == (k2).cookie \
				&& (k1.type == k2.type || IsANYTYPE(k1.type) || IsANYTYPE(k2.type)))

/*
 *	Symbol Table Data
 */

typedef struct SE {
	Identifier id;		/* Internal opt name.  0 => not in use */

	KeyType key;		/* key value for this table entry */
	char * id_text;		/* text of associated identifier */

	DAG_Node n;		/* Current Value */
	DAG_Node leaf_ref;	/* Most recent appearance as a leaf */
	DAG_Node last_killed;	/* Most recent node that killed this id */
	TempDefList tmp_defs;	/* For PCO objects, list of definitions */
	TreeNode t;		/* Prototypical tree */

	Boolean addressed;	/* Address has been taken (locals only) */
	Boolean use_before_set;	/* This var may be used before being set */
	Boolean transparent;	/* Is this a transparent temp? */
	Boolean allocated;	/* Has our allocator allocated this? */
	Boolean may_not_promote;/* For some reason, this object may not be
				   re-allocated (e.g., to a register */
	int benefit;		/* Gain by putting it in register */

	LongSet defs;		/* set of definitions of this var */
	LongSet uses;		/* set of uses of this var */

	LongSet NeedsSpace;	/* Set of blocks in which this needs space.
				 * If NULL, indicates this id is not interesting.
				 */
	int FirstNeed;		/* First element in NeedsSpace.
				 * NB: must be int (not FLowIndex) for NoElement.
				 */
# ifdef FORT
	Identifier shadow_id;	/* statname id for prime associated with this
				 * entry / entry associated with this prime
				 */
	TWORD actual_type;	/* Real type of the id (ARRY, not PTR) (f77) */
	Boolean	copy_eligible;	/* True if this id can be copied in/out (f77) */
	CONSZ common;		/* common block number if it exists. */
# endif /* FORT */
} SymbolEntry, *SymbolPtr;

/*
 *	The following declarations provide a dynamically allocated
 *	version of:
 *
 *	static SymbolEntry SymbolTable[NumTables][SymbolTableSize];
 *
 *	Dynamic allocation is used to reduce the size of PCO's object
 *	code.
 */

typedef SymbolEntry SymTable[SymbolTableSize];   /* one column of SymbolTable */
static SymTable *SymbolTable = NULL;	/* ref'd as SymbolTable[ns][id] */

static int NumEntries[NumTables];

/*
 *	The following variables are used to make things work before
 *	the symbol table is set up in advance.  They are needed to
 *	allow certain kinds of references to identifiers never
 *	before seen: LastKilled and NodeID
 */


/*
 *	Visible
 */

Identifier MaxIdentifier = FirstId - 1;	/* Largest valid Identifier value */
#ifdef	INLINER
Identifier LastRegId;
#endif	/* INLINER */

/*
 *	Forward Declarations
 */

/*
 *	Other Global Data
 */

#ifdef IdDebug
char * JunkId = "!Junk";
#endif

static TempDefList FreeTempDefs = NULL;		/* available TempDef records */

/*
 *	Implementation
 */

/*
 *	Expandable Array Implementation
 *	Note: This is similar to labels.c.  We could build a generic
 *	ExpandableArray package using m4, or with a little hacking,
 *	C macros.  I'm not sure the benefits are worth it.
 */

#define MinSize	1000
#define	SizeInc	1000

static SymbolPtr* IDs;		/* Expandable array of SymbolPtrs */

static int current_size = 0;

static void
Expand()
{
	int old_size, l;

	old_size = current_size;

	if (current_size == 0) {
		IDs = GetArray(s_Symbol, SizeInc, SymbolPtr);
		CheckStorage(IDs, "storage for %d symbol table entries", SizeInc);
	} else {
		IDs = Reallocate(s_Symbol, IDs, current_size,
					current_size + SizeInc, SymbolPtr);
		CheckStorage(IDs, "storage for %d symbol table entries", current_size + SizeInc);
	}
	current_size += SizeInc;
	for (l = old_size; l < current_size; ++l)
		IDs[l] = NULL;
}

static void
CheckRef(i)		/* Makes a[i] in-range */
	int i;
{
	while (i >= current_size) Expand();
}

/*
 *	End of Expandable Array Implementation
 */

static KeyType
MakeKey(op, cookie, ty)		/* Make a key from (op, cookie) pair */
	Operator op;
	CONSZ cookie;
	TWORD ty;
{
	KeyType k;

	k.op = op;
	k.cookie = cookie;
	k.type = ty;
	return k;
}

static SymbolPtr
TableLookUp(k)
	KeyType k;
{
	TableIndex ti;
	int s;		/* Which table space ? */

	ti = k.cookie % SymbolTableSize;
	if (ti < 0) ti += SymbolTableSize;
	s = space_map(k.op);
	while (SymbolTable[s][ti].id != NoId && !SameKey(k, SymbolTable[s][ti].key))
	{
		++ti;
		if (ti >= SymbolTableSize) ti = 0;
	}
	return &(SymbolTable[s][ti]);
}

static SymbolPtr
TableEntry(id)
	Identifier id;
{
	assert(id < current_size && IDs[id] != NULL && IDs[id]->id != NoId);
	return IDs[id];
}

Operator
IdOp(id)			/* What kind of id ? */
	Identifier id;
{
	assert(IDs[id] != NULL);
	return IDs[id]->key.op;
}

Identifier
IdLookUp(op, cookie, ty)		/* look up op, cookie */
	Operator op;
	CONSZ cookie;
	TWORD ty;
{
	SymbolPtr s;

	s = TableLookUp( MakeKey(op, cookie, ty) );
/*	assert(s->id != NoId); */
	return s->id;
}

# ifdef FORT

Identifier
ShadowLookUp( id )		/* Look for shadow id associated with id */
	Identifier id;
{
	assert( IDs[id] != NULL );
	return( IDs[id]->shadow_id );
}

void
SetShadowId( id, s_id )			/* link id and shadow id */
	Identifier id, s_id;
{
	assert( IDs[id] != NULL );
	assert( IDs[s_id] != NULL );
	IDs[id]->shadow_id = s_id;
	IDs[s_id]->shadow_id = id;
}

# endif /* FORT */

/*
 * Enter a symbol declared by pass 1. Main idea is to call EnterSymbol() and
 * print debugging info.
 */

Identifier
Pass1Symbol( op, cookie, type, name )
	Operator op;
	CONSZ cookie;		/* pass 1 cookie */
	TWORD type;
	char *name;
{
	Identifier entered;

	entered = EnterSymbol(op, cookie, type, name, False);

	if( pdebug )
	{
		printf("Enter symbol cookie %d type ", cookie);
		tprint(type);
		printf(" space %s", opst[op]);
		printf(" id %d name %s\n", entered, name);
	}
	return entered;
}

Identifier
EnterSymbol(op, cookie, type, name, match)		/* Enter (op, cookie), type into table */
	Operator op;
	CONSZ cookie;		/* unique cookie from pass1, NOT Identifier */
	TWORD type;		/* type of identifier */
	char * name;		/* if non-null, name of identifier */
	Boolean match;		/* Should we insist on matching types */
{
	SymbolPtr s;
	KeyType k;

	k = MakeKey(op, cookie, match ? type : ANYTYPE);
	s = TableLookUp(k);

	if (s->id == NoId) {

		/* Set up its Identifier */

		++MaxIdentifier;
		CheckRef( (int) MaxIdentifier);
		IDs[MaxIdentifier] = s;
		s->id = MaxIdentifier;

		/* Build the rest of the entry */

		++NumEntries[space_map(op)];
		assert(NumEntries[space_map(op)] < SymbolTableSize);
		s->key = MakeKey(op, cookie, type);
		if (name == NULL)
			s->id_text = NULL;
		else
			s->id_text = SaveIdText(name);
		s->n = NULL;
		s->leaf_ref = NULL;
		s->last_killed = NULL;
		s->tmp_defs = NULL;
		s->t = NULL;
		s->addressed = False;
		s->use_before_set = False;
		s->may_not_promote = False;
		s->transparent = False;
		s->benefit = 0;
		s->allocated = False;
		s->NeedsSpace = NULL;
		s->FirstNeed = NoElement;
		s->defs = NULL;
		s->uses = NULL;
# ifdef FORT
		s->shadow_id = NoId;
		s->actual_type = s->key.type;
		s->copy_eligible = False;
		s->common = 0;
# endif /* FORT */
	}
	return s->id;
}

/*
 *	Identifier Text Handling
 */

char *
SaveIdText(name)
	char *name;
{
	int len;
	char *saved;

	len = strlen(name) + 1;
	len = (len < MinIdSpace ? MinIdSpace : len);
	saved = (char *) malloc( (unsigned) len);
	CheckStorage(saved, "storage for identifier (%d chars)", len);
	IncreaseSpace(s_IdText, len);

	(void) strcpy(saved, name);

	return saved;
}

char *
IdText(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s != NULL);

	return s->id_text;
}

static void
FreeIdText(s)
	SymbolPtr s;
{
	int len;

	assert(s->id_text != NULL);
	len = strlen(s->id_text) + 1;
	len = len < MinIdSpace ? MinIdSpace : len;

#ifdef IdDebug
	assert(len >= strlen(JunkId) + 1);
	(void) strcpy(s->id_text, JunkId);		/* Debug help */
#endif

	free(s->id_text);
	DecreaseSpace(s_IdText, len);
}

DAG_Node
NodeID(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	if (s == NULL)
		return NULL;		/* Temporary */
	else {
		assert(s->id != NoId);
		return s->n;
	}
}

void
SetNodeID(id, n)
	Identifier id;
	DAG_Node n;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	s->n = n;
}

void
SetAddressed(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	s->addressed = True;
}

/*
 * Note that a parameter was addressed
 */

void
ParamAddressed()
{
	Identifier id;

	for (id=FirstId; id<=MaxIdentifier; ++id) {
		if (IdOp(id) == PNAME)
			SetAddressed(id);
	}
}

Boolean
WasAddressed(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	return s->addressed;
}

void
SetAllocated(id, value)
	Identifier id;
	Boolean value;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	s->allocated = value;
}

Boolean
IsAllocated(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	return s->allocated;
}

Boolean
HasAllocation(id)
	Identifier id;
{
	SymbolPtr s;

	assert(id != NoId);

	s = TableEntry(id);
	assert(s->id == id);
	assert(s->t != NULL);

	switch(s->t->in.op)
	{
	case LTEMP:
	case TNAME:
	case LNAME:
	case PNAME:
		return s->t->tn.lval != NOOFFSET;

	default:
		return True;
	}
	/* NOTREACHED */
}

void
SetNotPromotable(id, v)
	Identifier id;
	Boolean v;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id == id);
	s->may_not_promote = v;
}

Boolean
MayNotPromote(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id == id);
	return s->may_not_promote;
}

void
SetBenefit(id, value)
	Identifier id;
	int value;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	s->benefit = value;
}

void
ChangeBenefit(id, delta)
	Identifier id;
	int delta;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	s->benefit += delta;
}

int
Benefit(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	return s->benefit;
}

void
SetLeafRef(id, n)
	Identifier id;
	DAG_Node n;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);

	s->leaf_ref = n;
}

void
SetIdTree(id, t)
	Identifier id;
	TreeNode t;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	assert(s->t == NULL);
	s->t = TreeAllocate();
	*(s->t) = *t;
}

DAG_Node
LeafRef(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	return s->leaf_ref;
}

TreeNode
IdTree(id)		/* returns the tree that references this id */
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);

	return s->t;
}

void
SetLastKiller(id, n)
	Identifier id;
	DAG_Node n;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s != NULL && s->id != NoId);
	s->last_killed = n;
}

DAG_Node
LastKilled(id)		/* Node that last killed this one */
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s != NULL && s->id != NoId);
	return s->last_killed;
}

/*
 * Return the type of a symbol - usually as it was obtained from pass 1
 */
TWORD
SymType(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert( s != NULL  &&  s->id != NoId );
	return(s->key.type);
}

#ifdef FORT

void
SetCommonBlock(id, block)
	Identifier id;
	CONSZ block;
{
	assert( IDs[id] != NULL );
	IDs[id]->common = block;
}

CONSZ
InCommonBlock(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert( s != NULL  &&  s->id != NoId );
	return(s->common);
}


/*
 * Return the type of a symbol - its actual type
 */
TWORD
GetActualType(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert( s != NULL  &&  s->id != NoId );
	return(s->actual_type);
}

# endif /* FORT */

/*
 * Get cookie for a symbol
 */

CONSZ
IdCookie(id)
	Identifier id;
{
	SymbolPtr s = TableEntry(id);

	assert( s != NULL  &&  s->id != NoId );
	return s->key.cookie;
}

#ifdef FORT

/*
 *	f77 may make entries in the symbol table before the type
 *	is known for sure.
 */

void
SetType(id, type, actl_type)
	Identifier id;
	TWORD type;
	Boolean actl_type;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert( s != NULL  &&  s->id != NoId );
	s->key.type = type;
	if( actl_type )
		s->actual_type = type;
}

Boolean
IsCopyEligible( id )
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert( s != NULL  &&  s->id != NoId );
	return (s->copy_eligible );
}

void
SetCopyEligible( id )
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert( s != NULL  &&  s->id != NoId );
	s->copy_eligible = True;
}

#endif

/*
 * Return set of all definitions of an id
 */

LongSet
GetDefs(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	return s->defs;
}

/*
 * Initialize the set of all definitions of an id
 */

LongSet
InitDefs(id, nelements)
	Identifier id;
	int nelements;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	s->defs = CreateSet(nelements);
	NullSet(s->defs);
	return s->defs;
}

/*
 * Return set of all uses of an id
 */

LongSet
GetUses(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	return s->uses;
}

/*
 * Initialize the set of all definitions of an id
 */

LongSet
InitUses(id, nelements)
	Identifier id;
	int nelements;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	s->uses = CreateSet(nelements);
	NullSet(s->uses);
	return s->uses;
}

void
SetUsedFirst(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	s->use_before_set = True;
}

Boolean
UsedFirst(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	return s->use_before_set;
}

/* Mark a symbol - usually a temp - as transparent
 */
void
MarkTransparent(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	s->transparent = True;
}

/* Inverse of MarkTransparent()
 */
void
UndoTransparent(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	s->transparent = False;
}

Boolean
IsTransparent(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	return s->transparent;
}

/*
 * Add a node to the list of temporary definition points
 */

void
AddTempDef(id, n)
	Identifier id;
	DAG_Node n;
{
	SymbolPtr s;
	TempDefList p;

	s = TableEntry(id);
	assert(s->id != NoId);
	p = CreateTempDef();
	p->next = s->tmp_defs;
	s->tmp_defs = p;
	p->def = n;
}

/*
 * Return the list of definition points for a temporary
 */

TempDefList
GetTempDefs(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	assert(s->id != NoId);
	return s->tmp_defs;
}

/*
 *	Delete a definition point from list for id.
 */

void
DeleteTempDef(id, d)
	Identifier id;
	TempDefList d;
{
	SymbolPtr s;
	TempDefList lp, p;

	s = TableEntry(id);

	lp = NULL;
	p = s->tmp_defs;

	while (p != NULL && p != d) {
		lp = p;
		p = p->next;
	}

	assert(p != NULL);	/* fails if we didn't find it */

	/* p points to entry, lp points to predecessor */

	if (lp == NULL)
		s->tmp_defs = p->next;
	else
		lp->next = p->next;

	FreeTempDef(p);
}

/* NeedsSpace Operators */

void
EmptyNeeds(id)	/* set NeedsSpace to a new null set */
	Identifier id;
{
	SymbolPtr p = TableEntry(id);

	assert(p->NeedsSpace == NULL);
	p->NeedsSpace = CreateSet((int)NumFlowNodes);
	NullSet(p->NeedsSpace);
}

void
AddNeeds(flowindex,id)	/* add a block to a NeedSpace */
	FlowIndex flowindex;
	Identifier id;
{
	SymbolPtr p = TableEntry(id);

	assert(InitialNode<=flowindex && flowindex<NumFlowNodes);
	/* ignore uninteresting vars (with NULL needs) */
	if (p->NeedsSpace != NULL)
		Insert((int)flowindex, p->NeedsSpace);
}

void
FreeNeeds(id)	/* free a NeedsSpace */
	Identifier id;
{
	SymbolPtr p = TableEntry(id);

	assert(p->NeedsSpace != NULL);
	DestroySet(p->NeedsSpace);
	p->NeedsSpace = NULL;
	p->FirstNeed = NoElement;
}

LongSet
Needs(id)		/* return NeedsSpace */
	Identifier id;
{
	SymbolPtr p = TableEntry(id);

	return p->NeedsSpace;
}

void
SetFirstNeed(id)	/* calculate FirstNeed */
	Identifier id;
{
	SymbolPtr p = TableEntry(id);

	if (p->NeedsSpace != NULL)
		p->FirstNeed = FirstElement(p->NeedsSpace);
}

int
FirstNeed(id)	/* return pre-calculated FirstNeed */
	Identifier id;
{
	return TableEntry(id)->FirstNeed;
}

void
InitSymManager()		/* Initializes entire module */
{
	int tx, ix;
	SymbolPtr s;

	if (SymbolTable != NULL)
/**/		return;
	SymbolTable = (SymTable *) malloc((unsigned) (NumTables * SymbolTableSize *
				sizeof(SymbolEntry)));
	CheckStorage(SymbolTable, "storage for the symbol table", 0);

	for (tx = 0; tx < NumTables; ++tx) {
		NumEntries[tx] = 0;
		for (ix = 0; ix < SymbolTableSize; ++ix) {
			s = &SymbolTable[tx][ix];
			s->id = NoId;
			s->t = NULL;
		}
	}
	EnterRegisterDefinitions();
}

void
InitTable()			/* At function start (Reset not required) */
{
	TableIndex ix;
	SymbolPtr s;
	Identifier id;
	TempDefList p, p_n;

	/* Step 1: Clear Local Id's, reset Global ones */

	for (id = FirstId; id <= MaxIdentifier; ++id) {
		assert(id < current_size && IDs[id] != NULL);

		s = IDs[id];
		assert(s->id != NoId);

		if (s->defs != NULL )
			DestroySet(s->defs);
		s->defs = NULL;

		if (s->uses != NULL )
			DestroySet(s->uses);
		s->uses = NULL;

		/* Rudimentary storage management */

		if (s->t != NULL) {
			TreeFree(s->t);
			s->t = NULL;
		}

		if (space_map(s->key.op) == LocalSpace) {
			if (s->id_text != NULL)
				FreeIdText(s);
			s->id = NoId;
			s->n = NULL;

			/*	Free the list of temp defs */

			p = s->tmp_defs;
			while (p != NULL) {
				p_n = p->next;
				FreeTempDef(p);
				p = p_n;
			}
			s->tmp_defs = NULL;

			s->last_killed = NULL;
			s->leaf_ref = NULL;
		}
	}


	MaxIdentifier = FirstId - 1;	/* Clear the identifier map */
	NumEntries[LocalSpace] = 0;

	for (ix = 0; ix < SymbolTableSize; ++ix) {

		/* Restore global to map if it exists */

		s = &SymbolTable[GlobalSpace][ix];
		if (s->id != NoId) {
			++MaxIdentifier;
			CheckRef( (int) MaxIdentifier);
			IDs[MaxIdentifier] = s;
			s->id = MaxIdentifier;
		}
	}
	EnterRegisterDefinitions();
}

void
ResetTable()				/* At start of block */
{
	Identifier id;
	SymbolPtr s;

	for (id = FirstId; id <= MaxIdentifier; ++id) {
		assert(IDs[id] != NULL);
		s = IDs[id];
		s->last_killed = NULL;
		s->leaf_ref = NULL;
		s->n = NULL;
	}
}


/*
 *	Debugging
 */

#define node_num(n)	((n) == NULL ? 0 : (n)->order)
#define flag(b)		((b) ? 'Y' : 'N')

static void
PrintTblEntry(s)
	SymbolPtr s;
{
	printf("Table entry @ %o: ", s);
	printf("id %d ", s->id);
	printf("text ");
	if (s->id_text != NULL)
		printf("%s ", s->id_text);
	else
		printf("<none> ");
	printf("key (op %s cookie %d type ", opst[s->key.op], s->key.cookie);
	if (IsANYTYPE(s->key.type))
		printf("Any type");
	else
		tprint(s->key.type);
	printf(") ");
	printf("\n\t");
	printf("node # %d ", node_num(s->n));
	printf("leaf_ref %d last_killed %d ",
			node_num(s->leaf_ref),
			node_num(s->last_killed));
	printf("\n\t");
	printf("temp defs: ");
	{
		TempDefList p;

		p = s->tmp_defs;
		while (p != NULL) {
			printf("%d ", node_num(p->def));
			p = p->next;
		}
	}
	printf("\n\t");
	printf("addr? %c use before set %c transparent %c allocated %c ",
		flag(s->addressed), flag(s->use_before_set),
		flag(s->transparent), flag(s->allocated));

	printf("\n\t");
# ifdef FORT
	printf("shadow id %d, copy elig %c, actual type ",
		s->shadow_id, flag(s->copy_eligible));
	tprint(s->actual_type);
	printf(", common blk %d,\n\t", s->common );
# endif /* FORT */
	printf("can't promote %c, benefit %d tree (@ %lx):\n",
		flag(s->may_not_promote), s->benefit, (long) s->t);
	if (s->t != NULL)
		fwalk(s->t, eprint, 3);
}

/*
 *	Intended for use in sdb(1)
 */

void
PrintSymbol(id)
	Identifier id;
{
	SymbolPtr s;

	s = TableEntry(id);
	PrintTblEntry(s);
}

void
PrintTable()
{
	int tx, ix;
	SymbolPtr s;
	Identifier id;

	printf("\nSymbol Table\n");

	for (id = FirstId; id <= MaxIdentifier; ++id) {
		if (IDs[id] == NULL)
			printf("Null IDs[%d] !!\n", id);
		else
		if (IDs[id]->id != id)
			printf("Broken IDs pointer %d %d %o\n", id, IDs[id]->id, IDs[id]);
	}
	for (tx = 0; tx < NumTables; ++tx) {
		for (ix = 0; ix < SymbolTableSize; ++ix) {
			s = &SymbolTable[tx][ix];
			if (s->id != NoId) {
				PrintTblEntry(s);
				if (IDs[s->id] != s) {
					printf("***** Error %o %o", s, IDs[s->id]);
				}
				printf("\n");
			}
		}
	}
}


/*
 * Enter defintions of registers not used for register variables that may
 * appear in expressions.
 */

static void
EnterRegisterDefinitions()
{
	register int rx;

	/*
	 *	Scan the list of reserved registers, and enter them
	 *	into the symbol table.  The id cookie is the same
	 *	as the register number.
	 */

	rx = 0;
	while (ReservedRegs[rx].reg != -1) {
		(void) EnterSymbol(REG, (CONSZ) ReservedRegs[rx].reg,
					ReservedRegs[rx].type,
				   (char *) NULL, False);
		++rx;
	}

#ifdef	INLINER
	LastRegId = MaxIdentifier;
#endif	/* INLINER */
}

/*
 *	Submodule: TempDefList memory management
 */

static TempDefList
CreateTempDef()
{
	TempDefList a, c;
	int i;

	if (FreeTempDefs == NULL) {
		c = GetArray(s_TempDefs, TempDefChunk, TempDefType);
		CheckStorage(c, "storage for attached ids", 0);

		/* Turn array into backwards linked list */

		a = NULL;
		for (i = 0; i < TempDefChunk; ++i) {
			c->next = a;
			a = c++;
		}

		/* a points to the head of the new free list */
	} else
		a = FreeTempDefs;

	FreeTempDefs = a->next;
	return a;
}

static void
FreeTempDef(a)
	TempDefList a;
{
	a->next = FreeTempDefs;
	FreeTempDefs = a;
}

/*
 *	End of Storage management for TempDefList
 */
