/* dag.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */
/*
 *	Directed Acyclic Graph Representation of a Basic Block
 */

/*
 *	See: Aho and Ullman, Principles of Compiler Design, Section 12.3
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: dag.c,v 5.5 89/05/12 12:49:54 pcc Rel-3_0 $";
/* static char ID[] = "@(#)dag.c	15.7x(TK)	of 86/11/27"; */
#endif

/*	Import
 */

# include <assert.h>
# include <blocks.h>
# include <dag.h>
# include <flow.h>
# include <bool.h>
# include <udchain.h>
# include <temp.h>
# include <ops.h>
# include <comsubex.h>
# include <identifier.h>
# include <dagsymbol.h>
# include <daghash.h>
# include <storage.h>
# include <erroro.h>
# include <target.h>

/*	Export
 */

int ddebug = 0;

int GoodGlobals = DefaultGoodGlobals;	/* 0 => volatile */

DAG_Node FirstNodeCreated = NULL;	/* first node created */

/*	Forward
 */

static void BlockDAG();
static void InstDAG();
static DAG_Node TreeDAG();
static DAG_Node Assign();
static DAG_Node LeafDAG();
static DAG_Node UnaryDAG();
static DAG_Node BinaryDAG();
static DAG_Node ConditionalDAG();
static DAG_Node MakeSideEffectDag();
static DAG_Node NoHashDag();
static DAG_Node BuildLeaf();
static DAG_Node ForceConversion();
static DAG_Node AssignUMUL();
static Identifier WhatReferenced();
static Identifier RefWalk();
static void KillNodes();
static void KillConditional();
static Boolean DagDeleteID();
static DAG_Node BuildUNOP();

/*	Private
 */


/*
 *	Definitions that make it easier to read calls to routines
 *	with Boolean arguments.
 */

#define	Attach		True		/* BuildLeaf should attach the id */
#define NoAttach	False

#define Cascade		True		/* Cascading conditional op */
#define NoCascade	False

static DAG_type EmptyNode;		/* Actual node, not a pointer ! */

static DAG_Node LastNodeCreated  = NULL;
static DAG_Node free_list  = NULL;
static int NumLastNode   = 0;		/* Last node's order number */

static DAG_Node LastIStoreP = NULL;	/* Last node considered in indirect
					 * store dependencies
					 */
static DAG_Node LastDStoreP = NULL;	/* Last node considered in direct
					 * store dependencies
					 */
static DAG_Node LastFetchP = NULL;	/* Last node considered in fetch
					 * dependencies
					 */

static AttachedID FreeIds = NULL;	/* Free list of attached id's */

/*
 *	Current instruction being processed
 */

static LineNumber cur_line = 0;
static char * cur_file     = NULL;

/*
 *	Which conditional "zone of side effects" (The Twilight Zone)
 *	are we in ?
 */

static DagCount CondZone		= 0;

void
InitDagModule()			/* Overall initialization - called once */
{
	/*
	 *	We could explicitly initialize the globals above
	 *	in this module.  Maybe someday.
	 */

	/*
	 *	Initialize the generic empty DAG node.
	 *	This makes creating new nodes faster.
	 */

	EmptyNode.op = FREE;		/* reasonable, but silly! */
	EmptyNode.order = 0;
	EmptyNode.type = UNDEF;
	EmptyNode.next = NULL;
	EmptyNode.u.in.left = NULL; EmptyNode.u.in.right = NULL;
	EmptyNode.chain.hash_link = NULL;
	EmptyNode.leaf_id = NoId;
	EmptyNode.new_tree = NULL;
	EmptyNode.e_tree = NULL;
	EmptyNode.in_degree = 0;
	EmptyNode.refs = 0;
	EmptyNode.delay_count = 0;
	EmptyNode.count = 0;
	EmptyNode.visited = False;
	EmptyNode.evaluated = False;
	EmptyNode.attached = NULL;
	EmptyNode.carrier = NoId;
	EmptyNode.cost = 0;
	EmptyNode.hashable = False;
	EmptyNode.sticky = False;
	EmptyNode.is_fetch = True;
	EmptyNode.in_cond = False;
	EmptyNode.invariant = False;
	EmptyNode.line_number = 0;
	EmptyNode.file_name = NULL;
	EmptyNode.indirect = NoId;
	EmptyNode.UDIndex = BadDef;
	EmptyNode.In = NULL;
	EmptyNode.delayed = NULL;
	EmptyNode.special_delay = False;
	EmptyNode.prev_valid_carrier = False;
	EmptyNode.carrier_required = False;

}

void
InitDAG()
{
	FirstNodeCreated = NULL;
	LastNodeCreated  = NULL;
	NumLastNode = 0;
	LastIStoreP = NULL;
	LastDStoreP = NULL;
	LastFetchP = NULL;
}

static void
FreeNode(n)		/* Free up a node */
	DAG_Node n;
{
	n->next = free_list;
	free_list = n;
}

static AttachedID
CreateAid()
{
	AttachedID a, c;
	int i;

	if (FreeIds == NULL) {
		c = GetArray(s_Aids, AidChunk, AID_type);
		CheckStorage(c, "storage for attached ids", 0);

		/* Turn array into backwards linked list */

		a = NULL;
		for (i = 0; i < AidChunk; ++i) {
			c->next = a;
			a = c++;
		}

		/* a points to the head of the new free list */
	} else
		a = FreeIds;

	FreeIds = a->next;
	return a;
}

static void
FreeAid(a)
	AttachedID a;
{
	a->next = FreeIds;
	FreeIds = a;
}

static void
FreeComma(t)
	TreeNode t;
{
	/*
	 *	We know they tower to the left, because that is how
	 *	we build them up.
	 */

	TreeNode next;

	while (t->in.op == COMOP) {
		next = t->in.left;
		TreeFree(t);
		t = next;
	}
}

void
FreeDag(d)				/* Free an entire dag */
	DAG_Node d;	/* First in list */
{
	DAG_Node next;
	AttachedID a, next_a;
	TreeNode t, next_t;

	while (d != NULL) {

		next = d->next;

		a = d->attached;
		while (a != NULL) {
			next_a = a->next;
			FreeAid(a);
			a = next_a;
		}

		if (d->new_tree != NULL) {

			/* The tree will contain "count" assignment
			 * operators, used to assign the value to the
			 * attached id's.
			 * The node may also own one or two COMOP nodes
			 * if it is a :, && or ||.
			 * These are the only nodes owned by this DAG node
			 * field.
			 */

			t = d->new_tree;
			while (d->count != 0) {
				assert(t != NULL && t->in.op == ASSIGN);
				next_t = t->in.right;
				TreeFree(t);
				t = next_t;
				--(d->count);
			}

			/*
			 *	t is now the root of the tree that
			 *	was used to obtain the value of the node.
			 */

			if (t->in.op == COLON) {		/* might be two */
				FreeComma(t->in.left);
				FreeComma(t->in.right);
			} else
			if (t->in.op == ANDAND || t->in.op == OROR)
				FreeComma(t->in.right);

			d->new_tree = NULL;
		}

		if (d->e_tree != NULL) {
			/*
			 * This is the evaluation tree for the node.
			 * This field owns just this one tree node.
			 */

			TreeFree(d->e_tree);
			d->e_tree = NULL;
		}

		if (d->In != NULL) {
			DestroySet(d->In);
			d->In = NULL;
		}
		FreeNode(d);
		d = next;
	}
}

	
DAG_Node
CreateNode(o, t, inCond)
	Operator o;
	TWORD t;
	DagCount inCond;
{
	DAG_Node n, c;
	int i;

	if (free_list == NULL) {
		c = GetArray(s_DagNodes, DagChunk, DAG_type);
		CheckStorage(c, "storage for DAG nodes", 0);

		/* Turn array into linked list (backwards!) */

		n = NULL;
		for (i=0; i < DagChunk; ++i) {
			c->next = n;
			n = c++;
		}

		/*
		 * Now n points to the head of the free list (the
		 * last entry in the chunk.
		 */
		
	} else
		n = free_list;

	free_list = n->next;

	/*
	 *	Initialize the DAG node
	 */


	*n = EmptyNode;
	n->op = o;
	n->type = t;
	n->in_cond = inCond;
	n->order = ++NumLastNode;
	n->line_number = cur_line;
	n->file_name = cur_file;

	if (LastNodeCreated == NULL)
		FirstNodeCreated = n;
	else
		LastNodeCreated->next = n;
	LastNodeCreated = n;
	return n;
}

AttachedID
AttachID(id, n)		/* attach id to node n */
	Identifier id;
	DAG_Node n;
{
	AttachedID aid, alist;

	aid = CreateAid();

	aid->id = id;
	aid->leaf_ref = LeafRef(id);
	aid->killed_by = NULL;
	aid->prev_attach = NULL;
	aid->sticky = False;
	aid->UDIndex = 0;
	aid->valid_carrier = False;

	alist = n->attached;
	if( alist != NULL && IsTemp(alist->id) )
	{
		/* Ensure that temp is always first and that we don't
		 * attach two of them.
		 */
		assert(!IsTemp(id));
		aid->next = alist->next;
		alist->next = aid;
	}
	else
	{
		aid->next = alist;
		n->attached = aid;
	}
	return aid;
}

void
DeleteID(id, n, MustDelete)	/* Delete id from list attached to node n.
				 * Scream if it cannot be found */
	Identifier id;
	DAG_Node n;
	Boolean MustDelete;	/* Do we delete this id if it is stuck on? */
{
	AttachedID junk;
	Boolean was_attached;

	was_attached = DagDeleteID(id, n, MustDelete, &junk);
	assert(was_attached);
}

/* Delete id from list attached to node n.
 * Tell caller if we found something to delete
 */
static Boolean
DagDeleteID(id, n, MustDelete, AID_stuck )
	Identifier id;
	DAG_Node n;
	Boolean MustDelete;	/* Do we delete this id if it is stuck on? */
	AttachedID *AID_stuck;	/* address of attached ID if it is still stuck */
{
	AttachedID aid, prev;

	aid = n->attached; prev = NULL;
	while (aid != NULL && !SameId(aid->id, id)) {
		prev = aid;
		aid = aid->next;
	}

	/*
	 * Really delete this identifier if
	 * 1) We have been told to do so no matter what
	 * or
	 * 2) The identifier itself is not sticky and it is not the last
	 *    identifier on a sticky node
	 */
	if( aid != NULL )
	{
		if( MustDelete ||
		     (!aid->sticky && !(n->sticky && n->attached->next == NULL)) )
		{
			if (prev == NULL)
				n->attached = aid->next;
			else
				prev->next = aid->next;
			FreeAid(aid);
			*AID_stuck = NULL;
		}
		else
			*AID_stuck = aid;
	}

	return aid != NULL;
}

AttachedID
FindAttachedMember(list, id)
	AttachedID list;
	Identifier id;
{
	AttachedID aid;

	aid = list;
	while (aid != NULL && !SameId(aid->id, id)) {
		aid = aid->next;
	}
	return aid;
}

/*
 * Set up DAG dependencies required after a store through a pointer.
 * These include:
 *	Unhash all Unary Mul fetches (stores are never hashed)
 *	Make all Unary Mul fetches and stores sticky
 *	Make all NAME nodes sticky; also make sticky nodes of LNAMEs, PNAMEs
 *	and STATNAMEs whose addresses have been taken
 *	Propagate stickiness up through the DAG
 */

static void
IStoreDependencies(store)
	DAG_Node store;		/* DAG node that caused this call */
{
	Operator op;
	DAG_Node d, l, r;

	d = (LastIStoreP == NULL) ? FirstNodeCreated : LastIStoreP;

	while( d != NULL )
	{
		op = d->op;

		if( op == UNARY MUL && d != store &&
		   (d->indirect == NoId || store->indirect == NoId || d->indirect == store->indirect))
		{
			d->sticky = True;
			d->hashable = False;
		}
		else
		if( op == NAME ||
		  ((op == LNAME || op == PNAME || op == STATNAME) &&
		   (store->indirect == NoId || store->indirect == d->leaf_id)&&
		    WasAddressed(d->leaf_id) ))
		{
			d->sticky = True;
		}
		else
		switch(optype(op))
		{
		case UTYPE:
			l = d->u.in.left;
			d->sticky = d->sticky ||
				    (l->sticky && !WillHaveCarrier(l));
			break;

		case BITYPE:
			r = d->u.in.right;
			l = d->u.in.left;
			d->sticky = d->sticky ||
				    (l->sticky && !WillHaveCarrier(l)) ||
				    (r->sticky && !WillHaveCarrier(r));
			break;
		}

		d = d->next;
	}

	if( store->indirect == NoId )
		LastFetchP = LastDStoreP = LastIStoreP = store;
}

/*
 * Set up DAG dependencies required after a direct store to a global or
 * an LNAME, PNAME or STATNAME whose address has been taken.
 *	Unhash all Unary Mul fetches (stores are never hashed)
 *	Make all Unary Mul fetches sticky
 *	Propagate stickiness up through the DAG
 */

static void
DStoreDependencies(id)
	Identifier id;
{
	Operator op;
	DAG_Node d, l, r;

	d = (LastDStoreP == NULL) ? FirstNodeCreated : LastDStoreP;

	while( d != NULL )
	{
		op = d->op;

		if( op == UNARY MUL && d->is_fetch &&
		   (d->indirect == NoId || d->indirect == id))
		{
			d->sticky = True;
			d->hashable = False;
		}
		else
		switch(optype(op))
		{
		case UTYPE:
			l = d->u.in.left;
			d->sticky = d->sticky ||
				    (l->sticky && !WillHaveCarrier(l));
			break;

		case BITYPE:
			r = d->u.in.right;
			l = d->u.in.left;
			d->sticky = d->sticky ||
				    (l->sticky && !WillHaveCarrier(l)) ||
				    (r->sticky && !WillHaveCarrier(r));
			break;
		}

		d = d->next;
	}

	/* Can't do this at the moment, since there may be fetches that
	 * we have to look at again later
	 *
	 * LastDStoreP = LastNodeCreated;
	 */
}

/*
 * Common routine called to establish dependencies for both direct and
 * indirect fetches of globals and locals/parameters whose addresses have
 * been taken.  This routine guarantees that all indirect stores will be
 * done when code is generated for node "fetch"
 */

static void
FetchDependencies(fetch, id)
	DAG_Node fetch;
	Identifier id;
{
	Operator op;
	DAG_Node d, l, r;

	d = (LastFetchP == NULL) ? FirstNodeCreated : LastFetchP;

	while( d != NULL )
	{
		op = d->op;
		if( op == UNARY MUL && !d->is_fetch &&
		   (d->indirect == NoId || id == NoId || d->indirect == id))
		{
			d->sticky = True;
		}
		else
		switch(optype(op))
		{
		case UTYPE:
			l = d->u.in.left;
			d->sticky = d->sticky ||
				    (l->sticky && !WillHaveCarrier(l));
			break;
		case BITYPE:
			r = d->u.in.right;
			l = d->u.in.left;
			d->sticky = d->sticky ||
				    (l->sticky && !WillHaveCarrier(l)) ||
				    (r->sticky && !WillHaveCarrier(r));
			break;
		}

		d = d->next;
	}

	if( fetch->op != UNARY MUL )
		LastFetchP = fetch;
}

/*
 * Set up DAG dependencies required after a fetch through a pointer.
 * These include:
 *	Make all Unary Mul stores sticky
 *	Make all stores to NAME nodes sticky; also make sticky nodes of
 *	LNAMEs, PNAMEs and STATNAMEs whose addresses have been taken
 *	Propagate stickiness up through the DAG
 */

static void
IFetchDependencies(fetch)
	DAG_Node fetch;
{
	DAG_Node d;
	Identifier id;
	AttachedID a;

	assert(fetch->op == UNARY MUL);
	FetchDependencies(fetch, fetch->indirect);

	id = fetch->indirect;
	if( id != NoId )
	{
		d = NodeID(id);
		if( d != NULL && !IsTemp(id) )
		{
			a = d->attached;
			while (a != NULL && a->id != id) a = a->next;
			if( a != NULL )
				a->sticky = True;
		}
	}
	else
	{
	    for (id = FirstId; id <= MaxIdentifier; ++id) {
		d = NodeID(id);
		if ( d != NULL && !IsTemp(id) &&
			(WasAddressed(id) || IdOp(id) == NAME ) )
		{
			a = d->attached;
			while (a != NULL && a->id != id) a = a->next;
			if( a != NULL )
				a->sticky = True;
		}
	    }
	    LastFetchP = fetch;
	}

}

AttachedID
MakeTemp(n, code, ty)		/* Make a temp for n */
	DAG_Node n;
	char code;		/* part of name, for debugging */
	TWORD ty;		/* type of object to create */
{
	Identifier temp;
	AttachedID a;

	temp = GetTemp(TNAME, ty, code);
	SetNodeID(temp, n);
	AddTempDef(temp, n);
	
	a = AttachID(temp, n);
	a->valid_carrier = True;	/* ALL temps can be carriers */

	return a;
}

void
BuildDAGS()
{
	BasicBlock b;

	for (b = FirstBlock; b != NULL; b = b->next)
	    if( b->reachable ) {
		InitDAG();
		ResetTable();
		InitHash();
		BlockDAG(b);
		if (ddebug) {
			printf("Block # %d\n", b->blocknum);
			PrintGraph(b->Dag);
			if (ddebug > 1) {
				PrintTable();
				DumpHash();
			}
		}
	    }
}

static void
BlockDAG(b)
	BasicBlock b;
{
	Instruction i, ni;

	/*
	 *	Go through the code for the block, building a DAG.
	 *	Exprs are put into the DAG, all other instructions
	 *	are ignored.  They will be processed in dagtree.c.
	 */

	i = b->code.first;
	while (i != NULL) {
		ni = i->next;
		if (InstType(i) == Expr)
			InstDAG(i);
		i = ni;
	}
	b->Dag = FirstNodeCreated;
	InitFGindex(b);
}

static void
InstDAG(i)		/* Generate DAG for an instruction */
	Instruction i;
{
	switch (InstType(i)) {
		case Expr:
			cur_line = i->u.ex.lineno;
			cur_file = i->u.ex.filename;
			(void) TreeDAG(i->u.ex.root, NotConditional);
			if( !GoodGlobals )
			{
				/* Pretend there was a call after the ; of
				 * this statement.  The first arg of
				 * KillNodes() may need to be changed when it
				 * becomes smarter about pointers
				 */
				IStoreDependencies(LastNodeCreated);
				KillNodes(i->u.ex.root, LastNodeCreated);
			}
			break;
		default:
			InternalFault("Can't generate DAG for inst type %d\n",
				(int) InstType(i));
	}

	/*
 	 *	Prevent inadvertant misleading info
	 */

	cur_line = 0;
	cur_file = NULL;
}

static DAG_Node
TreeDAG(t, inCond)		/* Generate the DAG for an expression tree */
	TreeNode t;
	DagCount inCond;	/* Are we in right side of conditional ? */
{
	DAG_Node n;
	Operator op;

	if (asgop(t->in.op)) {
		n = Assign(t, inCond);
	} else {
		switch (optype(t->in.op)) {
			case LTYPE:
				n = LeafDAG(t, inCond);
				break;
			case BITYPE:
				n = BinaryDAG(t, inCond);
				break;
			case UTYPE:
				n = UnaryDAG(t, inCond);
				break;
			default:
				return NULL;	/* Error ? */
		}
	}
/***/
	/* Special case: copy of tree node kept to preserve special info */

	op = t->in.op;
	if (op == STASG || op == STARG || op == STCALL || op == UNARY STCALL )
	{
		assert(n->new_tree == NULL);
		n->new_tree = TreeAllocate();
		*(n->new_tree) = *t;
	}
	if (callop(t->in.op))
	{
		IStoreDependencies(n);
		KillNodes(t, n);
		n->sticky = True;
	}
	return n;
}

/*
 *		Handling For Assignments
 */

static DAG_Node
SimpleAssign(id, n, idtree, inCond)		/* id = n */
	Identifier id;
	DAG_Node n;
	TreeNode idtree;	/* "prototype" tree for this variable */
	DagCount inCond;	/* Conditional Zone number */
{
	DAG_Node ln, nop;
	DAG_Node killer;	/* node that may have killed this id */
	DAG_Node last_leaf;	/* last use as a leaf */
	DAG_Node real_assign;	/* node where this assign really happens */
	AttachedID a;
	AttachedID prev_attach;	/* previous attachment if we were unable to
				 * detach the identifier */
	Boolean rNAME;		/* if node is a rewritten NAME */
#ifdef FORT
	TreeNode tref;
#endif /* FORT */


	rNAME = False;
	if (idtree->in.op == NAME && GlobalAddrTransform(id)) {
		assert(idtree->tn.rval != 0);	/* not a NONAME */
		rNAME = True;
		RewriteNAME(idtree);
#ifdef FORT
		if( idtree->in.left->tn.lval != 0 ) {
			tref = IdTree( id );
			DoOffset( idtree->in.left, tref->tn.lval );
		}
#endif /* FORT */
	}

	if (IdTree(id) == NULL)
		SetIdTree(id, idtree);

	real_assign = n;
	prev_attach = NULL;
	ln = NodeID(id);
	if (rNAME) {
		/* It won't be attached, so don't try to do anything */
		if (ln != NULL)
			assert(FindAttachedMember(ln->attached, id) == NULL);
	} else
	if (inCond != NotConditional)
	{
		/* When an assignment is in a conditional, never remove
		 * attached identifiers from previous attachments.
		 */
		if( ln != NULL )
			prev_attach = FindAttachedMember(ln->attached, id);
	}
	else
	if (ln != NULL)
	{
		if( DagDeleteID(id, ln, NoForceDelete, &prev_attach))
		{
			/* If prev_attach is non-null, DagDeleteID was unable to remove
			 * this attached id because it was stuck on.  Force
			 * a delayed store if the new attachment precedes the
			 * old one
			 */
			if( prev_attach != NULL && n->order < ln->order )
			{
				SetNodeID(id, (DAG_Node) NULL);
				real_assign = BuildLeaf(id, NoAttach, idtree, inCond);
			}
		}
		else
		{
			/* Identifier not found on node.  Check that this
			 * is ok
			 */
			assert(IdOp(id) == ln->op  &&  ln->leaf_id == id);
		}
	}
	else
	{
		/*
		 *	If would be killed after store, or if last
		 *	leaf ref is later, but in a conditional, force
		 *	delay to here.
		 */

		killer = LastKilled(id);
		last_leaf = LeafRef(id);
		if ( (killer != NULL && n->order < killer->order) ||
		     (last_leaf != NULL && n->order < last_leaf->order &&
		      n->in_cond != last_leaf->in_cond) )
		{
			/* force delay */
			real_assign = BuildLeaf(id, NoAttach, idtree, inCond);
		}
	}

	/* If this is a NAME node, or an LNAME, PNAME or STATNAME that was
	 * addressd, we must ensure that all fetches (of this id) are done by
	 * the time this assignment is done.  Since we have no information
	 * about fetches, we just look at the last one.  IF it is later than n,
	 * we force a delayed store.  However, if we have already created
	 * or found a killing node that comes after the last fetch, then there
	 * is no need to create a new killer.   If we are rewriting NAMEs
	 * as * ADDR, then there is no need to generate a delayed store because
	 * the ASSIGN will be at the correct place in the DAG.
	 */

	if (rNAME) {
		DStoreDependencies(id);
	} else
	if( idtree->tn.op == NAME || WasAddressed(id) )
	{
		if( LastFetchP != NULL &&
		    real_assign->order < LastFetchP->order )
		{
			SetNodeID(id, (DAG_Node) NULL);
			real_assign = BuildLeaf(id, NoAttach, idtree, inCond);
		}
		DStoreDependencies(id);
	}

	if (inCond != NotConditional && (n->order != NumLastNode || n->in_cond == NotConditional)) {
		/*
		 * On the r.h.s. of cond'ls, assignments are not
		 * moved.
		 * If the assignment is not to the last node generated,
		 * or that last node isn't in the conditional
		 * a UNARYNOP is created to provide a link.
		 */

		nop = BuildUNOP(n, inCond);
		real_assign = n = nop;
	}

	/* If this id has remained attached to a node, record its killer
	 */

	if( prev_attach != NULL )
		prev_attach->killed_by = real_assign;

	SetNodeID(id, n);
	if (rNAME) {
		/*
		 *	Create a normal ADDR node, and an unhashed
		 *	Unary MUL and ASSIGN node to finish the job.
		 */

		DAG_Node addr, umul;

		addr = TreeDAG(idtree->in.left, inCond);
		umul = NoHashDag(idtree->in.op, idtree->in.type,
				 addr, (DAG_Node) NULL, inCond);
		umul->u.tn.rval = idtree->tn.rval;
		umul->is_fetch = False;
		umul->indirect = id;
		(void) NoHashDag(ASSIGN, idtree->in.type, umul, n, inCond);
	} else {
		a = AttachID(id, n);
		a->prev_attach = ln;
	}

	return n;
}

/*
 * Insert a conversion node to coerce dag node d to type ty
 */

static DAG_Node
ForceConversion(d,ty,inCond)
	DAG_Node d;
	TWORD ty;
	DagCount inCond;
{
	DAG_Node dnew;

	dnew = UnaryHashLook(OCONVTREE, ty, d, 0);
	if( dnew == NULL )
	{
		dnew = CreateNode(OCONVTREE, ty, inCond);
		dnew->u.in.left = d;
		d->in_degree++;
		dnew->u.tn.rval = 0;
		EnterHash(dnew);
		dnew->hash = dnew->order;
	}
	return(dnew);
}
static DAG_Node
Assign(t, inCond)
	TreeNode t;
	DagCount inCond;
{
	TreeNode left;
	DAG_Node rhs, lhs, n;
	Operator left_op, op;
	Identifier id;
	TWORD ty;
#ifdef FORT
	TreeNode tref;
#endif /* FORT */

	left = t->in.left;
	left_op = left->in.op;
	op = t->in.op;

	if (SideEffectOp(op))		/* Operator has side effects */
	{
		if (left_op == NAME) {	/* See if rewrite */
			id = IdLookUp(left_op, (CONSZ) left->tn.rval, ANYTYPE);
			if (GlobalAddrTransform(id)) {
				/*
				 * If going to rewrite this NAME, do it
				 * now.  Although we'll then treat it
				 * as a pointer operation, we'll do better
				 * at recovering the idiom.  Someday,
				 * when idioms is rewritten so that it
				 * doesn't matter, we can go ahead
				 * and treat it as a hybrid.  When that
				 * happens, remember that left will be
				 * rewritten in MakeSideEffectDAG and so
				 * the variables above (left_op, etc.) must
				 * be reset.
				 */
				RewriteNAME(left);
#ifdef FORT
				if( left->in.left->tn.lval != 0 ) {
					tref = IdTree( id );
					DoOffset( left->in.left, tref->tn.lval );
				}
#endif /* FORT */
				left_op = left->in.op;
			}
		}

		/* Look for type paintdowns.  If found, build the DAG
		 * for the assign without the paintdown first.
		 */
		if( t->in.type != left->in.type )
		{
			ty = t->in.type;
			t->in.type = left->in.type;
			n = Assign(t, inCond);
			n = ForceConversion(n, ty, inCond);
			return n;
		}
		n = MakeSideEffectDag(t, inCond); /* DAG as if no side effects */
	}
	else
		n = TreeDAG(t->in.right, inCond);


	switch(left_op) {
		case NAME:
		case LNAME:
		case PNAME:
		case STATNAME:
		case REG:
			if (left_op == REG) {
				assert(IsRegVar(left->tn.rval) || left->tn.rval == left->tn.lval);
				id = IdLookUp(left_op,left->tn.lval, ANYTYPE);
			} else
				id = IdLookUp(left_op, (CONSZ) left->tn.rval, ANYTYPE);
			ty = SymType(id);
			if( ty != n->type )
			{
				n = ForceConversion(n, ty, inCond);
				n = SimpleAssign(id, n, left, inCond);
				if( op == INCR || op == DECR )
					n = n->u.in.left->u.in.left;	/* Pre-side effect value */
			}
			else
			{
				n = SimpleAssign(id, n, left, inCond);
				if (op == INCR || op == DECR)
					n = n->u.in.left;	/* Pre-side effect value */
			}
			break;

		case UNARY MUL:
		case FLD:
			/* L.H.S is something difficult.  We'll not
			 * put it in the symbol table, but leave
			 * the assign in the DAG
			 */

			/* if this is an op with side effects,
			 */
			rhs = n;
			if( SideEffectOp(op) )
			{
				/* Make up a U MUL DAG node whose left subtree
				 * is a pointer to everything below the U MUL
				 * on the lhs of the original ASGOP.  This U
				 * MUL must ALWAYS be unhashed.  Finally, use
				 * this as the lhs in building an ASSIGN DAG
				 * node.
				 * Action for FLD is similar, but we also make
				 * up an unhashed FLD node, and leave "left"
				 * pointing at the UNARY MUL node.
				 */

				if( left_op == FLD )
				{
					lhs = AssignUMUL(left->in.left,rhs->u.in.left, inCond);
					lhs = NoHashDag(left->in.op,
						left->in.type,
						lhs, (DAG_Node)NULL,
						inCond);
					lhs->u.tn.rval = left->tn.rval;

					left = left->in.left;
				}
				else
					lhs = AssignUMUL(left, rhs, inCond);

				n = NoHashDag(ASSIGN,t->in.type,lhs,rhs,inCond);
			}
			else
		 	{
				/* Make a DAG for everything below the U MUL on
				 * the lhs.  Then add in a U MUL that is
				 * ALWAYS unhashed.  Finally, use this as the
				 * lhs in building a DAG node for the
				 * assignment
				 * Action for FLD is similar, but we also make
				 * up an unhashed FLD node, and leave "left"
				 * pointing at the UNARY MUL node.
				 */
				if( left_op == FLD )
				{
					lhs = AssignUMUL(left->in.left,
						(DAG_Node) NULL, inCond);
					lhs = NoHashDag(left->in.op,
						left->in.type,
						lhs, (DAG_Node)NULL,
						inCond);
					lhs->u.tn.rval = left->tn.rval;

					left = left->in.left;
				}
				else
					lhs = AssignUMUL(left,
						(DAG_Node) NULL, inCond);
				n = NoHashDag(op,t->in.type,lhs,rhs,inCond);
			}

			/* For INCR/DECR, must return pre-side effect value and
			 * make sure that it goes into a temp.
			 */
			if( op == INCR || op == DECR )
			{
				n = n->u.in.right->u.in.left;
				(void) MakeTemp(n, 'a', n->type);
			}
			break;
		default:		/*  Something complex */
			InternalFault("Can't assign to op %d\n",
							(int) left_op);
	}
	return n;
}

/*
 * Create the LHS of an ordinary assignment or a SideEffectOp when the lhs
 * lhs starts with a UNARY MUL.  "left" points to the UNARY MUL on the left
 * hand side of the original op.
 *
 * If the op is a side effect op, "rhs" points to the DAG that has just
 * been created containing the mapped op and its descendents.  Thus,
 * rhs->u.in.left points to the UNARY MUL on the left of the original ASGOP,
 * and rhs->u.in.left->in.left points to everything below it.
 *
 * If the op is not a side effect op, "rhs" is null, and we must build the
 * DAG from scratch.
 * 
 * NOTE: For fields ops, rhs points to the FLD operator.  However,
 * rhs->u.in.left should be a UNARY MUL and rhs->u.in.left->u.in.left will
 * be everything below it as before.
 */
static DAG_Node
AssignUMUL(left, rhs, inCond)
	TreeNode left;
	DAG_Node rhs;
	DagCount inCond;
{
	DAG_Node lhs;

	assert(left->in.op == UNARY MUL);
	if( rhs == NULL )
	{
		/* An assignment.  Must build entire DAG from scratch */
		lhs = TreeDAG(left->in.left, inCond);
		lhs = NoHashDag(left->in.op, left->in.type,
				lhs,
				(DAG_Node)NULL, inCond);
	}
	else
	{
		/* an assignment operator.  Most of DAG is already built */

		lhs = NoHashDag(left->in.op, left->in.type,
				rhs->u.in.left->u.in.left,
				(DAG_Node)NULL, inCond);
	}
	lhs->u.tn.rval = left->tn.rval;
	lhs->is_fetch = False;

	lhs->indirect = WhatReferenced(lhs->u.in.left);
	IStoreDependencies(lhs);
	KillNodes(left, lhs);

	return lhs;
}

/* Walk the DAG nodes that are descendents of a UNARY MUL in an attempt
 * to figure out what id is being referenced
 *
 * Life is complicated by the fact that we do not want to get fooled by
 * pathologies such as
 *		a[ &b[j] - &a[k] ]
 * which is really b[j-k].  If we find such a pathology, we return UnknownId
 * so that WhatReferenced() can say that the store doesn't go anywhere known.
 */
static Identifier
WhatReferenced(d)
	DAG_Node d;
{
	Identifier id;

	id = RefWalk(d);
	return (id == UnknownId) ? NoId : id;
}

static Identifier
RefWalk(d)
	DAG_Node d;
{
	Identifier rhs, lhs;
	TreeNode t;

	switch(d->op )
	{
	case OCONVTREE:
	case OCONVLEAF:
	case SCONV:
	case PCONV:
		return RefWalk(d->u.in.left);

	case PLUS:
		lhs = RefWalk(d->u.in.left);
		rhs = RefWalk(d->u.in.right);
		if( rhs == NoId )		/* usual case */
			return lhs;
		else
		if( lhs == NoId )
			return rhs;		/* usual case as commuted by
						 * pass one */
		else
			return UnknownId;	/* 1)at least one side unknown
						 *  or
						 * 2)array addresses on both
						 *   sides (??)
						 */

	case MINUS:
		lhs = RefWalk(d->u.in.left);
		rhs = RefWalk(d->u.in.right);
		if( rhs == NoId )		/* Usual case */
			return lhs;
		else
		if( rhs == UnknownId )
			return UnknownId;	/* something fishy on rhs -
						 * give up */
		else
		if( lhs == rhs )
			return NoId;		/* something like
						 *  (&b[i] - &b[j])
						 * fishy, but benign
						 */
		else
			return UnknownId;

	case ADDR:
	case LADDR:
	case PADDR:
	case STADDR:
#ifndef MPX
	case CADDR:
#endif
			/* GROSS way to get at corresponding name.
			 * Fix whenever ADDRs and NAMEs are rationalized
			 */
		t =  IdTree(d->leaf_id);
		return IdLookUp(ToNAME(t->in.op), (CONSZ) t->tn.rval,ANYTYPE);

	case UNARY MINUS:
		lhs = RefWalk(d->u.in.left);
		if( lhs == NoId )
			return NoId;
		else
			return UnknownId;	/* disallow unary minus in
						 * front of addresses
						 */

	default:
		return NoId;
	}
	/*NOTREACHED*/
}

static DAG_Node
BuildLeaf(id, attach, t, inCond)
	Identifier id;
	Boolean attach;
	TreeNode t;
	DagCount inCond;
{
	DAG_Node n;

	n = NodeID(id);
	if (n == NULL) {
		n = CreateNode(t->in.op, t->in.type, inCond);
		SetNodeID(id, n);
		n->leaf_id = id;
		SetLeafRef(id, n);
		if (IdTree(id) == NULL)
			SetIdTree(id, t);
		if (attach) (void) AttachID(id, n);
		n->hash = n->order;
	}
	return n;
}

static DAG_Node
LeafDAG(t, inCond)
	TreeNode t;
	DagCount inCond;
{
	Operator op;
	DAG_Node n, nprev;
	Identifier id;
#ifdef FORT
	TreeNode tref;
#endif /* FORT */

	op = t->in.op;
	switch (op) {
		case ADDR:
		case LADDR:
		case PADDR:
		case STADDR:
#ifndef MPX
		case CADDR:
#endif
			id = IdLookUp(op, (CONSZ) t->tn.rval, ANYTYPE);
			if (id == NoId) {
				/* Not in symbol table! */
				id = IdLookUp(ToNAME(op), (CONSZ) t->tn.rval, ANYTYPE);
				assert(id != NoId);
				id = EnterSymbol(op, (CONSZ) t->tn.rval, t->tn.type,
					IdText(id), False);
				if (ddebug)
					printf("?ADDR sym. entry made %d\n",
						id);
						
			}
			assert(id != NoId);
			n = BuildLeaf(id, NoAttach, t, inCond);
			break;

		case NAME:
			id = IdLookUp(op, (CONSZ) t->tn.rval, ANYTYPE);
			nprev = NodeID(id);
			if (GlobalAddrTransform(id)) {
				if (nprev == NULL) {	/* Must gen ref. */
					RewriteNAME(t);		/* -> * ADDR */
#ifdef FORT
					if( t->in.left->tn.lval != 0 ) {
						tref = IdTree( id );
						DoOffset( t->in.left, tref->tn.lval );
					}
#endif /* FORT */
					n = TreeDAG(t, inCond);
				} else {
					n = nprev;	/* Use the old value */
				}
			} else {
				n = BuildLeaf(id, NoAttach, t, inCond);
				if( nprev == NULL )
					FetchDependencies(n,id);
			}
			break;

		case LNAME:
		case PNAME:
		case STATNAME:
			id = IdLookUp(op, (CONSZ) t->tn.rval, ANYTYPE);
			nprev = NodeID(id);
			n = BuildLeaf(id, NoAttach, t, inCond);
			if( nprev == NULL && ( op == NAME || WasAddressed(id) ))
				FetchDependencies(n,id);
			break;

		case REG:
			assert(IsRegVar(t->tn.rval) || t->tn.rval == t->tn.lval);
			id = IdLookUp(op, t->tn.lval, ANYTYPE);
			n = BuildLeaf(id, NoAttach, t, inCond);
			break;

		case ICON:
		case LABCON:
#ifndef MPX
		case TCON:	/* (MEY) add TCON and HCON for Alpha C */
		case HCON:	
#endif
			id = EnterSymbol(op, t->tn.lval, t->tn.type, (char *) NULL, True);
			n = BuildLeaf(id, NoAttach, t, inCond);
			break;

		case FCON:
			n = FconDAG( op, t->in.type, t->fpn.dval, inCond );
			break;

		case STLABEL:	/* Treat STLABELs differently - no hashing */
			n = CreateNode(op, t->in.type, inCond);
			n->u.tn.lval = t->tn.lval;
			n->u.tn.rval = t->tn.rval;
			n->hash = n->order;
			break;

		default:
			InternalFault("Unrecognised Leaf op: %d\n",
					(int) op);
	}
	return n;
}

DAG_Node
FconDAG( op, type, dval, inCond )	/* If existing DAG found return	*
					 * it, Else make one up.	*/
	Operator op;
	TWORD type;
	FCONSZ dval;
	DagCount inCond;
{
	DAG_Node n;

	n = FconHashLook( type, dval );
	if( n == NULL ) {
		n = CreateNode( op, type, inCond );
		n = EnterFHash( n , dval );
		n->hash = n->order;
	}
	return n;
}

static DAG_Node
UnaryDAG(t, inCond)
	TreeNode t;
	DagCount inCond;
{
	DAG_Node Nleft, n;

	Nleft = TreeDAG(t->in.left, inCond);

	/* Check here if op, Nleft is in the DAG already */

	n = UnaryHashLook(t->in.op, t->in.type, Nleft, t->tn.rval);
	if (n == NULL) {	/* Not in the DAG, build a new one */
		n = CreateNode(t->in.op, t->in.type, inCond);
		n->u.in.left = Nleft; ++Nleft->in_degree;
		n->u.tn.rval = t->tn.rval;
		n->hash = n->order;
		EnterHash(n);
		if( n->op == UNARY MUL )
			n->indirect = WhatReferenced(Nleft);
	}
	n->sticky = Nleft->sticky && !WillHaveCarrier(Nleft);

	if( n->op == UNARY MUL )
		IFetchDependencies(n);

	return n;
}

static DAG_Node
ColonDAG(t, inCond)		/* Builds the tree for : */
	TreeNode t;
	DagCount inCond;
{
	DAG_Node left, right, n;
	DAG_Node fence;
	DagCount our_zone;		/* our new zone number */

	fence = LastNodeCreated;
	our_zone = ++CondZone;
	left = TreeDAG(t->in.left, our_zone);
	if (left->in_cond != our_zone)	/* place holder for zone */
		left = BuildUNOP(left, our_zone);
	KillConditional(fence, left);

	fence = LastNodeCreated;
	our_zone = ++CondZone;
	right = TreeDAG(t->in.right, our_zone);
	if (right->in_cond != our_zone)	/* need place holder */
		right = BuildUNOP(right, our_zone);
	KillConditional(fence, right);

	n = CreateNode(t->in.op, t->in.type, inCond);
	n->u.in.left = left; ++left->in_degree;
	n->u.in.right = right; ++right->in_degree;
	n->sticky = (left->sticky  && !WillHaveCarrier(left)) ||
		    (right->sticky && !WillHaveCarrier(right));
	n->hash = n->order;
	return n;
}

static DAG_Node
ConditionalDAG(t, inCond, cascading, returned_fence)
	TreeNode t;
	DagCount inCond;
	Boolean cascading;
	DAG_Node *returned_fence;		/* passed by ref */
{
	DAG_Node left, right, n;
	Operator left_op, right_op;
	DAG_Node fence;		/* to start cleanup */
	DAG_Node tmp_fence;
	DagCount our_zone;	/* zone number for this zone */

	fence = NULL;
	left_op = t->in.left->in.op;
	right_op = t->in.right->in.op;
	if (CascadeOp(left_op) && left_op == t->in.op)
		left = ConditionalDAG(t->in.left, inCond, Cascade, &fence);
	else
		left = TreeDAG(t->in.left, inCond);

	if (fence == NULL)		/* we are building the fence */
		fence = LastNodeCreated;

	if (right_op == COLON) {
		assert(t->in.op = QUEST);
		right = ColonDAG(t->in.right, inCond);
		fence = LastNodeCreated;
	} else
	if (CascadeOp(right_op) && right_op == t->in.op) {
		our_zone = ++CondZone;
		right = ConditionalDAG(t->in.right, our_zone, Cascade, &tmp_fence);
		if (fence == NULL)
			fence = tmp_fence;

		if (right->in_cond != our_zone)
			right = BuildUNOP(right, our_zone);
	} else {
		our_zone = ++CondZone;
		right = TreeDAG(t->in.right, our_zone);
		if (right->in_cond != our_zone)
			right = BuildUNOP(right, our_zone);
	}

	n = CreateNode(t->in.op, t->in.type, inCond);
	n->u.in.left = left; ++left->in_degree;
	n->u.in.right = right; ++right->in_degree;
	n->sticky = (left->sticky  && !WillHaveCarrier(left)) ||
		    (right->sticky && !WillHaveCarrier(right));
	n->hash = n->order;
	if (!cascading) {
		KillConditional(fence, n);	/* clean up for all */
		fence = NULL;
	}
	*returned_fence = fence;

	return n;
}

static DAG_Node
CommaDAG(t, inCond)		/* t is a binary tree with comma at root */
	TreeNode t;
	DagCount inCond;
{
	/*
	 * Comma operators are ditched.  The optimizer will re-create
	 * those that are necessary.
	 */

	(void) TreeDAG(t->in.left, inCond);	/* for side effects */
	return TreeDAG(t->in.right, inCond);	/* the value */
}


static DAG_Node	
BinaryDAG(t, inCond)			/* t is a binary tree */
	TreeNode t;
	DagCount inCond;
{
	DAG_Node Nleft, Nright, n;
	Operator op;
	DAG_Node fence;

	op = t->in.op;
	if (op == ANDAND || op == OROR || op == QUEST) {
		n = ConditionalDAG(t, inCond, NoCascade, &fence);
		assert(fence == NULL);
	} else
	if (op == COMOP)
		n = CommaDAG(t, inCond);
	else {				/* ordinary binary op */
		Nleft = TreeDAG(t->in.left, inCond);
		Nright = TreeDAG(t->in.right, inCond);

		/* Check here if op, Nleft, Nright is in the DAG already */

		n = BinaryHashLook(op, t->in.type, Nleft, Nright);

		/* For commutative ops, try commuting */
		if (n == NULL && (dope[op]&(ASGFLG|COMMFLG))==COMMFLG)
			n = BinaryHashLook(op, t->in.type, Nright, Nleft);

		if (n == NULL) {	/* Not in the DAG, build a new one */
			n = CreateNode(op, t->in.type, inCond);
			n->u.in.left = Nleft;    ++Nleft->in_degree;
			n->u.in.right = Nright;  ++Nright->in_degree;
			EnterHash(n);
			n->hash = n->order;
		}
		n->sticky = (Nleft->sticky  && !WillHaveCarrier(Nleft)) ||
			    (Nright->sticky && !WillHaveCarrier(Nright));
	}
	return n;
}

static DAG_Node
BuildUNOP(n, inCond)		/* Build a UNARY NOP that points at n */
	DAG_Node n;
	DagCount inCond;
{
	DAG_Node nop;

	nop = CreateNode(UNARYNOP, n->type, inCond);
	nop->u.in.left = n; ++n->in_degree;
	nop->u.tn.rval = 0;		/* not used */
	nop->sticky = n->sticky && !WillHaveCarrier(n);
	nop->hash = nop->order;
	return nop;
}

/*	This handles operators with side effects
 */

static DAG_Node
MakeSideEffectDag(t, inCond)
	TreeNode t;
	DagCount inCond;
{
	DAG_Node n;

	assert(SideEffectOp(t->in.op));
	t->in.op = OpMap(t->in.op);
	n = TreeDAG(t, inCond);

	/* The type on the ASGOP might not be the type in which the
	 * operation is to be performed.  If it is not, fix it up
	 * This is crude; there should be a better (machine dependent)
	 * way to check this
	 */

	switch( n->type )
	{
	case CHAR:
	case SHORT:
		n->type = INT;
		break;
	case UCHAR:
	case USHORT:
		n->type = UNSIGNED;
		break;
#ifndef HAS_SINGLE_ARITHMETIC
	case FLOAT:
		n->type = DOUBLE;
		break;
#endif
	}
	return n;
	
}

/*
 * Create a DAG node with no hashing
 * Paint over node op with that provided
 */

static DAG_Node
NoHashDag(op,type,lhs,rhs,inCond)
	Operator op;
	TWORD type;
	DAG_Node lhs,rhs;
	DagCount inCond;
{
	DAG_Node n;

	n = CreateNode(op, type, inCond);
	n->u.in.left = lhs;
	n->u.in.right = rhs;
	n->hash = n->order;
	if( lhs != NULL )
	{
		++lhs->in_degree;
		n->sticky = lhs->sticky && !WillHaveCarrier(lhs);
	}
	if( rhs != NULL )
	{
		++rhs->in_degree;
		n->sticky = n->sticky ||(rhs->sticky && !WillHaveCarrier(rhs));
	}

	return n;
}

/*	Debug
 */

static int
node_number(n)
	DAG_Node n;
{
	return (n == NULL ? 0 : n->order);
}

#define flagc(b)	((b) ? 'Y' : 'N')

static void
PrintNode(n)
	DAG_Node n;
{
	AttachedID a;
	char * tp;		/* text pointer */
	TreeNode t;

	printf("DAG # %d) %s ", n->order, opst[n->op]);
	tprint(n->type);
	printf(" leafid %d", n->leaf_id);
	if (n->leaf_id != NoId) {
		switch(n->op) {
			case ICON:
			case LABCON:
				t = IdTree(n->leaf_id);
				if (t != NULL)
					printf(" (%ld)", t->tn.lval);
				break;
			default:
				tp = IdText(n->leaf_id);
				if (tp != NULL)
					printf(" (%s)", tp);
				break;
		}
	} else
	if (n->op == FCON)
		printf(" (%f)", n->u.fpn.dval);
		
	printf("\n\t");
	printf(" addr %x count %d activity %d tree %x e_tree %x",
		n,n->count,n->activity,
		n->new_tree, n->e_tree);

	printf("\n\t");
	if( optype(n->op) == LTYPE )
		printf(" left ***");
	else
		printf(" left %d (0x%x)", node_number(n->u.in.left),
				 n->u.in.left);
	if( optype(n->op) != BITYPE )
		printf(" right *** ");
	else
		printf(" right %d (0x%x) ", node_number(n->u.in.right),
				n->u.in.right);

	printf("sticky %c ", flagc(n->sticky));
	printf("indegree %d refs %d cost %d carrier % d",
		n->in_degree,n->refs,n->cost,n->carrier);

	printf("\n\t");
	printf(" line %d file %s", n->line_number,n->file_name);
	printf(" is_fetch %c indirect %d", flagc(n->is_fetch),
		n->indirect);
	printf(" p_v_carrier %c", flagc(n->prev_valid_carrier));
	printf(" carrier reqd %c", flagc(n->carrier_required));
	printf("\n\t");
	printf(" hash %d AE %d", n->hash, n->AEIndex);

	printf(" invariant %c", flagc(n->invariant));
	printf(" in_cond %d", n->in_cond);
	printf("\n\t");
	printf(" delay count %d delayed %d special_delay %c",
		n->delay_count, node_number(n->delayed),
		flagc(n->special_delay));
	printf("\n\t");
	printf(" evaluated %c", flagc(n->evaluated));
	printf("\n\t");
	printf(" attached: ");
	for (a=n->attached; a != NULL; a=a->next) {
		printf(" (%d", a->id);
		if (a->id != NoId) {	/* ultra safe for debugging */
			tp = IdText(a->id);
			if (tp != NULL)
				printf(" (%s)", tp);
		}
		printf(", leaf %d, killed %d prev %d sticky %c carrier %c) ",
		    node_number(a->leaf_ref),
		    node_number(a->killed_by),
		    node_number(a->prev_attach),
		    flagc(a->sticky), flagc(a->valid_carrier));
	}
	putchar('\n');
}

void
PrintGraph(n)
	DAG_Node n;
{

	printf("DAG in generation order\n");
	for (; n != NULL; n = n->next)
		PrintNode(n);
}

/*
 *	End of debug
 */

void
ClearVisit(ln)
	DAG_Node ln;
{

	for (; ln != NULL; ln = ln->next)
		ln->visited = False;
}

/*ARGSUSED*/
static void
KillNodes(t, n)			/* Kill the nodes required for t */
	TreeNode t;		/* ignored for now */
	DAG_Node n;		/* node causing the kill */
{
	Identifier id;
	DAG_Node tmp;
	AttachedID a;

	for (id = FirstId; id <= MaxIdentifier; ++id) {
		tmp = NodeID(id);
		if ( !IsTemp(id) &&
			(WasAddressed(id) || IdOp(id) == NAME ) )
		{
			if (tmp != NULL) {	/* record killer */
				a = tmp->attached;
				while (a != NULL && a->id != id) a = a->next;
				if( a != NULL )
					a->killed_by = n;
			}
			SetNodeID(id, (DAG_Node) NULL);
			SetLastKiller(id, n);
		}
	}
}

/*ARGSUSED*/
static void
KillConditional(fence, root)
	DAG_Node fence;
	DAG_Node root;			/* unused for now */
{
	DAG_Node n;
	AttachedID a;

	/*
	 * 	Unhash all nodes, and kill all attached ids because
	 *	we don't know if this code will be executed.
	 *	Note: "Unhashing" a leaf means we have to clear
	 *	its symbol table entry.
	 */

	for (n = fence->next; n != NULL; n = n->next) {
		n->hashable = False;

		if (optype(n->op) == LTYPE && n->leaf_id != NoId)
			SetNodeID(n->leaf_id, (DAG_Node) NULL);
		
		for (a = n->attached; a != NULL; a = a->next) {
			SetNodeID(a->id, (DAG_Node) NULL);
		}
	}
}

Boolean
WillHaveCarrier(n)
	register DAG_Node n;
{
	if (n->carrier != NoId)		/* It has a carrier ... */
/**/		return !IsTransparent(n->carrier);	/* will it keep it ? */
	else
		/* Will it necessarily get a carrier ? */
		return
			(!Literal(n->op) && !Is_Addr(n) && !IsAddrConst(n) &&
			(n->attached != NULL ||
			 (n->delayed != NULL && n->in_degree > 0 &&
				n->special_delay) ||
			 n->delay_count > 0));
}
