/* carrier.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */
/*
 *	Choose Carriers.
 *	This routine analyzes the DAG and chooses an appropriate carrier
 *	for the value of each node.  If no suitable carrier exists,
 *	a temporary will be created for the node.
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: carrier.c,v 5.5 89/05/12 12:49:31 pcc Rel-3_0 $";
/* static char ID[] = "@(#)carrier.c	15.2	of 86/10/23"; */
#endif

/*	Import
 */

# include <assert.h>
# include <activity.h>
# include <identifier.h>
# include <dag.h>
# include <blocks.h>
# include <delay.h>
# include <cost.h>
# include <temp.h>

extern int rdebug;		/* temporary */

/*
 *	Forward
 */

static void ChooseDag();
static void AssignCarrier();

/*	Export
 */

/*
 *	Private
 */

static DAG_Node current_node;	/* "tunnel" for AssignCarrier() */

static
ProcessBlocks(f, debug)
	void (*f)();
	char *debug;
{
	BasicBlock b;

	for (b=FirstBlock; b != NULL; b = b->next) {
	    if( b->reachable ) {
		if (rdebug) {
			printf("%s", debug);
			printf(" for block #%d\n\n",
				 b->blocknum);
		}
		(*f)(b->Dag);
		if (rdebug)
			PrintGraph(b->Dag);
	    }
	}
}

void
ChooseCarrier()
{
	ProcessBlocks(ChooseDag, "Initial carrier choice");
}

static void
HandleDelay(n)
	DAG_Node n;
{
	assert(n->delayed->activity > 0);
	--(n->delayed->activity);
	if (n->delayed->activity == 0)
		AssignCarrier(n->delayed);
}

static Boolean
ChooseNode(n)
	DAG_Node n;
{
	current_node = n;

	UpdateActivity(n, AssignCarrier);

	if (n->special_delay) {
		assert(n->delayed != NULL);
		HandleDelay(n);
	}
	return False;
}

static void
ChooseDag(d)		/* Do the job for one DAG */
	DAG_Node d;
{

	(void) ActivityDag(d, ChooseNode);
}

static void
AssignCarrier(n)		/* Choose a carrier for n */
	DAG_Node n;
{
	AttachedID a, a_next;
	Identifier n_id;
	int last_valid;		/* Note at which an id becomes invalid */

	/*
	 * Note: current_node is the node that is being worked on
	 */

	/*
	 *	Try to find an attached id suitable for a carrier
	 *	It must survive until n is dead.  We will prefer
	 *	a cheaper carrier.
	 */

	a = n->attached;
	n_id = (optype(n->op) == LTYPE ? n->leaf_id : NoId);
	while (a != NULL) {

		last_valid = a->killed_by == NULL ? 0 : a->killed_by->order;
		if ((last_valid == 0 || last_valid > current_node->order)) {
		    a->valid_carrier = True;
		    if (n->carrier == NoId ||
			IdRefCost(n->carrier) > IdRefCost(a->id))
				n->carrier = a->id;
		}

		a_next = a->next;		/* save next pointer */
		if (n_id != NoId && a->id == n_id)	/* avoid a = a */
			DeleteID(n_id, n, False);
		a = a_next;
	}

	/*
	 * If there is a temp attached to this node, and if it was
	 * not chosen as the carrier, get rid of it
	 */

	a = n->attached;
	if( a != NULL && n->carrier != NoId && a->id != n->carrier &&
	    IsTemp(a->id) )
	{
		DeleteID(a->id, n, True); 
	}

	/*
	 *	We must have a carrier if:
	 *	  1) There are any attached identifiers.  If there
	 *	     are, and we don't have one, they were all
	 *	     killed before the last reference.
	 *	  2) This node will be overwritten by a delayed store,
	 *	     and the value will be needed again, and it is
	 *	     special.
	 *	  3) Delayed stores exist that need the value of
	 *	     this node.
	 *	Of course, literals and address constants never require
	 *	carriers.
	 */

	if (!Literal(n->op) && !Is_Addr(n) && !IsAddrConst(n) &&
	     	(n->attached != NULL ||				/* 1 */
	      	(n->delayed != NULL && n->in_degree > 0 &&
				n->special_delay) ||		/* 2 */
	      	n->delay_count > 0))				/* 3 */
	{
		n->carrier_required = True;
		if (n->carrier == NoId) {	/* create one */
			a = MakeTemp(n, 'b', n->type);
			n->carrier = a->id;
		}
	}

	/* Handle ordinary delayed stores */

	if (n->delayed != NULL && !(n->special_delay))
		HandleDelay(n);
}

/*
 *	Second (and final) pass at assigning and allocating carriers
 */

static void
CheckCarriers(d)
	DAG_Node d;
{
	AttachedID tmp;		/* the attached temp */
	AttachedID cvc;		/* cheapest valid carrier (if any) */
	AttachedID a;
	DAG_Node n;
	CostType cvc_cost,
		 cost;

	for (n = d; n != NULL; n = n->next) {

		if (n->op == LEAFNOP)
/***/			continue;	/* Don't process these */

		if (asgop(n->op) || n->op == INCR || n->op == DECR)
			/*
			 *	This is somewhat conservative.  In some
			 *	cases we could re-examine our choice
			 *	of carrier, but we'll leave it alone
			 *	for now, since we know the code will
			 *	be correct.
			 */

/****/			continue;		/* Skip further processing */

		/*
		 *	Check for unused p.v.c temp
 		 */

		if (n->carrier != NoId && n->prev_valid_carrier && 
		    !HasAllocation(n->carrier))
		{
			assert(IsTransparent(n->carrier));
			if (rdebug)
				printf("Unallocated pvc %d removed from node %d\n",
					 n->carrier, n->order);
			n->carrier = NoId;
			n->prev_valid_carrier = False;
		}

		/*
		 *	Scan for tmp and cvc
	 	 */

		tmp = NULL; cvc = NULL;
		cvc_cost = COST_INFINITY;	/* starting value */
		for (a = n->attached; a != NULL; a = a->next) {

			 /*	Save the temp if this is it. */

			if (IsTemp(a->id)) {
				assert(tmp == NULL); /* two temps! */
				tmp = a;
			}

			/* Find the cheapest carrier. temps lose ties */

			if (a->valid_carrier) {
				cost = IdRefCost(a->id);
				if ((cost < cvc_cost) ||	/*a cheaper*/
				    (cost == cvc_cost && cvc == tmp))
				{
						cvc = a;
						cvc_cost = cost;
				}
				
			}
		}

		if (rdebug)
			printf("Node %d cvc %d tmp %d\n",
				n->order,
				cvc == NULL ? 0 : cvc->id,
				tmp == NULL ? 0 : tmp->id);

		/*
		 *	Now, choose an appropriate carrier for the
		 *	node.
		 */

		if (n->carrier != NoId) {	/* it already has one */
			if (n->prev_valid_carrier) {
				/* The carrier is okay.  Eventually:
				 * If the cvc is enough cheaper than
				 * the pvc, then turn this node into a
				 * leaf referencing the pvc, and set
				 * the leaf carrier to the cvc, and
				 * get the best of both.
				 */
			} else {
				/*
				 *	Is cvc better than current choice?
				 *	It is better if cvc is cheaper than
				 *	current carrier, or if they are the
				 *	same cost and the current carrier is
				 *	is the temp.
				 */

				cost = IdRefCost(n->carrier);
				if (cvc != NULL &&
				   ((cvc_cost < cost) ||
				    (tmp != NULL && cvc != tmp &&
					cvc_cost == cost &&
					n->carrier == tmp->id)))
				{
					if (rdebug)
						printf("cvc %d replaces %d\n",
							cvc->id, n->carrier);
					n->carrier = cvc->id;
				}
			}
		} else {	/* Currently, no carrier */
			if (n->carrier_required) {
				assert(cvc != NULL);  /* someday, make one */
				n->carrier = cvc->id;
				if (rdebug)
					printf("Carrier set: %d\n", cvc->id);
			}
		}

		/*
		 *	We have now made the final choice of carrier.
		 *	Ensure that it is allocated if necessary, and
		 *	that there are no dangling temps.
		 */

		if (tmp != NULL) {
			if (n->carrier == tmp->id) {
				if (!HasAllocation(n->carrier)) {
					Boolean allocated;

					allocated = AllocResource(n->carrier,StackResource);
					assert(allocated);
					PccAdvanceAllocation();
					if (rdebug)
						printf("Allocate %d\n",
							 n->carrier);
				}
			} else {
				if (rdebug)
					printf("%d deleted from node %d\n",
						tmp->id, n->order);
				DeleteID(tmp->id, n, True);
			}
		}
	}
}

void
FinalCarriers()
{
	if (rdebug)
		printf("Checking carriers\n");

	ProcessBlocks(CheckCarriers, "Final carrier choice");
}

