/*
 * 	(c) Copyright 1987 Gould Inc.
 * 	    All Rights Reserved.
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: opcode.c,v 5.2 88/08/30 01:10:45 pcc Released $";
#endif

/*
* Gould Base Register Assembler
* Initial design & coding by D. J. Kopetzky 
* @ Compion Corporation 
* Thu May  5 10:04:39 CDT 1983
* modified for release 1 by R. J. Miller 
* Tue May 24 20:45:56 CDT 1983
*/

#include <stdio.h>
#include "parse.h"
#include "struct.h"
#include "operand.h"
#include "optable.h"
#include "ophash.h"

#ifndef lint
#endif

#define MAX_OPERS 4

OPND oper[MAX_OPERS];
static int immed_count;

/*
NAME: op1
PURPOSE: process an opcode and its operands
PRECONDITIONS:
token is the opcode. yylinept should be between the opcode and the
operands on the input line.
POSTCONDITIONS:
yylinept should be pointing after the operands.
guesses at constant segment size during pass 1, but this sizing is
handled by emit routines in pass 2.
dot -> svalue is adjusted by var_size or fix_align as appropriate.
code generation routines are called only during pass 2.
HISTORY:
part of initial coding RJM
bug fix from rmiller put into switch to fix appropriate segment size
*/

op1(token)
Tuchar *token;
{
    register struct opcode *opptr;
    register int comma, opnum, ops_ok;
    int index;

#ifdef DEBUG
    if (dbflag)
    {
	printf ("op1 entry \"%s\"\n", token);
	fflush (stdout);
    }
#endif

    if ((index = uncollide (token)) == MAX_HASH)
    {
	error ("unknown opcode %s", token);
	/* following code is identical to that in yyerror in yylex.c */
	while (yylinept && (*yylinept != '\n' && *yylinept != ';'))
	{
	    yylinept++;
	}
	return;
    }
    else
    {
	opptr = &optable[index];
    }

#ifdef DEBUG
    if (dbflag)
	dump_op (opptr);
#endif

    if (opptr -> otype & O_PSEUDO)
    {
#ifdef DEBUG
	if (dbflag)
	{
	    printf ("op1: pseudo op %s\n", yylinept);
	    fflush (stdout);
	}
#endif
	(*opptr -> o2data) (opptr);	/* call the function */
    }
    else
    {
	ops_ok = 1;
	immed_count = 0;
	opnum = 0;
	comma = 0;

	if (opptr -> onopnd != 0)
	{				/* check if any operands are expected */
	    do
	    {
		comma = 0;
		ops_ok = ops_ok && validate_op (opptr, opnum);
		SKIPSPACE (yylinept);
		if (*yylinept == ',')
		{
		    comma = 1;
		    yylinept++;
		    SKIPSPACE (yylinept);
		}
		opnum++;
	    } while (opnum < MAX_OPERS && comma);
	}

/* if comma == 1, then too many operands (or bad syntax) */
	if (!ops_ok || comma)
	{
	    SKIPEOS (yylinept);
	    return;
	}

	/* make sure we trap in the variable operand count cases */
	if (opnum < MAX_OPERS)
	{
	    oper[opnum].type = OPERROR;
	}

/* in pass 2, generate code for the opcode */
	if (pass == 2)
	{
#ifdef DEBUG
	    if (dbflag)
	    {
		int i;
		for (i = 0; (i < MAX_OPERS) && (oper[i].type != OPERROR); ++i)
		    popnd (&oper[i]);
	    }
#endif
	    (*opptr -> o2data) (opptr);
	}

/* fix all the sizes */
	if (pass == 1)
	{			/* size fixing done elsewhere in pass 2 */
	    if (immed_count)
	    {
#ifdef DEBUG
		if (dbflag)
		{
#ifdef POOL
		    printf ("op1: imm const seg fix: olength %d\n",
				opptr -> olength);
		    printf ("  old sizes %d %d %d %d %d\n",
			seginf[(int) S_Lconstant].si_dot,
			seginf[(int) S_Dconstant].si_dot,
			seginf[(int) S_Wconstant].si_dot,
			seginf[(int) S_Xconstant].si_dot,
			seginf[(int) S_Hconstant].si_dot);
#else not POOL
		    printf ("op1: imm const seg fix: olength %d old size %d\n",
			opptr -> olength,
			seginf[(int) S_constant].si_dot);
#endif POOL
		    fflush (stdout);
		}
#endif
		/* fix appropriate segment size */
		switch (opptr -> olength)
		{
		case NONE:
		case BYTE:
		case HALF:
#ifdef POOL
		    if (oper[0].haveseen == CONST) {
			seginf[(int) S_Hconstant].si_dot += 2;
			break;
		    }
#endif POOL
		case WORD:
#ifdef POOL
		/* may be local address constant, absolute constant */
		/*  or external address constant */
		    seginf[(int) S_Lconstant].si_dot += 4;
		    seginf[(int) S_Wconstant].si_dot += 4;
		    seginf[(int) S_Xconstant].si_dot += 4;
#else not POOL
		    seginf[(int) S_constant].si_dot += 4;
#endif POOL
		    break;
		case LONG:
#ifdef POOL
		    seginf[(int) S_Dconstant].si_dot += 8;
#else not POOL
		    seginf[(int) S_constant].si_dot += 12;
#endif POOL
		    break;
		}
	    }

	    /* set segment size accounting for this op */
	    /* ops that use multi instruction implementations use O_VAR */
	    if (opptr -> otype & O_VAR)
	    {
		var_size (opptr);
	    }
	    else
	    {
		fix_align (opptr, 0, 1);
	    }
	}
    }
}

/*
NAME: validate_op
PURPOSE: assure that operand has a valid address mode
PRECONDITIONS:
opptr is a pointer into the opcode table for the current opcode. opnum
is the (zero based) index of the operand (first operand has opnum==0,
second has opnum==1, etc).
POSTCONDITIONS:
return 0 if the comparison between the o1data field of the opcode and
the value of result.type (as returned from yyparse) fails.
if this comparison succeeds, copy the result structure into the oper
array so that result may be reused by the parser, increment the counter
immed_count if the type is immediate or constant, and return 1.
immed_count flags presence of immediate operands for op1 so that
constant segment set up will be done.
HISTORY:
part of initial code RJM
*/

validate_op(opptr, opnum)
register struct opcode *opptr;
register int opnum;
{
    register int field_data;

    if (yyparse())		/* returns 1 on failure, 0 on success */
    {
	return(0);
    }
    else
    {
    /* o1data has four 8 bit fields, one for each of up to 4 operands */
    /* the following statement breaks out the field for this operand */
	field_data = (opptr -> o1data >> (opnum * 8)) & 0x000000FF;

#ifdef DEBUG
	if (dbflag)
	{
	    printf (
    "validate_op: op %d field_data 0x%x result %d - %d - %d - %d - %d\n%s\n",
		opnum, field_data, result.type, result.haveseen, result.regval,
		result.baseval, result.constval,
		result.symname ? result.symname -> sname : (Tuchar *) "NULL");
	    fflush (stdout);
	}
#endif

	/* check the type */
	switch (result.type)
	{
	case OPERROR:
	    error ("validate_op: bad operand type");
	    return(0);
	case REGISTER:
	    if (!((field_data & R) || (field_data & E)))
	    {
		error ("validate_op: invalid operand (REGISTER)");
		return(0);
	    }
	    if ((field_data & E) && (result.regval % 2))
	    {
		error ("validate_op: invalid operand (EVEN REG)");
		return(0);
	    }
	    break;
	case BASE:
	    if (!(field_data & B))
	    {
		error ("validate_op: invalid operand (BASE)");
		return(0);
	    }
	    break;
	case IMMEDIATE:
	    if (!(field_data & I))
	    {
		error ("validate_op: invalid operand (IMMEDIATE)");
		return(0);
	    }
	    immed_count++;
	    break;
	case MEMORY:
	    if (!(field_data & M))
	    {
		error ("validate_op: invalid operand (MEMORY)");
		return(0);
	    }
	    break;
	case CONSTANT:
	    if (!(field_data & I))	/* like immediate in pass 1 */
	    {
		error ("validate_op: invalid operand (CONSTANT)");
		return(0);
	    }
	    immed_count++;
	    break;
	case INDEXED:
	    if (!(field_data & M))	/* like memory in both passes */
	    {
		error ("validate_op: invalid operand (INDEXED)");
		return(0);
	    }
	    break;
	}
    /* do the following in each case IF SUCCESSFUL */
	copy_result (&oper[opnum], &result);
    }
    return (1);
}

/*
NAME: fix_align
PURPOSE: fix dot -> svalue for the non O_VAR cases
PRECONDITIONS:
called from op1 if var_size not necessary.
ptr is the pointer to the opcode table for the current opcode. operA and
operB are pointers to the operands that the opcode requires.
POSTCONDITIONS:
sets dot -> svalue appropriately.
ALGORITHM:
if opcode is a halfword instruction or if opcode has and is a register
to register halfword variant of an instruction, add 2 (bytes) to svalue.
otherwise, the opcode MUST be a fullword instruction, add 6 to svalue
(to catch the case of previous instruction being a halfword instruction
on a fullword boundary) and truncate to the next fullword boundary.
HISTORY:
part of original coding RJM DAW
*/

fix_align (ptr, operA, operB)
register struct opcode *ptr;
register int operA, operB;
{
    if ( (ptr -> otype & O_HALF)
      || 
	 (   (ptr -> otype & O_RRHALF)
	  && (oper[operA].type == REGISTER)
	  && (oper[operB].type == REGISTER)
	 ))
    {					/* instr is a halfword */
#ifdef DEBUG
	if (dbflag)
	{
	    printf ("fix_align: op is half word alignment: dot %d\n",
		    dot -> svalue);
	    fflush (stdout);
	}
#endif
	dot -> svalue += 2;
    }
    else	/* actually the "normal" case */
    {
	do_align(4);		/* round to next word bound */
	dot -> svalue += 4;	/* and advance by a word */
    }
}

#ifndef OLDCALL
typedef struct e_stk *E_STKP;
typedef struct e_lchn *E_LCHNP;

struct e_stk {
	E_STKP e_next;
	enum SEGS e_space;
	SYM *e_sym0;
	int e_off0;
	SYM *e_sym1;
	int e_off1;
	E_LCHNP e_lfirst, e_llast;
	short int e_retns, e_entrys;
#ifdef R_SV
	short int e_funcs;
#endif R_SV
	unsigned char e_in, e_var;
};

struct e_lchn {
	E_LCHNP e_lnext;
	SYM *e_lab;
	short int e_lretns, e_lentrys;
#ifdef R_SV
	short int e_lfuncs;
#endif R_SV
	unsigned char e_lin;
};

static E_STKP e_stks = 0, e_stkp = 0;

#endif NOT OLDCALL
/*
NAME: var_size
PURPOSE: handle opcode alignment in cases where different sized variants
are possible.
PRECONDITIONS:
this routine can be used ONLY during pass 1.
variants must be hard coded here, there is no other place where data
about them can be stored.
the opcode taken from the optable for some variants (notably cvtcw) may
not be the correct opcode to use.
POSTCONDITIONS:
for the opcode func, the symbol "csav" must be defined in the symbol
table. this routine is the register save routine for a function call.
HISTORY:
part of initial code RJM
*/

static int e_const = 0, e_stack, e_rmsk, e_r2flg, e_save;
 
var_size (ptr)
register struct opcode *ptr;
{
    register SYM *d;
#ifndef OLDCALL
    register E_STKP ep;
    char *malloc();
#endif NOT OLDCALL

    d = dot;
    switch (ptr -> ocode & 0xffff)
    {
    /* bit ops - have half word if O2 is R, full if O2 is M */
    case 0x9808:
    case 0x9C08:
    case 0xA408:
    case 0xA008:
	if (oper[1].type == REGISTER)	/* note zero indexing on oper */
	{
	    d -> svalue += 2;
	}
	else
	{
	    do_align(4);		/* full word alignment */
	    d -> svalue += 4;
	}
	break;

#ifdef OLDCALL
    /* func requires 2 words of text plus 1 word of constant if */
    /* second operand is not indexed or if it is just [r3] (in this */
    /* case, r3 is assumed to be already loaded), in all other */
    /* indexed cases it needs 3 words of text plus 1 constant */
#else NOT OLDCALL
    /* func requires 2 words of text for each live register or register */
    /* pair variable to be saved and restored; then 1 constant plus 5 words */
    /* of text, except in special case of indexed func using b1 as index */
    /* where only 4 words of text are required */ 
#endif OLDCALL
    case 0xF880:
#ifdef OLDCALL
	if (defsym ("csav") == 0)
	{
	    error ("csav not definable for opcode func");
	}
	do_align(4);
	d -> svalue += 8;
	/* indexed case EXCEPT just [r3] needs extra word */
	if ((oper[1].type == INDEXED) &&
	    !((oper[1].haveseen == AREG) && (oper[1].regval == 3)))
	{
	    d -> svalue += 4;
	}
#else NOT OLDCALL
#ifdef R_SV
	if (e_stkp->e_in != 1)
	    error("var_size:(caller-save mode) func before enter or after .ef");
	if (e_stkp->e_var) {
	/* tally # of func's for save/restore code fixup later*/
	    if (e_stkp)
		++e_stkp->e_funcs;
	}
	else if (e_save)		/* saving live regs across call */
		d ->svalue += 8 * e_save;
#endif R_SV

	do_align(4);
	d -> svalue += 16;	/* four words are always generated*/

	/* one more word of text is generated except for func #n,[b1] */
	if (!((oper[1].type == INDEXED) &&
		(oper[1].haveseen == BREG) && (oper[1].baseval == 1)))
	    d -> svalue += 4;

#endif OLDCALL
#ifdef POOL
	/* may be local address constant, absolute constant */
	/*  or external address constant */
	seginf[(int) S_Lconstant].si_dot += 4;
	seginf[(int) S_Wconstant].si_dot += 4;
	seginf[(int) S_Xconstant].si_dot += 4;
#else not POOL
	seginf[(int) S_constant].si_dot += 7;
	seginf[(int) S_constant].si_dot &= ~3;
#endif POOL
	break;

#ifndef OLDCALL
    case 0xFFFE:	/* enter - usually generates two words */
#ifdef E_SV
			/* plus a word for each register save instruction */
#endif E_SV
	if (!(ep = e_stkp) || ep->e_in != 0)
	    error("var_size: enter repeated or not beween .bf and .ef");
	ep->e_in = 1;
	ep->e_off0 = oper[0].constval;
	ep->e_off1 = oper[1].constval;
	do_align(4);
	if (oper[0].symname || oper[1].symname) {
	    ep->e_var = 1;
	    /* stack enter info for later fixup */
	    ep->e_sym0 = oper[0].symname;
	    ep->e_sym1 = oper[1].symname;
	    d -> svalue += 8;
	}
	else {
	    register int m;

	    e_stack = oper[0].constval;
	    e_rmsk = oper[1].constval;
	    e_r2flg = e_rmsk & 0x05;
	    e_rmsk &= ~0x07;
	    for (e_save = 0, m = 0xc0; m > 0x04; m >>= 2)
		if (m & e_rmsk)
		    ++e_save;
	    d -> svalue += 4 + (e_stack ? 4 : 0);
#ifdef E_SV
	    if (e_save)
		d -> svalue += 4 * e_save;
#endif E_SV
	}
	break;

    case 0xFFFD:	/* retn */
	if (!(ep = e_stkp) || ep->e_in != 1)
	    error("var_size: retn without prior enter or outside .ef ... .bf");
	/* retn usually generates four words; fixup later will adjust */
#ifdef E_SV
	/* plus a word for each register restore instruction */
#endif E_SV
	do_align(4);
	if (ep->e_var) {
	    ++ep->e_retns;
	    d -> svalue += 16;
	}
	else {
#ifdef E_SV
	    if (e_save)
		d -> svalue += 4 * e_save;
#endif E_SV
	    d -> svalue += 8 + (e_stack ? 4 : 0)
		+ (e_r2flg == 0x05 ? 0 : 4);
	}
	break;

    case 0xFFFC:
	if (!(ep = e_stkp) || ep->e_in != 1)
	    error("var_size: entry without prior enter or outside .ef ... .bf");
	/* entry usually generates three words */
#ifdef E_SV
	/* plus a word for each register save instruction */
#endif E_SV
	do_align(4);
	if (ep->e_var) {
	    ++ep->e_entrys;
	    d -> svalue += 12;
	}
	else {
	    d -> svalue += 8 + (e_stack ? 4 : 0);
#ifdef E_SV
	    if (e_save)
		d -> svalue += 4 * e_save;
#endif E_SV
	}
	break;

#endif NOT OLDCALL
    /* cvtcw uses two halfword shifts */
    /* alignment is difficult, but not in pass 1! */
    case 0xFFFF:
	d -> svalue += 4;
	break;

    /* mov has a 3 operand variant that uses 2 mov instructions */
    /* either of those 2 mov ops could be a O_RRHALF */
    case 0xAC08:
    case 0xAC00:
	mov_align (ptr, 0, 1);

	if (oper[2].type != OPERROR)	/* note zero indexing on oper */
	{
	    mov_align (ptr, 1, 2);
	}
	break;

    /* shifts have a 3 instruction O_REGREG (needs 2 full words of */
    /* constant space) and a 1 instruction (half word) O_MEMREG */
    case 0x1C00:
    case 0x1C20:
    case 0x1C40:
    case 0x1C60:
    case 0x2000:
    case 0x2020:
    case 0x2040:
    case 0x2060:
    case 0x2440:
    case 0x2400:
	if (oper[0].type == REGISTER)	/* note zero indexing on oper */
	{
	/* 12 bytes of instruction */
	    do_align(4);
	    d -> svalue += 12;
#ifdef POOL
	/* and 4 bytes of halfword constant space */
	    seginf[(int) S_Hconstant].si_dot += 4;
#else not POOL
	/* and 4 of constant space */
	    seginf[(int) S_constant].si_dot += 5;
	    seginf[(int) S_constant].si_dot &= ~1;
#endif POOL
	}
	else
	{
	    d -> svalue += 2;
	}
	break;
    
    /* jmp to reg is halfword, else fullword */
    case 0xEC00:
	if (oper[0].type == REGISTER)
	    d -> svalue += 2;
	else
	{
	    do_align(4);	/* align it */
	    d -> svalue += 4;
	}
	break;

    default:
	/* this error due to incorrect optable setup */
	error ("var_size: illegal opcode with O_VAR attribute %s",
	    ptr -> oname);
	break;
    }
}

#ifndef OLDCALL
e_begin() {
    E_STKP ep;

    /* make new enter stack for later fixup */
    if ((ep = (E_STKP) calloc(1, sizeof(struct e_stk))) == 0)
	error("e_begin: no space for enter stack");
    if (e_stkp)
	e_stkp = (e_stkp->e_next = ep);
    else
	e_stkp = e_stks = ep;
    ep->e_space = dot->sspace;
}

e_end() {
    e_stkp->e_in = 2;
}

e_lab_append(lab)
register SYM *lab;
{
    register E_STKP ep;
    register E_LCHNP lp;

    if ((ep = e_stkp) && lab->sspace == ep->e_space) {
	if ((lp = (E_LCHNP) malloc(sizeof(struct e_lchn))) == 0)
	    error("e_lab_append: no space");
	if (!ep->e_lfirst)
	    ep->e_llast = ep->e_lfirst = lp;
	else
	    ep->e_llast = (ep->e_llast->e_lnext = lp);
	lp->e_lnext = 0;
	lp->e_lab = lab;
#ifdef R_SV
	lp->e_lfuncs = ep->e_funcs;
#endif R_SV
	lp->e_lretns = ep->e_retns;
	lp->e_lentrys = ep->e_entrys;
	lp->e_lin = ep->e_in;
    }
}

e_fixup()
{
    int frame, rmask, rsave;
    int dot_off[last_SEGS];
    register E_STKP ep;
    register int m, incr;
    register E_LCHNP lp;

    for (m = 0; m < last_SEGS; ++m)
	dot_off[m] = 0;
    for (ep = e_stks; ep;) {
	frame = (ep->e_sym0 ? ep->e_sym0->svalue : 0) + ep->e_off0;
	rsave = 0;
	rmask = (ep->e_sym1 ? ep->e_sym1->svalue : 0) + ep->e_off1;
	if (rmask)
	    for (m = 0xc0; m; m = (m >> 2) & ~0x07)
		if (m & rmask)
#ifdef R_SV
		    rsave += 8;
#else E_SV
		    rsave += 4;
#endif R_SV
	for (lp = ep->e_lfirst; lp;) {
	    incr = 0;
/*	    if (lp->e_lin) {
		if (frame)
		    incr = 0;
		else 
		    incr = -4 * (1 + lp->e_lretns + lp->e_lentrys);
	    }
 */
	    if (lp->e_lin && ep->e_var) {
		if (!frame)
		/* one less word less one per prior retn and entry */
		    incr = -4 * (1 + lp->e_lretns + lp->e_lentrys);
#ifdef R_SV
		incr += rsave * lp->e_lfuncs;
#else E_SV
		incr += rsave * (1 + lp->e_lretns + lp->e_lentrys);
#endif R_SV
		if ((rmask & 0x05) == 0x05)
		    incr -= 4 * lp->e_lretns;
	    }
	    lp->e_lab->svalue += dot_off[ep->e_space] + incr;
	    m = (int) lp;
	    lp = lp->e_lnext;
	    free((char *) m);
	}
	if (ep->e_var) {
		if (frame)
		    incr = 0;
		else 	/* one less word less one per retn and entry */
		    incr = -4 * (1 + ep->e_retns + ep->e_entrys);
#ifdef R_SV
		incr += rsave * ep->e_funcs;
#else E_SV
		incr += rsave * (1 + ep->e_retns + ep->e_entrys);
#endif R_SV
		if ((rmask & 0x05) == 0x05)
		    incr -= 4 * ep->e_retns;
		seginf[(int) ep->e_space].si_dot += incr;
		dot_off[ep->e_space] += incr;
	}
	m = (int) ep;
	ep = ep->e_next;
	free((char *) m);
    }
    e_stks = e_stkp = 0;
}

#endif NOT OLDCALL
/*
NAME: mov_align
PURPOSE: handle alignment for the various and sundry move ops
PRECONDITIONS:
called (only) from var_size. this routine exists to simplify var_size.
ptr is the optable pointer for this opcode. this is needed primarily
to prevent using register to register alignment for any opcode
other than mov[wdl] which have reg to reg instruction variants,
and to force a movl #0,<even reg> to use two halfword ZR instructions.
opA and opB are indices into the oper array of operand structures.
normally they are 0 and 1 respectively for the typical 2 operand move
opcode. there is a 3 operand move which uses 1 and 2 as parameters.
POSTCONDITIONS:
sets the value of dot -> svalue appropriately for whatever move opcode
is being parsed.
HISTORY:
part of original coding DAW
*/

mov_align (ptr, opA, opB)
register struct opcode *ptr;
register int opA, opB;
{
    register SYM *d;

    d = dot;
    if (
	((ptr -> otype & O_RRHALF) &&
	((oper[opA].type == REGISTER &&
	    (oper[opB].type == REGISTER || oper[opB].type == BASE)) ||
	    (oper[opA].type == BASE && oper[opB].type == REGISTER)))
    					/* this move is halfword aligned */
	||
	((oper[opA].haveseen == CONST) &&
	    (oper[opA].constval == 0) &&
	    (oper[opB].type == REGISTER))
					/* this will be a ZR */
       )
    {
	d -> svalue += 3;
	d -> svalue &= ~1;
	if (ptr -> olength == LONG)	/* this needs another HW instr */
	    d -> svalue += 2;
    }
    else
    {					/* otherwise, it takes a word */
	do_align(4);
	d -> svalue += 4;
    }
}

/*
NAME: copy_result
PURPOSE: copy an OPND structure to a more permanent place.
PRECONDITIONS:
required because of the global declaration of the result OPND structure.
used by validate_op to set up the oper array for further processing.
note destination is first argument, source is second.
HISTORY:
part of initial code RJM
*/

copy_result (dest, source)
register OPND *dest;
register OPND *source;
{
    dest -> type = source -> type;
    dest -> haveseen = source -> haveseen;
    dest -> regval = source -> regval;
    dest -> baseval = source -> baseval;
    dest -> constval = source -> constval;
    dest -> symname = source -> symname;
}

/*
NAME: uncollide
PURPOSE: handles chaining in opcode table for oplook
PRECONDITIONS:
parameter name is (hopefully) an opcode mnemonic.
the oname field of an opcode array entry must be zeroed (by builder.c)
if the entry does not contain a defined opcode.
POSTCONDITIONS:
the index into the opcode array is returned on success, MAX_HASH is
returned on failure (MAX_HASH defined in ophash.h and is the dimension
of the opcode array).
HISTORY:
part of release 2 coding RJM
Sun Jul 31 20:13:53 EDT 1983
*/

uncollide (name)
register Tuchar *name;
{
    register int i;
    register struct opcode *opptr;

    opptr = &optable[i = oplook (name)];

    while (opptr -> oname &&
	strcmp (opptr -> oname, name))
    {
	i = (++i < MAX_HASH) ? i : 0;
	opptr = &optable[i];
    }

    if (opptr -> oname)
    {
	return (i);
    }
    else
    {
	return (MAX_HASH);
    }
}

/*
NAME: dump_op, popnd
PURPOSE: debug printing routines
PRECONDITIONS:
dump_op prints pieces of the optable entry, popnd prints information
returned from the operand parser (note also pr_sym call for symbol table
data).
HISTORY:
part of initial code RJM DAW
*/

dump_op (ptr)
register struct opcode *ptr;
{
    printf ("dump_op: %s - 0x%x - 0x%x\n", ptr -> oname, ptr -> otype,
	ptr -> o1data);
    fflush (stdout);
}

popnd (op)
register OPND *op;
{		/* debug - print an operand */
    extern char *Optypes[];

    printf ("  operand tp %s, seen 0x%x, r %d, b %d, const 0x%x",
	Optypes[op -> type], op -> haveseen, op -> regval,
	op -> baseval, op -> constval);
    if (op -> symname)
	pr_sym (op -> symname);
    else
	printf ("\n");
    fflush (stdout);
}

/*
 * 	(c) Copyright 1987 Gould Inc.
 * 	    All Rights Reserved.
 */
